// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"
#include "ivrconfig.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

RPCTransaction rpclog;

RPCTransaction::RPCTransaction() : MemPager(1024), ThreadLock()
{
	unsigned i;

	count = 0;
	flist = NULL;
	for(i = 0; i < KEYWORD_INDEX_SIZE; ++i)
		hash[i] = NULL;
}

unsigned RPCTransaction::getIndex(const char *id)
{
	unsigned key = 0;

	while(*id)
		key ^= (key << 1) ^ (*(id++) & 0x1f);

	return key % KEYWORD_INDEX_SIZE;
}

void RPCTransaction::expire(void)
{
	time_t now;
	unsigned i;
	
	tlog *prev, *node;

	time(&now);
	for(i = 0; i < KEYWORD_INDEX_SIZE; ++i)
	{
		prev = NULL;
		writeLock();
		node = hash[i];
		while(node)
		{
			if(now < node->expires)
			{
				prev = node;
				node = node->next;
				continue;
			}
			--count;
			slog(Slog::levelDebug) << "rpclog: expire " << node->id << endl;
			if(prev)
				prev->next = node->next;
			else
				hash[i] = node->next;
			node = node->next;
			node->next = flist;
			flist = node;	
		}
		if(!count)
		{
			purge();
			flist = NULL;
		}
		unlock();
		Thread::yield();
	}
}		

void RPCTransaction::stop(const char *id)
{
	tlog *node;

	readLock();
	node = hash[getIndex(id)];
	while(node)
	{
		if(!stricmp(id, node->id))
			break;
		node = node->next;
	}
	if(node)
	{
		slog(Slog::levelDebug) << "rpclog: stop " << id << endl;
		node->running = false;
	}
	unlock();
}

bool RPCTransaction::isRunning(const char *id)
{
	tlog *node;
	bool rtn = false;

	readLock();
	node = hash[getIndex(id)];
	while(node)
	{
		if(!stricmp(id, node->id))
			break;
		node = node->next;
	}
	if(node)
		rtn = node->running;
	unlock();
	return rtn;
}

void RPCTransaction::setInfo(const char *id, const char *value)
{
        tlog *node;

        readLock();
        node = hash[getIndex(id)];
        while(node)
        {
                if(!stricmp(id, node->id))
                        break;
                node = node->next;
        }
        if(node)
		snprintf(node->info, 32, "%s", value);

        unlock();
        return;
}

char *RPCTransaction::getInfo(const char *id, char *buffer)
{
	tlog *node;
	char *rtn = NULL;

	readLock();
	node = hash[getIndex(id)];
	while(node)
	{
		if(!stricmp(id, node->id))
			break;
		node = node->next;
	}
	if(node)
	{
		strcpy(buffer, node->info);
		rtn = buffer;
	}
	unlock();
	return rtn;
}

const char *RPCTransaction::start(const char *pid, char **argv, time_t timeout, const char *login)
{
	TrunkEvent event;
	Trunk *trunk = driver->getTrunkId(pid);
	const char *gid;
	tlog *node;
	unsigned key;
	Script::Symbol *sym;
	char buf[12];

	if(!trunk)
		return NULL;

	event.id = TRUNK_START_SCRIPT;
	event.parm.argv = argv;
	trunk->enterMutex();
	if(!trunk->postEvent(&event))
	{
		trunk->leaveMutex();
		return NULL;
	}

	writeLock();
	node = flist;
	if(node)
		flist = node->next;
	else
		node = (tlog *)alloc(sizeof(node));
	unlock();

	gid = trunk->getSymbol(SYM_GID);
	if(!gid)
		gid = "error-error";

	gid = strchr(gid, '-');
	if(gid)
		++gid;

	strcpy(node->id, gid);
	gid = node->id;

	sym = trunk->getEntry("rpc.status", 32);
	if(sym)
	{
		sym->flags.initial = false;
		sym->flags.commit = true;
		sym->data[0] = 0;
	}

	trunk->setConst("rpc.loginid", login);
	sym = trunk->getEntry(SYM_LOGIN, 0);
	if(sym)
		snprintf(sym->data, sym->flags.size + 1, "%s", login);

	snprintf(buf, sizeof(buf) -1, "%ld", (long)timeout);
	strcat(buf, "s");
	trunk->setConst("rpc.expires", buf);

	trunk->leaveMutex();

	++count;
	node->info[0] = 0;
	node->running = true;
	time(&node->expires);
	node->expires += timeout;

	key = getIndex(gid);

	writeLock();
	node->next = hash[key];
	hash[key] = node->next;
	unlock();
	slog(Slog::levelDebug) << "rpclog: adding " << gid << endl;
	return gid;
}

#ifdef	CCXX_NAMESPACES
};
#endif
