// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include "server.h"

namespace server {
using namespace ost;
using namespace std;

const char *Checks::chkDial(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "dial uses no members";

	if(!useKeywords(line, "=timeout"))
		return "unknown keyword for dial";

	if(!getOption(line, &idx))
		return "dialed number argument missing";

	return NULL;
}

const char *Checks::chkChildSignal(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "signal uses no members";

	if(hasKeywords(line))
		return "signal has no keywords";

	if(getOption(line, &idx))
		return "signal has no arguments";

	return NULL;
}

const char *Checks::chkJoin(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "join no members used";

	if(!useKeywords(line, "=timeout"))
		return "invalid keyword for join";

	if(!getOption(line, &idx))
		return "no join session id requested";

	return NULL;
}

const char *Checks::chkWait(Line *line, ScriptImage *img)
{
        unsigned idx = 0;

        if(getMember(line))
                return "wait no members used";

        if(!useKeywords(line, "=timeout=id"))
                return "invalid keyword for wait";

        return NULL;
}

const char *Checks::chkCancel(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "cancel has no members";
	
	if(hasKeywords(line))
		return "cancel has no keywords";

	if(!getOption(line, &idx))
		return "cancel requires at least one timeslot argument";

	return NULL;
}

const char *Checks::chkTransfer(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "transfer has no members";

	if(hasKeywords(line))
		return "transfer uses no keywords";

	if(!getOption(line, &idx))
		return "transfer destination required";

	return NULL;
}

const char *Checks::chkDialer(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "dialers have no members";

	if(!useKeywords(line, "=interdigit=level"))
		return "unknown keyword for dialers";

	if(!getOption(line, &idx))
		return "no dial string argument";

	return NULL;
}

const char *Checks::chkTonegen(Line *line, ScriptImage *img)
{
	const char *cp;

	if(getMember(line))
		return "tone has no members";

	if(!useKeywords(line, "=level=freq1=freq2=frequency=duration"))
		return "unknown keyword for tone";

	cp = findKeyword(line, "frequency");
	if(!cp)
		cp = findKeyword(line, "freq1");

	if(!cp)
		return "no tone frequency specified";

	return NULL;
}	

const char *Checks::chkTone(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "tone has no members";

	if(!useKeywords(line, "=location=timeout=level"))
		return "unknown keyword for a tone";

	if(getOption(line, &idx))
		return "no arguments used for tones";

	return NULL;
}

const char *Checks::chkConnect(Line *line, ScriptImage *img)
{
        unsigned idx = 0;

        if(!useKeywords(line, "=caller=display=vars=session=timeout=tone=duration"))
                return "invalid keyword used for dial";

        if(!getOption(line, &idx))
                return "no start script specified";

	if(!getOption(line, &idx))
		return "no number specified for dial";

        return NULL;
}
	
const char *Checks::chkStart(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(!useKeywords(line, "=vars=session=timeout=duration"))
		return "invalid keyword used for start";

	if(!getOption(line, &idx))
		return "no start script specified";

	if(getOption(line, &idx))
		return "only one script name used in start";

	return NULL;
}

const char *Checks::chkDir(Line *line, ScriptImage *img)
{
	unsigned count = 0;
	const char *cp;

	if(getMember(line))
		return "dir has no members";

	if(hasKeywords(line))
		return "dir has no keywords";

	if(!line->argc)
		return "dir requires at least one prefix";

	while(NULL != (cp = getOption(line, &count)))
	{
		if(!isalnum(*cp))
			return "invalid directory name used";

		if(strstr(cp, ".."))
			return "invalid directory name used";

		if(strstr(cp, "/."))
			return "invalid directory name used";

		if(strchr(cp, ':'))
			return "invalid directory name used";

		Dir::create(cp);		
	}
	return "";
}

const char *Checks::chkLoad(Line *line, ScriptImage *img)
{
	unsigned count = 0;
	const char *cp;

	if(getMember(line))
		return "load has no members";

	if(hasKeywords(line))
		return "load has no keywords";

	if(!line->argc)
		return "load requires at least one module";

	while(count < line->argc)
	{ 
		cp = line->args[count++];
		if(!isalpha(*cp))
			return "an invalid module name was used";
	}	

	count = 0;
	while(count < line->argc)
	{
		cp = line->args[count++];
		loadPlugin(keypaths.getLast("modules"), cp);
	}
	BayonneService::start();
	
	return "";
}	

const char *Checks::chkLibexec(Line *line, ScriptImage *img)
{
	const char *cp = getMember(line);
	unsigned idx = 0;

	if(cp && atoi(++cp) < 1)
		return "libexec timeout member must be at least one second";

	cp = getOption(line, &idx);
	if(!cp)
		return "libexec requires command argument";

	if(*cp == '%' || *cp == '&' || *cp == '#')
		return "libexec command must be constant";

	if(strchr(cp, '/') || strchr(cp, '\\'))
		return "libexec command cannot be path";

	while(NULL != (cp = getOption(line, &idx)))
	{
		if(*cp != '%' && *cp != '&' && *cp != '#')
			return "libexec arguments must be variables";
	}

	return NULL;
}

const char *Checks::chkVoicelib(Line *line, ScriptImage *img)
{
	if(getMember(line))
		return "voice library has no members";

	if(hasKeywords(line))
		return "voice library has no keywords";

	if(line->argc != 1)
		return "voice library argument must be specified";

	if(*line->args[0] == '%' || *line->args[0] == '#' || *line->args[0] == '&')
		return "voice library cannot be variable";

	return NULL;
}

const char *Checks::chkErase(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "erase has no members";

	if(!useKeywords(line, "=extension=prefix"))
		return "invalid keyword for erase";

	if(!getOption(line, &idx))
		return "erase requires file argument";

	if(getOption(line, &idx))
		return "erase uses only one file argument";
	
	return NULL;
}	

const char *Checks::chkInfo(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	const char *cp;

	if(getMember(line))
		return "erase has no members";

	if(!useKeywords(line, 
"=extension=prefix=date=time=note=size=type=coding=rate=bitrate=count"))
		return "invalid keyword for info";

	cp = findKeyword(line, "date");
	if(cp && *cp != '%' && *cp != '&')
		return "info date must be symbol target";

	cp = findKeyword(line, "time");
	if(cp && *cp != '%' && *cp != '&')
		return "info time must be symbol target";

	cp = findKeyword(line, "size");
	if(cp && *cp != '%' && *cp != '&')
		return "info size must be symbol target";

	cp = findKeyword(line, "coding");
	if(cp && *cp != '%' && *cp != '&')
		return "info coding must be symbol target";

	cp = findKeyword(line, "type");
        if(cp && *cp != '%' && *cp != '&')
                return "info type must be symbol target";

        cp = findKeyword(line, "rate");
        if(cp && *cp != '%' && *cp != '&')
                return "info rate must be symbol target";  

        cp = findKeyword(line, "bitrate");
        if(cp && *cp != '%' && *cp != '&')
                return "info bitrate must be symbol target";  

        cp = findKeyword(line, "count");
        if(cp && *cp != '%' && *cp != '&')
                return "info count must be symbol target";  

        cp = findKeyword(line, "note");
        if(cp && *cp != '%' && *cp != '&')
                return "info note must be symbol target";

	if(!getOption(line, &idx))
		return "info requires file argument";

	if(getOption(line, &idx))
		return "info uses only one file argument";
	
	return NULL;
}	

const char *Checks::chkMove(Line *line, ScriptImage *img)
{
	unsigned idx = 0;

	if(getMember(line))
		return "command has no members";

	if(!useKeywords(line, "=extension=prefix"))
		return "invalid keyword for this command";

	if(!getOption(line, &idx))
		return "source argument missing";

	if(!getOption(line, &idx))
		return "target argument missing";

	if(getOption(line, &idx))
		return "only two arguments used";
	
	return NULL;
}	

const char *Checks::chkBuild(Line *line, ScriptImage *img)
{
        unsigned idx = 0;

	if(getMember(line))
		return "member not used in build";

	if(!useKeywords(line, "=extension=encoding=framing=prefix=voice=note"))
		return "invalid keyword for build";


	if(!getOption(line, &idx))
		return "build destination missing";

	if(!getOption(line, &idx))
		return "build requires at least one file source";	

	return NULL;
}

const char *Checks::chkRecord(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	const char *cp;

	cp = getMember(line);
	if(cp && stricmp(cp, ".vox"))
		return "use no member or .vox only";

	if(!useKeywords(line, "=extension=encoding=prefix=framing=silence=intersilence=note=position=exit=timeout=menu"))
		return "invalid keyword used";

	if(!getOption(line, &idx))
		return "requires one file argument";

	if(getOption(line, &idx))
		return "uses only one file argument";

	return NULL;
}

const char *Checks::chkReplay(Line *line, ScriptImage *img)
{
        unsigned idx = 0;

        if(getMember(line))
		return "no member used in replay";

        if(!useKeywords(line, "=extension=encoding=prefix=framing=exit=menu=position"))
                return "invalid keyword used for replay";

        if(!getOption(line, &idx))
                return "replay requires one file argument";

        if(getOption(line, &idx))
                return "replay uses only one file argument";

        return NULL;
}

const char *Checks::chkAppend(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	const char *cp;

	cp = getMember(line);
	if(cp && stricmp(cp, ".vox"))
		return "use no member or .vox only";

	if(!useKeywords(line, "=extension=encoding=prefix=framing=silence=intersilence=exit=timeout"))
		return "invalid keyword used";

	if(!getOption(line, &idx))
		return "requires one file argument";

	if(getOption(line, &idx))
		return "uses only one file argument";

	return NULL;
}

const char *Checks::chkPathname(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	
	if(getMember(line))
		return "command has no members";

	if(!useKeywords(line, "=extension=voice=encoding=prefix"))
		return "invalid keyword used";

	if(!getOption(line, &idx))
		return "requires at least one prompt argument";

	return NULL;
}		

const char *Checks::chkCollect(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	const char *cp;

	if(!useKeywords(line, "=count=timeout=interdigit=ignore=exit=format"))
		return "invalid keyword for collect";

	cp = getOption(line, &idx);

	if(cp && *cp != '%' && *cp != '&')
		return "collect argument must be variable";

	return NULL;
}

const char *Checks::chkInput(Line *line, ScriptImage *img)
{
	unsigned idx = 0;
	const char *cp;

	if(getMember(line))
		return "input has no members";

	if(!useKeywords(line, "=count=timeout=interdigit=exit=format"))
		return "invalid keyword for input, use timeout, interdigit, count, format, and exit";

	cp = getOption(line, &idx);	
	if(!cp)
		return "input missing variable argument";

	if(*cp != '%' && *cp != '&')
		return "invalid argument for variable name";

	return NULL;
}

const char *Checks::chkKeyinput(Line *line, ScriptImage *img)
{
	const char *cp;
	unsigned idx = 0;

	if(getMember(line))
		return "inkey has no members";

	if(!useKeywords(line, "=timeout=menu"))
		return "invalid keyword for keyinput, use timeout and menu";

	cp = getOption(line, &idx);
	if(!cp)
		return NULL;

	if(*cp != '%' && *cp != '&')
		return "optional keyinput argument must be variable"; 

	return NULL;
}

const char *Checks::chkCleardigits(Line *line, ScriptImage *img)
{
	if(getMember(line))
		return "cleardigits does not use members";

	if(line->argc > 1)
		return "cleardigits uses one optional timeout";

	return NULL;
}

const char *Checks::chkRoute(Line *line, ScriptImage *img)
{
	if(hasKeywords(line))
		return "no keywords used for digits routing";

	if(!line->argc)
		return "route requires at least one route value";
	
	return NULL;
}

const char *Checks::chkSleep(Line *line, ScriptImage *img)
{
	if(getMember(line))
		return "member not used for this command";

	if(!line->argc || line->argc > 1)
		return "requires one timeout and uses no keywords";

	return NULL;
}

const char *Checks::chkRegister(Line *line, ScriptImage *img)
{
	const char *cp = getMember(line);
	Name *scr = img->getCurrent();
	BayonneDriver *driver;

	if(scr->access != ScriptInterp::scrPUBLIC)
		return "cannot register non-public script";

	if(cp)
		++cp;

	driver = BayonneDriver::get(cp);
		
	if(!driver)
	{
		slog.warn("cannot register %s with %s", scr->name, cp);
		return "";
	}
	return driver->registerScript(img, line);
}
		
const char *Checks::chkAssign(Line *line, ScriptImage *img)
{
	const char *cp = getMember(line);
	Name *scr = img->getCurrent();
	unsigned idx = 0, spid;
	char tag[65];
	BayonneDriver *driver;
	BayonneSession *s;

	if(hasKeywords(line))
		return "keywords not used for assign";

	if(!line->argc)
		return "missing entries to assign";

	if(scr->access != ScriptInterp::scrPUBLIC)
		return "cannot assign non-public script";	

	if(!stricmp(line->cmd, "incoming"))
		goto dnis;

	if(!stricmp(line->cmd, "incoming.timeslot"))
		goto timeslot;

	if(!stricmp(line->cmd, "incoming.span"))
		goto span;

	if(!strnicmp(line->cmd, "incoming.", 9))
		return "invalid member for incoming"; 

	if(cp && *cp == '.')
		++cp;

	if(cp)
	{
		if(!stricmp(cp, "timeslot"))
			goto timeslot;

		if(!stricmp(cp, "span"))
			goto span;

		if(!stricmp(cp, "dnis"))
			goto dnis;

		if(!stricmp(cp, "did"))
			goto dnis;

		driver = BayonneDriver::get(cp);
		if(!driver)
		{
			slog.warn("cannot assign %s with %s", scr->name, cp);
			return "";
		}
		return driver->assignScript(img, line);
	}

timeslot:
	while(NULL != (cp = getOption(line, &idx)))
	{
		if(!stricmp(cp, "default"))
		{
			slog.debug("assigning unassigned timeslots to %s", scr->name);
			img->setPointer("ts.default", scr);
			continue;
		}
		s = getSid(cp);
		if(s && s->getInterface() == IF_INET)
			s = NULL;
		if(!s)
		{
			slog.error("server cannot assign %s; invalid timeslot", cp);
			continue;
		}
		slog.debug("assigning timeslot %s to %s", cp, scr->name);
		snprintf(tag, sizeof(tag), "ts.%d", s->getSlot());
		img->setPointer(tag, scr);
	}
	return "";

span:
	while(NULL != (cp = getOption(line, &idx)))
	{
		spid = atoi(cp);
		slog.debug("assigning span %s to %s", cp, scr->name);
		snprintf(tag, sizeof(tag), "span.%d", spid);
		img->setPointer(tag, scr);
	}
	return "";
dnis:
        while(NULL != (cp = getOption(line, &idx)))
        {
		if(!stricmp(cp, "default"))
		{
			cp = "default";
			slog.debug("assigning inbound default %s", scr->name);
		}
		else
	                slog.debug("assigning dialed number %s to %s", cp, scr->name);
                snprintf(tag, sizeof(tag), "did.%s", cp);
                img->setPointer(tag, scr);
        }
        return "";
}	

const char *Checks::chkSelect(Line *line, ScriptImage *img)
{
	const char *cp = getMember(line);
	Name *scr = img->getCurrent();
	timeslot_t ts;
	unsigned idx = 0, spid;
	interface_t iface;
	BayonneSession *session;

	if(hasKeywords(line))
		return "keywords not used for select";

	if(!line->argc)
		return "missing entries to select";

	if(scr->access != ScriptInterp::scrPUBLIC)
		return "cannot select from non-public script";	

	if(cp && *cp == '.')
		++cp;

	if(cp)
	{
		if(!stricmp(cp, "timeslot"))
			goto timeslot;

		if(!stricmp(cp, "span"))
			goto span;

		return "unknown form of select";
	}

timeslot:
	while(NULL != (cp = getOption(line, &idx)))
	{
		ts = toTimeslot(cp);
		if(ts == NO_TIMESLOT)
		{
			slog.error("server cannot select %s; no such timeslot", cp);
			continue;
		}
		session = getSession(ts);
		if(!session)
		{
			slog.error("server cannot select %s; timeslot not allocated", cp);
			continue;
		}
		iface = session->getInterface();
		switch(iface)
		{
		case IF_INET:
			slog.error("server cannot select %s; non-physical port", cp);
			break;
		default:
			slog.debug("selecting timeslot %s for %s", cp, scr->name);
			img->addSelect(line);
		}
	}
	return "";

span:
	while(NULL != (cp = getOption(line, &idx)))
	{
		spid = atoi(cp);
		if(BayonneSpan::get(spid))
		{
			slog.debug("selecting span %s for %s", cp, scr->name);
			img->addSelect(line);
		}
		else
			slog.error("cannot select span %d; does not exist", spid);

	}
	return "";
}	


} // end namespace
