/*****************************************************************************\
 * Copyright 2005, 2006 Niels Lohmann, Christian Gierds, Dennis Reinert      *
 *                                                                           *
 * This file is part of BPEL2oWFN.                                           *
 *                                                                           *
 * BPEL2oWFN is free software; you can redistribute it and/or modify it      *
 * under the terms of the GNU General Public License as published by the     *
 * Free Software Foundation; either version 2 of the License, or (at your    *
 * option) any later version.                                                *
 *                                                                           *
 * BPEL2oWFN is distributed in the hope that it will be useful, but WITHOUT  *
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  *
 * more details.                                                             *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with BPEL2oWFN; if not, write to the Free Software Foundation, Inc., 51   *
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.                      *
\*****************************************************************************/

/**
 * \file cfg.h
 *
 * \brief Functions for the Control Flow Graph (interface)
 *
 * This file provides the necessary classes for building a Control Flow Graph
 * 
 * \author  
 *          - responsible: Christian Gierds <gierds@informatik.hu-berlin.de>
 *          - last changes of: \$Author: gierds $
 *          
 * \date
 *          - created: 2006-01-19
 *          - last changed: \$Date: 2006/01/26 15:15:34 $
 * 
 * \note    This file is part of the tool BPEL2oWFN and was created during the
 *          project "Tools4BPEL" at the Humboldt-Universitt zu Berlin. See
 *          http://www.informatik.hu-berlin.de/top/forschung/projekte/tools4bpel
 *          for details.
 *
 * \version \$Revision: 1.3 $
 */

#ifndef CFG_H
#define CFG_H

#include <string>
#include <iostream>
#include <list>
#include "exception.h"
#include "helpers.h"
#include "bpel-kc-k.h"		// generated by kimwitu++
#include "bpel-kc-rk.h"		// generated by kimwitu++
#include "bpel-kc-unpk.h"	// generated by kimwitu++

extern std::string filename;

// forward declarations
class CFGBlock;
class CFGProcess;
class CFGEmpty;
class CFGInvoke;
class CFGReceive;
class CFGReply;
class CFGAssign;
class CFGCopy;

/**
 * This class is a generic template for all blocks within
 * a CFG.
 * 
 */
class CFGBlock {
  public:
    /// pointer to the previous block (for backward analysis)	  
    CFGBlock * prevBlock;
    /// pointer to the next block (for forward analysis) 
    CFGBlock * nextBlock;

    /// label of the block
    std::string label;

    /// constructor
    CFGBlock();
    /// generic constructor
    CFGBlock(std::string id);
    /// generic constructor
    CFGBlock(kc::integer id);
    /// generic destructor
    virtual ~CFGBlock();
    
    /// returns the concrete type
    std::string getType(); 
    /// dot_output
    virtual void print_dot();
    virtual void dot_nextBlock();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();

    std::string type;
	  
};


/******************************************************************************
  PROCESS
******************************************************************************/

class CFGProcess : public CFGBlock {
  public:
    CFGProcess(std::string id);
    CFGProcess(kc::integer id);
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  EMPTY
******************************************************************************/

class CFGEmpty : public CFGBlock {
  public:
    CFGEmpty(std::string id);
    CFGEmpty(kc::integer id);

};


/******************************************************************************
  INVOKE
******************************************************************************/

class CFGInvoke : public CFGBlock {
  public:
    CFGInvoke(std::string id);
    CFGInvoke(kc::integer id);

};


/******************************************************************************
  RECEIVE
******************************************************************************/

class CFGReceive : public CFGBlock {
  public:
    CFGReceive(std::string id);
    CFGReceive(kc::integer id);

};


/******************************************************************************
  REPLY
******************************************************************************/

class CFGReply : public CFGBlock {
  public:
    CFGReply(std::string id);
    CFGReply(kc::integer id);

};


/******************************************************************************
  ASSIGN
******************************************************************************/

class CFGAssign : public CFGBlock {
  public:
    list<CFGCopy *> copyList;

    CFGAssign(std::string id);
    CFGAssign(kc::integer id);
    virtual ~CFGAssign();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};

class CFGCopy : public CFGBlock {
  public:
    CFGCopy(std::string id);
    CFGCopy(kc::integer id);
    
};


/******************************************************************************
  WAIT
******************************************************************************/

class CFGWait : public CFGBlock {
  public:
    CFGWait(std::string id);
    CFGWait(kc::integer id);

};


/******************************************************************************
  THROW
******************************************************************************/

class CFGThrow : public CFGBlock {
  public:
    CFGThrow(std::string id);
    CFGThrow(kc::integer id);

};


/******************************************************************************
  COMPENSATE
******************************************************************************/

class CFGCompensate : public CFGBlock {
  public:
    CFGCompensate(std::string id);
    CFGCompensate(kc::integer id);

};


/******************************************************************************
  TERMINATE
******************************************************************************/

class CFGTerminate : public CFGBlock {
  public:
    CFGTerminate(std::string id);
    CFGTerminate(kc::integer id);

};


/******************************************************************************
  FLOW
******************************************************************************/

class CFGFlow : public CFGBlock {
  public:
    list<CFGBlock *> activityList;

    CFGFlow(std::string id);
    CFGFlow(kc::integer id);
    virtual ~CFGFlow();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  SWITCH
******************************************************************************/

class CFGSwitch : public CFGBlock {
  public:
    list<CFGBlock *> caseList;
    CFGBlock * otherwise;

    CFGSwitch(std::string id);
    CFGSwitch(kc::integer id);
    virtual ~CFGSwitch();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  WHILE
******************************************************************************/

class CFGWhile : public CFGBlock {
  public:
    CFGBlock * loopActivity;

    CFGWhile(std::string id);
    CFGWhile(kc::integer id);
    virtual ~CFGWhile();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  SEQUENCE
******************************************************************************/

class CFGSequence : public CFGBlock {
  public:
    list<CFGBlock *> activityList;

    CFGSequence(std::string id);
    CFGSequence(kc::integer id);
    virtual ~CFGSequence();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  PICK
******************************************************************************/

class CFGPick : public CFGBlock {
  public:
    list<CFGBlock *> messageList;
    list<CFGBlock *> alarmList;

    CFGPick(std::string id);
    CFGPick(kc::integer id);
    virtual ~CFGPick();
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


/******************************************************************************
  SCOPE
******************************************************************************/

class CFGScope : public CFGBlock {
  public:
    CFGBlock * innerActivity;

    CFGScope(std::string id);
    CFGScope(kc::integer id);
    
    virtual void print_dot();
    /// returns the name of the first node in this block
    virtual std::string firstNodeName();
    /// returns the name of the last node in this block
    virtual std::string lastNodeName();
};


#endif

