/* Copyright (C) 1997-2017 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Andreas Jaeger <aj@suse.de>, 1997.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

/* Part of testsuite for libm.

   This file is processed by a perl script.  The resulting file has to
   be included by a master file that defines:

   Macros:
   FUNC(function): converts general function name (like cos) to
   name with correct suffix (e.g. cosl or cosf)
   FLOAT:	   floating point type to test
   - TEST_MSG:	   informal message to be displayed
   chooses one of the parameters as delta for testing
   equality
   PREFIX A macro which defines the prefix for common macros for the
   type (i.e LDBL, DBL, or FLT).
   LIT A function which appends the correct suffix to a literal.
   TYPE_STR A macro which defines a stringitized name of the type.
   FTOSTR This macro defines a function similar in type to strfromf
   which converts a FLOAT to a string.  */

/* This testsuite has currently tests for:
   acos, acosh, asin, asinh, atan, atan2, atanh,
   canonicalize, cbrt, ceil, copysign, cos, cosh, drem,
   erf, erfc, exp, exp10, exp2, expm1,
   fabs, fdim, finite, floor, fma, fmax, fmaxmag, fmin, fminmag,
   fmod, fpclassify, frexp, fromfp, fromfpx, gamma, getpayload, hypot,
   ilogb, iscanonical, isfinite, isinf, isnan, isnormal, issignaling,
   issubnormal, iszero, iseqsig, isless, islessequal, isgreater,
   isgreaterequal, islessgreater, isunordered, j0, j1, jn,
   ldexp, lgamma, llogb, log, log10, log1p, log2, logb,
   modf, nearbyint, nextafter, nexttoward,
   pow, pow10, remainder, remquo, rint, lrint, llrint,
   round, roundeven, lround, llround,
   scalb, scalbn, scalbln, setpayload, setpayloadsig, signbit,
   sin, sincos, sinh, sqrt,
   tan, tanh, tgamma, totalorder, totalordermag, trunc,
   ufromfp, ufromfpx,
   y0, y1, yn, significand

   and for the following complex math functions:
   cabs, cacos, cacosh, carg, casin, casinh, catan, catanh,
   ccos, ccosh, cexp, cimag, clog, clog10, conj, cpow, cproj, creal,
   csin, csinh, csqrt, ctan, ctanh.

   At the moment the following functions and macros aren't tested:
   lgamma_r,
   nan.

   Parameter handling is primitive in the moment:
   --verbose=[0..3] for different levels of output:
   0: only error count
   1: basic report on failed tests (default)
   2: full report on all tests
   -v for full output (equals --verbose=3)
   -u for generation of an ULPs file
 */

/* "Philosophy":

   This suite tests some aspects of the correct implementation of
   mathematical functions in libm.  Some simple, specific parameters
   are tested for correctness but there's no exhaustive
   testing.  Handling of specific inputs (e.g. infinity, not-a-number)
   is also tested.  Correct handling of exceptions is checked
   against.  These implemented tests should check all cases that are
   specified in ISO C99.

   NaN values: The payload of NaNs is set in inputs for functions
   where it is significant, and is examined in the outputs of some
   functions.

   Inline functions: Inlining functions should give an improvement in
   speed - but not in precission.  The inlined functions return
   reasonable values for a reasonable range of input values.  The
   result is not necessarily correct for all values and exceptions are
   not correctly raised in all cases.  Problematic input and return
   values are infinity, not-a-number and minus zero.  This suite
   therefore does not check these specific inputs and the exception
   handling for inlined mathematical functions - just the "reasonable"
   values are checked.

   Beware: The tests might fail for any of the following reasons:
   - Tests are wrong
   - Functions are wrong
   - Floating Point Unit not working properly
   - Compiler has errors

   With e.g. gcc 2.7.2.2 the test for cexp fails because of a compiler error.


   To Do: All parameter should be numbers that can be represented as
   exact floating point values.  Currently some values cannot be
   represented exactly and therefore the result is not the expected
   result.  For this we will use 36 digits so that numbers can be
   represented exactly.  */

#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif

#include <complex.h>
#include <math.h>
#include <float.h>
#include <fenv.h>
#include <limits.h>

#include <errno.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <argp.h>
#include <tininess.h>
#include <math-tests.h>
#include <math-tests-arch.h>
#include <nan-high-order-bit.h>

/* This header defines func_ulps, func_real_ulps and func_imag_ulps
   arrays.  */
#include "libm-test-ulps.h"

/* Allow platforms without all rounding modes to test properly,
   assuming they provide an __FE_UNDEFINED in <bits/fenv.h> which
   causes fesetround() to return failure.  */
#ifndef FE_TONEAREST
# define FE_TONEAREST	__FE_UNDEFINED
#endif
#ifndef FE_TOWARDZERO
# define FE_TOWARDZERO	__FE_UNDEFINED
#endif
#ifndef FE_UPWARD
# define FE_UPWARD	__FE_UNDEFINED
#endif
#ifndef FE_DOWNWARD
# define FE_DOWNWARD	__FE_UNDEFINED
#endif

/* Possible exceptions */
#define NO_EXCEPTION			0x0
#define INVALID_EXCEPTION		0x1
#define DIVIDE_BY_ZERO_EXCEPTION	0x2
#define OVERFLOW_EXCEPTION		0x4
#define UNDERFLOW_EXCEPTION		0x8
#define INEXACT_EXCEPTION		0x10
/* The next flags signals that those exceptions are allowed but not required.   */
#define INVALID_EXCEPTION_OK		0x20
#define DIVIDE_BY_ZERO_EXCEPTION_OK	0x40
#define OVERFLOW_EXCEPTION_OK		0x80
#define UNDERFLOW_EXCEPTION_OK		0x100
/* For "inexact" exceptions, the default is allowed but not required
   unless INEXACT_EXCEPTION or NO_INEXACT_EXCEPTION is specified.  */
#define NO_INEXACT_EXCEPTION		0x200
#define EXCEPTIONS_OK INVALID_EXCEPTION_OK+DIVIDE_BY_ZERO_EXCEPTION_OK
/* Some special test flags, passed together with exceptions.  */
#define IGNORE_ZERO_INF_SIGN		0x400
#define TEST_NAN_SIGN			0x800
#define TEST_NAN_PAYLOAD		0x1000
#define NO_TEST_INLINE			0x2000
#define XFAIL_TEST			0x4000
/* Indicate errno settings required or disallowed.  */
#define ERRNO_UNCHANGED			0x8000
#define ERRNO_EDOM			0x10000
#define ERRNO_ERANGE			0x20000
/* Flags generated by gen-libm-test.pl, not entered here manually.  */
#define IGNORE_RESULT			0x40000
#define NON_FINITE			0x80000
#define TEST_SNAN			0x100000
#define NO_TEST_MATHVEC			0x200000

#define TEST_NAN_PAYLOAD_CANONICALIZE	(SNAN_TESTS_PRESERVE_PAYLOAD	\
					 ? TEST_NAN_PAYLOAD		\
					 : 0)

#define __CONCATX(a,b) __CONCAT(a,b)

#define TYPE_MIN __CONCATX (PREFIX, _MIN)
#define TYPE_TRUE_MIN __CONCATX (PREFIX, _TRUE_MIN)
#define TYPE_MAX __CONCATX (PREFIX, _MAX)
#define MIN_EXP __CONCATX (PREFIX, _MIN_EXP)
#define MAX_EXP __CONCATX (PREFIX, _MAX_EXP)
#define MANT_DIG __CONCATX (PREFIX, _MANT_DIG)

/* Maximum character buffer to store a stringitized FLOAT value.  */
#define FSTR_MAX (128)

#if TEST_INLINE
# define ULP_IDX __CONCATX (ULP_I_, PREFIX)
# define QTYPE_STR "i" TYPE_STR
#else
# define ULP_IDX __CONCATX (ULP_, PREFIX)
# define QTYPE_STR TYPE_STR
#endif

/* Format specific test macros.  */
#define TEST_COND_binary32 (MANT_DIG == 24	\
			    && MIN_EXP == -125	\
			    && MAX_EXP == 128)

#define TEST_COND_binary64 (MANT_DIG == 53	\
			    && MIN_EXP == -1021	\
			    && MAX_EXP == 1024)

#define TEST_COND_binary128 (MANT_DIG == 113		\
			     && MIN_EXP == -16381	\
			     && MAX_EXP == 16384)

#define TEST_COND_ibm128 (MANT_DIG == 106)

#define TEST_COND_intel96 (MANT_DIG == 64	\
			   && MIN_EXP == -16381	\
			   && MAX_EXP == 16384)

#define TEST_COND_m68k96 (MANT_DIG == 64	\
			  && MIN_EXP == -16382	\
			  && MAX_EXP == 16384)

/* The condition ibm128-libgcc is used instead of ibm128 to mark tests
   where in principle the glibc code is OK but the tests fail because
   of limitations of the libgcc support for that format (e.g. GCC bug
   59666, in non-default rounding modes).  */
#define TEST_COND_ibm128_libgcc TEST_COND_ibm128

/* Mark a test as expected to fail for ibm128-libgcc.  This is used
   via XFAIL_ROUNDING_IBM128_LIBGCC, which gen-libm-test.pl transforms
   appropriately for each rounding mode.  */
#define XFAIL_IBM128_LIBGCC (TEST_COND_ibm128_libgcc ? XFAIL_TEST : 0)

/* Number of bits in NaN payload.  */
#if TEST_COND_ibm128
# define PAYLOAD_DIG (DBL_MANT_DIG - 2)
#else
# define PAYLOAD_DIG (MANT_DIG - 2)
#endif

/* Values underflowing only for float.  */
#if TEST_COND_binary32
# define UNDERFLOW_EXCEPTION_FLOAT	UNDERFLOW_EXCEPTION
# define UNDERFLOW_EXCEPTION_OK_FLOAT	UNDERFLOW_EXCEPTION_OK
#else
# define UNDERFLOW_EXCEPTION_FLOAT	0
# define UNDERFLOW_EXCEPTION_OK_FLOAT	0
#endif

/* Values underflowing only for double or types with a larger least
   positive normal value.  */
#if TEST_COND_binary32 || TEST_COND_binary64 || TEST_COND_ibm128
# define UNDERFLOW_EXCEPTION_DOUBLE	UNDERFLOW_EXCEPTION
# define UNDERFLOW_EXCEPTION_OK_DOUBLE	UNDERFLOW_EXCEPTION_OK
#else
# define UNDERFLOW_EXCEPTION_DOUBLE	0
# define UNDERFLOW_EXCEPTION_OK_DOUBLE	0
#endif

/* Values underflowing only for IBM long double or types with a larger least
   positive normal value.  */
#if TEST_COND_binary32 || TEST_COND_ibm128
# define UNDERFLOW_EXCEPTION_LDOUBLE_IBM	UNDERFLOW_EXCEPTION
#else
# define UNDERFLOW_EXCEPTION_LDOUBLE_IBM	0
#endif

/* Values underflowing on architectures detecting tininess before
   rounding, but not on those detecting tininess after rounding.  */
#define UNDERFLOW_EXCEPTION_BEFORE_ROUNDING	(TININESS_AFTER_ROUNDING \
						 ? 0			\
						 : UNDERFLOW_EXCEPTION)

#if LONG_MAX == 0x7fffffff
# define TEST_COND_long32	1
# define TEST_COND_long64	0
#else
# define TEST_COND_long32	0
# define TEST_COND_long64	1
#endif
#define TEST_COND_before_rounding	(!TININESS_AFTER_ROUNDING)
#define TEST_COND_after_rounding	TININESS_AFTER_ROUNDING

/* Various constants derived from pi.  We must supply them precalculated for
   accuracy.  They are written as a series of postfix operations to keep
   them concise yet somewhat readable.  */

/* (pi * 3) / 4 */
#define lit_pi_3_m_4_d		LIT (2.356194490192344928846982537459627163)
/* pi * 3 / (4 * ln(10)) */
#define lit_pi_3_m_4_ln10_m_d	LIT (1.023282265381381010614337719073516828)
/* pi / (2 * ln(10)) */
#define lit_pi_2_ln10_m_d	LIT (0.682188176920920673742891812715677885)
/* pi / (4 * ln(10)) */
#define lit_pi_4_ln10_m_d	LIT (0.341094088460460336871445906357838943)
/* pi / ln(10) */
#define lit_pi_ln10_d		LIT (1.364376353841841347485783625431355770)
/* pi / 2 */
#define lit_pi_2_d		LITM (M_PI_2)
/* pi / 4 */
#define lit_pi_4_d		LITM (M_PI_4)
/* pi */
#define lit_pi			LITM (M_PI)

/* Other useful constants.  */

/* e */
#define lit_e			LITM (M_E)

#define ulps_file_name "ULPs"	/* Name of the ULPs file.  */
static FILE *ulps_file;		/* File to document difference.  */
static int output_ulps;		/* Should ulps printed?  */
static char *output_dir;	/* Directory where generated files will be written.  */

static int noErrors;	/* number of errors */
static int noTests;	/* number of tests (without testing exceptions) */
static int noExcTests;	/* number of tests for exception flags */
static int noErrnoTests;/* number of tests for errno values */

static int verbose;
static int output_max_error;	/* Should the maximal errors printed?  */
static int output_points;	/* Should the single function results printed?  */
static int ignore_max_ulp;	/* Should we ignore max_ulp?  */

#define plus_zero	LIT (0.0)
#define minus_zero	LIT (-0.0)
#define plus_infty	FUNC (__builtin_inf) ()
#define minus_infty	-(FUNC (__builtin_inf) ())
#define qnan_value_pl(S)	FUNC (__builtin_nan) (S)
#define qnan_value	qnan_value_pl ("")
#define snan_value_pl(S)	FUNC (__builtin_nans) (S)
#define snan_value	snan_value_pl ("")
#define max_value	TYPE_MAX
#define min_value	TYPE_MIN
#define min_subnorm_value TYPE_TRUE_MIN

/* For nexttoward tests.  */
#define snan_value_ld	__builtin_nansl ("")

static FLOAT max_error, real_max_error, imag_max_error;

static FLOAT prev_max_error, prev_real_max_error, prev_imag_max_error;

static FLOAT max_valid_error;

/* Sufficient numbers of digits to represent any floating-point value
   unambiguously (for any choice of the number of bits in the first
   hex digit, in the case of TYPE_HEX_DIG).  When used with printf
   formats where the precision counts only digits after the point, 1
   is subtracted from these values. */
#define TYPE_DECIMAL_DIG __CONCATX (PREFIX, _DECIMAL_DIG)
#define TYPE_HEX_DIG ((MANT_DIG + 6) / 4)

/* Converts VALUE (a floating-point number) to string and writes it to DEST.
   PRECISION specifies the number of fractional digits that should be printed.
   CONVERSION is the conversion specifier, such as in printf, e.g. 'f' or 'a'.
   The output is prepended with an empty space if VALUE is non-negative.  */
static void
fmt_ftostr (char *dest, size_t size, int precision, const char *conversion,
	    FLOAT value)
{
  char format[64];
  char *ptr_format;
  int ret;

  /* Generate the format string.  */
  ptr_format = stpcpy (format, "%.");
  ret = sprintf (ptr_format, "%d", precision);
  ptr_format += ret;
  ptr_format = stpcpy (ptr_format, conversion);

  /* Add a space to the beginning of the output string, if the floating-point
     number is non-negative.  This mimics the behavior of the space (' ') flag
     in snprintf, which is not available on strfrom.  */
  if (! signbit (value))
    {
      *dest = ' ';
      dest++;
      size--;
    }

  /* Call the float to string conversion function, e.g.: strfromd.  */
  FTOSTR (dest, size, format, value);
}

/* Compare KEY (a string, with the name of a function) with ULP (a
   pointer to a struct ulp_data structure), returning a value less
   than, equal to or greater than zero for use in bsearch.  */

static int
compare_ulp_data (const void *key, const void *ulp)
{
  const char *keystr = key;
  const struct ulp_data *ulpdat = ulp;
  return strcmp (keystr, ulpdat->name);
}

/* Return the ulps for NAME in array DATA with NMEMB elements, or 0 if
   no ulps listed.  */

static FLOAT
find_ulps (const char *name, const struct ulp_data *data, size_t nmemb)
{
  const struct ulp_data *entry = bsearch (name, data, nmemb, sizeof (*data),
					  compare_ulp_data);
  if (entry == NULL)
    return 0;
  else
    return entry->max_ulp[ULP_IDX];
}

static void
init_max_error (const char *name, int exact)
{
  max_error = 0;
  real_max_error = 0;
  imag_max_error = 0;
  prev_max_error = find_ulps (name, func_ulps,
			      sizeof (func_ulps) / sizeof (func_ulps[0]));
  prev_real_max_error = find_ulps (name, func_real_ulps,
				   (sizeof (func_real_ulps)
				    / sizeof (func_real_ulps[0])));
  prev_imag_max_error = find_ulps (name, func_imag_ulps,
				   (sizeof (func_imag_ulps)
				    / sizeof (func_imag_ulps[0])));
#if TEST_COND_ibm128
  /* The documented accuracy of IBM long double division is 3ulp (see
     libgcc/config/rs6000/ibm-ldouble-format), so do not require
     better accuracy for libm functions that are exactly defined for
     other formats.  */
  max_valid_error = exact ? 3 : 16;
#else
  max_valid_error = exact ? 0 : 9;
#endif
  prev_max_error = (prev_max_error <= max_valid_error
		    ? prev_max_error
		    : max_valid_error);
  prev_real_max_error = (prev_real_max_error <= max_valid_error
			 ? prev_real_max_error
			 : max_valid_error);
  prev_imag_max_error = (prev_imag_max_error <= max_valid_error
			 ? prev_imag_max_error
			 : max_valid_error);
  feclearexcept (FE_ALL_EXCEPT);
  errno = 0;
}

static void
set_max_error (FLOAT current, FLOAT *curr_max_error)
{
  if (current > *curr_max_error && current <= max_valid_error)
    *curr_max_error = current;
}


/* Print a FLOAT.  */
static void
print_float (FLOAT f)
{
  /* As printf doesn't differ between a sNaN and a qNaN, do this manually.  */
  if (issignaling (f))
    printf ("sNaN\n");
  else if (isnan (f))
    printf ("qNaN\n");
  else
    {
      char fstrn[FSTR_MAX], fstrx[FSTR_MAX];
      fmt_ftostr (fstrn, FSTR_MAX, TYPE_DECIMAL_DIG - 1, "e", f);
      fmt_ftostr (fstrx, FSTR_MAX, TYPE_HEX_DIG - 1, "a", f);
      printf ("%s  %s\n", fstrn, fstrx);
    }
}

/* Should the message print to screen?  This depends on the verbose flag,
   and the test status.  */
static int
print_screen (int ok)
{
  if (output_points
      && (verbose > 1
	  || (verbose == 1 && ok == 0)))
    return 1;
  return 0;
}


/* Should the message print to screen?  This depends on the verbose flag,
   and the test status.  */
static int
print_screen_max_error (int ok)
{
  if (output_max_error
      && (verbose > 1
	  || ((verbose == 1) && (ok == 0))))
    return 1;
  return 0;
}

/* Update statistic counters.  */
static void
update_stats (int ok)
{
  ++noTests;
  if (!ok)
    ++noErrors;
}

static void
print_function_ulps (const char *function_name, FLOAT ulp)
{
  if (output_ulps)
    {
      char ustrn[FSTR_MAX];
      FTOSTR (ustrn, FSTR_MAX, "%.0f", FUNC (ceil) (ulp));
      fprintf (ulps_file, "Function: \"%s\":\n", function_name);
      fprintf (ulps_file, QTYPE_STR ": %s\n", ustrn);
    }
}


static void
print_complex_function_ulps (const char *function_name, FLOAT real_ulp,
			     FLOAT imag_ulp)
{
  if (output_ulps)
    {
      char fstrn[FSTR_MAX];
      if (real_ulp != 0.0)
	{
	  FTOSTR (fstrn, FSTR_MAX, "%.0f", FUNC (ceil) (real_ulp));
	  fprintf (ulps_file, "Function: Real part of \"%s\":\n", function_name);
	  fprintf (ulps_file, QTYPE_STR ": %s\n", fstrn);
	}
      if (imag_ulp != 0.0)
	{
	  FTOSTR (fstrn, FSTR_MAX, "%.0f", FUNC (ceil) (imag_ulp));
	  fprintf (ulps_file, "Function: Imaginary part of \"%s\":\n", function_name);
	  fprintf (ulps_file, QTYPE_STR ": %s\n", fstrn);
	}


    }
}



/* Test if Floating-Point stack hasn't changed */
static void
fpstack_test (const char *test_name)
{
#if defined (__i386__) || defined (__x86_64__)
  static int old_stack;
  int sw;

  asm ("fnstsw" : "=a" (sw));
  sw >>= 11;
  sw &= 7;

  if (sw != old_stack)
    {
      printf ("FP-Stack wrong after test %s (%d, should be %d)\n",
	      test_name, sw, old_stack);
      ++noErrors;
      old_stack = sw;
    }
#endif
}


static void
print_max_error (const char *func_name)
{
  int ok = 0;

  if (max_error == 0.0 || (max_error <= prev_max_error && !ignore_max_ulp))
    {
      ok = 1;
    }

  if (!ok)
    print_function_ulps (func_name, max_error);


  if (print_screen_max_error (ok))
    {
      char mestr[FSTR_MAX], pmestr[FSTR_MAX];
      FTOSTR (mestr, FSTR_MAX, "%.0f", FUNC (ceil) (max_error));
      FTOSTR (pmestr, FSTR_MAX, "%.0f", FUNC (ceil) (prev_max_error));
      printf ("Maximal error of `%s'\n", func_name);
      printf (" is      : %s ulp\n", mestr);
      printf (" accepted: %s ulp\n", pmestr);
    }

  update_stats (ok);
}


static void
print_complex_max_error (const char *func_name)
{
  int real_ok = 0, imag_ok = 0, ok;

  if (real_max_error == 0
      || (real_max_error <= prev_real_max_error && !ignore_max_ulp))
    {
      real_ok = 1;
    }

  if (imag_max_error == 0
      || (imag_max_error <= prev_imag_max_error && !ignore_max_ulp))
    {
      imag_ok = 1;
    }

  ok = real_ok && imag_ok;

  if (!ok)
    print_complex_function_ulps (func_name,
				 real_ok ? 0 : real_max_error,
				 imag_ok ? 0 : imag_max_error);

  if (print_screen_max_error (ok))
    {
      char rmestr[FSTR_MAX], prmestr[FSTR_MAX];
      char imestr[FSTR_MAX], pimestr[FSTR_MAX];
      FTOSTR (rmestr, FSTR_MAX, "%.0f", FUNC (ceil) (real_max_error));
      FTOSTR (prmestr, FSTR_MAX, "%.0f", FUNC (ceil) (prev_real_max_error));
      FTOSTR (imestr, FSTR_MAX, "%.0f", FUNC (ceil) (imag_max_error));
      FTOSTR (pimestr, FSTR_MAX, "%.0f", FUNC (ceil) (prev_imag_max_error));
      printf ("Maximal error of real part of: %s\n", func_name);
      printf (" is      : %s ulp\n", rmestr);
      printf (" accepted: %s ulp\n", prmestr);
      printf ("Maximal error of imaginary part of: %s\n", func_name);
      printf (" is      : %s ulp\n", imestr);
      printf (" accepted: %s ulp\n", pimestr);
    }

  update_stats (ok);
}


#if FE_ALL_EXCEPT
/* Test whether a given exception was raised.  */
static void
test_single_exception (const char *test_name,
		       int exception,
		       int exc_flag,
		       int fe_flag,
		       const char *flag_name)
{
  int ok = 1;
  if (exception & exc_flag)
    {
      if (fetestexcept (fe_flag))
	{
	  if (print_screen (1))
	    printf ("Pass: %s: Exception \"%s\" set\n", test_name, flag_name);
	}
      else
	{
	  ok = 0;
	  if (print_screen (0))
	    printf ("Failure: %s: Exception \"%s\" not set\n",
		    test_name, flag_name);
	}
    }
  else
    {
      if (fetestexcept (fe_flag))
	{
	  ok = 0;
	  if (print_screen (0))
	    printf ("Failure: %s: Exception \"%s\" set\n",
		    test_name, flag_name);
	}
      else
	{
	  if (print_screen (1))
	    printf ("%s: Exception \"%s\" not set\n", test_name,
		    flag_name);
	}
    }
  if (!ok)
    ++noErrors;
}
#endif

/* Test whether exceptions given by EXCEPTION are raised.  Ignore thereby
   allowed but not required exceptions.
*/
static void
test_exceptions (const char *test_name, int exception)
{
  if (TEST_EXCEPTIONS && EXCEPTION_TESTS (FLOAT))
    {
      ++noExcTests;
#ifdef FE_DIVBYZERO
      if ((exception & DIVIDE_BY_ZERO_EXCEPTION_OK) == 0)
	test_single_exception (test_name, exception,
			       DIVIDE_BY_ZERO_EXCEPTION, FE_DIVBYZERO,
			       "Divide by zero");
#endif
#ifdef FE_INVALID
      if ((exception & INVALID_EXCEPTION_OK) == 0)
	test_single_exception (test_name, exception,
			       INVALID_EXCEPTION, FE_INVALID,
			       "Invalid operation");
#endif
#ifdef FE_OVERFLOW
      if ((exception & OVERFLOW_EXCEPTION_OK) == 0)
	test_single_exception (test_name, exception, OVERFLOW_EXCEPTION,
			       FE_OVERFLOW, "Overflow");
#endif
      /* Spurious "underflow" and "inexact" exceptions are always
	 allowed for IBM long double, in line with the underlying
	 arithmetic.  */
#ifdef FE_UNDERFLOW
      if ((exception & UNDERFLOW_EXCEPTION_OK) == 0
	  && !(TEST_COND_ibm128
	       && (exception & UNDERFLOW_EXCEPTION) == 0))
	test_single_exception (test_name, exception, UNDERFLOW_EXCEPTION,
			       FE_UNDERFLOW, "Underflow");
#endif
#ifdef FE_INEXACT
      if ((exception & (INEXACT_EXCEPTION | NO_INEXACT_EXCEPTION)) != 0
	  && !(TEST_COND_ibm128
	       && (exception & NO_INEXACT_EXCEPTION) != 0))
	test_single_exception (test_name, exception, INEXACT_EXCEPTION,
			       FE_INEXACT, "Inexact");
#endif
    }
  feclearexcept (FE_ALL_EXCEPT);
}

/* Test whether errno for TEST_NAME, set to ERRNO_VALUE, has value
   EXPECTED_VALUE (description EXPECTED_NAME).  */
static void
test_single_errno (const char *test_name, int errno_value,
		   int expected_value, const char *expected_name)
{
  if (errno_value == expected_value)
    {
      if (print_screen (1))
	printf ("Pass: %s: errno set to %d (%s)\n", test_name, errno_value,
		expected_name);
    }
  else
    {
      ++noErrors;
      if (print_screen (0))
	printf ("Failure: %s: errno set to %d, expected %d (%s)\n",
		test_name, errno_value, expected_value, expected_name);
    }
}

/* Test whether errno (value ERRNO_VALUE) has been for TEST_NAME set
   as required by EXCEPTIONS.  */
static void
test_errno (const char *test_name, int errno_value, int exceptions)
{
  if (TEST_ERRNO)
    {
      ++noErrnoTests;
      if (exceptions & ERRNO_UNCHANGED)
	test_single_errno (test_name, errno_value, 0, "unchanged");
      if (exceptions & ERRNO_EDOM)
	test_single_errno (test_name, errno_value, EDOM, "EDOM");
      if (exceptions & ERRNO_ERANGE)
	test_single_errno (test_name, errno_value, ERANGE, "ERANGE");
    }
}

/* Returns the number of ulps that GIVEN is away from EXPECTED.  */
#define ULPDIFF(given, expected) \
	(FUNC(fabs) ((given) - (expected)) / ulp (expected))

/* Returns the size of an ulp for VALUE.  */
static FLOAT
ulp (FLOAT value)
{
  FLOAT ulp;

  switch (fpclassify (value))
    {
      case FP_ZERO:
	/* We compute the distance to the next FP which is the same as the
	   value of the smallest subnormal number. Previously we used
	   2^-(MANT_DIG - 1) which is too large a value to be useful. Note that we
	   can't use ilogb(0), since that isn't a valid thing to do. As a point
	   of comparison Java's ulp returns the next normal value e.g.
	   2^(1 - MAX_EXP) for ulp(0), but that is not what we want for
	   glibc.  */
	/* Fall through...  */
      case FP_SUBNORMAL:
        /* The next closest subnormal value is a constant distance away.  */
	ulp = FUNC(ldexp) (1.0, MIN_EXP - MANT_DIG);
	break;

      case FP_NORMAL:
	ulp = FUNC(ldexp) (1.0, FUNC(ilogb) (value) - MANT_DIG + 1);
	break;

      default:
	/* It should never happen. */
	abort ();
	break;
    }
  return ulp;
}

static void
check_float_internal (const char *test_name, FLOAT computed, FLOAT expected,
		      int exceptions,
		      FLOAT *curr_max_error, FLOAT max_ulp)
{
  int ok = 0;
  int print_diff = 0;
  FLOAT diff = 0;
  FLOAT ulps = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  if (issignaling (computed) && issignaling (expected))
    {
      if ((exceptions & TEST_NAN_SIGN) != 0
	  && signbit (computed) != signbit (expected))
	{
	  ok = 0;
	  printf ("signaling NaN has wrong sign.\n");
	}
      else if ((exceptions & TEST_NAN_PAYLOAD) != 0
	       && (FUNC (getpayload) (&computed)
		   != FUNC (getpayload) (&expected)))
	{
	  ok = 0;
	  printf ("signaling NaN has wrong payload.\n");
	}
      else
	ok = 1;
    }
  else if (issignaling (computed) || issignaling (expected))
    ok = 0;
  else if (isnan (computed) && isnan (expected))
    {
      if ((exceptions & TEST_NAN_SIGN) != 0
	  && signbit (computed) != signbit (expected))
	{
	  ok = 0;
	  printf ("quiet NaN has wrong sign.\n");
	}
      else if ((exceptions & TEST_NAN_PAYLOAD) != 0
	       && (FUNC (getpayload) (&computed)
		   != FUNC (getpayload) (&expected)))
	{
	  ok = 0;
	  printf ("quiet NaN has wrong payload.\n");
	}
      else
	ok = 1;
    }
  else if (isinf (computed) && isinf (expected))
    {
      /* Test for sign of infinities.  */
      if ((exceptions & IGNORE_ZERO_INF_SIGN) == 0
	  && signbit (computed) != signbit (expected))
	{
	  ok = 0;
	  printf ("infinity has wrong sign.\n");
	}
      else
	ok = 1;
    }
  /* Don't calculate ULPs for infinities or any kind of NaNs.  */
  else if (isinf (computed) || isnan (computed)
	   || isinf (expected) || isnan (expected))
    ok = 0;
  else
    {
      diff = FUNC(fabs) (computed - expected);
      ulps = ULPDIFF (computed, expected);
      set_max_error (ulps, curr_max_error);
      print_diff = 1;
      if ((exceptions & IGNORE_ZERO_INF_SIGN) == 0
	  && computed == 0.0 && expected == 0.0
	  && signbit(computed) != signbit (expected))
	ok = 0;
      else if (ulps <= max_ulp && !ignore_max_ulp)
	ok = 1;
      else
	ok = 0;
    }
  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure: ");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         ");
      print_float (computed);
      printf (" should be:  ");
      print_float (expected);
      if (print_diff)
	{
	  char dstrn[FSTR_MAX], dstrx[FSTR_MAX];
	  char ustrn[FSTR_MAX], mustrn[FSTR_MAX];
	  fmt_ftostr (dstrn, FSTR_MAX, TYPE_DECIMAL_DIG - 1, "e", diff);
	  fmt_ftostr (dstrx, FSTR_MAX, TYPE_HEX_DIG - 1, "a", diff);
	  fmt_ftostr (ustrn, FSTR_MAX, 4, "f", ulps);
	  fmt_ftostr (mustrn, FSTR_MAX, 4, "f", max_ulp);
	  printf (" difference: %s  %s\n", dstrn, dstrx);
	  printf (" ulp       : %s\n", ustrn);
	  printf (" max.ulp   : %s\n", mustrn);
	}
    }
  update_stats (ok);

 out:
  fpstack_test (test_name);
  errno = 0;
}


static void
check_float (const char *test_name, FLOAT computed, FLOAT expected,
	     int exceptions)
{
  check_float_internal (test_name, computed, expected,
			exceptions, &max_error, prev_max_error);
}


static void
check_complex (const char *test_name, __complex__ FLOAT computed,
	       __complex__ FLOAT expected,
	       int exception)
{
  FLOAT part_comp, part_exp;
  char *str;

  if (asprintf (&str, "Real part of: %s", test_name) == -1)
    abort ();

  part_comp = __real__ computed;
  part_exp = __real__ expected;

  check_float_internal (str, part_comp, part_exp,
			exception, &real_max_error, prev_real_max_error);
  free (str);

  if (asprintf (&str, "Imaginary part of: %s", test_name) == -1)
    abort ();

  part_comp = __imag__ computed;
  part_exp = __imag__ expected;

  /* Don't check again for exceptions or errno, just pass through the
     other relevant flags.  */
  check_float_internal (str, part_comp, part_exp,
			exception & (IGNORE_ZERO_INF_SIGN
				     | TEST_NAN_SIGN
				     | IGNORE_RESULT),
			&imag_max_error, prev_imag_max_error);
  free (str);
}


/* Check that computed and expected values are equal (int values).  */
static void
check_int (const char *test_name, int computed, int expected,
	   int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if (computed == expected)
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure: ");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %d\n", computed);
      printf (" should be:  %d\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}


/* Check that computed and expected values are equal (long int values).  */
static void
check_long (const char *test_name, long int computed, long int expected,
	    int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if (computed == expected)
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure: ");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %ld\n", computed);
      printf (" should be:  %ld\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}


/* Check that computed value is true/false.  */
static void
check_bool (const char *test_name, int computed, int expected,
	    int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if ((computed == 0) == (expected == 0))
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure: ");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %d\n", computed);
      printf (" should be:  %d\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}


/* check that computed and expected values are equal (long int values) */
static void
check_longlong (const char *test_name, long long int computed,
		long long int expected,
		int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if (computed == expected)
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure:");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %lld\n", computed);
      printf (" should be:  %lld\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}


/* Check that computed and expected values are equal (intmax_t values).  */
static void
check_intmax_t (const char *test_name, intmax_t computed,
		intmax_t expected, int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if (computed == expected)
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure:");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %jd\n", computed);
      printf (" should be:  %jd\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}


/* Check that computed and expected values are equal (uintmax_t values).  */
static void
check_uintmax_t (const char *test_name, uintmax_t computed,
		 uintmax_t expected, int exceptions)
{
  int ok = 0;
  int errno_value = errno;

  test_exceptions (test_name, exceptions);
  test_errno (test_name, errno_value, exceptions);
  if (exceptions & IGNORE_RESULT)
    goto out;
  noTests++;
  if (computed == expected)
    ok = 1;

  if (print_screen (ok))
    {
      if (!ok)
	printf ("Failure:");
      printf ("Test: %s\n", test_name);
      printf ("Result:\n");
      printf (" is:         %ju\n", computed);
      printf (" should be:  %ju\n", expected);
    }

  update_stats (ok);
 out:
  fpstack_test (test_name);
  errno = 0;
}

/* Return whether a test with flags EXCEPTIONS should be run.  */
static int
enable_test (int exceptions)
{
  if (exceptions & XFAIL_TEST)
    return 0;
  if (TEST_INLINE && (exceptions & NO_TEST_INLINE))
    return 0;
  if (TEST_FINITE && (exceptions & NON_FINITE) != 0)
    return 0;
  if (!SNAN_TESTS (FLOAT) && (exceptions & TEST_SNAN) != 0)
    return 0;
  if (TEST_MATHVEC && (exceptions & NO_TEST_MATHVEC) != 0)
    return 0;

  return 1;
}

/* Structures for each kind of test.  */
/* Used for both RUN_TEST_LOOP_f_f and RUN_TEST_LOOP_fp_f.  */
struct test_f_f_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_ff_f_data
{
  const char *arg_str;
  FLOAT arg1, arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
/* Strictly speaking, a j type argument is one gen-libm-test.pl will not
   attempt to muck with.  For now, it is only used to prevent it from
   mucking up an explicitly long double argument.  */
struct test_fj_f_data
{
  const char *arg_str;
  FLOAT arg1;
  long double arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fi_f_data
{
  const char *arg_str;
  FLOAT arg1;
  int arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fl_f_data
{
  const char *arg_str;
  FLOAT arg1;
  long int arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_if_f_data
{
  const char *arg_str;
  int arg1;
  FLOAT arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fff_f_data
{
  const char *arg_str;
  FLOAT arg1, arg2, arg3;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fiu_M_data
{
  const char *arg_str;
  FLOAT arg1;
  int arg2;
  unsigned int arg3;
  struct
  {
    intmax_t expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fiu_U_data
{
  const char *arg_str;
  FLOAT arg1;
  int arg2;
  unsigned int arg3;
  struct
  {
    uintmax_t expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_c_f_data
{
  const char *arg_str;
  FLOAT argr, argc;
  struct
  {
    FLOAT expected;
    int exceptions;
  } rd, rn, rz, ru;
};
/* Used for both RUN_TEST_LOOP_f_f1 and RUN_TEST_LOOP_fI_f1.  */
struct test_f_f1_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    FLOAT expected;
    int exceptions;
    int extra_test;
    int extra_expected;
  } rd, rn, rz, ru;
};
struct test_fF_f1_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    FLOAT expected;
    int exceptions;
    int extra_test;
    FLOAT extra_expected;
  } rd, rn, rz, ru;
};
struct test_ffI_f1_data
{
  const char *arg_str;
  FLOAT arg1, arg2;
  struct
  {
    FLOAT expected;
    int exceptions;
    int extra_test;
    int extra_expected;
  } rd, rn, rz, ru;
};
struct test_c_c_data
{
  const char *arg_str;
  FLOAT argr, argc;
  struct
  {
    FLOAT expr, expc;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_cc_c_data
{
  const char *arg_str;
  FLOAT arg1r, arg1c, arg2r, arg2c;
  struct
  {
    FLOAT expr, expc;
    int exceptions;
  } rd, rn, rz, ru;
};
/* Used for all of RUN_TEST_LOOP_f_i, RUN_TEST_LOOP_f_i_tg,
   RUN_TEST_LOOP_f_b and RUN_TEST_LOOP_f_b_tg.  */
struct test_f_i_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    int expected;
    int exceptions;
  } rd, rn, rz, ru;
};
/* Used for both RUN_TEST_LOOP_ff_b and RUN_TEST_LOOP_ff_i_tg.  */
struct test_ff_i_data
{
  const char *arg_str;
  FLOAT arg1, arg2;
  struct
  {
    int expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_f_l_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    long int expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_f_L_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    long long int expected;
    int exceptions;
  } rd, rn, rz, ru;
};
struct test_fFF_11_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    int exceptions;
    int extra1_test;
    FLOAT extra1_expected;
    int extra2_test;
    FLOAT extra2_expected;
  } rd, rn, rz, ru;
};
/* Used for both RUN_TEST_LOOP_Ff_b1 and RUN_TEST_LOOP_Ffp_b1.  */
struct test_Ff_b1_data
{
  const char *arg_str;
  FLOAT arg;
  struct
  {
    int expected;
    int exceptions;
    int extra_test;
    FLOAT extra_expected;
  } rd, rn, rz, ru;
};

/* Set the rounding mode, or restore the saved value.  */
#define IF_ROUND_INIT_	/* Empty.  */
#define IF_ROUND_INIT_FE_DOWNWARD		\
  int save_round_mode = fegetround ();		\
  if (ROUNDING_TESTS (FLOAT, FE_DOWNWARD)	\
      && fesetround (FE_DOWNWARD) == 0)
#define IF_ROUND_INIT_FE_TONEAREST		\
  int save_round_mode = fegetround ();		\
  if (ROUNDING_TESTS (FLOAT, FE_TONEAREST)	\
      && fesetround (FE_TONEAREST) == 0)
#define IF_ROUND_INIT_FE_TOWARDZERO		\
  int save_round_mode = fegetround ();		\
  if (ROUNDING_TESTS (FLOAT, FE_TOWARDZERO)	\
      && fesetround (FE_TOWARDZERO) == 0)
#define IF_ROUND_INIT_FE_UPWARD			\
  int save_round_mode = fegetround ();		\
  if (ROUNDING_TESTS (FLOAT, FE_UPWARD)		\
      && fesetround (FE_UPWARD) == 0)
#define ROUND_RESTORE_	/* Empty.  */
#define ROUND_RESTORE_FE_DOWNWARD		\
  fesetround (save_round_mode)
#define ROUND_RESTORE_FE_TONEAREST		\
  fesetround (save_round_mode)
#define ROUND_RESTORE_FE_TOWARDZERO		\
  fesetround (save_round_mode)
#define ROUND_RESTORE_FE_UPWARD			\
  fesetround (save_round_mode)

/* Field name to use for a given rounding mode.  */
#define RM_			rn
#define RM_FE_DOWNWARD		rd
#define RM_FE_TONEAREST		rn
#define RM_FE_TOWARDZERO	rz
#define RM_FE_UPWARD		ru

/* Common setup for an individual test.  */
#define COMMON_TEST_SETUP(ARG_STR)					\
  char *test_name;							\
  if (asprintf (&test_name, "%s (%s)", this_func, (ARG_STR)) == -1)	\
    abort ()

/* Setup for a test with an extra output.  */
#define EXTRA_OUTPUT_TEST_SETUP(ARG_STR, N)			\
  char *extra##N##_name;					\
  if (asprintf (&extra##N##_name, "%s (%s) extra output " #N,	\
		this_func, (ARG_STR)) == -1)			\
    abort ()

/* Common cleanup after an individual test.  */
#define COMMON_TEST_CLEANUP			\
  free (test_name)

/* Cleanup for a test with an extra output.  */
#define EXTRA_OUTPUT_TEST_CLEANUP(N)		\
  free (extra##N##_name)

/* Run an individual test, including any required setup and checking
   of results, or loop over all tests in an array.  */
#define RUN_TEST_f_f(ARG_STR, FUNC_NAME, ARG, EXPECTED,			\
		     EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_float (test_name,	FUNC_TEST (FUNC_NAME) (ARG),		\
		     EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_f(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_f ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fp_f(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
		     EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_float (test_name,	FUNC_TEST (FUNC_NAME) (&(ARG)),		\
		     EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fp_f(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fp_f ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_2_f(ARG_STR, FUNC_NAME, ARG1, ARG2, EXPECTED,	\
		     EXCEPTIONS)				\
  do								\
    if (enable_test (EXCEPTIONS))				\
      {								\
	COMMON_TEST_SETUP (ARG_STR);				\
	check_float (test_name, FUNC_TEST (FUNC_NAME) (ARG1, ARG2),	\
		     EXPECTED, EXCEPTIONS);			\
	COMMON_TEST_CLEANUP;					\
      }								\
  while (0)
#define RUN_TEST_LOOP_2_f(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_2_f ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg1,	\
		    (ARRAY)[i].arg2,					\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_ff_f RUN_TEST_2_f
#define RUN_TEST_LOOP_ff_f RUN_TEST_LOOP_2_f
#define RUN_TEST_LOOP_fj_f RUN_TEST_LOOP_2_f
#define RUN_TEST_fi_f RUN_TEST_2_f
#define RUN_TEST_LOOP_fi_f RUN_TEST_LOOP_2_f
#define RUN_TEST_fl_f RUN_TEST_2_f
#define RUN_TEST_LOOP_fl_f RUN_TEST_LOOP_2_f
#define RUN_TEST_if_f RUN_TEST_2_f
#define RUN_TEST_LOOP_if_f RUN_TEST_LOOP_2_f
#define RUN_TEST_fff_f(ARG_STR, FUNC_NAME, ARG1, ARG2, ARG3,		\
		       EXPECTED, EXCEPTIONS)				\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_float (test_name, FUNC_TEST (FUNC_NAME) (ARG1, ARG2, ARG3),	\
		     EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fff_f(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fff_f ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg1,	\
		      (ARRAY)[i].arg2, (ARRAY)[i].arg3,			\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fiu_M(ARG_STR, FUNC_NAME, ARG1, ARG2, ARG3,		\
		       EXPECTED, EXCEPTIONS)				\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_intmax_t (test_name,					\
			FUNC_TEST (FUNC_NAME) (ARG1, ARG2, ARG3),	\
			EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fiu_M(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fiu_M ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg1,	\
		      (ARRAY)[i].arg2, (ARRAY)[i].arg3,			\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fiu_U(ARG_STR, FUNC_NAME, ARG1, ARG2, ARG3,		\
		       EXPECTED, EXCEPTIONS)				\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_uintmax_t (test_name,					\
			 FUNC_TEST (FUNC_NAME) (ARG1, ARG2, ARG3),	\
			 EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fiu_U(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fiu_U ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg1,	\
		      (ARRAY)[i].arg2, (ARRAY)[i].arg3,			\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_c_f(ARG_STR, FUNC_NAME, ARG1, ARG2, EXPECTED,		\
		     EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_float (test_name,						\
		     FUNC_TEST (FUNC_NAME) (BUILD_COMPLEX (ARG1, ARG2)),\
		     EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_c_f(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_c_f ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].argr,	\
		    (ARRAY)[i].argc,					\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_f1(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
		      EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
		      EXTRA_EXPECTED)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	check_float (test_name, FUNC_TEST (FUNC_NAME) (ARG), EXPECTED,	\
		     EXCEPTIONS);					\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_int (extra1_name, EXTRA_VAR, EXTRA_EXPECTED, 0);	\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_f1(FUNC_NAME, ARRAY, ROUNDING_MODE, EXTRA_VAR)	\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_f1 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		     (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		     (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,		\
		     EXTRA_VAR,						\
		     (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,		\
		     (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fF_f1(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
		       EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
		       EXTRA_EXPECTED)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	check_float (test_name, FUNC_TEST (FUNC_NAME) (ARG, &(EXTRA_VAR)),	\
		     EXPECTED, EXCEPTIONS);				\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_float (extra1_name, EXTRA_VAR, EXTRA_EXPECTED, 0);	\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fF_f1(FUNC_NAME, ARRAY, ROUNDING_MODE, EXTRA_VAR)	\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fF_f1 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,		\
		      EXTRA_VAR,					\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fI_f1(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
		       EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
		       EXTRA_EXPECTED)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	check_float (test_name, FUNC_TEST (FUNC_NAME) (ARG, &(EXTRA_VAR)),	\
		     EXPECTED, EXCEPTIONS);				\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_int (extra1_name, EXTRA_VAR, EXTRA_EXPECTED, 0);	\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fI_f1(FUNC_NAME, ARRAY, ROUNDING_MODE, EXTRA_VAR)	\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fI_f1 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,		\
		      EXTRA_VAR,					\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_ffI_f1_mod8(ARG_STR, FUNC_NAME, ARG1, ARG2, EXPECTED,	\
			     EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
			     EXTRA_EXPECTED)				\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	check_float (test_name,						\
		     FUNC_TEST (FUNC_NAME) (ARG1, ARG2, &(EXTRA_VAR)),	\
		     EXPECTED, EXCEPTIONS);				\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_int (extra1_name, (EXTRA_VAR) % 8, EXTRA_EXPECTED, 0);	\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_ffI_f1_mod8(FUNC_NAME, ARRAY, ROUNDING_MODE,	\
				  EXTRA_VAR)				\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_ffI_f1_mod8 ((ARRAY)[i].arg_str, FUNC_NAME,		\
			    (ARRAY)[i].arg1, (ARRAY)[i].arg2,		\
			    (ARRAY)[i].RM_##ROUNDING_MODE.expected,	\
			    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,	\
			    EXTRA_VAR,					\
			    (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,	\
			    (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected); \
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_Ff_b1(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
		       EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
		       EXTRA_EXPECTED)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	/* Clear any exceptions from comparison involving sNaN		\
	   EXTRA_EXPECTED.  */						\
	feclearexcept (FE_ALL_EXCEPT);					\
	check_bool (test_name, FUNC_TEST (FUNC_NAME) (&(EXTRA_VAR),	\
						      (ARG)),		\
		    EXPECTED, EXCEPTIONS);				\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_float (extra1_name, EXTRA_VAR, EXTRA_EXPECTED,		\
		       (EXCEPTIONS) & TEST_NAN_PAYLOAD);		\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_Ff_b1(FUNC_NAME, ARRAY, ROUNDING_MODE,		\
			    EXTRA_VAR)					\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_Ff_b1 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		      (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,		\
		      EXTRA_VAR,					\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,		\
		      (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_Ffp_b1(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
			EXCEPTIONS, EXTRA_VAR, EXTRA_TEST,		\
			EXTRA_EXPECTED)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	(EXTRA_VAR) = (EXTRA_EXPECTED) == 0 ? 1 : 0;			\
	check_bool (test_name, FUNC_TEST (FUNC_NAME) (&(EXTRA_VAR),	\
						      &(ARG)),		\
		    EXPECTED, EXCEPTIONS);				\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA_TEST)							\
	  check_float (extra1_name, EXTRA_VAR, EXTRA_EXPECTED,		\
		       (EXCEPTIONS) & TEST_NAN_PAYLOAD);		\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_Ffp_b1(FUNC_NAME, ARRAY, ROUNDING_MODE,		\
			     EXTRA_VAR)					\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_Ffp_b1 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		       (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,	\
		       EXTRA_VAR,					\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra_test,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_c_c(ARG_STR, FUNC_NAME, ARGR, ARGC, EXPR, EXPC,	\
		     EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_complex (test_name,					\
		       FUNC_TEST (FUNC_NAME) (BUILD_COMPLEX (ARGR, ARGC)),	\
		       BUILD_COMPLEX (EXPR, EXPC), EXCEPTIONS);		\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_c_c(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_c_c ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].argr,	\
		    (ARRAY)[i].argc,					\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expr,			\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expc,			\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_cc_c(ARG_STR, FUNC_NAME, ARG1R, ARG1C, ARG2R, ARG2C,	\
		      EXPR, EXPC, EXCEPTIONS)				\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_complex (test_name,					\
		       FUNC_TEST (FUNC_NAME) (BUILD_COMPLEX (ARG1R, ARG1C),	\
					      BUILD_COMPLEX (ARG2R, ARG2C)),	\
		       BUILD_COMPLEX (EXPR, EXPC), EXCEPTIONS);		\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_cc_c(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_cc_c ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg1r,	\
		     (ARRAY)[i].arg1c, (ARRAY)[i].arg2r,		\
		     (ARRAY)[i].arg2c,					\
		     (ARRAY)[i].RM_##ROUNDING_MODE.expr,		\
		     (ARRAY)[i].RM_##ROUNDING_MODE.expc,		\
		     (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_i(ARG_STR, FUNC_NAME, ARG, EXPECTED, EXCEPTIONS)	\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_int (test_name, FUNC_TEST (FUNC_NAME) (ARG), EXPECTED,	\
		   EXCEPTIONS);						\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_i(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_i ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_i_tg(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
			EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_int (test_name, FUNC_NAME (ARG), EXPECTED, EXCEPTIONS);	\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_i_tg(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_i_tg ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		       (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_ff_b(ARG_STR, FUNC_NAME, ARG1, ARG2, EXPECTED,		\
		      EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_bool (test_name, FUNC_TEST (FUNC_NAME) (ARG1, ARG2),	\
		    EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_ff_b(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_ff_b ((ARRAY)[i].arg_str, FUNC_NAME,			\
		     (ARRAY)[i].arg1, (ARRAY)[i].arg2,			\
		     (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		     (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_ff_i_tg(ARG_STR, FUNC_NAME, ARG1, ARG2, EXPECTED,	\
			 EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_int (test_name, FUNC_NAME (ARG1, ARG2), EXPECTED,		\
		   EXCEPTIONS);						\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_ff_i_tg(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_ff_i_tg ((ARRAY)[i].arg_str, FUNC_NAME,			\
			(ARRAY)[i].arg1, (ARRAY)[i].arg2,		\
			(ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
			(ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_b(ARG_STR, FUNC_NAME, ARG, EXPECTED, EXCEPTIONS)	\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_bool (test_name, FUNC_TEST (FUNC_NAME) (ARG), EXPECTED,	\
		    EXCEPTIONS);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_b(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_b ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_b_tg(ARG_STR, FUNC_NAME, ARG, EXPECTED,		\
			EXCEPTIONS)					\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_bool (test_name, FUNC_NAME (ARG), EXPECTED, EXCEPTIONS);	\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_b_tg(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_b_tg ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		       (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);	\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_l(ARG_STR, FUNC_NAME, ARG, EXPECTED, EXCEPTIONS)	\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_long (test_name, FUNC_TEST (FUNC_NAME) (ARG), EXPECTED,	\
		    EXCEPTIONS);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_l(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_l ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_f_L(ARG_STR, FUNC_NAME, ARG, EXPECTED, EXCEPTIONS)	\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	check_longlong (test_name, FUNC_TEST (FUNC_NAME) (ARG),		\
			EXPECTED, EXCEPTIONS);				\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_f_L(FUNC_NAME, ARRAY, ROUNDING_MODE)		\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_f_L ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		    (ARRAY)[i].RM_##ROUNDING_MODE.expected,		\
		    (ARRAY)[i].RM_##ROUNDING_MODE.exceptions);		\
  ROUND_RESTORE_ ## ROUNDING_MODE
#define RUN_TEST_fFF_11(ARG_STR, FUNC_NAME, ARG, EXCEPTIONS,		\
			EXTRA1_VAR, EXTRA1_TEST,			\
			EXTRA1_EXPECTED, EXTRA2_VAR,			\
			EXTRA2_TEST, EXTRA2_EXPECTED)			\
  do									\
    if (enable_test (EXCEPTIONS))					\
      {									\
	COMMON_TEST_SETUP (ARG_STR);					\
	FUNC_TEST (FUNC_NAME) (ARG, &(EXTRA1_VAR), &(EXTRA2_VAR));	\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 1);				\
	if (EXTRA1_TEST)						\
	  check_float (extra1_name, EXTRA1_VAR, EXTRA1_EXPECTED,	\
		       EXCEPTIONS);					\
	EXTRA_OUTPUT_TEST_CLEANUP (1);					\
	EXTRA_OUTPUT_TEST_SETUP (ARG_STR, 2);				\
	if (EXTRA2_TEST)						\
	  check_float (extra2_name, EXTRA2_VAR, EXTRA2_EXPECTED, 0);	\
	EXTRA_OUTPUT_TEST_CLEANUP (2);					\
	COMMON_TEST_CLEANUP;						\
      }									\
  while (0)
#define RUN_TEST_LOOP_fFF_11(FUNC_NAME, ARRAY, ROUNDING_MODE,		\
			     EXTRA1_VAR, EXTRA2_VAR)			\
  IF_ROUND_INIT_ ## ROUNDING_MODE					\
    for (size_t i = 0; i < sizeof (ARRAY) / sizeof (ARRAY)[0]; i++)	\
      RUN_TEST_fFF_11 ((ARRAY)[i].arg_str, FUNC_NAME, (ARRAY)[i].arg,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.exceptions,	\
		       EXTRA1_VAR,					\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra1_test,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra1_expected,	\
		       EXTRA2_VAR,					\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra2_test,	\
		       (ARRAY)[i].RM_##ROUNDING_MODE.extra2_expected);	\
  ROUND_RESTORE_ ## ROUNDING_MODE

#if !TEST_MATHVEC
# define VEC_SUFF
#endif

#define STR_CONCAT(a, b, c) __STRING (a##b##c)
#define STR_CON3(a, b, c) STR_CONCAT (a, b, c)

/* This generated header defines series of macros started with HAVE_VECTOR_. */
#include "libm-have-vector-test.h"

#define HAVE_VECTOR(func) __CONCAT (HAVE_VECTOR_, func)

/* Start and end the tests for a given function.  */
#define START(FUN, SUFF, EXACT)					\
  CHECK_ARCH_EXT;						\
  if (TEST_MATHVEC && !HAVE_VECTOR (FUNC (FUN))) return;	\
  const char *this_func = STR_CON3 (FUN, SUFF, VEC_SUFF);	\
  init_max_error (this_func, EXACT)
#define END					\
  print_max_error (this_func)
#define END_COMPLEX				\
  print_complex_max_error (this_func)

/* Run tests for a given function in all rounding modes.  */
#define ALL_RM_TEST(FUNC, EXACT, ARRAY, LOOP_MACRO, END_MACRO, ...)	\
  do									\
    {									\
      do								\
	{								\
	  START (FUNC,, EXACT);						\
	  LOOP_MACRO (FUNC, ARRAY, , ## __VA_ARGS__);			\
	  END_MACRO;							\
	}								\
      while (0);							\
      do								\
	{								\
	  START (FUNC, _downward, EXACT);				\
	  LOOP_MACRO (FUNC, ARRAY, FE_DOWNWARD, ## __VA_ARGS__);	\
	  END_MACRO;							\
	}								\
      while (0);							\
      do								\
	{								\
	  START (FUNC, _towardzero, EXACT);				\
	  LOOP_MACRO (FUNC, ARRAY, FE_TOWARDZERO, ## __VA_ARGS__);	\
	  END_MACRO;							\
	}								\
      while (0);							\
      do								\
	{								\
	  START (FUNC, _upward, EXACT);				\
	  LOOP_MACRO (FUNC, ARRAY, FE_UPWARD, ## __VA_ARGS__);		\
	  END_MACRO;							\
	}								\
      while (0);							\
    }									\
  while (0);

/* This is to prevent messages from the SVID libm emulation.  */
int
matherr (struct exception *x __attribute__ ((unused)))
{
  return 1;
}


/****************************************************************************
  Tests for single functions of libm.
  Please keep them alphabetically sorted!
****************************************************************************/

static const struct test_f_f_data acos_test_data[] =
  {
    TEST_f_f (acos, plus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acos, minus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acos, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (acos, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (acos, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (acos, -snan_value, qnan_value, INVALID_EXCEPTION),

    /* |x| > 1: */
    TEST_f_f (acos, 1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acos, -1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acos, max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acos, -max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),

    AUTO_TESTS_f_f (acos),
  };

static void
acos_test (void)
{
  ALL_RM_TEST (acos, 0, acos_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data acosh_test_data[] =
  {
    TEST_f_f (acosh, plus_infty, plus_infty, ERRNO_UNCHANGED),
    TEST_f_f (acosh, minus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (acosh, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (acosh, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (acosh, -snan_value, qnan_value, INVALID_EXCEPTION),

    /* x < 1:  */
    TEST_f_f (acosh, 0.75L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, min_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, min_subnorm_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, plus_zero, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, minus_zero, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, -min_subnorm_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, -min_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, -1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (acosh, -max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),

    AUTO_TESTS_f_f (acosh),
  };

static void
acosh_test (void)
{
  ALL_RM_TEST (acosh, 0, acosh_test_data, RUN_TEST_LOOP_f_f, END);
}

static const struct test_f_f_data asin_test_data[] =
  {
    TEST_f_f (asin, plus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (asin, minus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (asin, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (asin, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (asin, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (asin, -snan_value, qnan_value, INVALID_EXCEPTION),

    /* asin x == qNaN plus invalid exception for |x| > 1.  */
    TEST_f_f (asin, 1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (asin, -1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (asin, max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (asin, -max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),

    AUTO_TESTS_f_f (asin),
  };

static void
asin_test (void)
{
  ALL_RM_TEST (asin, 0, asin_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data asinh_test_data[] =
  {
    TEST_f_f (asinh, plus_infty, plus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (asinh, minus_infty, minus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (asinh, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (asinh, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (asinh, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (asinh, -snan_value, qnan_value, INVALID_EXCEPTION),
    AUTO_TESTS_f_f (asinh),
  };

static void
asinh_test (void)
{
  ALL_RM_TEST (asinh, 0, asinh_test_data, RUN_TEST_LOOP_f_f, END);
}

static const struct test_f_f_data atan_test_data[] =
  {
    TEST_f_f (atan, plus_infty, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_f_f (atan, minus_infty, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_f_f (atan, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (atan, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (atan, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (atan, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (atan),
  };

static void
atan_test (void)
{
  ALL_RM_TEST (atan, 0, atan_test_data, RUN_TEST_LOOP_f_f, END);
}



static const struct test_f_f_data atanh_test_data[] =
  {
    TEST_f_f (atanh, 1, plus_infty, DIVIDE_BY_ZERO_EXCEPTION|ERRNO_ERANGE),
    TEST_f_f (atanh, -1, minus_infty, DIVIDE_BY_ZERO_EXCEPTION|ERRNO_ERANGE),
    TEST_f_f (atanh, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (atanh, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (atanh, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (atanh, -snan_value, qnan_value, INVALID_EXCEPTION),

    /* atanh (x) == qNaN plus invalid exception if |x| > 1.  */
    TEST_f_f (atanh, 1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (atanh, -1.125L, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (atanh, max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM|NO_TEST_INLINE),
    TEST_f_f (atanh, -max_value, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM|NO_TEST_INLINE),
    TEST_f_f (atanh, plus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (atanh, minus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),

    AUTO_TESTS_f_f (atanh),
  };

static void
atanh_test (void)
{
  ALL_RM_TEST (atanh, 0, atanh_test_data, RUN_TEST_LOOP_f_f, END);
}

static const struct test_ff_f_data atan2_test_data[] =
  {
    /* atan2 (y,inf) == +0 for finite y > 0 or +0.  */
    TEST_ff_f (atan2, 1, plus_infty, 0, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_zero, plus_infty, 0, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_value, plus_infty, 0, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_subnorm_value, plus_infty, 0, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, max_value, plus_infty, 0, ERRNO_UNCHANGED),

    /* atan2 (y,inf) == -0 for finite y < 0 or -0.  */
    TEST_ff_f (atan2, -1, plus_infty, minus_zero, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_zero, plus_infty, minus_zero, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_value, plus_infty, minus_zero, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_subnorm_value, plus_infty, minus_zero, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -max_value, plus_infty, minus_zero, ERRNO_UNCHANGED),

    /* atan2(+inf, x) == pi/2 for finite x.  */
    TEST_ff_f (atan2, plus_infty, 1, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, plus_zero, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, min_value, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, min_subnorm_value, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, max_value, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, -1, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, minus_zero, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, -min_value, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, -min_subnorm_value, lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, -max_value, lit_pi_2_d, ERRNO_UNCHANGED),

    /* atan2(-inf, x) == -pi/2 for finite x.  */
    TEST_ff_f (atan2, minus_infty, 1, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, plus_zero, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, min_value, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, min_subnorm_value, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, max_value, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, -1, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, minus_zero, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, -min_value, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, -min_subnorm_value, -lit_pi_2_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, -max_value, -lit_pi_2_d, ERRNO_UNCHANGED),

    /* atan2 (y,-inf) == +pi for finite y > 0 or +0.  */
    TEST_ff_f (atan2, 1, minus_infty, lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_zero, minus_infty, lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_value, minus_infty, lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_subnorm_value, minus_infty, lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, max_value, minus_infty, lit_pi, ERRNO_UNCHANGED),

    /* atan2 (y,-inf) == -pi for finite y < 0 or -0.  */
    TEST_ff_f (atan2, -1, minus_infty, -lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_zero, minus_infty, -lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_value, minus_infty, -lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_subnorm_value, minus_infty, -lit_pi, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -max_value, minus_infty, -lit_pi, ERRNO_UNCHANGED),

    TEST_ff_f (atan2, plus_infty, plus_infty, lit_pi_4_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, plus_infty, -lit_pi_4_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, minus_infty, lit_pi_3_m_4_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, minus_infty, -lit_pi_3_m_4_d, ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, snan_value, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, qnan_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, qnan_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -qnan_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -qnan_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, plus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, -1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, min_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, -min_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, -min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, max_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, qnan_value, -max_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, plus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, -1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, min_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, -min_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, -min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, max_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -qnan_value, -max_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, snan_value, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, minus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, plus_zero, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, minus_zero, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, 1, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -1, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, min_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -min_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, min_subnorm_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -min_subnorm_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, max_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, snan_value, -max_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, minus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, plus_zero, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, minus_zero, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, 1, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -1, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, min_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -min_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, min_subnorm_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -min_subnorm_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, max_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -snan_value, -max_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, plus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_zero, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_zero, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, 1, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -1, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_subnorm_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_subnorm_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, max_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -max_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_zero, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, minus_zero, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, 1, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -1, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, min_subnorm_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -min_subnorm_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, max_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, -max_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (atan2, plus_infty, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, minus_infty, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, plus_zero, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, minus_zero, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, 1, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -1, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, min_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -min_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, min_subnorm_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -min_subnorm_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, max_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -max_value, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, plus_infty, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, minus_infty, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, plus_zero, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, minus_zero, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, 1, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -1, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, min_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -min_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, min_subnorm_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -min_subnorm_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, max_value, -snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_ff_f (atan2, -max_value, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_ff_f (atan2),
  };

static void
atan2_test (void)
{
  ALL_RM_TEST (atan2, 0, atan2_test_data, RUN_TEST_LOOP_ff_f, END);
}

static const struct test_c_f_data cabs_test_data[] =
  {
    /* cabs (x + iy) is specified as hypot (x,y) */

    /* cabs (+inf + i x) == +inf.  */
    TEST_c_f (cabs, plus_infty, 1.0, plus_infty),
    /* cabs (-inf + i x) == +inf.  */
    TEST_c_f (cabs, minus_infty, 1.0, plus_infty),

    TEST_c_f (cabs, minus_infty, qnan_value, plus_infty),
    TEST_c_f (cabs, minus_infty, qnan_value, plus_infty),

    TEST_c_f (cabs, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_f (cabs),
  };

static void
cabs_test (void)
{
  ALL_RM_TEST (cabs, 0, cabs_test_data, RUN_TEST_LOOP_c_f, END);
}


static const struct test_c_c_data cacos_test_data[] =
  {
    TEST_c_c (cacos, 0, 0, lit_pi_2_d, minus_zero),
    TEST_c_c (cacos, minus_zero, 0, lit_pi_2_d, minus_zero),
    TEST_c_c (cacos, minus_zero, minus_zero, lit_pi_2_d, 0.0),
    TEST_c_c (cacos, 0, minus_zero, lit_pi_2_d, 0.0),

    TEST_c_c (cacos, minus_infty, plus_infty, lit_pi_3_m_4_d, minus_infty),
    TEST_c_c (cacos, minus_infty, minus_infty, lit_pi_3_m_4_d, plus_infty),

    TEST_c_c (cacos, plus_infty, plus_infty, lit_pi_4_d, minus_infty),
    TEST_c_c (cacos, plus_infty, minus_infty, lit_pi_4_d, plus_infty),

    TEST_c_c (cacos, -10.0, plus_infty, lit_pi_2_d, minus_infty),
    TEST_c_c (cacos, -10.0, minus_infty, lit_pi_2_d, plus_infty),
    TEST_c_c (cacos, 0, plus_infty, lit_pi_2_d, minus_infty),
    TEST_c_c (cacos, 0, minus_infty, lit_pi_2_d, plus_infty),
    TEST_c_c (cacos, 0.1L, plus_infty, lit_pi_2_d, minus_infty),
    TEST_c_c (cacos, 0.1L, minus_infty, lit_pi_2_d, plus_infty),

    TEST_c_c (cacos, minus_infty, 0, lit_pi, minus_infty),
    TEST_c_c (cacos, minus_infty, minus_zero, lit_pi, plus_infty),
    TEST_c_c (cacos, minus_infty, 100, lit_pi, minus_infty),
    TEST_c_c (cacos, minus_infty, -100, lit_pi, plus_infty),

    TEST_c_c (cacos, plus_infty, 0, 0.0, minus_infty),
    TEST_c_c (cacos, plus_infty, minus_zero, 0.0, plus_infty),
    TEST_c_c (cacos, plus_infty, 0.5, 0.0, minus_infty),
    TEST_c_c (cacos, plus_infty, -0.5, 0.0, plus_infty),

    TEST_c_c (cacos, plus_infty, qnan_value, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (cacos, minus_infty, qnan_value, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (cacos, 0, qnan_value, lit_pi_2_d, qnan_value),
    TEST_c_c (cacos, minus_zero, qnan_value, lit_pi_2_d, qnan_value),

    TEST_c_c (cacos, qnan_value, plus_infty, qnan_value, minus_infty),
    TEST_c_c (cacos, qnan_value, minus_infty, qnan_value, plus_infty),

    TEST_c_c (cacos, 10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cacos, -10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (cacos, qnan_value, 0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cacos, qnan_value, -0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (cacos, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (cacos, plus_zero, -1.5L, lit_pi_2_d, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, minus_zero, -1.5L, lit_pi_2_d, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, plus_zero, -1.0L, lit_pi_2_d, 0.8813735870195430252326093249797923090282L),
    TEST_c_c (cacos, minus_zero, -1.0L, lit_pi_2_d, 0.8813735870195430252326093249797923090282L),
    TEST_c_c (cacos, plus_zero, -0.5L, lit_pi_2_d, 0.4812118250596034474977589134243684231352L),
    TEST_c_c (cacos, minus_zero, -0.5L, lit_pi_2_d, 0.4812118250596034474977589134243684231352L),
    TEST_c_c (cacos, plus_zero, 0.5L, lit_pi_2_d, -0.4812118250596034474977589134243684231352L),
    TEST_c_c (cacos, minus_zero, 0.5L, lit_pi_2_d, -0.4812118250596034474977589134243684231352L),
    TEST_c_c (cacos, plus_zero, 1.0L, lit_pi_2_d, -0.8813735870195430252326093249797923090282L),
    TEST_c_c (cacos, minus_zero, 1.0L, lit_pi_2_d, -0.8813735870195430252326093249797923090282L),
    TEST_c_c (cacos, plus_zero, 1.5L, lit_pi_2_d, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, minus_zero, 1.5L, lit_pi_2_d, -1.194763217287109304111930828519090523536L),

    TEST_c_c (cacos, -1.5L, plus_zero, lit_pi, -0.9624236501192068949955178268487368462704L),
    TEST_c_c (cacos, -1.5L, minus_zero, lit_pi, 0.9624236501192068949955178268487368462704L),
    TEST_c_c (cacos, -1.0L, plus_zero, lit_pi, minus_zero),
    TEST_c_c (cacos, -1.0L, minus_zero, lit_pi, plus_zero),
    TEST_c_c (cacos, -0.5L, plus_zero, 2.094395102393195492308428922186335256131L, minus_zero),
    TEST_c_c (cacos, -0.5L, minus_zero, 2.094395102393195492308428922186335256131L, plus_zero),
    TEST_c_c (cacos, 0.5L, plus_zero, 1.047197551196597746154214461093167628066L, minus_zero),
    TEST_c_c (cacos, 0.5L, minus_zero, 1.047197551196597746154214461093167628066L, plus_zero),
    TEST_c_c (cacos, 1.0L, plus_zero, plus_zero, minus_zero),
    TEST_c_c (cacos, 1.0L, minus_zero, plus_zero, plus_zero),
    TEST_c_c (cacos, 1.5L, plus_zero, plus_zero, -0.9624236501192068949955178268487368462704L),
    TEST_c_c (cacos, 1.5L, minus_zero, plus_zero, 0.9624236501192068949955178268487368462704L),

    TEST_c_c (cacos, 0x1p50L, 1.0L, 8.881784197001252323389053344727730248720e-16L, -3.535050620855721078027883819436720218708e1L),
    TEST_c_c (cacos, 0x1p50L, -1.0L, 8.881784197001252323389053344727730248720e-16L, 3.535050620855721078027883819436720218708e1L),
    TEST_c_c (cacos, -0x1p50L, 1.0L, 3.141592653589792350284223683154270545292L, -3.535050620855721078027883819436720218708e1L),
    TEST_c_c (cacos, -0x1p50L, -1.0L, 3.141592653589792350284223683154270545292L, 3.535050620855721078027883819436720218708e1L),
    TEST_c_c (cacos, 1.0L, 0x1p50L, 1.570796326794895731052901991514519103193L, -3.535050620855721078027883819436759661753e1L),
    TEST_c_c (cacos, -1.0L, 0x1p50L, 1.570796326794897507409741391764983781004L, -3.535050620855721078027883819436759661753e1L),
    TEST_c_c (cacos, 1.0L, -0x1p50L, 1.570796326794895731052901991514519103193L, 3.535050620855721078027883819436759661753e1L),
    TEST_c_c (cacos, -1.0L, -0x1p50L, 1.570796326794897507409741391764983781004L, 3.535050620855721078027883819436759661753e1L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1p500L, 1.0L, 3.054936363499604682051979393213617699789e-151L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, 0x1p500L, -1.0L, 3.054936363499604682051979393213617699789e-151L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, -0x1p500L, 1.0L, 3.141592653589793238462643383279502884197L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, -0x1p500L, -1.0L, 3.141592653589793238462643383279502884197L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, 1.0L, 0x1p500L, 1.570796326794896619231321691639751442099L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, -1.0L, 0x1p500L, 1.570796326794896619231321691639751442099L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, 1.0L, -0x1p500L, 1.570796326794896619231321691639751442099L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (cacos, -1.0L, -0x1p500L, 1.570796326794896619231321691639751442099L, 3.472667374605326000180332928505464606058e2L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (cacos, 0x1p5000L, 1.0L, 7.079811261048172892385615158694057552948e-1506L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, 0x1p5000L, -1.0L, 7.079811261048172892385615158694057552948e-1506L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, -0x1p5000L, 1.0L, 3.141592653589793238462643383279502884197L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, -0x1p5000L, -1.0L, 3.141592653589793238462643383279502884197L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, 1.0L, 0x1p5000L, 1.570796326794896619231321691639751442099L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, -1.0L, 0x1p5000L, 1.570796326794896619231321691639751442099L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, 1.0L, -0x1p5000L, 1.570796326794896619231321691639751442099L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (cacos, -1.0L, -0x1p5000L, 1.570796326794896619231321691639751442099L, 3.466429049980286492395577839412341016946e3L),
#endif

    TEST_c_c (cacos, 0x1.fp127L, 0x1.fp127L, 7.853981633974483096156608458198757210493e-1L, -8.973081118419833726837456344608533993585e1L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp1023L, 0x1.fp1023L, 7.853981633974483096156608458198757210493e-1L, -7.107906849659093345062145442726115449315e2L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (cacos, 0x1.fp16383L, 0x1.fp16383L, 7.853981633974483096156608458198757210493e-1L, -1.135753137836666928715489992987020363057e4L),
#endif

    TEST_c_c (cacos, 0x1.fp-129L, 1.5L, 1.570796326794896619231321691639751442097L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 0x1.fp-129L, -1.5L, 1.570796326794896619231321691639751442097L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-129L, 1.5L, 1.570796326794896619231321691639751442100L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-129L, -1.5L, 1.570796326794896619231321691639751442100L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 1.5L, 0x1.fp-129L, 2.546345110742945032959687790021055102355e-39L, -9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -1.5L, 0x1.fp-129L, 3.141592653589793238462643383279502884195L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (cacos, 1.5L, -0x1.fp-129L, 2.546345110742945032959687790021055102355e-39L, 9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -1.5L, -0x1.fp-129L, 3.141592653589793238462643383279502884195L, 9.624236501192068949955178268487368462704e-1L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1025L, 1.5L, 1.570796326794896619231321691639751442099L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 0x1.fp-1025L, -1.5L, 1.570796326794896619231321691639751442099L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-1025L, 1.5L, 1.570796326794896619231321691639751442099L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-1025L, -1.5L, 1.570796326794896619231321691639751442099L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 1.5L, 0x1.fp-1025L, 4.819934639999230680322935210539402497827e-309L, -9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -1.5L, 0x1.fp-1025L, 3.141592653589793238462643383279502884197L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (cacos, 1.5L, -0x1.fp-1025L, 4.819934639999230680322935210539402497827e-309L, 9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -1.5L, -0x1.fp-1025L, 3.141592653589793238462643383279502884197L, 9.624236501192068949955178268487368462704e-1L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 1.5L, 1.570796326794896619231321691639751442099L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 0x1.fp-16385L, -1.5L, 1.570796326794896619231321691639751442099L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-16385L, 1.5L, 1.570796326794896619231321691639751442099L, -1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, -0x1.fp-16385L, -1.5L, 1.570796326794896619231321691639751442099L, 1.194763217287109304111930828519090523536L),
    TEST_c_c (cacos, 1.5L, 0x1.fp-16385L, 7.282957076134209141226696333885150260319e-4933L, -9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -1.5L, 0x1.fp-16385L, 3.141592653589793238462643383279502884197L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (cacos, 1.5L, -0x1.fp-16385L, 7.282957076134209141226696333885150260319e-4933L, 9.624236501192068949955178268487368462704e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -1.5L, -0x1.fp-16385L, 3.141592653589793238462643383279502884197L, 9.624236501192068949955178268487368462704e-1L),
#endif

    TEST_c_c (cacos, 0.5L, 1.0L, 1.221357263937683325603909865564381489366L, -9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (cacos, 0.5L, -1.0L, 1.221357263937683325603909865564381489366L, 9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (cacos, -0.5L, 1.0L, 1.920235389652109912858733517715121394831L, -9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (cacos, -0.5L, -1.0L, 1.920235389652109912858733517715121394831L, 9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (cacos, 1.0L, 0.5L, 6.748888455860063801646649673121744318756e-1L, -7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (cacos, -1.0L, 0.5L, 2.466703808003786858297978415967328452322L, -7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (cacos, 1.0L, -0.5L, 6.748888455860063801646649673121744318756e-1L, 7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (cacos, -1.0L, -0.5L, 2.466703808003786858297978415967328452322L, 7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (cacos, 0.25L, 1.0L, 1.394493894017929688812643125003661339452L, -8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (cacos, 0.25L, -1.0L, 1.394493894017929688812643125003661339452L, 8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (cacos, -0.25L, 1.0L, 1.747098759571863549650000258275841544745L, -8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (cacos, -0.25L, -1.0L, 1.747098759571863549650000258275841544745L, 8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (cacos, 1.0L, 0.25L, 4.890443302710802929202843732146540079124e-1L, -5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (cacos, -1.0L, 0.25L, 2.652548323318712945542359010064848876285L, -5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (cacos, 1.0L, -0.25L, 4.890443302710802929202843732146540079124e-1L, 5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (cacos, -1.0L, -0.25L, 2.652548323318712945542359010064848876285L, 5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (cacos, 0x1.fp-10L, 1.0L, 1.569458417435338878318763342108699202986L, -8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (cacos, 0x1.fp-10L, -1.0L, 1.569458417435338878318763342108699202986L, 8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (cacos, -0x1.fp-10L, 1.0L, 1.572134236154454360143880041170803681211L, -8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (cacos, -0x1.fp-10L, -1.0L, 1.572134236154454360143880041170803681211L, 8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-10L, 4.349129763101882771258049954181971959031e-2L, -4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-10L, 3.098101355958774410750062883737683164607L, -4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-10L, 4.349129763101882771258049954181971959031e-2L, 4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-10L, 3.098101355958774410750062883737683164607L, 4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (cacos, 0x1.fp-30L, 1.0L, 1.570796325518966635014803151387033957091L, -8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (cacos, 0x1.fp-30L, -1.0L, 1.570796325518966635014803151387033957091L, 8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (cacos, -0x1.fp-30L, 1.0L, 1.570796328070826603447840231892468927106L, -8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (cacos, -0x1.fp-30L, -1.0L, 1.570796328070826603447840231892468927106L, 8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-30L, 4.247867097467650115899790787875186617316e-5L, -4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-30L, 3.141550174918818561961484385371624132331L, -4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-30L, 4.247867097467650115899790787875186617316e-5L, 4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-30L, 3.141550174918818561961484385371624132331L, 4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (cacos, 0x1.fp-100L, 1.0L, 1.570796326794896619231321691638670687364L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-100L, -1.0L, 1.570796326794896619231321691638670687364L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-100L, 1.0L, 1.570796326794896619231321691640832196834L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-100L, -1.0L, 1.570796326794896619231321691640832196834L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-100L, 1.236292038260260888664514866456887257525e-15L, -1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-100L, 3.141592653589792002170605123018614219682L, -1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-100L, 1.236292038260260888664514866456887257525e-15L, 1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-100L, 3.141592653589792002170605123018614219682L, 1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (cacos, 0x1.fp-129L, 1.0L, 1.570796326794896619231321691639751442097L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-129L, -1.0L, 1.570796326794896619231321691639751442097L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, 1.0L, 1.570796326794896619231321691639751442101L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, -1.0L, 1.570796326794896619231321691639751442101L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-129L, 5.335635276982233498398987585285818977930e-20L, -5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-129L, 3.141592653589793238409287030509680549213L, -5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-129L, 5.335635276982233498398987585285818977930e-20L, 5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-129L, 3.141592653589793238409287030509680549213L, 5.335635276982233498398987585285818977933e-20L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1000L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-1000L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-1000L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-1000L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, -4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-1000L, 3.141592653589793238462643383279502884197L, -4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, 4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-1000L, 3.141592653589793238462643383279502884197L, 4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacos, 0x1.fp-1025L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, -7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-1025L, 3.141592653589793238462643383279502884197L, -7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, 7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-1025L, 3.141592653589793238462643383279502884197L, 7.340879205566679497036857179189356754017e-155L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-10000L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-10000L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-10000L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-10000L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, -9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-10000L, 3.141592653589793238462643383279502884197L, -9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, 9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-10000L, 3.141592653589793238462643383279502884197L, 9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacos, 0x1.fp-16385L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 1.0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -1.0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (cacos, 1.0L, 0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, -9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (cacos, -1.0L, 0x1.fp-16385L, 3.141592653589793238462643383279502884197L, -9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (cacos, 1.0L, -0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, 9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (cacos, -1.0L, -0x1.fp-16385L, 3.141592653589793238462643383279502884197L, 9.023632056840860275214893047597614177639e-2467L),
#endif

    TEST_c_c (cacos, 0x1p-23L, 0x1.000002p0L, 1.570796242501204621739026081991856762292L, -8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (cacos, 0x1p-23L, -0x1.000002p0L, 1.570796242501204621739026081991856762292L, 8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (cacos, -0x1p-23L, 0x1.000002p0L, 1.570796411088588616723617301287646121905L, -8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (cacos, -0x1p-23L, -0x1.000002p0L, 1.570796411088588616723617301287646121905L, 8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (cacos, 0x1.000002p0L, 0x1p-23L, 2.222118384408546368406374049167636760903e-4L, -5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, 0x1p-23L, 3.141370441751352383825802745874586120521L, -5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (cacos, 0x1.000002p0L, -0x1p-23L, 2.222118384408546368406374049167636760903e-4L, 5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, -0x1p-23L, 3.141370441751352383825802745874586120521L, 5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (cacos, 0x1.fp-129L, 0x1.000002p0L, 1.570796326794896619231321691639751442097L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, 0x1.fp-129L, -0x1.000002p0L, 1.570796326794896619231321691639751442097L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, 0x1.000002p0L, 1.570796326794896619231321691639751442101L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, -0x1.000002p0L, 1.570796326794896619231321691639751442101L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, 0x1.000002p0L, 0x1.fp-129L, 5.830451806317544230969669308596361881467e-36L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, 0x1.fp-129L, 3.141592653589793238462643383279502878367L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, 0x1.000002p0L, -0x1.fp-129L, 5.830451806317544230969669308596361881467e-36L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, -0x1.fp-129L, 3.141592653589793238462643383279502878367L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, 0.0L, 0x1.000002p0L, 1.570796326794896619231321691639751442099L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, 0.0L, -0x1.000002p0L, 1.570796326794896619231321691639751442099L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, -0.0L, 0x1.000002p0L, 1.570796326794896619231321691639751442099L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, -0.0L, -0x1.000002p0L, 1.570796326794896619231321691639751442099L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (cacos, 0x1.000002p0L, 0.0L, 0.0L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, 0.0L, 3.141592653589793238462643383279502884197L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, 0x1.000002p0L, -0.0L, 0.0L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (cacos, -0x1.000002p0L, -0.0L, 3.141592653589793238462643383279502884197L, 4.882812451493617206486388134172712975070e-4L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1p-52L, 0x1.0000000000001p0L, 1.570796326794896462222075823262262934288L, -8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (cacos, 0x1p-52L, -0x1.0000000000001p0L, 1.570796326794896462222075823262262934288L, 8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (cacos, -0x1p-52L, 0x1.0000000000001p0L, 1.570796326794896776240567560017239949909L, -8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (cacos, -0x1p-52L, -0x1.0000000000001p0L, 1.570796326794896776240567560017239949909L, 8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, 0x1p-52L, 9.590301705980041385828904092662391018164e-9L, -2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (cacos, -0x1.0000000000001p0L, 0x1p-52L, 3.141592643999491532482601997450598791535L, -2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, -0x1p-52L, 9.590301705980041385828904092662391018164e-9L, 2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (cacos, -0x1.0000000000001p0L, -0x1p-52L, 3.141592643999491532482601997450598791535L, 2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (cacos, 0x1.fp-1025L, 0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, 0x1.fp-1025L, 2.557178503953494342609835913586108008322e-301L, -2.107342425544701550354780375182800088393e-8L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x1.0000000000001p0L, 0x1.fp-1025L, 3.141592653589793238462643383279502884197L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, -0x1.fp-1025L, 2.557178503953494342609835913586108008322e-301L, 2.107342425544701550354780375182800088393e-8L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x1.0000000000001p0L, -0x1.fp-1025L, 3.141592653589793238462643383279502884197L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (cacos, 0.0L, 0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, 0.0L, -0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, -0.0L, 0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, -0.0L, -0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, 0.0L, 0.0L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (cacos, -0x1.0000000000001p0L, 0.0L, 3.141592653589793238462643383279502884197L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (cacos, 0x1.0000000000001p0L, -0.0L, 0.0L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (cacos, -0x1.0000000000001p0L, -0.0L, 3.141592653589793238462643383279502884197L, 2.107342425544701550354780375182800088393e-8L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (cacos, 0x1p-63L, 0x1.0000000000000002p0L, 1.570796326794896619154657020805582738031L, -8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (cacos, 0x1p-63L, -0x1.0000000000000002p0L, 1.570796326794896619154657020805582738031L, 8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (cacos, -0x1p-63L, 0x1.0000000000000002p0L, 1.570796326794896619307986362473920146166L, -8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (cacos, -0x1p-63L, -0x1.0000000000000002p0L, 1.570796326794896619307986362473920146166L, 8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, 0x1p-63L, 2.119177303101063432592523199680782317447e-10L, -5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, 0x1p-63L, 3.141592653377875508152537040020250564229L, -5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, -0x1p-63L, 2.119177303101063432592523199680782317447e-10L, 5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, -0x1p-63L, 3.141592653377875508152537040020250564229L, 5.116146586219826555037807251857670783420e-10L),
# if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, 0x1.fp-16385L, 1.748608650034385653922359120438227480943e-4923L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, 0x1.fp-16385L, 3.141592653589793238462643383279502884197L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, -0x1.fp-16385L, 1.748608650034385653922359120438227480943e-4923L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, -0x1.fp-16385L, 3.141592653589793238462643383279502884197L, 4.656612873077392578082927418388212703712e-10L),
# endif
    TEST_c_c (cacos, 0.0L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, 0.0L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, -0.0L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, -0.0L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, 0.0L, 0.0L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, 0.0L, 3.141592653589793238462643383279502884197L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, 0x1.0000000000000002p0L, -0.0L, 0.0L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (cacos, -0x1.0000000000000002p0L, -0.0L, 3.141592653589793238462643383279502884197L, 4.656612873077392578082927418388212703712e-10L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (cacos, 0x1p-106L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639742726335L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0x1p-106L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639742726335L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0x1p-106L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639760157863L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0x1p-106L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639760157863L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, 0x1p-106L, 5.394221422390606848017034778914096659726e-17L, -2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, 0x1p-106L, 3.141592653589793184520429159373434404027L, -2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, -0x1p-106L, 5.394221422390606848017034778914096659726e-17L, 2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, -0x1p-106L, 3.141592653589793184520429159373434404027L, 2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (cacos, 0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 2.426922623448365473354662093431821897807e-293L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 3.141592653589793238462643383279502884197L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 2.426922623448365473354662093431821897807e-293L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 3.141592653589793238462643383279502884197L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, 0.0L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0.0L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0.0L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, -0.0L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, 0.0L, 0.0L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, 0.0L, 3.141592653589793238462643383279502884197L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, 0x1.000000000000000000000000008p0L, -0.0L, 0.0L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (cacos, -0x1.000000000000000000000000008p0L, -0.0L, 3.141592653589793238462643383279502884197L, 2.220446049250313080847263336181636063482e-16L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (cacos, 0x1p-113L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751374007L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0x1p-113L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751374007L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0x1p-113L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751510190L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0x1p-113L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751510190L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, 0x1p-113L, 4.767863183742677481693563511435642755521e-18L, -2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, 0x1p-113L, 3.141592653589793233694780199536825402504L, -2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, -0x1p-113L, 4.767863183742677481693563511435642755521e-18L, 2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, -0x1p-113L, 3.141592653589793233694780199536825402504L, 2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (cacos, 0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 4.148847925325683229178506116378864361396e-4916L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 3.141592653589793238462643383279502884197L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 4.148847925325683229178506116378864361396e-4916L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 3.141592653589793238462643383279502884197L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, 0.0L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0.0L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0.0L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, -0.0L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, 0.0L, 0.0L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, 0.0L, 3.141592653589793238462643383279502884197L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, 0x1.0000000000000000000000000001p0L, -0.0L, 0.0L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (cacos, -0x1.0000000000000000000000000001p0L, -0.0L, 3.141592653589793238462643383279502884197L, 1.962615573354718824241727964954454332780e-17L),
#endif

    TEST_c_c (cacos, 0x1p-23L, 0x0.ffffffp0L, 1.570796242501197085295336701632142060969L, -8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (cacos, 0x1p-23L, -0x0.ffffffp0L, 1.570796242501197085295336701632142060969L, 8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (cacos, -0x1p-23L, 0x0.ffffffp0L, 1.570796411088596153167306681647360823228L, -8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (cacos, -0x1p-23L, -0x0.ffffffp0L, 1.570796411088596153167306681647360823228L, 8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (cacos, 0x0.ffffffp0L, 0x1p-23L, 4.391863861910487109445187743978204002614e-4L, -2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (cacos, -0x0.ffffffp0L, 0x1p-23L, 3.141153467203602189751698864505105063797L, -2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (cacos, 0x0.ffffffp0L, -0x1p-23L, 4.391863861910487109445187743978204002614e-4L, 2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (cacos, -0x0.ffffffp0L, -0x1p-23L, 3.141153467203602189751698864505105063797L, 2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (cacos, 0x1.fp-129L, 0x0.ffffffp0L, 1.570796326794896619231321691639751442097L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, 0x1.fp-129L, -0x0.ffffffp0L, 1.570796326794896619231321691639751442097L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, 0x0.ffffffp0L, 1.570796326794896619231321691639751442101L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, -0x0.ffffffp0L, 1.570796326794896619231321691639751442101L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, 0x0.ffffffp0L, 0x1.fp-129L, 3.452669847162035876032494826387364972849e-4L, -8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (cacos, -0x0.ffffffp0L, 0x1.fp-129L, 3.141247386605077034875040133796864147700L, -8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (cacos, 0x0.ffffffp0L, -0x1.fp-129L, 3.452669847162035876032494826387364972849e-4L, 8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (cacos, -0x0.ffffffp0L, -0x1.fp-129L, 3.141247386605077034875040133796864147700L, 8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (cacos, 0.0L, 0x0.ffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, 0.0L, -0x0.ffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, -0.0L, 0x0.ffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, -0.0L, -0x0.ffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (cacos, 0x0.ffffffp0L, 0.0L, 3.452669847162035876032494826387364972849e-4L, -0.0L),
    TEST_c_c (cacos, -0x0.ffffffp0L, 0.0L, 3.141247386605077034875040133796864147700L, -0.0L),
    TEST_c_c (cacos, 0x0.ffffffp0L, -0.0L, 3.452669847162035876032494826387364972849e-4L, 0.0L),
    TEST_c_c (cacos, -0x0.ffffffp0L, -0.0L, 3.141247386605077034875040133796864147700L, 0.0L),
    TEST_c_c (cacos, 0x1p-23L, 0.5L, 1.570796220170866625230343643673321150378L, -4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (cacos, 0x1p-23L, -0.5L, 1.570796220170866625230343643673321150378L, 4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (cacos, -0x1p-23L, 0.5L, 1.570796433418926613232299739606181733819L, -4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (cacos, -0x1p-23L, -0.5L, 1.570796433418926613232299739606181733819L, 4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (cacos, 0.5L, 0x1p-23L, 1.047197551196603215914744618665204532273L, -1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (cacos, -0.5L, 0x1p-23L, 2.094395102393190022547898764614298351924L, -1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (cacos, 0.5L, -0x1p-23L, 1.047197551196603215914744618665204532273L, 1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (cacos, -0.5L, -0x1p-23L, 2.094395102393190022547898764614298351924L, 1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (cacos, 0x1.fp-129L, 0.5L, 1.570796326794896619231321691639751442096L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0x1.fp-129L, -0.5L, 1.570796326794896619231321691639751442096L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, 0.5L, 1.570796326794896619231321691639751442101L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-129L, -0.5L, 1.570796326794896619231321691639751442101L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0.5L, 0x1.fp-129L, 1.047197551196597746154214461093167628066L, -3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0.5L, 0x1.fp-129L, 2.094395102393195492308428922186335256131L, -3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, 0.5L, -0x1.fp-129L, 1.047197551196597746154214461093167628066L, 3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0.5L, -0x1.fp-129L, 2.094395102393195492308428922186335256131L, 3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, 0x1p-23L, 0x1p-23L, 1.570796207585607068450636380271254316214L, -1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (cacos, 0x1p-23L, -0x1p-23L, 1.570796207585607068450636380271254316214L, 1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (cacos, -0x1p-23L, 0x1p-23L, 1.570796446004186170012007003008248567984L, -1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (cacos, -0x1p-23L, -0x1p-23L, 1.570796446004186170012007003008248567984L, 1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (cacos, 0x1.fp-129L, 0x1p-23L, 1.570796326794896619231321691639751442096L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, 0x1.fp-129L, -0x1p-23L, 1.570796326794896619231321691639751442096L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, -0x1.fp-129L, 0x1p-23L, 1.570796326794896619231321691639751442101L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, -0x1.fp-129L, -0x1p-23L, 1.570796326794896619231321691639751442101L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, 0x1p-23L, 0x1.fp-129L, 1.570796207585607068449789347324000006847L, -2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0x1p-23L, 0x1.fp-129L, 1.570796446004186170012854035955502877351L, -2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, 0x1p-23L, -0x1.fp-129L, 1.570796207585607068449789347324000006847L, 2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0x1p-23L, -0x1.fp-129L, 1.570796446004186170012854035955502877351L, 2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, 0.0L, 0x1p-23L, 1.570796326794896619231321691639751442099L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, 0.0L, -0x1p-23L, 1.570796326794896619231321691639751442099L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, -0.0L, 0x1p-23L, 1.570796326794896619231321691639751442099L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, -0.0L, -0x1p-23L, 1.570796326794896619231321691639751442099L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (cacos, 0x1p-23L, 0.0L, 1.570796207585607068449789347324000006847L, -0.0L),
    TEST_c_c (cacos, -0x1p-23L, 0.0L, 1.570796446004186170012854035955502877351L, -0.0L),
    TEST_c_c (cacos, 0x1p-23L, -0.0L, 1.570796207585607068449789347324000006847L, 0.0L),
    TEST_c_c (cacos, -0x1p-23L, -0.0L, 1.570796446004186170012854035955502877351L, 0.0L),
    TEST_c_c (cacos, 0x1.fp-129L, 0x1.fp-129L, 1.570796326794896619231321691639751442096L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, 0x1.fp-129L, -0x1.fp-129L, 1.570796326794896619231321691639751442096L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0x1.fp-129L, 0x1.fp-129L, 1.570796326794896619231321691639751442101L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacos, -0x1.fp-129L, -0x1.fp-129L, 1.570796326794896619231321691639751442101L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1p-52L, 0x0.fffffffffffff8p0L, 1.570796326794896462222075823262236786996L, -8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (cacos, 0x1p-52L, -0x0.fffffffffffff8p0L, 1.570796326794896462222075823262236786996L, 8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (cacos, -0x1p-52L, 0x0.fffffffffffff8p0L, 1.570796326794896776240567560017266097201L, -8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (cacos, -0x1p-52L, -0x0.fffffffffffff8p0L, 1.570796326794896776240567560017266097201L, 8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, 0x1p-52L, 1.895456983915074112227925127005564372844e-8L, -1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, 0x1p-52L, 3.141592634635223399311902261000251614142L, -1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, -0x1p-52L, 1.895456983915074112227925127005564372844e-8L, 1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, -0x1p-52L, 3.141592634635223399311902261000251614142L, 1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (cacos, 0x1.fp-1025L, 0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, 0x1.fp-1025L, 1.490116119384765638786343542550461592240e-8L, -3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, 0x1.fp-1025L, 3.141592638688632044614986995416067458693L, -3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, -0x1.fp-1025L, 1.490116119384765638786343542550461592240e-8L, 3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, -0x1.fp-1025L, 3.141592638688632044614986995416067458693L, 3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, 0.0L, 0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, 0.0L, -0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, -0.0L, 0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, -0.0L, -0x0.fffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, 0.0L, 1.490116119384765638786343542550461592240e-8L, -0.0L),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, 0.0L, 3.141592638688632044614986995416067458693L, -0.0L),
    TEST_c_c (cacos, 0x0.fffffffffffff8p0L, -0.0L, 1.490116119384765638786343542550461592240e-8L, 0.0L),
    TEST_c_c (cacos, -0x0.fffffffffffff8p0L, -0.0L, 3.141592638688632044614986995416067458693L, 0.0L),
#endif
    TEST_c_c (cacos, 0x1p-52L, 0.5L, 1.570796326794896420628589431857911675917L, -4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (cacos, 0x1p-52L, -0.5L, 1.570796326794896420628589431857911675917L, 4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (cacos, -0x1p-52L, 0.5L, 1.570796326794896817834053951421591208280L, -4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (cacos, -0x1p-52L, -0.5L, 1.570796326794896817834053951421591208280L, 4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (cacos, 0.5L, 0x1p-52L, 1.047197551196597746154214461093186605110L, -2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (cacos, -0.5L, 0x1p-52L, 2.094395102393195492308428922186316279087L, -2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (cacos, 0.5L, -0x1p-52L, 1.047197551196597746154214461093186605110L, 2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (cacos, -0.5L, -0x1p-52L, 2.094395102393195492308428922186316279087L, 2.563950248511418570403591756798643804971e-16L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1025L, 0.5L, 1.570796326794896619231321691639751442099L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0.5L, 1.570796326794896619231321691639751442099L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0.5L, 1.570796326794896619231321691639751442099L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0.5L, 1.570796326794896619231321691639751442099L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0.5L, 0x1.fp-1025L, 1.047197551196597746154214461093167628066L, -6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0.5L, 0x1.fp-1025L, 2.094395102393195492308428922186335256131L, -6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, 0.5L, -0x1.fp-1025L, 1.047197551196597746154214461093167628066L, 6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0.5L, -0x1.fp-1025L, 2.094395102393195492308428922186335256131L, 6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacos, 0x1p-52L, 0x1p-52L, 1.570796326794896397186716766608443357372L, -2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (cacos, 0x1p-52L, -0x1p-52L, 1.570796326794896397186716766608443357372L, 2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (cacos, -0x1p-52L, 0x1p-52L, 1.570796326794896841275926616671059526825L, -2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (cacos, -0x1p-52L, -0x1p-52L, 1.570796326794896841275926616671059526825L, 2.220446049250313080847263336181677117148e-16L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1025L, 0x1p-52L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x1p-52L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x1p-52L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x1p-52L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, 0x1p-52L, 0x1.fp-1025L, 1.570796326794896397186716766608443357372L, -5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1p-52L, 0x1.fp-1025L, 1.570796326794896841275926616671059526825L, -5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, 0x1p-52L, -0x1.fp-1025L, 1.570796326794896397186716766608443357372L, 5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1p-52L, -0x1.fp-1025L, 1.570796326794896841275926616671059526825L, 5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacos, 0.0L, 0x1p-52L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, 0.0L, -0x1p-52L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, -0.0L, 0x1p-52L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, -0.0L, -0x1p-52L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (cacos, 0x1p-52L, 0.0L, 1.570796326794896397186716766608443357372L, -0.0L),
    TEST_c_c (cacos, -0x1p-52L, 0.0L, 1.570796326794896841275926616671059526825L, -0.0L),
    TEST_c_c (cacos, 0x1p-52L, -0.0L, 1.570796326794896397186716766608443357372L, 0.0L),
    TEST_c_c (cacos, -0x1p-52L, -0.0L, 1.570796326794896841275926616671059526825L, 0.0L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1025L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 64
    TEST_c_c (cacos, 0x1p-63L, 0x0.ffffffffffffffffp0L, 1.570796326794896619154657020805582738025L, -8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (cacos, 0x1p-63L, -0x0.ffffffffffffffffp0L, 1.570796326794896619154657020805582738025L, 8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (cacos, -0x1p-63L, 0x0.ffffffffffffffffp0L, 1.570796326794896619307986362473920146173L, -8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (cacos, -0x1p-63L, -0x0.ffffffffffffffffp0L, 1.570796326794896619307986362473920146173L, 8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, 0x1p-63L, 4.188407771167967636741951941902992986043e-10L, -2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, 0x1p-63L, 3.141592653170952461345846619605307690007L, -2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, -0x1p-63L, 4.188407771167967636741951941902992986043e-10L, 2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, -0x1p-63L, 3.141592653170952461345846619605307690007L, 2.588578361325995866221775673638805081337e-10L),
# if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 3.292722539913596233371825532007990724506e-10L, -2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 3.141592653260520984471283759942320330996L, -2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 3.292722539913596233371825532007990724506e-10L, 2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 3.141592653260520984471283759942320330996L, 2.472906068161537187835415298076415423459e-4923L),
# endif
    TEST_c_c (cacos, 0.0L, 0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, 0.0L, -0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, -0.0L, 0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, -0.0L, -0x0.ffffffffffffffffp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, 0.0L, 3.292722539913596233371825532007990724506e-10L, -0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, 0.0L, 3.141592653260520984471283759942320330996L, -0.0L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffp0L, -0.0L, 3.292722539913596233371825532007990724506e-10L, 0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffp0L, -0.0L, 3.141592653260520984471283759942320330996L, 0.0L),
#endif
    TEST_c_c (cacos, 0x1p-63L, 0.5L, 1.570796326794896619134347701278529840650L, -4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (cacos, 0x1p-63L, -0.5L, 1.570796326794896619134347701278529840650L, 4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (cacos, -0x1p-63L, 0.5L, 1.570796326794896619328295682000973043547L, -4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (cacos, -0x1p-63L, -0.5L, 1.570796326794896619328295682000973043547L, 4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (cacos, 0.5L, 0x1p-63L, 1.047197551196597746154214461093167628070L, -1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (cacos, -0.5L, 0x1p-63L, 2.094395102393195492308428922186335256127L, -1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (cacos, 0.5L, -0x1p-63L, 1.047197551196597746154214461093167628070L, 1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (cacos, -0.5L, -0x1p-63L, 2.094395102393195492308428922186335256127L, 1.251928832280966098829878787499365228659e-19L),
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0.5L, 1.570796326794896619231321691639751442099L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0.5L, 1.570796326794896619231321691639751442099L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0.5L, 1.570796326794896619231321691639751442099L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0.5L, 1.570796326794896619231321691639751442099L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0.5L, 0x1.fp-16385L, 1.047197551196597746154214461093167628066L, -9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0.5L, 0x1.fp-16385L, 2.094395102393195492308428922186335256131L, -9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, 0.5L, -0x1.fp-16385L, 1.047197551196597746154214461093167628066L, 9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0.5L, -0x1.fp-16385L, 2.094395102393195492308428922186335256131L, 9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacos, 0x1p-63L, 0x1p-63L, 1.570796326794896619122901474391200998698L, -1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (cacos, 0x1p-63L, -0x1p-63L, 1.570796326794896619122901474391200998698L, 1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (cacos, -0x1p-63L, 0x1p-63L, 1.570796326794896619339741908888301885499L, -1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (cacos, -0x1p-63L, -0x1p-63L, 1.570796326794896619339741908888301885499L, 1.084202172485504434007452800869941711430e-19L),
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0x1p-63L, 1.570796326794896619231321691639751442099L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x1p-63L, 1.570796326794896619231321691639751442099L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x1p-63L, 1.570796326794896619231321691639751442099L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x1p-63L, 1.570796326794896619231321691639751442099L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, 0x1p-63L, 0x1.fp-16385L, 1.570796326794896619122901474391200998698L, -8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1p-63L, 0x1.fp-16385L, 1.570796326794896619339741908888301885499L, -8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, 0x1p-63L, -0x1.fp-16385L, 1.570796326794896619122901474391200998698L, 8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1p-63L, -0x1.fp-16385L, 1.570796326794896619339741908888301885499L, 8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacos, 0.0L, 0x1p-63L, 1.570796326794896619231321691639751442099L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, 0.0L, -0x1p-63L, 1.570796326794896619231321691639751442099L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, -0.0L, 0x1p-63L, 1.570796326794896619231321691639751442099L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, -0.0L, -0x1p-63L, 1.570796326794896619231321691639751442099L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (cacos, 0x1p-63L, 0.0L, 1.570796326794896619122901474391200998698L, -0.0L),
    TEST_c_c (cacos, -0x1p-63L, 0.0L, 1.570796326794896619339741908888301885499L, -0.0L),
    TEST_c_c (cacos, 0x1p-63L, -0.0L, 1.570796326794896619122901474391200998698L, 0.0L),
    TEST_c_c (cacos, -0x1p-63L, -0.0L, 1.570796326794896619339741908888301885499L, 0.0L),
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
#if MANT_DIG >= 106
    TEST_c_c (cacos, 0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639734010571L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639734010571L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639768873627L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639768873627L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 1.997188458999618357060632049675702684031e-16L, -1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 3.141592653589793038743797483317667178134L, -1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 1.997188458999618357060632049675702684031e-16L, 1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 3.141592653589793038743797483317667178134L, 1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (cacos, 0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 1.570092458683775059393382371963565104164e-16L, -3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 3.141592653589793081453397514901996944859L, -3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 1.570092458683775059393382371963565104164e-16L, 3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 3.141592653589793081453397514901996944859L, 3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacos, 0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, -0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 1.570092458683775059393382371963565104164e-16L, -0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 3.141592653589793081453397514901996944859L, -0.0L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 1.570092458683775059393382371963565104164e-16L, 0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 3.141592653589793081453397514901996944859L, 0.0L),
#endif
    TEST_c_c (cacos, 0x1p-105L, 0.5L, 1.570796326794896619231321691639729392766L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0x1p-105L, -0.5L, 1.570796326794896619231321691639729392766L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1p-105L, 0.5L, 1.570796326794896619231321691639773491431L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1p-105L, -0.5L, 1.570796326794896619231321691639773491431L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0.5L, 0x1p-105L, 1.047197551196597746154214461093167628066L, -2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (cacos, -0.5L, 0x1p-105L, 2.094395102393195492308428922186335256131L, -2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (cacos, 0.5L, -0x1p-105L, 1.047197551196597746154214461093167628066L, 2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (cacos, -0.5L, -0x1p-105L, 2.094395102393195492308428922186335256131L, 2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (cacos, 0x1p-105L, 0x1p-105L, 1.570796326794896619231321691639726790195L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, 0x1p-105L, -0x1p-105L, 1.570796326794896619231321691639726790195L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0x1p-105L, 0x1p-105L, 1.570796326794896619231321691639776094002L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0x1p-105L, -0x1p-105L, 1.570796326794896619231321691639776094002L, 2.465190328815661891911651766508706967729e-32L),
#if !TEST_COND_binary32
    TEST_c_c (cacos, 0x1.fp-1025L, 0x1p-105L, 1.570796326794896619231321691639751442099L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, 0x1.fp-1025L, -0x1p-105L, 1.570796326794896619231321691639751442099L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0x1.fp-1025L, 0x1p-105L, 1.570796326794896619231321691639751442099L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0x1.fp-1025L, -0x1p-105L, 1.570796326794896619231321691639751442099L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, 0x1p-105L, 0x1.fp-1025L, 1.570796326794896619231321691639726790195L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1p-105L, 0x1.fp-1025L, 1.570796326794896619231321691639776094002L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, 0x1p-105L, -0x1.fp-1025L, 1.570796326794896619231321691639726790195L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacos, -0x1p-105L, -0x1.fp-1025L, 1.570796326794896619231321691639776094002L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacos, 0.0L, 0x1p-105L, 1.570796326794896619231321691639751442099L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, 0.0L, -0x1p-105L, 1.570796326794896619231321691639751442099L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0.0L, 0x1p-105L, 1.570796326794896619231321691639751442099L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, -0.0L, -0x1p-105L, 1.570796326794896619231321691639751442099L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (cacos, 0x1p-105L, 0.0L, 1.570796326794896619231321691639726790195L, -0.0L),
    TEST_c_c (cacos, -0x1p-105L, 0.0L, 1.570796326794896619231321691639776094002L, -0.0L),
    TEST_c_c (cacos, 0x1p-105L, -0.0L, 1.570796326794896619231321691639726790195L, 0.0L),
    TEST_c_c (cacos, -0x1p-105L, -0.0L, 1.570796326794896619231321691639776094002L, 0.0L),
#if MANT_DIG >= 113
    TEST_c_c (cacos, 0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751305915L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751305915L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751578282L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751578282L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 1.765281878332676474347460257221369818872e-17L, -1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 3.141592653589793220809824599952738140723L, -1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 1.765281878332676474347460257221369818872e-17L, 1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 3.141592653589793220809824599952738140723L, 1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (cacos, 0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 1.387778780781445675529539585113525401762e-17L, -5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 3.141592653589793224584855575465046128902L, -5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 1.387778780781445675529539585113525401762e-17L, 5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 3.141592653589793224584855575465046128902L, 5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (cacos, 0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, -0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.570796326794896619231321691639751442099L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 1.387778780781445675529539585113525401762e-17L, -0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 3.141592653589793224584855575465046128902L, -0.0L),
    TEST_c_c (cacos, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 1.387778780781445675529539585113525401762e-17L, 0.0L),
    TEST_c_c (cacos, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 3.141592653589793224584855575465046128902L, 0.0L),
#endif
    TEST_c_c (cacos, 0x1p-112L, 0.5L, 1.570796326794896619231321691639751269838L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0x1p-112L, -0.5L, 1.570796326794896619231321691639751269838L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1p-112L, 0.5L, 1.570796326794896619231321691639751614359L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, -0x1p-112L, -0.5L, 1.570796326794896619231321691639751614359L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (cacos, 0.5L, 0x1p-112L, 1.047197551196597746154214461093167628066L, -2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (cacos, -0.5L, 0x1p-112L, 2.094395102393195492308428922186335256131L, -2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (cacos, 0.5L, -0x1p-112L, 1.047197551196597746154214461093167628066L, 2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (cacos, -0.5L, -0x1p-112L, 2.094395102393195492308428922186335256131L, 2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (cacos, 0x1p-112L, 0x1p-112L, 1.570796326794896619231321691639751249506L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, 0x1p-112L, -0x1p-112L, 1.570796326794896619231321691639751249506L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0x1p-112L, 0x1p-112L, 1.570796326794896619231321691639751634692L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0x1p-112L, -0x1p-112L, 1.570796326794896619231321691639751634692L, 1.925929944387235853055977942584927318538e-34L),
#if MIN_EXP <= -16381
    TEST_c_c (cacos, 0x1.fp-16385L, 0x1p-112L, 1.570796326794896619231321691639751442099L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, 0x1.fp-16385L, -0x1p-112L, 1.570796326794896619231321691639751442099L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0x1.fp-16385L, 0x1p-112L, 1.570796326794896619231321691639751442099L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0x1.fp-16385L, -0x1p-112L, 1.570796326794896619231321691639751442099L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, 0x1p-112L, 0x1.fp-16385L, 1.570796326794896619231321691639751249506L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1p-112L, 0x1.fp-16385L, 1.570796326794896619231321691639751634692L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, 0x1p-112L, -0x1.fp-16385L, 1.570796326794896619231321691639751249506L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacos, -0x1p-112L, -0x1.fp-16385L, 1.570796326794896619231321691639751634692L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacos, 0.0L, 0x1p-112L, 1.570796326794896619231321691639751442099L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, 0.0L, -0x1p-112L, 1.570796326794896619231321691639751442099L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0.0L, 0x1p-112L, 1.570796326794896619231321691639751442099L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, -0.0L, -0x1p-112L, 1.570796326794896619231321691639751442099L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (cacos, 0x1p-112L, 0.0L, 1.570796326794896619231321691639751249506L, -0.0L),
    TEST_c_c (cacos, -0x1p-112L, 0.0L, 1.570796326794896619231321691639751634692L, -0.0L),
    TEST_c_c (cacos, 0x1p-112L, -0.0L, 1.570796326794896619231321691639751249506L, 0.0L),
    TEST_c_c (cacos, -0x1p-112L, -0.0L, 1.570796326794896619231321691639751634692L, 0.0L),

    TEST_c_c (cacos, 0.75L, 1.25L, 1.11752014915610270578240049553777969L, -1.13239363160530819522266333696834467L),
    TEST_c_c (cacos, -2, -3, 2.1414491111159960199416055713254211L, 1.9833870299165354323470769028940395L),
  };

static void
cacos_test (void)
{
  ALL_RM_TEST (cacos, 0, cacos_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}

static const struct test_c_c_data cacosh_test_data[] =
  {
    TEST_c_c (cacosh, 0, 0, 0.0, lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, 0, 0.0, lit_pi_2_d),
    TEST_c_c (cacosh, 0, minus_zero, 0.0, -lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, minus_zero, 0.0, -lit_pi_2_d),
    TEST_c_c (cacosh, minus_infty, plus_infty, plus_infty, lit_pi_3_m_4_d),
    TEST_c_c (cacosh, minus_infty, minus_infty, plus_infty, -lit_pi_3_m_4_d),

    TEST_c_c (cacosh, plus_infty, plus_infty, plus_infty, lit_pi_4_d),
    TEST_c_c (cacosh, plus_infty, minus_infty, plus_infty, -lit_pi_4_d),

    TEST_c_c (cacosh, -10.0, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (cacosh, -10.0, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (cacosh, 0, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (cacosh, 0, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (cacosh, 0.1L, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (cacosh, 0.1L, minus_infty, plus_infty, -lit_pi_2_d),

    TEST_c_c (cacosh, minus_infty, 0, plus_infty, lit_pi),
    TEST_c_c (cacosh, minus_infty, minus_zero, plus_infty, -lit_pi),
    TEST_c_c (cacosh, minus_infty, 100, plus_infty, lit_pi),
    TEST_c_c (cacosh, minus_infty, -100, plus_infty, -lit_pi),

    TEST_c_c (cacosh, plus_infty, 0, plus_infty, 0.0),
    TEST_c_c (cacosh, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (cacosh, plus_infty, 0.5, plus_infty, 0.0),
    TEST_c_c (cacosh, plus_infty, -0.5, plus_infty, minus_zero),

    TEST_c_c (cacosh, plus_infty, qnan_value, plus_infty, qnan_value),
    TEST_c_c (cacosh, minus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (cacosh, 0, qnan_value, qnan_value, qnan_value),
    TEST_c_c (cacosh, minus_zero, qnan_value, qnan_value, qnan_value),

    TEST_c_c (cacosh, qnan_value, plus_infty, plus_infty, qnan_value),
    TEST_c_c (cacosh, qnan_value, minus_infty, plus_infty, qnan_value),

    TEST_c_c (cacosh, 10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cacosh, -10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (cacosh, qnan_value, 0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cacosh, qnan_value, -0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (cacosh, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (cacosh, plus_zero, -1.5L, 1.194763217287109304111930828519090523536L, -lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, -1.5L, 1.194763217287109304111930828519090523536L, -lit_pi_2_d),
    TEST_c_c (cacosh, plus_zero, -1.0L, 0.8813735870195430252326093249797923090282L, -lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, -1.0L, 0.8813735870195430252326093249797923090282L, -lit_pi_2_d),
    TEST_c_c (cacosh, plus_zero, -0.5L, 0.4812118250596034474977589134243684231352L, -lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, -0.5L, 0.4812118250596034474977589134243684231352L, -lit_pi_2_d),
    TEST_c_c (cacosh, plus_zero, 0.5L, 0.4812118250596034474977589134243684231352L, lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, 0.5L, 0.4812118250596034474977589134243684231352L, lit_pi_2_d),
    TEST_c_c (cacosh, plus_zero, 1.0L, 0.8813735870195430252326093249797923090282L, lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, 1.0L, 0.8813735870195430252326093249797923090282L, lit_pi_2_d),
    TEST_c_c (cacosh, plus_zero, 1.5L, 1.194763217287109304111930828519090523536L, lit_pi_2_d),
    TEST_c_c (cacosh, minus_zero, 1.5L, 1.194763217287109304111930828519090523536L, lit_pi_2_d),

    TEST_c_c (cacosh, -1.5L, plus_zero, 0.9624236501192068949955178268487368462704L, lit_pi),
    TEST_c_c (cacosh, -1.5L, minus_zero, 0.9624236501192068949955178268487368462704L, -lit_pi),
    TEST_c_c (cacosh, -1.0L, plus_zero, plus_zero, lit_pi),
    TEST_c_c (cacosh, -1.0L, minus_zero, plus_zero, -lit_pi),
    TEST_c_c (cacosh, -0.5L, plus_zero, plus_zero, 2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, -0.5L, minus_zero, plus_zero, -2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, 0.5L, plus_zero, plus_zero, 1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, 0.5L, minus_zero, plus_zero, -1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, 1.0L, plus_zero, plus_zero, plus_zero),
    TEST_c_c (cacosh, 1.0L, minus_zero, plus_zero, minus_zero),
    TEST_c_c (cacosh, 1.5L, plus_zero, 0.9624236501192068949955178268487368462704L, plus_zero),
    TEST_c_c (cacosh, 1.5L, minus_zero, 0.9624236501192068949955178268487368462704L, minus_zero),

    TEST_c_c (cacosh, 0x1p50L, 1.0L, 3.535050620855721078027883819436720218708e1L, 8.881784197001252323389053344727730248720e-16L),
    TEST_c_c (cacosh, 0x1p50L, -1.0L, 3.535050620855721078027883819436720218708e1L, -8.881784197001252323389053344727730248720e-16L),
    TEST_c_c (cacosh, -0x1p50L, 1.0L, 3.535050620855721078027883819436720218708e1L, 3.141592653589792350284223683154270545292L),
    TEST_c_c (cacosh, -0x1p50L, -1.0L, 3.535050620855721078027883819436720218708e1L, -3.141592653589792350284223683154270545292L),
    TEST_c_c (cacosh, 1.0L, 0x1p50L, 3.535050620855721078027883819436759661753e1L, 1.570796326794895731052901991514519103193L),
    TEST_c_c (cacosh, -1.0L, 0x1p50L, 3.535050620855721078027883819436759661753e1L, 1.570796326794897507409741391764983781004L),
    TEST_c_c (cacosh, 1.0L, -0x1p50L, 3.535050620855721078027883819436759661753e1L, -1.570796326794895731052901991514519103193L),
    TEST_c_c (cacosh, -1.0L, -0x1p50L, 3.535050620855721078027883819436759661753e1L, -1.570796326794897507409741391764983781004L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1p500L, 1.0L, 3.472667374605326000180332928505464606058e2L, 3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (cacosh, 0x1p500L, -1.0L, 3.472667374605326000180332928505464606058e2L, -3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (cacosh, -0x1p500L, 1.0L, 3.472667374605326000180332928505464606058e2L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, -0x1p500L, -1.0L, 3.472667374605326000180332928505464606058e2L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, 0x1p500L, 3.472667374605326000180332928505464606058e2L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -1.0L, 0x1p500L, 3.472667374605326000180332928505464606058e2L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, -0x1p500L, 3.472667374605326000180332928505464606058e2L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -1.0L, -0x1p500L, 3.472667374605326000180332928505464606058e2L, -1.570796326794896619231321691639751442099L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (cacosh, 0x1p5000L, 1.0L, 3.466429049980286492395577839412341016946e3L, 7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (cacosh, 0x1p5000L, -1.0L, 3.466429049980286492395577839412341016946e3L, -7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (cacosh, -0x1p5000L, 1.0L, 3.466429049980286492395577839412341016946e3L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, -0x1p5000L, -1.0L, 3.466429049980286492395577839412341016946e3L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, 0x1p5000L, 3.466429049980286492395577839412341016946e3L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -1.0L, 0x1p5000L, 3.466429049980286492395577839412341016946e3L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, -0x1p5000L, 3.466429049980286492395577839412341016946e3L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -1.0L, -0x1p5000L, 3.466429049980286492395577839412341016946e3L, -1.570796326794896619231321691639751442099L),
#endif
    TEST_c_c (cacosh, 0x1.fp127L, 0x1.fp127L, 8.973081118419833726837456344608533993585e1L, 7.853981633974483096156608458198757210493e-1L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp1023L, 0x1.fp1023L, 7.107906849659093345062145442726115449315e2L, 7.853981633974483096156608458198757210493e-1L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (cacosh, 0x1.fp16383L, 0x1.fp16383L, 1.135753137836666928715489992987020363057e4L, 7.853981633974483096156608458198757210493e-1L),
#endif
    TEST_c_c (cacosh, 0x1.fp-129L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, 0x1.fp-129L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, -0x1.fp-129L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442100L),
    TEST_c_c (cacosh, -0x1.fp-129L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442100L),
    TEST_c_c (cacosh, 1.5L, 0x1.fp-129L, 9.624236501192068949955178268487368462704e-1L, 2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -1.5L, 0x1.fp-129L, 9.624236501192068949955178268487368462704e-1L, 3.141592653589793238462643383279502884195L),
    TEST_c_c (cacosh, 1.5L, -0x1.fp-129L, 9.624236501192068949955178268487368462704e-1L, -2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -1.5L, -0x1.fp-129L, 9.624236501192068949955178268487368462704e-1L, -3.141592653589793238462643383279502884195L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1025L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.5L, 0x1.fp-1025L, 9.624236501192068949955178268487368462704e-1L, 4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -1.5L, 0x1.fp-1025L, 9.624236501192068949955178268487368462704e-1L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.5L, -0x1.fp-1025L, 9.624236501192068949955178268487368462704e-1L, -4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -1.5L, -0x1.fp-1025L, 9.624236501192068949955178268487368462704e-1L, -3.141592653589793238462643383279502884197L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 1.5L, 1.194763217287109304111930828519090523536L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -1.5L, 1.194763217287109304111930828519090523536L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.5L, 0x1.fp-16385L, 9.624236501192068949955178268487368462704e-1L, 7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -1.5L, 0x1.fp-16385L, 9.624236501192068949955178268487368462704e-1L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.5L, -0x1.fp-16385L, 9.624236501192068949955178268487368462704e-1L, -7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -1.5L, -0x1.fp-16385L, 9.624236501192068949955178268487368462704e-1L, -3.141592653589793238462643383279502884197L),
#endif
    TEST_c_c (cacosh, 0.5L, 1.0L, 9.261330313501824245501244453057873152694e-1L, 1.221357263937683325603909865564381489366L),
    TEST_c_c (cacosh, 0.5L, -1.0L, 9.261330313501824245501244453057873152694e-1L, -1.221357263937683325603909865564381489366L),
    TEST_c_c (cacosh, -0.5L, 1.0L, 9.261330313501824245501244453057873152694e-1L, 1.920235389652109912858733517715121394831L),
    TEST_c_c (cacosh, -0.5L, -1.0L, 9.261330313501824245501244453057873152694e-1L, -1.920235389652109912858733517715121394831L),
    TEST_c_c (cacosh, 1.0L, 0.5L, 7.328576759736452608886724437653071523305e-1L, 6.748888455860063801646649673121744318756e-1L),
    TEST_c_c (cacosh, -1.0L, 0.5L, 7.328576759736452608886724437653071523305e-1L, 2.466703808003786858297978415967328452322L),
    TEST_c_c (cacosh, 1.0L, -0.5L, 7.328576759736452608886724437653071523305e-1L, -6.748888455860063801646649673121744318756e-1L),
    TEST_c_c (cacosh, -1.0L, -0.5L, 7.328576759736452608886724437653071523305e-1L, -2.466703808003786858297978415967328452322L),
    TEST_c_c (cacosh, 0.25L, 1.0L, 8.924633639033482359562124741744951972772e-1L, 1.394493894017929688812643125003661339452L),
    TEST_c_c (cacosh, 0.25L, -1.0L, 8.924633639033482359562124741744951972772e-1L, -1.394493894017929688812643125003661339452L),
    TEST_c_c (cacosh, -0.25L, 1.0L, 8.924633639033482359562124741744951972772e-1L, 1.747098759571863549650000258275841544745L),
    TEST_c_c (cacosh, -0.25L, -1.0L, 8.924633639033482359562124741744951972772e-1L, -1.747098759571863549650000258275841544745L),
    TEST_c_c (cacosh, 1.0L, 0.25L, 5.097911466811016354623559941115413499164e-1L, 4.890443302710802929202843732146540079124e-1L),
    TEST_c_c (cacosh, -1.0L, 0.25L, 5.097911466811016354623559941115413499164e-1L, 2.652548323318712945542359010064848876285L),
    TEST_c_c (cacosh, 1.0L, -0.25L, 5.097911466811016354623559941115413499164e-1L, -4.890443302710802929202843732146540079124e-1L),
    TEST_c_c (cacosh, -1.0L, -0.25L, 5.097911466811016354623559941115413499164e-1L, -2.652548323318712945542359010064848876285L),
    TEST_c_c (cacosh, 0x1.fp-10L, 1.0L, 8.813742198809567991336704287826445879025e-1L, 1.569458417435338878318763342108699202986L),
    TEST_c_c (cacosh, 0x1.fp-10L, -1.0L, 8.813742198809567991336704287826445879025e-1L, -1.569458417435338878318763342108699202986L),
    TEST_c_c (cacosh, -0x1.fp-10L, 1.0L, 8.813742198809567991336704287826445879025e-1L, 1.572134236154454360143880041170803681211L),
    TEST_c_c (cacosh, -0x1.fp-10L, -1.0L, 8.813742198809567991336704287826445879025e-1L, -1.572134236154454360143880041170803681211L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-10L, 4.350501469856803800217957402220976497152e-2L, 4.349129763101882771258049954181971959031e-2L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-10L, 4.350501469856803800217957402220976497152e-2L, 3.098101355958774410750062883737683164607L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-10L, 4.350501469856803800217957402220976497152e-2L, -4.349129763101882771258049954181971959031e-2L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-10L, 4.350501469856803800217957402220976497152e-2L, -3.098101355958774410750062883737683164607L),
    TEST_c_c (cacosh, 0x1.fp-30L, 1.0L, 8.813735870195430258081932989769495326854e-1L, 1.570796325518966635014803151387033957091L),
    TEST_c_c (cacosh, 0x1.fp-30L, -1.0L, 8.813735870195430258081932989769495326854e-1L, -1.570796325518966635014803151387033957091L),
    TEST_c_c (cacosh, -0x1.fp-30L, 1.0L, 8.813735870195430258081932989769495326854e-1L, 1.570796328070826603447840231892468927106L),
    TEST_c_c (cacosh, -0x1.fp-30L, -1.0L, 8.813735870195430258081932989769495326854e-1L, -1.570796328070826603447840231892468927106L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-30L, 4.247867098745151888768727039216644758847e-5L, 4.247867097467650115899790787875186617316e-5L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-30L, 4.247867098745151888768727039216644758847e-5L, 3.141550174918818561961484385371624132331L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-30L, 4.247867098745151888768727039216644758847e-5L, -4.247867097467650115899790787875186617316e-5L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-30L, 4.247867098745151888768727039216644758847e-5L, -3.141550174918818561961484385371624132331L),
    TEST_c_c (cacosh, 0x1.fp-100L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691638670687364L),
    TEST_c_c (cacosh, 0x1.fp-100L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691638670687364L),
    TEST_c_c (cacosh, -0x1.fp-100L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691640832196834L),
    TEST_c_c (cacosh, -0x1.fp-100L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691640832196834L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-100L, 1.236292038260260888664514866457202186027e-15L, 1.236292038260260888664514866456887257525e-15L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-100L, 1.236292038260260888664514866457202186027e-15L, 3.141592653589792002170605123018614219682L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-100L, 1.236292038260260888664514866457202186027e-15L, -1.236292038260260888664514866456887257525e-15L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-100L, 1.236292038260260888664514866457202186027e-15L, -3.141592653589792002170605123018614219682L),
    TEST_c_c (cacosh, 0x1.fp-129L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, 0x1.fp-129L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, -0x1.fp-129L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, -0x1.fp-129L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-129L, 5.335635276982233498398987585285818977933e-20L, 5.335635276982233498398987585285818977930e-20L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-129L, 5.335635276982233498398987585285818977933e-20L, 3.141592653589793238409287030509680549213L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-129L, 5.335635276982233498398987585285818977933e-20L, -5.335635276982233498398987585285818977930e-20L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-129L, 5.335635276982233498398987585285818977933e-20L, -3.141592653589793238409287030509680549213L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1000L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1000L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1000L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1000L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, 4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, -4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-1000L, 4.252291453851660175550490409247739011867e-151L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.fp-1025L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, 7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, -7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-1025L, 7.340879205566679497036857179189356754017e-155L, -3.141592653589793238462643383279502884197L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-10000L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-10000L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-10000L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-10000L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, 9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, -9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-10000L, 9.854680208706673586644342922051388714633e-1506L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.fp-16385L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 1.0L, 8.813735870195430252326093249797923090282e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -1.0L, 8.813735870195430252326093249797923090282e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 1.0L, 0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, 9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (cacosh, -1.0L, 0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 1.0L, -0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, -9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (cacosh, -1.0L, -0x1.fp-16385L, 9.023632056840860275214893047597614177639e-2467L, -3.141592653589793238462643383279502884197L),
#endif
    TEST_c_c (cacosh, 0x1p-23L, 0x1.000002p0L, 8.813736713132400470205730751186547909968e-1L, 1.570796242501204621739026081991856762292L),
    TEST_c_c (cacosh, 0x1p-23L, -0x1.000002p0L, 8.813736713132400470205730751186547909968e-1L, -1.570796242501204621739026081991856762292L),
    TEST_c_c (cacosh, -0x1p-23L, 0x1.000002p0L, 8.813736713132400470205730751186547909968e-1L, 1.570796411088588616723617301287646121905L),
    TEST_c_c (cacosh, -0x1p-23L, -0x1.000002p0L, 8.813736713132400470205730751186547909968e-1L, -1.570796411088588616723617301287646121905L),
    TEST_c_c (cacosh, 0x1.000002p0L, 0x1p-23L, 5.364668491573609633134147164031476452679e-4L, 2.222118384408546368406374049167636760903e-4L),
    TEST_c_c (cacosh, -0x1.000002p0L, 0x1p-23L, 5.364668491573609633134147164031476452679e-4L, 3.141370441751352383825802745874586120521L),
    TEST_c_c (cacosh, 0x1.000002p0L, -0x1p-23L, 5.364668491573609633134147164031476452679e-4L, -2.222118384408546368406374049167636760903e-4L),
    TEST_c_c (cacosh, -0x1.000002p0L, -0x1p-23L, 5.364668491573609633134147164031476452679e-4L, -3.141370441751352383825802745874586120521L),
    TEST_c_c (cacosh, 0x1.fp-129L, 0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, 1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, 0x1.fp-129L, -0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, -1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, -0x1.fp-129L, 0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, 1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, -0x1.fp-129L, -0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, -1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, 0x1.000002p0L, 0x1.fp-129L, 4.882812451493617206486388134172712975070e-4L, 5.830451806317544230969669308596361881467e-36L),
    TEST_c_c (cacosh, -0x1.000002p0L, 0x1.fp-129L, 4.882812451493617206486388134172712975070e-4L, 3.141592653589793238462643383279502878367L),
    TEST_c_c (cacosh, 0x1.000002p0L, -0x1.fp-129L, 4.882812451493617206486388134172712975070e-4L, -5.830451806317544230969669308596361881467e-36L),
    TEST_c_c (cacosh, -0x1.000002p0L, -0x1.fp-129L, 4.882812451493617206486388134172712975070e-4L, -3.141592653589793238462643383279502878367L),
    TEST_c_c (cacosh, 0.0L, 0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1.000002p0L, 8.813736713132375348727889167749389235161e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.000002p0L, 0.0L, 4.882812451493617206486388134172712975070e-4L, 0.0L),
    TEST_c_c (cacosh, -0x1.000002p0L, 0.0L, 4.882812451493617206486388134172712975070e-4L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.000002p0L, -0.0L, 4.882812451493617206486388134172712975070e-4L, -0.0L),
    TEST_c_c (cacosh, -0x1.000002p0L, -0.0L, 4.882812451493617206486388134172712975070e-4L, -3.141592653589793238462643383279502884197L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1p-52L, 0x1.0000000000001p0L, 8.813735870195431822418551933572982483664e-1L, 1.570796326794896462222075823262262934288L),
    TEST_c_c (cacosh, 0x1p-52L, -0x1.0000000000001p0L, 8.813735870195431822418551933572982483664e-1L, -1.570796326794896462222075823262262934288L),
    TEST_c_c (cacosh, -0x1p-52L, 0x1.0000000000001p0L, 8.813735870195431822418551933572982483664e-1L, 1.570796326794896776240567560017239949909L),
    TEST_c_c (cacosh, -0x1p-52L, -0x1.0000000000001p0L, 8.813735870195431822418551933572982483664e-1L, -1.570796326794896776240567560017239949909L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, 0x1p-52L, 2.315303644582684770975188768022139415020e-8L, 9.590301705980041385828904092662391018164e-9L),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, 0x1p-52L, 2.315303644582684770975188768022139415020e-8L, 3.141592643999491532482601997450598791535L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, -0x1p-52L, 2.315303644582684770975188768022139415020e-8L, -9.590301705980041385828904092662391018164e-9L),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, -0x1p-52L, 2.315303644582684770975188768022139415020e-8L, -3.141592643999491532482601997450598791535L),
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, 0x1.fp-1025L, 2.107342425544701550354780375182800088393e-8L, 2.557178503953494342609835913586108008322e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, 0x1.fp-1025L, 2.107342425544701550354780375182800088393e-8L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, -0x1.fp-1025L, 2.107342425544701550354780375182800088393e-8L, -2.557178503953494342609835913586108008322e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, -0x1.fp-1025L, 2.107342425544701550354780375182800088393e-8L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0.0L, 0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1.0000000000001p0L, 8.813735870195431822418551933572895326024e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, 0.0L, 2.107342425544701550354780375182800088393e-8L, 0.0L),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, 0.0L, 2.107342425544701550354780375182800088393e-8L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000001p0L, -0.0L, 2.107342425544701550354780375182800088393e-8L, -0.0L),
    TEST_c_c (cacosh, -0x1.0000000000001p0L, -0.0L, 2.107342425544701550354780375182800088393e-8L, -3.141592653589793238462643383279502884197L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (cacosh, 0x1p-63L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610131001e-1L, 1.570796326794896619154657020805582738031L),
    TEST_c_c (cacosh, 0x1p-63L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610131001e-1L, -1.570796326794896619154657020805582738031L),
    TEST_c_c (cacosh, -0x1p-63L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610131001e-1L, 1.570796326794896619307986362473920146166L),
    TEST_c_c (cacosh, -0x1p-63L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610131001e-1L, -1.570796326794896619307986362473920146166L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, 0x1p-63L, 5.116146586219826555037807251857670783420e-10L, 2.119177303101063432592523199680782317447e-10L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, 0x1p-63L, 5.116146586219826555037807251857670783420e-10L, 3.141592653377875508152537040020250564229L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, -0x1p-63L, 5.116146586219826555037807251857670783420e-10L, -2.119177303101063432592523199680782317447e-10L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, -0x1p-63L, 5.116146586219826555037807251857670783420e-10L, -3.141592653377875508152537040020250564229L),
# if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, 0x1.fp-16385L, 4.656612873077392578082927418388212703712e-10L, 1.748608650034385653922359120438227480943e-4923L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, 0x1.fp-16385L, 4.656612873077392578082927418388212703712e-10L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, -0x1.fp-16385L, 4.656612873077392578082927418388212703712e-10L, -1.748608650034385653922359120438227480943e-4923L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, -0x1.fp-16385L, 4.656612873077392578082927418388212703712e-10L, -3.141592653589793238462643383279502884197L),
# endif
    TEST_c_c (cacosh, 0.0L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1.0000000000000002p0L, 8.813735870195430253092739958139610130980e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, 0.0L, 4.656612873077392578082927418388212703712e-10L, 0.0L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, 0.0L, 4.656612873077392578082927418388212703712e-10L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000000002p0L, -0.0L, 4.656612873077392578082927418388212703712e-10L, -0.0L),
    TEST_c_c (cacosh, -0x1.0000000000000002p0L, -0.0L, 4.656612873077392578082927418388212703712e-10L, -3.141592653589793238462643383279502884197L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (cacosh, 0x1p-106L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639742726335L),
    TEST_c_c (cacosh, 0x1p-106L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639742726335L),
    TEST_c_c (cacosh, -0x1p-106L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639760157863L),
    TEST_c_c (cacosh, -0x1p-106L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639760157863L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, 0x1p-106L, 2.285028863093221674154232933662774454211e-16L, 5.394221422390606848017034778914096659726e-17L),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, 0x1p-106L, 2.285028863093221674154232933662774454211e-16L, 3.141592653589793184520429159373434404027L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, -0x1p-106L, 2.285028863093221674154232933662774454211e-16L, -5.394221422390606848017034778914096659726e-17L),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, -0x1p-106L, 2.285028863093221674154232933662774454211e-16L, -3.141592653589793184520429159373434404027L),
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 2.220446049250313080847263336181636063482e-16L, 2.426922623448365473354662093431821897807e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 2.220446049250313080847263336181636063482e-16L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 2.220446049250313080847263336181636063482e-16L, -2.426922623448365473354662093431821897807e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 2.220446049250313080847263336181636063482e-16L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0.0L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1.000000000000000000000000008p0L, 8.813735870195430252326093249798097405561e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, 0.0L, 2.220446049250313080847263336181636063482e-16L, 0.0L),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, 0.0L, 2.220446049250313080847263336181636063482e-16L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.000000000000000000000000008p0L, -0.0L, 2.220446049250313080847263336181636063482e-16L, -0.0L),
    TEST_c_c (cacosh, -0x1.000000000000000000000000008p0L, -0.0L, 2.220446049250313080847263336181636063482e-16L, -3.141592653589793238462643383279502884197L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (cacosh, 0x1p-113L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751374007L),
    TEST_c_c (cacosh, 0x1p-113L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751374007L),
    TEST_c_c (cacosh, -0x1p-113L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751510190L),
    TEST_c_c (cacosh, -0x1p-113L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751510190L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, 0x1p-113L, 2.019699255375255198156433710951064632386e-17L, 4.767863183742677481693563511435642755521e-18L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, 0x1p-113L, 2.019699255375255198156433710951064632386e-17L, 3.141592653589793233694780199536825402504L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, -0x1p-113L, 2.019699255375255198156433710951064632386e-17L, -4.767863183742677481693563511435642755521e-18L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, -0x1p-113L, 2.019699255375255198156433710951064632386e-17L, -3.141592653589793233694780199536825402504L),
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 1.962615573354718824241727964954454332780e-17L, 4.148847925325683229178506116378864361396e-4916L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 1.962615573354718824241727964954454332780e-17L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 1.962615573354718824241727964954454332780e-17L, -4.148847925325683229178506116378864361396e-4916L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 1.962615573354718824241727964954454332780e-17L, -3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0.0L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1.0000000000000000000000000001p0L, 8.813735870195430252326093249797924452120e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, 0.0L, 1.962615573354718824241727964954454332780e-17L, 0.0L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, 0.0L, 1.962615573354718824241727964954454332780e-17L, 3.141592653589793238462643383279502884197L),
    TEST_c_c (cacosh, 0x1.0000000000000000000000000001p0L, -0.0L, 1.962615573354718824241727964954454332780e-17L, -0.0L),
    TEST_c_c (cacosh, -0x1.0000000000000000000000000001p0L, -0.0L, 1.962615573354718824241727964954454332780e-17L, -3.141592653589793238462643383279502884197L),
#endif
    TEST_c_c (cacosh, 0x1p-23L, 0x0.ffffffp0L, 8.813735448726963984495965873956465777250e-1L, 1.570796242501197085295336701632142060969L),
    TEST_c_c (cacosh, 0x1p-23L, -0x0.ffffffp0L, 8.813735448726963984495965873956465777250e-1L, -1.570796242501197085295336701632142060969L),
    TEST_c_c (cacosh, -0x1p-23L, 0x0.ffffffp0L, 8.813735448726963984495965873956465777250e-1L, 1.570796411088596153167306681647360823228L),
    TEST_c_c (cacosh, -0x1p-23L, -0x0.ffffffp0L, 8.813735448726963984495965873956465777250e-1L, -1.570796411088596153167306681647360823228L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, 0x1p-23L, 2.714321200917194650737217746780928423385e-4L, 4.391863861910487109445187743978204002614e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, 0x1p-23L, 2.714321200917194650737217746780928423385e-4L, 3.141153467203602189751698864505105063797L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, -0x1p-23L, 2.714321200917194650737217746780928423385e-4L, -4.391863861910487109445187743978204002614e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, -0x1p-23L, 2.714321200917194650737217746780928423385e-4L, -3.141153467203602189751698864505105063797L),
    TEST_c_c (cacosh, 0x1.fp-129L, 0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, 1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, 0x1.fp-129L, -0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, -1.570796326794896619231321691639751442097L),
    TEST_c_c (cacosh, -0x1.fp-129L, 0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, 1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, -0x1.fp-129L, -0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, -1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, 0x1.fp-129L, 8.245504387859737323891843603996428534945e-36L, 3.452669847162035876032494826387364972849e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, 0x1.fp-129L, 8.245504387859737323891843603996428534945e-36L, 3.141247386605077034875040133796864147700L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, -0x1.fp-129L, 8.245504387859737323891843603996428534945e-36L, -3.452669847162035876032494826387364972849e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, -0x1.fp-129L, 8.245504387859737323891843603996428534945e-36L, -3.141247386605077034875040133796864147700L),
    TEST_c_c (cacosh, 0.0L, 0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x0.ffffffp0L, 8.813735448726938863015878255140556727969e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, 0.0L, 0.0L, 3.452669847162035876032494826387364972849e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, 0.0L, 0.0L, 3.141247386605077034875040133796864147700L),
    TEST_c_c (cacosh, 0x0.ffffffp0L, -0.0L, 0.0L, -3.452669847162035876032494826387364972849e-4L),
    TEST_c_c (cacosh, -0x0.ffffffp0L, -0.0L, 0.0L, -3.141247386605077034875040133796864147700L),
    TEST_c_c (cacosh, 0x1p-23L, 0.5L, 4.812118250596059896127318386463676808798e-1L, 1.570796220170866625230343643673321150378L),
    TEST_c_c (cacosh, 0x1p-23L, -0.5L, 4.812118250596059896127318386463676808798e-1L, -1.570796220170866625230343643673321150378L),
    TEST_c_c (cacosh, -0x1p-23L, 0.5L, 4.812118250596059896127318386463676808798e-1L, 1.570796433418926613232299739606181733819L),
    TEST_c_c (cacosh, -0x1p-23L, -0.5L, 4.812118250596059896127318386463676808798e-1L, -1.570796433418926613232299739606181733819L),
    TEST_c_c (cacosh, 0.5L, 0x1p-23L, 1.376510308240943236356088341381173571841e-7L, 1.047197551196603215914744618665204532273L),
    TEST_c_c (cacosh, -0.5L, 0x1p-23L, 1.376510308240943236356088341381173571841e-7L, 2.094395102393190022547898764614298351924L),
    TEST_c_c (cacosh, 0.5L, -0x1p-23L, 1.376510308240943236356088341381173571841e-7L, -1.047197551196603215914744618665204532273L),
    TEST_c_c (cacosh, -0.5L, -0x1p-23L, 1.376510308240943236356088341381173571841e-7L, -2.094395102393190022547898764614298351924L),
    TEST_c_c (cacosh, 0x1.fp-129L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (cacosh, 0x1.fp-129L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (cacosh, -0x1.fp-129L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, -0x1.fp-129L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, 0.5L, 0x1.fp-129L, 3.287317402534702257036015056278368274737e-39L, 1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0.5L, 0x1.fp-129L, 3.287317402534702257036015056278368274737e-39L, 2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, 0.5L, -0x1.fp-129L, 3.287317402534702257036015056278368274737e-39L, -1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0.5L, -0x1.fp-129L, 3.287317402534702257036015056278368274737e-39L, -2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, 0x1p-23L, 0x1p-23L, 1.192092895507818146886315028596704749235e-7L, 1.570796207585607068450636380271254316214L),
    TEST_c_c (cacosh, 0x1p-23L, -0x1p-23L, 1.192092895507818146886315028596704749235e-7L, -1.570796207585607068450636380271254316214L),
    TEST_c_c (cacosh, -0x1p-23L, 0x1p-23L, 1.192092895507818146886315028596704749235e-7L, 1.570796446004186170012007003008248567984L),
    TEST_c_c (cacosh, -0x1p-23L, -0x1p-23L, 1.192092895507818146886315028596704749235e-7L, -1.570796446004186170012007003008248567984L),
    TEST_c_c (cacosh, 0x1.fp-129L, 0x1p-23L, 1.192092895507809676556842485683592032154e-7L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (cacosh, 0x1.fp-129L, -0x1p-23L, 1.192092895507809676556842485683592032154e-7L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (cacosh, -0x1.fp-129L, 0x1p-23L, 1.192092895507809676556842485683592032154e-7L, 1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, -0x1.fp-129L, -0x1p-23L, 1.192092895507809676556842485683592032154e-7L, -1.570796326794896619231321691639751442101L),
    TEST_c_c (cacosh, 0x1p-23L, 0x1.fp-129L, 2.846900380897747786805634596726756660388e-39L, 1.570796207585607068449789347324000006847L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0x1p-23L, 0x1.fp-129L, 2.846900380897747786805634596726756660388e-39L, 1.570796446004186170012854035955502877351L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, 0x1p-23L, -0x1.fp-129L, 2.846900380897747786805634596726756660388e-39L, -1.570796207585607068449789347324000006847L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0x1p-23L, -0x1.fp-129L, 2.846900380897747786805634596726756660388e-39L, -1.570796446004186170012854035955502877351L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, 0.0L, 0x1p-23L, 1.192092895507809676556842485683592032154e-7L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1p-23L, 1.192092895507809676556842485683592032154e-7L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1p-23L, 1.192092895507809676556842485683592032154e-7L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1p-23L, 1.192092895507809676556842485683592032154e-7L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-23L, 0.0L, 0.0L, 1.570796207585607068449789347324000006847L),
    TEST_c_c (cacosh, -0x1p-23L, 0.0L, 0.0L, 1.570796446004186170012854035955502877351L),
    TEST_c_c (cacosh, 0x1p-23L, -0.0L, 0.0L, -1.570796207585607068449789347324000006847L),
    TEST_c_c (cacosh, -0x1p-23L, -0.0L, 0.0L, -1.570796446004186170012854035955502877351L),
    TEST_c_c (cacosh, 0x1.fp-129L, 0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, 1.570796326794896619231321691639751442096L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, 0x1.fp-129L, -0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, -1.570796326794896619231321691639751442096L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0x1.fp-129L, 0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, 1.570796326794896619231321691639751442101L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (cacosh, -0x1.fp-129L, -0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, -1.570796326794896619231321691639751442101L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1p-52L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910458761820e-1L, 1.570796326794896462222075823262236786996L),
    TEST_c_c (cacosh, 0x1p-52L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910458761820e-1L, -1.570796326794896462222075823262236786996L),
    TEST_c_c (cacosh, -0x1p-52L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910458761820e-1L, 1.570796326794896776240567560017266097201L),
    TEST_c_c (cacosh, -0x1p-52L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910458761820e-1L, -1.570796326794896776240567560017266097201L),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, 0x1p-52L, 1.171456840272878582596796205397918831268e-8L, 1.895456983915074112227925127005564372844e-8L),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, 0x1p-52L, 1.171456840272878582596796205397918831268e-8L, 3.141592634635223399311902261000251614142L),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, -0x1p-52L, 1.171456840272878582596796205397918831268e-8L, -1.895456983915074112227925127005564372844e-8L),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, -0x1p-52L, 1.171456840272878582596796205397918831268e-8L, -3.141592634635223399311902261000251614142L),
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, 0x1.fp-1025L, 3.616396521699973256461764099945789620604e-301L, 1.490116119384765638786343542550461592240e-8L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, 0x1.fp-1025L, 3.616396521699973256461764099945789620604e-301L, 3.141592638688632044614986995416067458693L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, -0x1.fp-1025L, 3.616396521699973256461764099945789620604e-301L, -1.490116119384765638786343542550461592240e-8L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, -0x1.fp-1025L, 3.616396521699973256461764099945789620604e-301L, -3.141592638688632044614986995416067458693L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, 0.0L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x0.fffffffffffff8p0L, 8.813735870195429467279863907910371604180e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, 0.0L, 0.0L, 1.490116119384765638786343542550461592240e-8L),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, 0.0L, 0.0L, 3.141592638688632044614986995416067458693L),
    TEST_c_c (cacosh, 0x0.fffffffffffff8p0L, -0.0L, 0.0L, -1.490116119384765638786343542550461592240e-8L),
    TEST_c_c (cacosh, -0x0.fffffffffffff8p0L, -0.0L, 0.0L, -3.141592638688632044614986995416067458693L),
#endif
    TEST_c_c (cacosh, 0x1p-52L, 0.5L, 4.812118250596034474977589134243772428682e-1L, 1.570796326794896420628589431857911675917L),
    TEST_c_c (cacosh, 0x1p-52L, -0.5L, 4.812118250596034474977589134243772428682e-1L, -1.570796326794896420628589431857911675917L),
    TEST_c_c (cacosh, -0x1p-52L, 0.5L, 4.812118250596034474977589134243772428682e-1L, 1.570796326794896817834053951421591208280L),
    TEST_c_c (cacosh, -0x1p-52L, -0.5L, 4.812118250596034474977589134243772428682e-1L, -1.570796326794896817834053951421591208280L),
    TEST_c_c (cacosh, 0.5L, 0x1p-52L, 2.563950248511418570403591756798643804971e-16L, 1.047197551196597746154214461093186605110L),
    TEST_c_c (cacosh, -0.5L, 0x1p-52L, 2.563950248511418570403591756798643804971e-16L, 2.094395102393195492308428922186316279087L),
    TEST_c_c (cacosh, 0.5L, -0x1p-52L, 2.563950248511418570403591756798643804971e-16L, -1.047197551196597746154214461093186605110L),
    TEST_c_c (cacosh, -0.5L, -0x1p-52L, 2.563950248511418570403591756798643804971e-16L, -2.094395102393195492308428922186316279087L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1025L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.5L, 0x1.fp-1025L, 6.222508863508420569166420770843207333493e-309L, 1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0.5L, 0x1.fp-1025L, 6.222508863508420569166420770843207333493e-309L, 2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, 0.5L, -0x1.fp-1025L, 6.222508863508420569166420770843207333493e-309L, -1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0.5L, -0x1.fp-1025L, 6.222508863508420569166420770843207333493e-309L, -2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacosh, 0x1p-52L, 0x1p-52L, 2.220446049250313080847263336181677117148e-16L, 1.570796326794896397186716766608443357372L),
    TEST_c_c (cacosh, 0x1p-52L, -0x1p-52L, 2.220446049250313080847263336181677117148e-16L, -1.570796326794896397186716766608443357372L),
    TEST_c_c (cacosh, -0x1p-52L, 0x1p-52L, 2.220446049250313080847263336181677117148e-16L, 1.570796326794896841275926616671059526825L),
    TEST_c_c (cacosh, -0x1p-52L, -0x1p-52L, 2.220446049250313080847263336181677117148e-16L, -1.570796326794896841275926616671059526825L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x1p-52L, 2.220446049250313080847263336181622378926e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x1p-52L, 2.220446049250313080847263336181622378926e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x1p-52L, 2.220446049250313080847263336181622378926e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x1p-52L, 2.220446049250313080847263336181622378926e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-52L, 0x1.fp-1025L, 5.388850751072128349671657362289548938458e-309L, 1.570796326794896397186716766608443357372L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1p-52L, 0x1.fp-1025L, 5.388850751072128349671657362289548938458e-309L, 1.570796326794896841275926616671059526825L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, 0x1p-52L, -0x1.fp-1025L, 5.388850751072128349671657362289548938458e-309L, -1.570796326794896397186716766608443357372L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1p-52L, -0x1.fp-1025L, 5.388850751072128349671657362289548938458e-309L, -1.570796326794896841275926616671059526825L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacosh, 0.0L, 0x1p-52L, 2.220446049250313080847263336181622378926e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1p-52L, 2.220446049250313080847263336181622378926e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1p-52L, 2.220446049250313080847263336181622378926e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1p-52L, 2.220446049250313080847263336181622378926e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-52L, 0.0L, 0.0L, 1.570796326794896397186716766608443357372L),
    TEST_c_c (cacosh, -0x1p-52L, 0.0L, 0.0L, 1.570796326794896841275926616671059526825L),
    TEST_c_c (cacosh, 0x1p-52L, -0.0L, 0.0L, -1.570796326794896397186716766608443357372L),
    TEST_c_c (cacosh, -0x1p-52L, -0.0L, 0.0L, -1.570796326794896841275926616671059526825L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 64
    TEST_c_c (cacosh, 0x1p-63L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569937e-1L, 1.570796326794896619154657020805582738025L),
    TEST_c_c (cacosh, 0x1p-63L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569937e-1L, -1.570796326794896619154657020805582738025L),
    TEST_c_c (cacosh, -0x1p-63L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569937e-1L, 1.570796326794896619307986362473920146173L),
    TEST_c_c (cacosh, -0x1p-63L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569937e-1L, -1.570796326794896619307986362473920146173L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, 0x1p-63L, 2.588578361325995866221775673638805081337e-10L, 4.188407771167967636741951941902992986043e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, 0x1p-63L, 2.588578361325995866221775673638805081337e-10L, 3.141592653170952461345846619605307690007L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, -0x1p-63L, 2.588578361325995866221775673638805081337e-10L, -4.188407771167967636741951941902992986043e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, -0x1p-63L, 2.588578361325995866221775673638805081337e-10L, -3.141592653170952461345846619605307690007L),
# if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 2.472906068161537187835415298076415423459e-4923L, 3.292722539913596233371825532007990724506e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 2.472906068161537187835415298076415423459e-4923L, 3.141592653260520984471283759942320330996L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 2.472906068161537187835415298076415423459e-4923L, -3.292722539913596233371825532007990724506e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 2.472906068161537187835415298076415423459e-4923L, -3.141592653260520984471283759942320330996L),
# endif
    TEST_c_c (cacosh, 0.0L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x0.ffffffffffffffffp0L, 8.813735870195430251942769895627079569917e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, 0.0L, 0.0L, 3.292722539913596233371825532007990724506e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, 0.0L, 0.0L, 3.141592653260520984471283759942320330996L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffp0L, -0.0L, 0.0L, -3.292722539913596233371825532007990724506e-10L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffp0L, -0.0L, 0.0L, -3.141592653260520984471283759942320330996L),
#endif
    TEST_c_c (cacosh, 0x1p-63L, 0.5L, 4.812118250596034474977589134243684231373e-1L, 1.570796326794896619134347701278529840650L),
    TEST_c_c (cacosh, 0x1p-63L, -0.5L, 4.812118250596034474977589134243684231373e-1L, -1.570796326794896619134347701278529840650L),
    TEST_c_c (cacosh, -0x1p-63L, 0.5L, 4.812118250596034474977589134243684231373e-1L, 1.570796326794896619328295682000973043547L),
    TEST_c_c (cacosh, -0x1p-63L, -0.5L, 4.812118250596034474977589134243684231373e-1L, -1.570796326794896619328295682000973043547L),
    TEST_c_c (cacosh, 0.5L, 0x1p-63L, 1.251928832280966098829878787499365228659e-19L, 1.047197551196597746154214461093167628070L),
    TEST_c_c (cacosh, -0.5L, 0x1p-63L, 1.251928832280966098829878787499365228659e-19L, 2.094395102393195492308428922186335256127L),
    TEST_c_c (cacosh, 0.5L, -0x1p-63L, 1.251928832280966098829878787499365228659e-19L, -1.047197551196597746154214461093167628070L),
    TEST_c_c (cacosh, -0.5L, -0x1p-63L, 1.251928832280966098829878787499365228659e-19L, -2.094395102393195492308428922186335256127L),
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.5L, 0x1.fp-16385L, 9.402257155670418112159334875199821342890e-4933L, 1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0.5L, 0x1.fp-16385L, 9.402257155670418112159334875199821342890e-4933L, 2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, 0.5L, -0x1.fp-16385L, 9.402257155670418112159334875199821342890e-4933L, -1.047197551196597746154214461093167628066L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0.5L, -0x1.fp-16385L, 9.402257155670418112159334875199821342890e-4933L, -2.094395102393195492308428922186335256131L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacosh, 0x1p-63L, 0x1p-63L, 1.084202172485504434007452800869941711430e-19L, 1.570796326794896619122901474391200998698L),
    TEST_c_c (cacosh, 0x1p-63L, -0x1p-63L, 1.084202172485504434007452800869941711430e-19L, -1.570796326794896619122901474391200998698L),
    TEST_c_c (cacosh, -0x1p-63L, 0x1p-63L, 1.084202172485504434007452800869941711430e-19L, 1.570796326794896619339741908888301885499L),
    TEST_c_c (cacosh, -0x1p-63L, -0x1p-63L, 1.084202172485504434007452800869941711430e-19L, -1.570796326794896619339741908888301885499L),
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x1p-63L, 1.084202172485504434007452800869941711424e-19L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x1p-63L, 1.084202172485504434007452800869941711424e-19L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x1p-63L, 1.084202172485504434007452800869941711424e-19L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x1p-63L, 1.084202172485504434007452800869941711424e-19L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-63L, 0x1.fp-16385L, 8.142593549724601460479922838826119584465e-4933L, 1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1p-63L, 0x1.fp-16385L, 8.142593549724601460479922838826119584465e-4933L, 1.570796326794896619339741908888301885499L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, 0x1p-63L, -0x1.fp-16385L, 8.142593549724601460479922838826119584465e-4933L, -1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1p-63L, -0x1.fp-16385L, 8.142593549724601460479922838826119584465e-4933L, -1.570796326794896619339741908888301885499L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacosh, 0.0L, 0x1p-63L, 1.084202172485504434007452800869941711424e-19L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1p-63L, 1.084202172485504434007452800869941711424e-19L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1p-63L, 1.084202172485504434007452800869941711424e-19L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1p-63L, 1.084202172485504434007452800869941711424e-19L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-63L, 0.0L, 0.0L, 1.570796326794896619122901474391200998698L),
    TEST_c_c (cacosh, -0x1p-63L, 0.0L, 0.0L, 1.570796326794896619339741908888301885499L),
    TEST_c_c (cacosh, 0x1p-63L, -0.0L, 0.0L, -1.570796326794896619122901474391200998698L),
    TEST_c_c (cacosh, -0x1p-63L, -0.0L, 0.0L, -1.570796326794896619339741908888301885499L),
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
#endif
#if MANT_DIG >= 106
    TEST_c_c (cacosh, 0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639734010571L),
    TEST_c_c (cacosh, 0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639734010571L),
    TEST_c_c (cacosh, -0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639768873627L),
    TEST_c_c (cacosh, -0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639768873627L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 1.234330349600789959989661887846005659983e-16L, 1.997188458999618357060632049675702684031e-16L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 1.234330349600789959989661887846005659983e-16L, 3.141592653589793038743797483317667178134L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 1.234330349600789959989661887846005659983e-16L, -1.997188458999618357060632049675702684031e-16L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 1.234330349600789959989661887846005659983e-16L, -3.141592653589793038743797483317667178134L),
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 3.432186888910770473784103112009399360940e-293L, 1.570092458683775059393382371963565104164e-16L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 3.432186888910770473784103112009399360940e-293L, 3.141592653589793081453397514901996944859L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 3.432186888910770473784103112009399360940e-293L, -1.570092458683775059393382371963565104164e-16L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 3.432186888910770473784103112009399360940e-293L, -3.141592653589793081453397514901996944859L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (cacosh, 0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 8.813735870195430252326093249797835932642e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 0.0L, 1.570092458683775059393382371963565104164e-16L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 0.0L, 3.141592653589793081453397514901996944859L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 0.0L, -1.570092458683775059393382371963565104164e-16L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 0.0L, -3.141592653589793081453397514901996944859L),
#endif
    TEST_c_c (cacosh, 0x1p-105L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639729392766L),
    TEST_c_c (cacosh, 0x1p-105L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639729392766L),
    TEST_c_c (cacosh, -0x1p-105L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639773491431L),
    TEST_c_c (cacosh, -0x1p-105L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639773491431L),
    TEST_c_c (cacosh, 0.5L, 0x1p-105L, 2.846556599890768890940941645338546615370e-32L, 1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, -0.5L, 0x1p-105L, 2.846556599890768890940941645338546615370e-32L, 2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, 0.5L, -0x1p-105L, 2.846556599890768890940941645338546615370e-32L, -1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, -0.5L, -0x1p-105L, 2.846556599890768890940941645338546615370e-32L, -2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, 0x1p-105L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639726790195L),
    TEST_c_c (cacosh, 0x1p-105L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639726790195L),
    TEST_c_c (cacosh, -0x1p-105L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639776094002L),
    TEST_c_c (cacosh, -0x1p-105L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639776094002L),
#if !TEST_COND_binary32
    TEST_c_c (cacosh, 0x1.fp-1025L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-1025L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-1025L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-105L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 1.570796326794896619231321691639726790195L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1p-105L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 1.570796326794896619231321691639776094002L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, 0x1p-105L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -1.570796326794896619231321691639726790195L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (cacosh, -0x1p-105L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -1.570796326794896619231321691639776094002L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (cacosh, 0.0L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-105L, 0.0L, 0.0L, 1.570796326794896619231321691639726790195L),
    TEST_c_c (cacosh, -0x1p-105L, 0.0L, 0.0L, 1.570796326794896619231321691639776094002L),
    TEST_c_c (cacosh, 0x1p-105L, -0.0L, 0.0L, -1.570796326794896619231321691639726790195L),
    TEST_c_c (cacosh, -0x1p-105L, -0.0L, 0.0L, -1.570796326794896619231321691639776094002L),
#if MANT_DIG >= 113
    TEST_c_c (cacosh, 0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751305915L),
    TEST_c_c (cacosh, 0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751305915L),
    TEST_c_c (cacosh, -0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751578282L),
    TEST_c_c (cacosh, -0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751578282L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 1.091004200533850618345330909857103457781e-17L, 1.765281878332676474347460257221369818872e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 1.091004200533850618345330909857103457781e-17L, 3.141592653589793220809824599952738140723L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 1.091004200533850618345330909857103457781e-17L, -1.765281878332676474347460257221369818872e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 1.091004200533850618345330909857103457781e-17L, -3.141592653589793220809824599952738140723L),
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 5.867357004219059108790368268900490184505e-4916L, 1.387778780781445675529539585113525401762e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 5.867357004219059108790368268900490184505e-4916L, 3.141592653589793224584855575465046128902L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 5.867357004219059108790368268900490184505e-4916L, -1.387778780781445675529539585113525401762e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 5.867357004219059108790368268900490184505e-4916L, -3.141592653589793224584855575465046128902L),
    TEST_c_c (cacosh, 0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 8.813735870195430252326093249797922409363e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 0.0L, 1.387778780781445675529539585113525401762e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 0.0L, 3.141592653589793224584855575465046128902L),
    TEST_c_c (cacosh, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 0.0L, -1.387778780781445675529539585113525401762e-17L),
    TEST_c_c (cacosh, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 0.0L, -3.141592653589793224584855575465046128902L),
#endif
    TEST_c_c (cacosh, 0x1p-112L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751269838L),
    TEST_c_c (cacosh, 0x1p-112L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751269838L),
    TEST_c_c (cacosh, -0x1p-112L, 0.5L, 4.812118250596034474977589134243684231352e-1L, 1.570796326794896619231321691639751614359L),
    TEST_c_c (cacosh, -0x1p-112L, -0.5L, 4.812118250596034474977589134243684231352e-1L, -1.570796326794896619231321691639751614359L),
    TEST_c_c (cacosh, 0.5L, 0x1p-112L, 2.223872343664663196047610660420739543258e-34L, 1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, -0.5L, 0x1p-112L, 2.223872343664663196047610660420739543258e-34L, 2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, 0.5L, -0x1p-112L, 2.223872343664663196047610660420739543258e-34L, -1.047197551196597746154214461093167628066L),
    TEST_c_c (cacosh, -0.5L, -0x1p-112L, 2.223872343664663196047610660420739543258e-34L, -2.094395102393195492308428922186335256131L),
    TEST_c_c (cacosh, 0x1p-112L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751249506L),
    TEST_c_c (cacosh, 0x1p-112L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751249506L),
    TEST_c_c (cacosh, -0x1p-112L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751634692L),
    TEST_c_c (cacosh, -0x1p-112L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751634692L),
#if MIN_EXP <= -16381
    TEST_c_c (cacosh, 0x1.fp-16385L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1.fp-16385L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0x1.fp-16385L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-112L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 1.570796326794896619231321691639751249506L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1p-112L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 1.570796326794896619231321691639751634692L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, 0x1p-112L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -1.570796326794896619231321691639751249506L, UNDERFLOW_EXCEPTION),
    TEST_c_c (cacosh, -0x1p-112L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -1.570796326794896619231321691639751634692L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (cacosh, 0.0L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0.0L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, -0.0L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (cacosh, 0x1p-112L, 0.0L, 0.0L, 1.570796326794896619231321691639751249506L),
    TEST_c_c (cacosh, -0x1p-112L, 0.0L, 0.0L, 1.570796326794896619231321691639751634692L),
    TEST_c_c (cacosh, 0x1p-112L, -0.0L, 0.0L, -1.570796326794896619231321691639751249506L),
    TEST_c_c (cacosh, -0x1p-112L, -0.0L, 0.0L, -1.570796326794896619231321691639751634692L),

    TEST_c_c (cacosh, 0.75L, 1.25L, 1.13239363160530819522266333696834467L, 1.11752014915610270578240049553777969L),
    TEST_c_c (cacosh, -2, -3, 1.9833870299165354323470769028940395L, -2.1414491111159960199416055713254211L),
  };

static void
cacosh_test (void)
{
  ALL_RM_TEST (cacosh, 0, cacosh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_Ff_b1_data canonicalize_test_data[] =
  {
    TEST_Ffp_b1 (canonicalize, plus_infty, 0, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, minus_infty, 0, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, plus_zero, 0, plus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, minus_zero, 0, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, 1000, 0, 1000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, max_value, 0, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, -max_value, 0, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, min_value, 0, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, -min_value, 0, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, min_subnorm_value, 0, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, -min_subnorm_value, 0, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_Ffp_b1 (canonicalize, qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value, 0, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_Ffp_b1 (canonicalize, -snan_value, 0, -qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
#if HIGH_ORDER_BIT_IS_SET_FOR_SNAN
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x0"), 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x0"), 0, -qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
#else
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x0"), 0, qnan_value_pl ("0x0"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x0"), 0, -qnan_value_pl ("0x0"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
#endif
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x1"), 0, qnan_value_pl ("0x1"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x1"), 0, -qnan_value_pl ("0x1"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x1"), 0, qnan_value_pl ("0x1"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x1"), 0, -qnan_value_pl ("0x1"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x2"), 0, qnan_value_pl ("0x2"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x2"), 0, -qnan_value_pl ("0x2"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x2"), 0, qnan_value_pl ("0x2"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x2"), 0, -qnan_value_pl ("0x2"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x3fffff"), 0, qnan_value_pl ("0x3fffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x3fffff"), 0, -qnan_value_pl ("0x3fffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x3fffff"), 0, qnan_value_pl ("0x3fffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x3fffff"), 0, -qnan_value_pl ("0x3fffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
#if PAYLOAD_DIG >= 51
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x7ffffffffffff"), 0, qnan_value_pl ("0x7ffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x7ffffffffffff"), 0, -qnan_value_pl ("0x7ffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x7ffffffffffff"), 0, qnan_value_pl ("0x7ffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x7ffffffffffff"), 0, -qnan_value_pl ("0x7ffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
#endif
#if PAYLOAD_DIG >= 62
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x3fffffffffffffff"), 0, qnan_value_pl ("0x3fffffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x3fffffffffffffff"), 0, -qnan_value_pl ("0x3fffffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x3fffffffffffffff"), 0, qnan_value_pl ("0x3fffffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x3fffffffffffffff"), 0, -qnan_value_pl ("0x3fffffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
#endif
#if PAYLOAD_DIG >= 111
    TEST_Ffp_b1 (canonicalize, qnan_value_pl ("0x7fffffffffffffffffffffffffff"), 0, qnan_value_pl ("0x7fffffffffffffffffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, -qnan_value_pl ("0x7fffffffffffffffffffffffffff"), 0, -qnan_value_pl ("0x7fffffffffffffffffffffffffff"), NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD),
    TEST_Ffp_b1 (canonicalize, snan_value_pl ("0x7fffffffffffffffffffffffffff"), 0, qnan_value_pl ("0x7fffffffffffffffffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
    TEST_Ffp_b1 (canonicalize, -snan_value_pl ("0x7fffffffffffffffffffffffffff"), 0, -qnan_value_pl ("0x7fffffffffffffffffffffffffff"), NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN|TEST_NAN_PAYLOAD_CANONICALIZE),
#endif
  };

static void
canonicalize_test (void)
{
  FLOAT x;

  ALL_RM_TEST (canonicalize, 1, canonicalize_test_data, RUN_TEST_LOOP_Ffp_b1, END, x);
}


static const struct test_c_f_data carg_test_data[] =
  {
    /* carg (x + iy) is specified as atan2 (y, x) */

    /* carg (inf + i y) == +0 for finite y > 0.  */
    TEST_c_f (carg, plus_infty, 2.0, 0),

    /* carg (inf + i y) == -0 for finite y < 0.  */
    TEST_c_f (carg, plus_infty, -2.0, minus_zero),

    /* carg(x + i inf) == pi/2 for finite x.  */
    TEST_c_f (carg, 10.0, plus_infty, lit_pi_2_d),

    /* carg(x - i inf) == -pi/2 for finite x.  */
    TEST_c_f (carg, 10.0, minus_infty, -lit_pi_2_d),

    /* carg (-inf + i y) == +pi for finite y > 0.  */
    TEST_c_f (carg, minus_infty, 10.0, lit_pi),

    /* carg (-inf + i y) == -pi for finite y < 0.  */
    TEST_c_f (carg, minus_infty, -10.0, -lit_pi),

    TEST_c_f (carg, plus_infty, plus_infty, lit_pi_4_d),

    TEST_c_f (carg, plus_infty, minus_infty, -lit_pi_4_d),

    TEST_c_f (carg, minus_infty, plus_infty, lit_pi_3_m_4_d),

    TEST_c_f (carg, minus_infty, minus_infty, -lit_pi_3_m_4_d),

    TEST_c_f (carg, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_f (carg),
  };

static void
carg_test (void)
{
  ALL_RM_TEST (carg, 0, carg_test_data, RUN_TEST_LOOP_c_f, END);
}

static const struct test_c_c_data casin_test_data[] =
  {
    TEST_c_c (casin, 0, 0, 0.0, 0.0),
    TEST_c_c (casin, minus_zero, 0, minus_zero, 0.0),
    TEST_c_c (casin, 0, minus_zero, 0.0, minus_zero),
    TEST_c_c (casin, minus_zero, minus_zero, minus_zero, minus_zero),

    TEST_c_c (casin, plus_infty, plus_infty, lit_pi_4_d, plus_infty),
    TEST_c_c (casin, plus_infty, minus_infty, lit_pi_4_d, minus_infty),
    TEST_c_c (casin, minus_infty, plus_infty, -lit_pi_4_d, plus_infty),
    TEST_c_c (casin, minus_infty, minus_infty, -lit_pi_4_d, minus_infty),

    TEST_c_c (casin, -10.0, plus_infty, minus_zero, plus_infty),
    TEST_c_c (casin, -10.0, minus_infty, minus_zero, minus_infty),
    TEST_c_c (casin, 0, plus_infty, 0.0, plus_infty),
    TEST_c_c (casin, 0, minus_infty, 0.0, minus_infty),
    TEST_c_c (casin, minus_zero, plus_infty, minus_zero, plus_infty),
    TEST_c_c (casin, minus_zero, minus_infty, minus_zero, minus_infty),
    TEST_c_c (casin, 0.1L, plus_infty, 0.0, plus_infty),
    TEST_c_c (casin, 0.1L, minus_infty, 0.0, minus_infty),

    TEST_c_c (casin, minus_infty, 0, -lit_pi_2_d, plus_infty),
    TEST_c_c (casin, minus_infty, minus_zero, -lit_pi_2_d, minus_infty),
    TEST_c_c (casin, minus_infty, 100, -lit_pi_2_d, plus_infty),
    TEST_c_c (casin, minus_infty, -100, -lit_pi_2_d, minus_infty),

    TEST_c_c (casin, plus_infty, 0, lit_pi_2_d, plus_infty),
    TEST_c_c (casin, plus_infty, minus_zero, lit_pi_2_d, minus_infty),
    TEST_c_c (casin, plus_infty, 0.5, lit_pi_2_d, plus_infty),
    TEST_c_c (casin, plus_infty, -0.5, lit_pi_2_d, minus_infty),

    TEST_c_c (casin, qnan_value, plus_infty, qnan_value, plus_infty),
    TEST_c_c (casin, qnan_value, minus_infty, qnan_value, minus_infty),

    TEST_c_c (casin, 0.0, qnan_value, 0.0, qnan_value),
    TEST_c_c (casin, minus_zero, qnan_value, minus_zero, qnan_value),

    TEST_c_c (casin, plus_infty, qnan_value, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (casin, minus_infty, qnan_value, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (casin, qnan_value, 10.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (casin, qnan_value, -10.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (casin, 0.75, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (casin, -0.75, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (casin, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (casin, plus_zero, -1.5L, plus_zero, -1.194763217287109304111930828519090523536L),
    TEST_c_c (casin, minus_zero, -1.5L, minus_zero, -1.194763217287109304111930828519090523536L),
    TEST_c_c (casin, plus_zero, -1.0L, plus_zero, -0.8813735870195430252326093249797923090282L),
    TEST_c_c (casin, minus_zero, -1.0L, minus_zero, -0.8813735870195430252326093249797923090282L),
    TEST_c_c (casin, plus_zero, -0.5L, plus_zero, -0.4812118250596034474977589134243684231352L),
    TEST_c_c (casin, minus_zero, -0.5L, minus_zero, -0.4812118250596034474977589134243684231352L),
    TEST_c_c (casin, plus_zero, 0.5L, plus_zero, 0.4812118250596034474977589134243684231352L),
    TEST_c_c (casin, minus_zero, 0.5L, minus_zero, 0.4812118250596034474977589134243684231352L),
    TEST_c_c (casin, plus_zero, 1.0L, plus_zero, 0.8813735870195430252326093249797923090282L),
    TEST_c_c (casin, minus_zero, 1.0L, minus_zero, 0.8813735870195430252326093249797923090282L),
    TEST_c_c (casin, plus_zero, 1.5L, plus_zero, 1.194763217287109304111930828519090523536L),
    TEST_c_c (casin, minus_zero, 1.5L, minus_zero, 1.194763217287109304111930828519090523536L),

    TEST_c_c (casin, -1.5L, plus_zero, -lit_pi_2_d, 0.9624236501192068949955178268487368462704L),
    TEST_c_c (casin, -1.5L, minus_zero, -lit_pi_2_d, -0.9624236501192068949955178268487368462704L),
    TEST_c_c (casin, -1.0L, plus_zero, -lit_pi_2_d, plus_zero),
    TEST_c_c (casin, -1.0L, minus_zero, -lit_pi_2_d, minus_zero),
    TEST_c_c (casin, -0.5L, plus_zero, -0.5235987755982988730771072305465838140329L, plus_zero),
    TEST_c_c (casin, -0.5L, minus_zero, -0.5235987755982988730771072305465838140329L, minus_zero),
    TEST_c_c (casin, 0.5L, plus_zero, 0.5235987755982988730771072305465838140329L, plus_zero),
    TEST_c_c (casin, 0.5L, minus_zero, 0.5235987755982988730771072305465838140329L, minus_zero),
    TEST_c_c (casin, 1.0L, plus_zero, lit_pi_2_d, plus_zero),
    TEST_c_c (casin, 1.0L, minus_zero, lit_pi_2_d, minus_zero),
    TEST_c_c (casin, 1.5L, plus_zero, lit_pi_2_d, 0.9624236501192068949955178268487368462704L),
    TEST_c_c (casin, 1.5L, minus_zero, lit_pi_2_d, -0.9624236501192068949955178268487368462704L),

    TEST_c_c (casin, 0x1p50L, 1.0L, 1.570796326794895731052901991514519103193L, 3.535050620855721078027883819436720218708e1L),
    TEST_c_c (casin, 0x1p50L, -1.0L, 1.570796326794895731052901991514519103193L, -3.535050620855721078027883819436720218708e1L),
    TEST_c_c (casin, -0x1p50L, 1.0L, -1.570796326794895731052901991514519103193L, 3.535050620855721078027883819436720218708e1L),
    TEST_c_c (casin, -0x1p50L, -1.0L, -1.570796326794895731052901991514519103193L, -3.535050620855721078027883819436720218708e1L),
    TEST_c_c (casin, 1.0L, 0x1p50L, 8.881784197001252323389053344720723756399e-16L, 3.535050620855721078027883819436759661753e1L),
    TEST_c_c (casin, -1.0L, 0x1p50L, -8.881784197001252323389053344720723756399e-16L, 3.535050620855721078027883819436759661753e1L),
    TEST_c_c (casin, 1.0L, -0x1p50L, 8.881784197001252323389053344720723756399e-16L, -3.535050620855721078027883819436759661753e1L),
    TEST_c_c (casin, -1.0L, -0x1p50L, -8.881784197001252323389053344720723756399e-16L, -3.535050620855721078027883819436759661753e1L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1p500L, 1.0L, 1.570796326794896619231321691639751442099L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, 0x1p500L, -1.0L, 1.570796326794896619231321691639751442099L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, -0x1p500L, 1.0L, -1.570796326794896619231321691639751442099L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, -0x1p500L, -1.0L, -1.570796326794896619231321691639751442099L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, 1.0L, 0x1p500L, 3.054936363499604682051979393213617699789e-151L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, -1.0L, 0x1p500L, -3.054936363499604682051979393213617699789e-151L, 3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, 1.0L, -0x1p500L, 3.054936363499604682051979393213617699789e-151L, -3.472667374605326000180332928505464606058e2L),
    TEST_c_c (casin, -1.0L, -0x1p500L, -3.054936363499604682051979393213617699789e-151L, -3.472667374605326000180332928505464606058e2L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (casin, 0x1p5000L, 1.0L, 1.570796326794896619231321691639751442099L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, 0x1p5000L, -1.0L, 1.570796326794896619231321691639751442099L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, -0x1p5000L, 1.0L, -1.570796326794896619231321691639751442099L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, -0x1p5000L, -1.0L, -1.570796326794896619231321691639751442099L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, 1.0L, 0x1p5000L, 7.079811261048172892385615158694057552948e-1506L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, -1.0L, 0x1p5000L, -7.079811261048172892385615158694057552948e-1506L, 3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, 1.0L, -0x1p5000L, 7.079811261048172892385615158694057552948e-1506L, -3.466429049980286492395577839412341016946e3L),
    TEST_c_c (casin, -1.0L, -0x1p5000L, -7.079811261048172892385615158694057552948e-1506L, -3.466429049980286492395577839412341016946e3L),
#endif

    TEST_c_c (casin, 0x1.fp127L, 0x1.fp127L, 7.853981633974483096156608458198757210493e-1L, 8.973081118419833726837456344608533993585e1L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp1023L, 0x1.fp1023L, 7.853981633974483096156608458198757210493e-1L, 7.107906849659093345062145442726115449315e2L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (casin, 0x1.fp16383L, 0x1.fp16383L, 7.853981633974483096156608458198757210493e-1L, 1.135753137836666928715489992987020363057e4L),
#endif

    TEST_c_c (casin, 0x1.fp-129L, 1.5L, 1.579176199917649005841160751101628985741e-39L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -1.5L, 1.579176199917649005841160751101628985741e-39L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 1.5L, -1.579176199917649005841160751101628985741e-39L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -1.5L, -1.579176199917649005841160751101628985741e-39L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 1.5L, 0x1.fp-129L, 1.570796326794896619231321691639751442096L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, 0x1.fp-129L, -1.570796326794896619231321691639751442096L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, 1.5L, -0x1.fp-129L, 1.570796326794896619231321691639751442096L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, -0x1.fp-129L, -1.570796326794896619231321691639751442096L, -9.624236501192068949955178268487368462704e-1L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1025L, 1.5L, 2.989196569048182929051881765490354365918e-309L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -1.5L, 2.989196569048182929051881765490354365918e-309L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 1.5L, -2.989196569048182929051881765490354365918e-309L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -1.5L, -2.989196569048182929051881765490354365918e-309L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 1.5L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, 0x1.fp-1025L, -1.570796326794896619231321691639751442099L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, 1.5L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, -0x1.fp-1025L, -1.570796326794896619231321691639751442099L, -9.624236501192068949955178268487368462704e-1L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 1.5L, 4.516698239814521372306784062043266700598e-4933L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -1.5L, 4.516698239814521372306784062043266700598e-4933L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 1.5L, -4.516698239814521372306784062043266700598e-4933L, 1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -1.5L, -4.516698239814521372306784062043266700598e-4933L, -1.194763217287109304111930828519090523536L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 1.5L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, 0x1.fp-16385L, -1.570796326794896619231321691639751442099L, 9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, 1.5L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -9.624236501192068949955178268487368462704e-1L),
    TEST_c_c (casin, -1.5L, -0x1.fp-16385L, -1.570796326794896619231321691639751442099L, -9.624236501192068949955178268487368462704e-1L),
#endif

    TEST_c_c (casin, 0.5L, 1.0L, 3.494390628572132936274118260753699527325e-1L, 9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (casin, 0.5L, -1.0L, 3.494390628572132936274118260753699527325e-1L, -9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (casin, -0.5L, 1.0L, -3.494390628572132936274118260753699527325e-1L, 9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (casin, -0.5L, -1.0L, -3.494390628572132936274118260753699527325e-1L, -9.261330313501824245501244453057873152694e-1L),
    TEST_c_c (casin, 1.0L, 0.5L, 8.959074812088902390666567243275770102229e-1L, 7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (casin, -1.0L, 0.5L, -8.959074812088902390666567243275770102229e-1L, 7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (casin, 1.0L, -0.5L, 8.959074812088902390666567243275770102229e-1L, -7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (casin, -1.0L, -0.5L, -8.959074812088902390666567243275770102229e-1L, -7.328576759736452608886724437653071523305e-1L),
    TEST_c_c (casin, 0.25L, 1.0L, 1.763024327769669304186785666360901026468e-1L, 8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (casin, 0.25L, -1.0L, 1.763024327769669304186785666360901026468e-1L, -8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (casin, -0.25L, 1.0L, -1.763024327769669304186785666360901026468e-1L, 8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (casin, -0.25L, -1.0L, -1.763024327769669304186785666360901026468e-1L, -8.924633639033482359562124741744951972772e-1L),
    TEST_c_c (casin, 1.0L, 0.25L, 1.081751996523816326311037318425097434186L, 5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (casin, -1.0L, 0.25L, -1.081751996523816326311037318425097434186L, 5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (casin, 1.0L, -0.25L, 1.081751996523816326311037318425097434186L, -5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (casin, -1.0L, -0.25L, -1.081751996523816326311037318425097434186L, -5.097911466811016354623559941115413499164e-1L),
    TEST_c_c (casin, 0x1.fp-10L, 1.0L, 1.337909359557740912558349531052239112857e-3L, 8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (casin, 0x1.fp-10L, -1.0L, 1.337909359557740912558349531052239112857e-3L, -8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (casin, -0x1.fp-10L, 1.0L, -1.337909359557740912558349531052239112857e-3L, 8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (casin, -0x1.fp-10L, -1.0L, -1.337909359557740912558349531052239112857e-3L, -8.813742198809567991336704287826445879025e-1L),
    TEST_c_c (casin, 1.0L, 0x1.fp-10L, 1.527305029163877791518741192097931722508L, 4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (casin, -1.0L, 0x1.fp-10L, -1.527305029163877791518741192097931722508L, 4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (casin, 1.0L, -0x1.fp-10L, 1.527305029163877791518741192097931722508L, -4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (casin, -1.0L, -0x1.fp-10L, -1.527305029163877791518741192097931722508L, -4.350501469856803800217957402220976497152e-2L),
    TEST_c_c (casin, 0x1.fp-30L, 1.0L, 1.275929984216518540252717485007112529021e-9L, 8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (casin, 0x1.fp-30L, -1.0L, 1.275929984216518540252717485007112529021e-9L, -8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (casin, -0x1.fp-30L, 1.0L, -1.275929984216518540252717485007112529021e-9L, 8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (casin, -0x1.fp-30L, -1.0L, -1.275929984216518540252717485007112529021e-9L, -8.813735870195430258081932989769495326854e-1L),
    TEST_c_c (casin, 1.0L, 0x1.fp-30L, 1.570753848123921942730162693731872690232L, 4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (casin, -1.0L, 0x1.fp-30L, -1.570753848123921942730162693731872690232L, 4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (casin, 1.0L, -0x1.fp-30L, 1.570753848123921942730162693731872690232L, -4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (casin, -1.0L, -0x1.fp-30L, -1.570753848123921942730162693731872690232L, -4.247867098745151888768727039216644758847e-5L),
    TEST_c_c (casin, 0x1.fp-100L, 1.0L, 1.080754735021050612990719608916167354321e-30L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, 0x1.fp-100L, -1.0L, 1.080754735021050612990719608916167354321e-30L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, -0x1.fp-100L, 1.0L, -1.080754735021050612990719608916167354321e-30L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, -0x1.fp-100L, -1.0L, -1.080754735021050612990719608916167354321e-30L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, 1.0L, 0x1.fp-100L, 1.570796326794895382939283431378862777584L, 1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (casin, -1.0L, 0x1.fp-100L, -1.570796326794895382939283431378862777584L, 1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (casin, 1.0L, -0x1.fp-100L, 1.570796326794895382939283431378862777584L, -1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (casin, -1.0L, -0x1.fp-100L, -1.570796326794895382939283431378862777584L, -1.236292038260260888664514866457202186027e-15L),
    TEST_c_c (casin, 0x1.fp-129L, 1.0L, 2.013062564695348242280482517399205554874e-39L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -1.0L, 2.013062564695348242280482517399205554874e-39L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 1.0L, -2.013062564695348242280482517399205554874e-39L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -1.0L, -2.013062564695348242280482517399205554874e-39L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 1.0L, 0x1.fp-129L, 1.570796326794896619177965338869929107115L, 5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (casin, -1.0L, 0x1.fp-129L, -1.570796326794896619177965338869929107115L, 5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (casin, 1.0L, -0x1.fp-129L, 1.570796326794896619177965338869929107115L, -5.335635276982233498398987585285818977933e-20L),
    TEST_c_c (casin, -1.0L, -0x1.fp-129L, -1.570796326794896619177965338869929107115L, -5.335635276982233498398987585285818977933e-20L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1000L, 1.0L, 1.278589251976747242280879285935084814093e-301L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 0x1.fp-1000L, -1.0L, 1.278589251976747242280879285935084814093e-301L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x1.fp-1000L, 1.0L, -1.278589251976747242280879285935084814093e-301L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x1.fp-1000L, -1.0L, -1.278589251976747242280879285935084814093e-301L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 1.0L, 0x1.fp-1000L, 1.570796326794896619231321691639751442099L, 4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (casin, -1.0L, 0x1.fp-1000L, -1.570796326794896619231321691639751442099L, 4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (casin, 1.0L, -0x1.fp-1000L, 1.570796326794896619231321691639751442099L, -4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (casin, -1.0L, -0x1.fp-1000L, -1.570796326794896619231321691639751442099L, -4.252291453851660175550490409247739011867e-151L),
    TEST_c_c (casin, 0x1.fp-1025L, 1.0L, 3.810492908885321743133304375216617626230e-309L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -1.0L, 3.810492908885321743133304375216617626230e-309L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 1.0L, -3.810492908885321743133304375216617626230e-309L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -1.0L, -3.810492908885321743133304375216617626230e-309L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 1.0L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (casin, -1.0L, 0x1.fp-1025L, -1.570796326794896619231321691639751442099L, 7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (casin, 1.0L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -7.340879205566679497036857179189356754017e-155L),
    TEST_c_c (casin, -1.0L, -0x1.fp-1025L, -1.570796326794896619231321691639751442099L, -7.340879205566679497036857179189356754017e-155L),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-10000L, 1.0L, 6.867047849047171855399183659351043150871e-3011L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, 0x1.fp-10000L, -1.0L, 6.867047849047171855399183659351043150871e-3011L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, -0x1.fp-10000L, 1.0L, -6.867047849047171855399183659351043150871e-3011L, 8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, -0x1.fp-10000L, -1.0L, -6.867047849047171855399183659351043150871e-3011L, -8.813735870195430252326093249797923090282e-1L),
    TEST_c_c (casin, 1.0L, 0x1.fp-10000L, 1.570796326794896619231321691639751442099L, 9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (casin, -1.0L, 0x1.fp-10000L, -1.570796326794896619231321691639751442099L, 9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (casin, 1.0L, -0x1.fp-10000L, 1.570796326794896619231321691639751442099L, -9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (casin, -1.0L, -0x1.fp-10000L, -1.570796326794896619231321691639751442099L, -9.854680208706673586644342922051388714633e-1506L),
    TEST_c_c (casin, 0x1.fp-16385L, 1.0L, 5.757683115456107044131264955348448954458e-4933L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -1.0L, 5.757683115456107044131264955348448954458e-4933L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 1.0L, -5.757683115456107044131264955348448954458e-4933L, 8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -1.0L, -5.757683115456107044131264955348448954458e-4933L, -8.813735870195430252326093249797923090282e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 1.0L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (casin, -1.0L, 0x1.fp-16385L, -1.570796326794896619231321691639751442099L, 9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (casin, 1.0L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -9.023632056840860275214893047597614177639e-2467L),
    TEST_c_c (casin, -1.0L, -0x1.fp-16385L, -1.570796326794896619231321691639751442099L, -9.023632056840860275214893047597614177639e-2467L),
#endif

    TEST_c_c (casin, 0x1p-23L, 0x1.000002p0L, 8.429369199749229560964789467980644296420e-8L, 8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (casin, 0x1p-23L, -0x1.000002p0L, 8.429369199749229560964789467980644296420e-8L, -8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (casin, -0x1p-23L, 0x1.000002p0L, -8.429369199749229560964789467980644296420e-8L, 8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (casin, -0x1p-23L, -0x1.000002p0L, -8.429369199749229560964789467980644296420e-8L, -8.813736713132400470205730751186547909968e-1L),
    TEST_c_c (casin, 0x1.000002p0L, 0x1p-23L, 1.570574114956455764594481054234834678422L, 5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (casin, -0x1.000002p0L, 0x1p-23L, -1.570574114956455764594481054234834678422L, 5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (casin, 0x1.000002p0L, -0x1p-23L, 1.570574114956455764594481054234834678422L, -5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (casin, -0x1.000002p0L, -0x1p-23L, -1.570574114956455764594481054234834678422L, -5.364668491573609633134147164031476452679e-4L),
    TEST_c_c (casin, 0x1.fp-129L, 0x1.000002p0L, 2.013062444707472738895109955455676357057e-39L, 8.813736713132375348727889167749389235161e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -0x1.000002p0L, 2.013062444707472738895109955455676357057e-39L, -8.813736713132375348727889167749389235161e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 0x1.000002p0L, -2.013062444707472738895109955455676357057e-39L, 8.813736713132375348727889167749389235161e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -0x1.000002p0L, -2.013062444707472738895109955455676357057e-39L, -8.813736713132375348727889167749389235161e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.000002p0L, 0x1.fp-129L, 1.570796326794896619231321691639751436268L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, -0x1.000002p0L, 0x1.fp-129L, -1.570796326794896619231321691639751436268L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, 0x1.000002p0L, -0x1.fp-129L, 1.570796326794896619231321691639751436268L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, -0x1.000002p0L, -0x1.fp-129L, -1.570796326794896619231321691639751436268L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, 0.0L, 0x1.000002p0L, 0.0L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (casin, 0.0L, -0x1.000002p0L, 0.0L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (casin, -0.0L, 0x1.000002p0L, -0.0L, 8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (casin, -0.0L, -0x1.000002p0L, -0.0L, -8.813736713132375348727889167749389235161e-1L),
    TEST_c_c (casin, 0x1.000002p0L, 0.0L, 1.570796326794896619231321691639751442099L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, -0x1.000002p0L, 0.0L, -1.570796326794896619231321691639751442099L, 4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, 0x1.000002p0L, -0.0L, 1.570796326794896619231321691639751442099L, -4.882812451493617206486388134172712975070e-4L),
    TEST_c_c (casin, -0x1.000002p0L, -0.0L, -1.570796326794896619231321691639751442099L, -4.882812451493617206486388134172712975070e-4L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1p-52L, 0x1.0000000000001p0L, 1.570092458683774885078102529858632363236e-16L, 8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (casin, 0x1p-52L, -0x1.0000000000001p0L, 1.570092458683774885078102529858632363236e-16L, -8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (casin, -0x1p-52L, 0x1.0000000000001p0L, -1.570092458683774885078102529858632363236e-16L, 8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (casin, -0x1p-52L, -0x1.0000000000001p0L, -1.570092458683774885078102529858632363236e-16L, -8.813735870195431822418551933572982483664e-1L),
    TEST_c_c (casin, 0x1.0000000000001p0L, 0x1p-52L, 1.570796317204594913251280305810847349436L, 2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, 0x1p-52L, -1.570796317204594913251280305810847349436L, 2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (casin, 0x1.0000000000001p0L, -0x1p-52L, 1.570796317204594913251280305810847349436L, -2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, -0x1p-52L, -1.570796317204594913251280305810847349436L, -2.315303644582684770975188768022139415020e-8L),
    TEST_c_c (casin, 0x1.fp-1025L, 0x1.0000000000001p0L, 3.810492908885321320083608113679347200012e-309L, 8.813735870195431822418551933572895326024e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x1.0000000000001p0L, 3.810492908885321320083608113679347200012e-309L, -8.813735870195431822418551933572895326024e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x1.0000000000001p0L, -3.810492908885321320083608113679347200012e-309L, 8.813735870195431822418551933572895326024e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x1.0000000000001p0L, -3.810492908885321320083608113679347200012e-309L, -8.813735870195431822418551933572895326024e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.0000000000001p0L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, 0x1.fp-1025L, -1.570796326794896619231321691639751442099L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, 0x1.0000000000001p0L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, -0x1.fp-1025L, -1.570796326794896619231321691639751442099L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, 0.0L, 0x1.0000000000001p0L, 0.0L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (casin, 0.0L, -0x1.0000000000001p0L, 0.0L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (casin, -0.0L, 0x1.0000000000001p0L, -0.0L, 8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (casin, -0.0L, -0x1.0000000000001p0L, -0.0L, -8.813735870195431822418551933572895326024e-1L),
    TEST_c_c (casin, 0x1.0000000000001p0L, 0.0L, 1.570796326794896619231321691639751442099L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, 0.0L, -1.570796326794896619231321691639751442099L, 2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, 0x1.0000000000001p0L, -0.0L, 1.570796326794896619231321691639751442099L, -2.107342425544701550354780375182800088393e-8L),
    TEST_c_c (casin, -0x1.0000000000001p0L, -0.0L, -1.570796326794896619231321691639751442099L, -2.107342425544701550354780375182800088393e-8L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (casin, 0x1p-63L, 0x1.0000000000000002p0L, 7.666467083416870406778649849746878368519e-20L, 8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (casin, 0x1p-63L, -0x1.0000000000000002p0L, 7.666467083416870406778649849746878368519e-20L, -8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (casin, -0x1p-63L, 0x1.0000000000000002p0L, -7.666467083416870406778649849746878368519e-20L, 8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (casin, -0x1p-63L, -0x1.0000000000000002p0L, -7.666467083416870406778649849746878368519e-20L, -8.813735870195430253092739958139610131001e-1L),
    TEST_c_c (casin, 0x1.0000000000000002p0L, 0x1p-63L, 1.570796326582978888921215348380499122131L, 5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, 0x1p-63L, -1.570796326582978888921215348380499122131L, 5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (casin, 0x1.0000000000000002p0L, -0x1p-63L, 1.570796326582978888921215348380499122131L, -5.116146586219826555037807251857670783420e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, -0x1p-63L, -1.570796326582978888921215348380499122131L, -5.116146586219826555037807251857670783420e-10L),
# if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0x1.0000000000000002p0L, 5.757683115456107043819140328235418018963e-4933L, 8.813735870195430253092739958139610130980e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x1.0000000000000002p0L, 5.757683115456107043819140328235418018963e-4933L, -8.813735870195430253092739958139610130980e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x1.0000000000000002p0L, -5.757683115456107043819140328235418018963e-4933L, 8.813735870195430253092739958139610130980e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x1.0000000000000002p0L, -5.757683115456107043819140328235418018963e-4933L, -8.813735870195430253092739958139610130980e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.0000000000000002p0L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, 0x1.fp-16385L, -1.570796326794896619231321691639751442099L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, 0x1.0000000000000002p0L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, -0x1.fp-16385L, -1.570796326794896619231321691639751442099L, -4.656612873077392578082927418388212703712e-10L),
# endif
    TEST_c_c (casin, 0.0L, 0x1.0000000000000002p0L, 0.0L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (casin, 0.0L, -0x1.0000000000000002p0L, 0.0L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (casin, -0.0L, 0x1.0000000000000002p0L, -0.0L, 8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (casin, -0.0L, -0x1.0000000000000002p0L, -0.0L, -8.813735870195430253092739958139610130980e-1L),
    TEST_c_c (casin, 0x1.0000000000000002p0L, 0.0L, 1.570796326794896619231321691639751442099L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, 0.0L, -1.570796326794896619231321691639751442099L, 4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, 0x1.0000000000000002p0L, -0.0L, 1.570796326794896619231321691639751442099L, -4.656612873077392578082927418388212703712e-10L),
    TEST_c_c (casin, -0x1.0000000000000002p0L, -0.0L, -1.570796326794896619231321691639751442099L, -4.656612873077392578082927418388212703712e-10L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (casin, 0x1p-106L, 0x1.000000000000000000000000008p0L, 8.715763992105246878957416200936726072500e-33L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, 0x1p-106L, -0x1.000000000000000000000000008p0L, 8.715763992105246878957416200936726072500e-33L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, -0x1p-106L, 0x1.000000000000000000000000008p0L, -8.715763992105246878957416200936726072500e-33L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, -0x1p-106L, -0x1.000000000000000000000000008p0L, -8.715763992105246878957416200936726072500e-33L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, 0x1p-106L, 1.570796326794896565289107467733682961928L, 2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, 0x1p-106L, -1.570796326794896565289107467733682961928L, 2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, -0x1p-106L, 1.570796326794896565289107467733682961928L, -2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, -0x1p-106L, -1.570796326794896565289107467733682961928L, -2.285028863093221674154232933662774454211e-16L),
    TEST_c_c (casin, 0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 3.810492908885321743133304375216570658278e-309L, 8.813735870195430252326093249798097405561e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 3.810492908885321743133304375216570658278e-309L, -8.813735870195430252326093249798097405561e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x1.000000000000000000000000008p0L, -3.810492908885321743133304375216570658278e-309L, 8.813735870195430252326093249798097405561e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x1.000000000000000000000000008p0L, -3.810492908885321743133304375216570658278e-309L, -8.813735870195430252326093249798097405561e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, 0x1.fp-1025L, -1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, -0x1.fp-1025L, -1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, 0.0L, 0x1.000000000000000000000000008p0L, 0.0L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, 0.0L, -0x1.000000000000000000000000008p0L, 0.0L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, -0.0L, 0x1.000000000000000000000000008p0L, -0.0L, 8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, -0.0L, -0x1.000000000000000000000000008p0L, -0.0L, -8.813735870195430252326093249798097405561e-1L),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, 0.0L, 1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, 0.0L, -1.570796326794896619231321691639751442099L, 2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, 0x1.000000000000000000000000008p0L, -0.0L, 1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181636063482e-16L),
    TEST_c_c (casin, -0x1.000000000000000000000000008p0L, -0.0L, -1.570796326794896619231321691639751442099L, -2.220446049250313080847263336181636063482e-16L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (casin, 0x1p-113L, 0x1.0000000000000000000000000001p0L, 6.809190618832224124185481406981900518193e-35L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, 0x1p-113L, -0x1.0000000000000000000000000001p0L, 6.809190618832224124185481406981900518193e-35L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, -0x1p-113L, 0x1.0000000000000000000000000001p0L, -6.809190618832224124185481406981900518193e-35L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, -0x1p-113L, -0x1.0000000000000000000000000001p0L, -6.809190618832224124185481406981900518193e-35L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, 0x1p-113L, 1.570796326794896614463458507897073960405L, 2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, 0x1p-113L, -1.570796326794896614463458507897073960405L, 2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, -0x1p-113L, 1.570796326794896614463458507897073960405L, -2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, -0x1p-113L, -1.570796326794896614463458507897073960405L, -2.019699255375255198156433710951064632386e-17L),
    TEST_c_c (casin, 0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 5.757683115456107044131264955348448400014e-4933L, 8.813735870195430252326093249797924452120e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 5.757683115456107044131264955348448400014e-4933L, -8.813735870195430252326093249797924452120e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, -5.757683115456107044131264955348448400014e-4933L, 8.813735870195430252326093249797924452120e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, -5.757683115456107044131264955348448400014e-4933L, -8.813735870195430252326093249797924452120e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 1.570796326794896619231321691639751442099L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, -1.570796326794896619231321691639751442099L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 1.570796326794896619231321691639751442099L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, -1.570796326794896619231321691639751442099L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, 0.0L, 0x1.0000000000000000000000000001p0L, 0.0L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, 0.0L, -0x1.0000000000000000000000000001p0L, 0.0L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, -0.0L, 0x1.0000000000000000000000000001p0L, -0.0L, 8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, -0.0L, -0x1.0000000000000000000000000001p0L, -0.0L, -8.813735870195430252326093249797924452120e-1L),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, 0.0L, 1.570796326794896619231321691639751442099L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, 0.0L, -1.570796326794896619231321691639751442099L, 1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, 0x1.0000000000000000000000000001p0L, -0.0L, 1.570796326794896619231321691639751442099L, -1.962615573354718824241727964954454332780e-17L),
    TEST_c_c (casin, -0x1.0000000000000000000000000001p0L, -0.0L, -1.570796326794896619231321691639751442099L, -1.962615573354718824241727964954454332780e-17L),
#endif

    TEST_c_c (casin, 0x1p-23L, 0x0.ffffffp0L, 8.429369953393598499000760938112968755257e-8L, 8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (casin, 0x1p-23L, -0x0.ffffffp0L, 8.429369953393598499000760938112968755257e-8L, -8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (casin, -0x1p-23L, 0x0.ffffffp0L, -8.429369953393598499000760938112968755257e-8L, 8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (casin, -0x1p-23L, -0x0.ffffffp0L, -8.429369953393598499000760938112968755257e-8L, -8.813735448726963984495965873956465777250e-1L),
    TEST_c_c (casin, 0x0.ffffffp0L, 0x1p-23L, 1.570357140408705570520377172865353621698L, 2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (casin, -0x0.ffffffp0L, 0x1p-23L, -1.570357140408705570520377172865353621698L, 2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (casin, 0x0.ffffffp0L, -0x1p-23L, 1.570357140408705570520377172865353621698L, -2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (casin, -0x0.ffffffp0L, -0x1p-23L, -1.570357140408705570520377172865353621698L, -2.714321200917194650737217746780928423385e-4L),
    TEST_c_c (casin, 0x1.fp-129L, 0x0.ffffffp0L, 2.013062624689288675911339929224634593425e-39L, 8.813735448726938863015878255140556727969e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -0x0.ffffffp0L, 2.013062624689288675911339929224634593425e-39L, -8.813735448726938863015878255140556727969e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 0x0.ffffffp0L, -2.013062624689288675911339929224634593425e-39L, 8.813735448726938863015878255140556727969e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -0x0.ffffffp0L, -2.013062624689288675911339929224634593425e-39L, -8.813735448726938863015878255140556727969e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x0.ffffffp0L, 0x1.fp-129L, 1.570451059810180415643718442157112705601L, 8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (casin, -0x0.ffffffp0L, 0x1.fp-129L, -1.570451059810180415643718442157112705601L, 8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (casin, 0x0.ffffffp0L, -0x1.fp-129L, 1.570451059810180415643718442157112705601L, -8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (casin, -0x0.ffffffp0L, -0x1.fp-129L, -1.570451059810180415643718442157112705601L, -8.245504387859737323891843603996428534945e-36L),
    TEST_c_c (casin, 0.0L, 0x0.ffffffp0L, 0.0L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (casin, 0.0L, -0x0.ffffffp0L, 0.0L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (casin, -0.0L, 0x0.ffffffp0L, -0.0L, 8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (casin, -0.0L, -0x0.ffffffp0L, -0.0L, -8.813735448726938863015878255140556727969e-1L),
    TEST_c_c (casin, 0x0.ffffffp0L, 0.0L, 1.570451059810180415643718442157112705601L, 0.0L),
    TEST_c_c (casin, -0x0.ffffffp0L, 0.0L, -1.570451059810180415643718442157112705601L, 0.0L),
    TEST_c_c (casin, 0x0.ffffffp0L, -0.0L, 1.570451059810180415643718442157112705601L, -0.0L),
    TEST_c_c (casin, -0x0.ffffffp0L, -0.0L, -1.570451059810180415643718442157112705601L, -0.0L),
    TEST_c_c (casin, 0x1p-23L, 0.5L, 1.066240299940009780479664302917201696886e-7L, 4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (casin, 0x1p-23L, -0.5L, 1.066240299940009780479664302917201696886e-7L, -4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (casin, -0x1p-23L, 0.5L, -1.066240299940009780479664302917201696886e-7L, 4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (casin, -0x1p-23L, -0.5L, -1.066240299940009780479664302917201696886e-7L, -4.812118250596059896127318386463676808798e-1L),
    TEST_c_c (casin, 0.5L, 0x1p-23L, 5.235987755982934033165770729745469098254e-1L, 1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (casin, -0.5L, 0x1p-23L, -5.235987755982934033165770729745469098254e-1L, 1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (casin, 0.5L, -0x1p-23L, 5.235987755982934033165770729745469098254e-1L, -1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (casin, -0.5L, -0x1p-23L, -5.235987755982934033165770729745469098254e-1L, -1.376510308240943236356088341381173571841e-7L),
    TEST_c_c (casin, 0x1.fp-129L, 0.5L, 2.546345110742945032959687790021055102355e-39L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -0.5L, 2.546345110742945032959687790021055102355e-39L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 0.5L, -2.546345110742945032959687790021055102355e-39L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -0.5L, -2.546345110742945032959687790021055102355e-39L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0.5L, 0x1.fp-129L, 5.235987755982988730771072305465838140329e-1L, 3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0.5L, 0x1.fp-129L, -5.235987755982988730771072305465838140329e-1L, 3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0.5L, -0x1.fp-129L, 5.235987755982988730771072305465838140329e-1L, -3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0.5L, -0x1.fp-129L, -5.235987755982988730771072305465838140329e-1L, -3.287317402534702257036015056278368274737e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1p-23L, 0x1p-23L, 1.192092895507806853113684971258850504935e-7L, 1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (casin, 0x1p-23L, -0x1p-23L, 1.192092895507806853113684971258850504935e-7L, -1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (casin, -0x1p-23L, 0x1p-23L, -1.192092895507806853113684971258850504935e-7L, 1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (casin, -0x1p-23L, -0x1p-23L, -1.192092895507806853113684971258850504935e-7L, -1.192092895507818146886315028596704749235e-7L),
    TEST_c_c (casin, 0x1.fp-129L, 0x1p-23L, 2.846900380897707329917933005874691056564e-39L, 1.192092895507809676556842485683592032154e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -0x1p-23L, 2.846900380897707329917933005874691056564e-39L, -1.192092895507809676556842485683592032154e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 0x1p-23L, -2.846900380897707329917933005874691056564e-39L, 1.192092895507809676556842485683592032154e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -0x1p-23L, -2.846900380897707329917933005874691056564e-39L, -1.192092895507809676556842485683592032154e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1p-23L, 0x1.fp-129L, 1.192092895507815323443157514352519154304e-7L, 2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1p-23L, 0x1.fp-129L, -1.192092895507815323443157514352519154304e-7L, 2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1p-23L, -0x1.fp-129L, 1.192092895507815323443157514352519154304e-7L, -2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1p-23L, -0x1.fp-129L, -1.192092895507815323443157514352519154304e-7L, -2.846900380897747786805634596726756660388e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0.0L, 0x1p-23L, 0.0L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (casin, 0.0L, -0x1p-23L, 0.0L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (casin, -0.0L, 0x1p-23L, -0.0L, 1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (casin, -0.0L, -0x1p-23L, -0.0L, -1.192092895507809676556842485683592032154e-7L),
    TEST_c_c (casin, 0x1p-23L, 0.0L, 1.192092895507815323443157514352519154304e-7L, 0.0L),
    TEST_c_c (casin, -0x1p-23L, 0.0L, -1.192092895507815323443157514352519154304e-7L, 0.0L),
    TEST_c_c (casin, 0x1p-23L, -0.0L, 1.192092895507815323443157514352519154304e-7L, -0.0L),
    TEST_c_c (casin, -0x1p-23L, -0.0L, -1.192092895507815323443157514352519154304e-7L, -0.0L),
    TEST_c_c (casin, 0x1.fp-129L, 0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, 0x1.fp-129L, -0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, 0x1.fp-129L, -2.846900380897727558361783801085126250967e-39L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casin, -0x1.fp-129L, -0x1.fp-129L, -2.846900380897727558361783801085126250967e-39L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1p-52L, 0x0.fffffffffffff8p0L, 1.570092458683775146551022293016031474627e-16L, 8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (casin, 0x1p-52L, -0x0.fffffffffffff8p0L, 1.570092458683775146551022293016031474627e-16L, -8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (casin, -0x1p-52L, 0x0.fffffffffffff8p0L, -1.570092458683775146551022293016031474627e-16L, 8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (casin, -0x1p-52L, -0x0.fffffffffffff8p0L, -1.570092458683775146551022293016031474627e-16L, -8.813735870195429467279863907910458761820e-1L),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, 0x1p-52L, 1.570796307840326780080580569360500172043L, 1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, 0x1p-52L, -1.570796307840326780080580569360500172043L, 1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, -0x1p-52L, 1.570796307840326780080580569360500172043L, -1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, -0x1p-52L, -1.570796307840326780080580569360500172043L, -1.171456840272878582596796205397918831268e-8L),
    TEST_c_c (casin, 0x1.fp-1025L, 0x0.fffffffffffff8p0L, 3.810492908885321954658152505985270452320e-309L, 8.813735870195429467279863907910371604180e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x0.fffffffffffff8p0L, 3.810492908885321954658152505985270452320e-309L, -8.813735870195429467279863907910371604180e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x0.fffffffffffff8p0L, -3.810492908885321954658152505985270452320e-309L, 8.813735870195429467279863907910371604180e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x0.fffffffffffff8p0L, -3.810492908885321954658152505985270452320e-309L, -8.813735870195429467279863907910371604180e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, 0x1.fp-1025L, 1.570796311893735425383665303776316016594L, 3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, 0x1.fp-1025L, -1.570796311893735425383665303776316016594L, 3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, -0x1.fp-1025L, 1.570796311893735425383665303776316016594L, -3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, -0x1.fp-1025L, -1.570796311893735425383665303776316016594L, -3.616396521699973256461764099945789620604e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 0.0L, 0x0.fffffffffffff8p0L, 0.0L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (casin, 0.0L, -0x0.fffffffffffff8p0L, 0.0L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (casin, -0.0L, 0x0.fffffffffffff8p0L, -0.0L, 8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (casin, -0.0L, -0x0.fffffffffffff8p0L, -0.0L, -8.813735870195429467279863907910371604180e-1L),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, 0.0L, 1.570796311893735425383665303776316016594L, 0.0L),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, 0.0L, -1.570796311893735425383665303776316016594L, 0.0L),
    TEST_c_c (casin, 0x0.fffffffffffff8p0L, -0.0L, 1.570796311893735425383665303776316016594L, -0.0L),
    TEST_c_c (casin, -0x0.fffffffffffff8p0L, -0.0L, -1.570796311893735425383665303776316016594L, -0.0L),
#endif
    TEST_c_c (casin, 0x1p-52L, 0.5L, 1.986027322597818397661816253158024383707e-16L, 4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (casin, 0x1p-52L, -0.5L, 1.986027322597818397661816253158024383707e-16L, -4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (casin, -0x1p-52L, 0.5L, -1.986027322597818397661816253158024383707e-16L, 4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (casin, -0x1p-52L, -0.5L, -1.986027322597818397661816253158024383707e-16L, -4.812118250596034474977589134243772428682e-1L),
    TEST_c_c (casin, 0.5L, 0x1p-52L, 5.235987755982988730771072305465648369889e-1L, 2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (casin, -0.5L, 0x1p-52L, -5.235987755982988730771072305465648369889e-1L, 2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (casin, 0.5L, -0x1p-52L, 5.235987755982988730771072305465648369889e-1L, -2.563950248511418570403591756798643804971e-16L),
    TEST_c_c (casin, -0.5L, -0x1p-52L, -5.235987755982988730771072305465648369889e-1L, -2.563950248511418570403591756798643804971e-16L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1025L, 0.5L, 4.819934639999230680322935210539402497827e-309L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0.5L, 4.819934639999230680322935210539402497827e-309L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0.5L, -4.819934639999230680322935210539402497827e-309L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0.5L, -4.819934639999230680322935210539402497827e-309L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0.5L, 0x1.fp-1025L, 5.235987755982988730771072305465838140329e-1L, 6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0.5L, 0x1.fp-1025L, -5.235987755982988730771072305465838140329e-1L, 6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0.5L, -0x1.fp-1025L, 5.235987755982988730771072305465838140329e-1L, -6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0.5L, -0x1.fp-1025L, -5.235987755982988730771072305465838140329e-1L, -6.222508863508420569166420770843207333493e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casin, 0x1p-52L, 0x1p-52L, 2.220446049250313080847263336181604132852e-16L, 2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (casin, 0x1p-52L, -0x1p-52L, 2.220446049250313080847263336181604132852e-16L, -2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (casin, -0x1p-52L, 0x1p-52L, -2.220446049250313080847263336181604132852e-16L, 2.220446049250313080847263336181677117148e-16L),
    TEST_c_c (casin, -0x1p-52L, -0x1p-52L, -2.220446049250313080847263336181604132852e-16L, -2.220446049250313080847263336181677117148e-16L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1025L, 0x1p-52L, 5.388850751072128349671657362289283247603e-309L, 2.220446049250313080847263336181622378926e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x1p-52L, 5.388850751072128349671657362289283247603e-309L, -2.220446049250313080847263336181622378926e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x1p-52L, -5.388850751072128349671657362289283247603e-309L, 2.220446049250313080847263336181622378926e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x1p-52L, -5.388850751072128349671657362289283247603e-309L, -2.220446049250313080847263336181622378926e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1p-52L, 0x1.fp-1025L, 2.220446049250313080847263336181658871074e-16L, 5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1p-52L, 0x1.fp-1025L, -2.220446049250313080847263336181658871074e-16L, 5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1p-52L, -0x1.fp-1025L, 2.220446049250313080847263336181658871074e-16L, -5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1p-52L, -0x1.fp-1025L, -2.220446049250313080847263336181658871074e-16L, -5.388850751072128349671657362289548938458e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casin, 0.0L, 0x1p-52L, 0.0L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (casin, 0.0L, -0x1p-52L, 0.0L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (casin, -0.0L, 0x1p-52L, -0.0L, 2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (casin, -0.0L, -0x1p-52L, -0.0L, -2.220446049250313080847263336181622378926e-16L),
    TEST_c_c (casin, 0x1p-52L, 0.0L, 2.220446049250313080847263336181658871074e-16L, 0.0L),
    TEST_c_c (casin, -0x1p-52L, 0.0L, -2.220446049250313080847263336181658871074e-16L, 0.0L),
    TEST_c_c (casin, 0x1p-52L, -0.0L, 2.220446049250313080847263336181658871074e-16L, -0.0L),
    TEST_c_c (casin, -0x1p-52L, -0.0L, -2.220446049250313080847263336181658871074e-16L, -0.0L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1025L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x1.fp-1025L, -5.388850751072128349671657362289416093031e-309L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x1.fp-1025L, -5.388850751072128349671657362289416093031e-309L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 64
    TEST_c_c (casin, 0x1p-63L, 0x0.ffffffffffffffffp0L, 7.666467083416870407402049869781566856442e-20L, 8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (casin, 0x1p-63L, -0x0.ffffffffffffffffp0L, 7.666467083416870407402049869781566856442e-20L, -8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (casin, -0x1p-63L, 0x0.ffffffffffffffffp0L, -7.666467083416870407402049869781566856442e-20L, 8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (casin, -0x1p-63L, -0x0.ffffffffffffffffp0L, -7.666467083416870407402049869781566856442e-20L, -8.813735870195430251942769895627079569937e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, 0x1p-63L, 1.570796326376055842114524927965556247908L, 2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, 0x1p-63L, -1.570796326376055842114524927965556247908L, 2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, -0x1p-63L, 1.570796326376055842114524927965556247908L, -2.588578361325995866221775673638805081337e-10L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, -0x1p-63L, -1.570796326376055842114524927965556247908L, -2.588578361325995866221775673638805081337e-10L),
# if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 5.757683115456107044287327268904964422212e-4933L, 8.813735870195430251942769895627079569917e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 5.757683115456107044287327268904964422212e-4933L, -8.813735870195430251942769895627079569917e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x0.ffffffffffffffffp0L, -5.757683115456107044287327268904964422212e-4933L, 8.813735870195430251942769895627079569917e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x0.ffffffffffffffffp0L, -5.757683115456107044287327268904964422212e-4933L, -8.813735870195430251942769895627079569917e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 1.570796326465624365239962068302568888898L, 2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, 0x1.fp-16385L, -1.570796326465624365239962068302568888898L, 2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 1.570796326465624365239962068302568888898L, -2.472906068161537187835415298076415423459e-4923L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, -0x1.fp-16385L, -1.570796326465624365239962068302568888898L, -2.472906068161537187835415298076415423459e-4923L),
# endif
    TEST_c_c (casin, 0.0L, 0x0.ffffffffffffffffp0L, 0.0L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (casin, 0.0L, -0x0.ffffffffffffffffp0L, 0.0L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (casin, -0.0L, 0x0.ffffffffffffffffp0L, -0.0L, 8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (casin, -0.0L, -0x0.ffffffffffffffffp0L, -0.0L, -8.813735870195430251942769895627079569917e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, 0.0L, 1.570796326465624365239962068302568888898L, 0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, 0.0L, -1.570796326465624365239962068302568888898L, 0.0L),
    TEST_c_c (casin, 0x0.ffffffffffffffffp0L, -0.0L, 1.570796326465624365239962068302568888898L, -0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffp0L, -0.0L, -1.570796326465624365239962068302568888898L, -0.0L),
#endif
    TEST_c_c (casin, 0x1p-63L, 0.5L, 9.697399036122160144833087173623140436413e-20L, 4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (casin, 0x1p-63L, -0.5L, 9.697399036122160144833087173623140436413e-20L, -4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (casin, -0x1p-63L, 0.5L, -9.697399036122160144833087173623140436413e-20L, 4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (casin, -0x1p-63L, -0.5L, -9.697399036122160144833087173623140436413e-20L, -4.812118250596034474977589134243684231373e-1L),
    TEST_c_c (casin, 0.5L, 0x1p-63L, 5.235987755982988730771072305465838140283e-1L, 1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (casin, -0.5L, 0x1p-63L, -5.235987755982988730771072305465838140283e-1L, 1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (casin, 0.5L, -0x1p-63L, 5.235987755982988730771072305465838140283e-1L, -1.251928832280966098829878787499365228659e-19L),
    TEST_c_c (casin, -0.5L, -0x1p-63L, -5.235987755982988730771072305465838140283e-1L, -1.251928832280966098829878787499365228659e-19L),
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0.5L, 7.282957076134209141226696333885150260319e-4933L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0.5L, 7.282957076134209141226696333885150260319e-4933L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0.5L, -7.282957076134209141226696333885150260319e-4933L, 4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0.5L, -7.282957076134209141226696333885150260319e-4933L, -4.812118250596034474977589134243684231352e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0.5L, 0x1.fp-16385L, 5.235987755982988730771072305465838140329e-1L, 9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0.5L, 0x1.fp-16385L, -5.235987755982988730771072305465838140329e-1L, 9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0.5L, -0x1.fp-16385L, 5.235987755982988730771072305465838140329e-1L, -9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0.5L, -0x1.fp-16385L, -5.235987755982988730771072305465838140329e-1L, -9.402257155670418112159334875199821342890e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casin, 0x1p-63L, 0x1p-63L, 1.084202172485504434007452800869941711422e-19L, 1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (casin, 0x1p-63L, -0x1p-63L, 1.084202172485504434007452800869941711422e-19L, -1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (casin, -0x1p-63L, 0x1p-63L, -1.084202172485504434007452800869941711422e-19L, 1.084202172485504434007452800869941711430e-19L),
    TEST_c_c (casin, -0x1p-63L, -0x1p-63L, -1.084202172485504434007452800869941711422e-19L, -1.084202172485504434007452800869941711430e-19L),
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0x1p-63L, 8.142593549724601460479922838826119584369e-4933L, 1.084202172485504434007452800869941711424e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x1p-63L, 8.142593549724601460479922838826119584369e-4933L, -1.084202172485504434007452800869941711424e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x1p-63L, -8.142593549724601460479922838826119584369e-4933L, 1.084202172485504434007452800869941711424e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x1p-63L, -8.142593549724601460479922838826119584369e-4933L, -1.084202172485504434007452800869941711424e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1p-63L, 0x1.fp-16385L, 1.084202172485504434007452800869941711428e-19L, 8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1p-63L, 0x1.fp-16385L, -1.084202172485504434007452800869941711428e-19L, 8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1p-63L, -0x1.fp-16385L, 1.084202172485504434007452800869941711428e-19L, -8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1p-63L, -0x1.fp-16385L, -1.084202172485504434007452800869941711428e-19L, -8.142593549724601460479922838826119584465e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casin, 0.0L, 0x1p-63L, 0.0L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (casin, 0.0L, -0x1p-63L, 0.0L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (casin, -0.0L, 0x1p-63L, -0.0L, 1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (casin, -0.0L, -0x1p-63L, -0.0L, -1.084202172485504434007452800869941711424e-19L),
    TEST_c_c (casin, 0x1p-63L, 0.0L, 1.084202172485504434007452800869941711428e-19L, 0.0L),
    TEST_c_c (casin, -0x1p-63L, 0.0L, -1.084202172485504434007452800869941711428e-19L, 0.0L),
    TEST_c_c (casin, 0x1p-63L, -0.0L, 1.084202172485504434007452800869941711428e-19L, -0.0L),
    TEST_c_c (casin, -0x1p-63L, -0.0L, -1.084202172485504434007452800869941711428e-19L, -0.0L),
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x1.fp-16385L, -8.142593549724601460479922838826119584417e-4933L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x1.fp-16385L, -8.142593549724601460479922838826119584417e-4933L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
#if MANT_DIG >= 106
    TEST_c_c (casin, 0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 1.743152798421049375791483240187377443526e-32L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, 0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 1.743152798421049375791483240187377443526e-32L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, -0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, -1.743152798421049375791483240187377443526e-32L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, -0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, -1.743152798421049375791483240187377443526e-32L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 1.570796326794896419512475791677915736035L, 1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, -1.570796326794896419512475791677915736035L, 1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 1.570796326794896419512475791677915736035L, -1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, -1.570796326794896419512475791677915736035L, -1.234330349600789959989661887846005659983e-16L),
    TEST_c_c (casin, 0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 3.810492908885321743133304375216641110205e-309L, 8.813735870195430252326093249797835932642e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 3.810492908885321743133304375216641110205e-309L, -8.813735870195430252326093249797835932642e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, -3.810492908885321743133304375216641110205e-309L, 8.813735870195430252326093249797835932642e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, -3.810492908885321743133304375216641110205e-309L, -8.813735870195430252326093249797835932642e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 1.570796326794896462222075823262245502760L, 3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, -1.570796326794896462222075823262245502760L, 3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 1.570796326794896462222075823262245502760L, -3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, -1.570796326794896462222075823262245502760L, -3.432186888910770473784103112009399360940e-293L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casin, 0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, 0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, -0.0L, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, -0.0L, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, -8.813735870195430252326093249797835932642e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 1.570796326794896462222075823262245502760L, 0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, -1.570796326794896462222075823262245502760L, 0.0L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 1.570796326794896462222075823262245502760L, -0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, -1.570796326794896462222075823262245502760L, -0.0L),
#endif
    TEST_c_c (casin, 0x1p-105L, 0.5L, 2.204933261082751447436996198490110245167e-32L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, 0x1p-105L, -0.5L, 2.204933261082751447436996198490110245167e-32L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, -0x1p-105L, 0.5L, -2.204933261082751447436996198490110245167e-32L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, -0x1p-105L, -0.5L, -2.204933261082751447436996198490110245167e-32L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, 0.5L, 0x1p-105L, 5.235987755982988730771072305465838140329e-1L, 2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (casin, -0.5L, 0x1p-105L, -5.235987755982988730771072305465838140329e-1L, 2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (casin, 0.5L, -0x1p-105L, 5.235987755982988730771072305465838140329e-1L, -2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (casin, -0.5L, -0x1p-105L, -5.235987755982988730771072305465838140329e-1L, -2.846556599890768890940941645338546615370e-32L),
    TEST_c_c (casin, 0x1p-105L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, 0x1p-105L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, -0x1p-105L, 0x1p-105L, -2.465190328815661891911651766508706967729e-32L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, -0x1p-105L, -0x1p-105L, -2.465190328815661891911651766508706967729e-32L, -2.465190328815661891911651766508706967729e-32L),
#if !TEST_COND_binary32
    TEST_c_c (casin, 0x1.fp-1025L, 0x1p-105L, 5.388850751072128349671657362289416093031e-309L, 2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1.fp-1025L, -0x1p-105L, 5.388850751072128349671657362289416093031e-309L, -2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, 0x1p-105L, -5.388850751072128349671657362289416093031e-309L, 2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1.fp-1025L, -0x1p-105L, -5.388850751072128349671657362289416093031e-309L, -2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1p-105L, 0x1.fp-1025L, 2.465190328815661891911651766508706967729e-32L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1p-105L, 0x1.fp-1025L, -2.465190328815661891911651766508706967729e-32L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, 0x1p-105L, -0x1.fp-1025L, 2.465190328815661891911651766508706967729e-32L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casin, -0x1p-105L, -0x1.fp-1025L, -2.465190328815661891911651766508706967729e-32L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casin, 0.0L, 0x1p-105L, 0.0L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, 0.0L, -0x1p-105L, 0.0L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, -0.0L, 0x1p-105L, -0.0L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, -0.0L, -0x1p-105L, -0.0L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casin, 0x1p-105L, 0.0L, 2.465190328815661891911651766508706967729e-32L, 0.0L),
    TEST_c_c (casin, -0x1p-105L, 0.0L, -2.465190328815661891911651766508706967729e-32L, 0.0L),
    TEST_c_c (casin, 0x1p-105L, -0.0L, 2.465190328815661891911651766508706967729e-32L, -0.0L),
    TEST_c_c (casin, -0x1p-105L, -0.0L, -2.465190328815661891911651766508706967729e-32L, -0.0L),
#if MANT_DIG >= 113
    TEST_c_c (casin, 0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.361838123766444824837096281396380300349e-34L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, 0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.361838123766444824837096281396380300349e-34L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, -0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, -1.361838123766444824837096281396380300349e-34L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, -0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, -1.361838123766444824837096281396380300349e-34L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 1.570796326794896601578502908312986698624L, 1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, -1.570796326794896601578502908312986698624L, 1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 1.570796326794896601578502908312986698624L, -1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, -1.570796326794896601578502908312986698624L, -1.091004200533850618345330909857103457781e-17L),
    TEST_c_c (casin, 0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 5.757683115456107044131264955348449231681e-4933L, 8.813735870195430252326093249797922409363e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 5.757683115456107044131264955348449231681e-4933L, -8.813735870195430252326093249797922409363e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, -5.757683115456107044131264955348449231681e-4933L, 8.813735870195430252326093249797922409363e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, -5.757683115456107044131264955348449231681e-4933L, -8.813735870195430252326093249797922409363e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 1.570796326794896605353533883825294686803L, 5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, -1.570796326794896605353533883825294686803L, 5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 1.570796326794896605353533883825294686803L, -5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, -1.570796326794896605353533883825294686803L, -5.867357004219059108790368268900490184505e-4916L),
    TEST_c_c (casin, 0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, 0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, -0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, -0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, -8.813735870195430252326093249797922409363e-1L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 1.570796326794896605353533883825294686803L, 0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, -1.570796326794896605353533883825294686803L, 0.0L),
    TEST_c_c (casin, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 1.570796326794896605353533883825294686803L, -0.0L),
    TEST_c_c (casin, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, -1.570796326794896605353533883825294686803L, -0.0L),
#endif
    TEST_c_c (casin, 0x1p-112L, 0.5L, 1.722604110220899568310153280070398629037e-34L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, 0x1p-112L, -0.5L, 1.722604110220899568310153280070398629037e-34L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, -0x1p-112L, 0.5L, -1.722604110220899568310153280070398629037e-34L, 4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, -0x1p-112L, -0.5L, -1.722604110220899568310153280070398629037e-34L, -4.812118250596034474977589134243684231352e-1L),
    TEST_c_c (casin, 0.5L, 0x1p-112L, 5.235987755982988730771072305465838140329e-1L, 2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (casin, -0.5L, 0x1p-112L, -5.235987755982988730771072305465838140329e-1L, 2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (casin, 0.5L, -0x1p-112L, 5.235987755982988730771072305465838140329e-1L, -2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (casin, -0.5L, -0x1p-112L, -5.235987755982988730771072305465838140329e-1L, -2.223872343664663196047610660420739543258e-34L),
    TEST_c_c (casin, 0x1p-112L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, 0x1p-112L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, -0x1p-112L, 0x1p-112L, -1.925929944387235853055977942584927318538e-34L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, -0x1p-112L, -0x1p-112L, -1.925929944387235853055977942584927318538e-34L, -1.925929944387235853055977942584927318538e-34L),
#if MIN_EXP <= -16381
    TEST_c_c (casin, 0x1.fp-16385L, 0x1p-112L, 8.142593549724601460479922838826119584417e-4933L, 1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1.fp-16385L, -0x1p-112L, 8.142593549724601460479922838826119584417e-4933L, -1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, 0x1p-112L, -8.142593549724601460479922838826119584417e-4933L, 1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1.fp-16385L, -0x1p-112L, -8.142593549724601460479922838826119584417e-4933L, -1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1p-112L, 0x1.fp-16385L, 1.925929944387235853055977942584927318538e-34L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1p-112L, 0x1.fp-16385L, -1.925929944387235853055977942584927318538e-34L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, 0x1p-112L, -0x1.fp-16385L, 1.925929944387235853055977942584927318538e-34L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casin, -0x1p-112L, -0x1.fp-16385L, -1.925929944387235853055977942584927318538e-34L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casin, 0.0L, 0x1p-112L, 0.0L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, 0.0L, -0x1p-112L, 0.0L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, -0.0L, 0x1p-112L, -0.0L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, -0.0L, -0x1p-112L, -0.0L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casin, 0x1p-112L, 0.0L, 1.925929944387235853055977942584927318538e-34L, 0.0L),
    TEST_c_c (casin, -0x1p-112L, 0.0L, -1.925929944387235853055977942584927318538e-34L, 0.0L),
    TEST_c_c (casin, 0x1p-112L, -0.0L, 1.925929944387235853055977942584927318538e-34L, -0.0L),
    TEST_c_c (casin, -0x1p-112L, -0.0L, -1.925929944387235853055977942584927318538e-34L, -0.0L),

    TEST_c_c (casin, 0.75L, 1.25L, 0.453276177638793913448921196101971749L, 1.13239363160530819522266333696834467L),
    TEST_c_c (casin, -2, -3, -0.57065278432109940071028387968566963L, -1.9833870299165354323470769028940395L),
  };

static void
casin_test (void)
{
  ALL_RM_TEST (casin, 0, casin_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data casinh_test_data[] =
  {
    TEST_c_c (casinh, 0, 0, 0.0, 0.0),
    TEST_c_c (casinh, minus_zero, 0, minus_zero, 0),
    TEST_c_c (casinh, 0, minus_zero, 0.0, minus_zero),
    TEST_c_c (casinh, minus_zero, minus_zero, minus_zero, minus_zero),

    TEST_c_c (casinh, plus_infty, plus_infty, plus_infty, lit_pi_4_d),
    TEST_c_c (casinh, plus_infty, minus_infty, plus_infty, -lit_pi_4_d),
    TEST_c_c (casinh, minus_infty, plus_infty, minus_infty, lit_pi_4_d),
    TEST_c_c (casinh, minus_infty, minus_infty, minus_infty, -lit_pi_4_d),

    TEST_c_c (casinh, -10.0, plus_infty, minus_infty, lit_pi_2_d),
    TEST_c_c (casinh, -10.0, minus_infty, minus_infty, -lit_pi_2_d),
    TEST_c_c (casinh, 0, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (casinh, 0, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, plus_infty, minus_infty, lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, minus_infty, minus_infty, -lit_pi_2_d),
    TEST_c_c (casinh, 0.1L, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (casinh, 0.1L, minus_infty, plus_infty, -lit_pi_2_d),

    TEST_c_c (casinh, minus_infty, 0, minus_infty, 0.0),
    TEST_c_c (casinh, minus_infty, minus_zero, minus_infty, minus_zero),
    TEST_c_c (casinh, minus_infty, 100, minus_infty, 0.0),
    TEST_c_c (casinh, minus_infty, -100, minus_infty, minus_zero),

    TEST_c_c (casinh, plus_infty, 0, plus_infty, 0.0),
    TEST_c_c (casinh, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (casinh, plus_infty, 0.5, plus_infty, 0.0),
    TEST_c_c (casinh, plus_infty, -0.5, plus_infty, minus_zero),

    TEST_c_c (casinh, plus_infty, qnan_value, plus_infty, qnan_value),
    TEST_c_c (casinh, minus_infty, qnan_value, minus_infty, qnan_value),

    TEST_c_c (casinh, qnan_value, 0, qnan_value, 0.0),
    TEST_c_c (casinh, qnan_value, minus_zero, qnan_value, minus_zero),

    TEST_c_c (casinh, qnan_value, plus_infty, plus_infty, qnan_value, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (casinh, qnan_value, minus_infty, plus_infty, qnan_value, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (casinh, 10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (casinh, -10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (casinh, qnan_value, 0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (casinh, -0.75, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (casinh, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (casinh, plus_zero, -1.5L, 0.9624236501192068949955178268487368462704L, -lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, -1.5L, -0.9624236501192068949955178268487368462704L, -lit_pi_2_d),
    TEST_c_c (casinh, plus_zero, -1.0L, plus_zero, -lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, -1.0L, minus_zero, -lit_pi_2_d),
    TEST_c_c (casinh, plus_zero, -0.5L, plus_zero, -0.5235987755982988730771072305465838140329L),
    TEST_c_c (casinh, minus_zero, -0.5L, minus_zero, -0.5235987755982988730771072305465838140329L),
    TEST_c_c (casinh, plus_zero, 0.5L, plus_zero, 0.5235987755982988730771072305465838140329L),
    TEST_c_c (casinh, minus_zero, 0.5L, minus_zero, 0.5235987755982988730771072305465838140329L),
    TEST_c_c (casinh, plus_zero, 1.0L, plus_zero, lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, 1.0L, minus_zero, lit_pi_2_d),
    TEST_c_c (casinh, plus_zero, 1.5L, 0.9624236501192068949955178268487368462704L, lit_pi_2_d),
    TEST_c_c (casinh, minus_zero, 1.5L, -0.9624236501192068949955178268487368462704L, lit_pi_2_d),

    TEST_c_c (casinh, -1.5L, plus_zero, -1.194763217287109304111930828519090523536L, plus_zero),
    TEST_c_c (casinh, -1.5L, minus_zero, -1.194763217287109304111930828519090523536L, minus_zero),
    TEST_c_c (casinh, -1.0L, plus_zero, -0.8813735870195430252326093249797923090282L, plus_zero),
    TEST_c_c (casinh, -1.0L, minus_zero, -0.8813735870195430252326093249797923090282L, minus_zero),
    TEST_c_c (casinh, -0.5L, plus_zero, -0.4812118250596034474977589134243684231352L, plus_zero),
    TEST_c_c (casinh, -0.5L, minus_zero, -0.4812118250596034474977589134243684231352L, minus_zero),
    TEST_c_c (casinh, 0.5L, plus_zero, 0.4812118250596034474977589134243684231352L, plus_zero),
    TEST_c_c (casinh, 0.5L, minus_zero, 0.4812118250596034474977589134243684231352L, minus_zero),
    TEST_c_c (casinh, 1.0L, plus_zero, 0.8813735870195430252326093249797923090282L, plus_zero),
    TEST_c_c (casinh, 1.0L, minus_zero, 0.8813735870195430252326093249797923090282L, minus_zero),
    TEST_c_c (casinh, 1.5L, plus_zero, 1.194763217287109304111930828519090523536L, plus_zero),
    TEST_c_c (casinh, 1.5L, minus_zero, 1.194763217287109304111930828519090523536L, minus_zero),

    TEST_c_c (casinh, 0x1p50L, 1.0L, 3.535050620855721078027883819436759661753e1L, 8.881784197001252323389053344720723756399e-16L),
    TEST_c_c (casinh, 0x1p50L, -1.0L, 3.535050620855721078027883819436759661753e1L, -8.881784197001252323389053344720723756399e-16L),
    TEST_c_c (casinh, -0x1p50L, 1.0L, -3.535050620855721078027883819436759661753e1L, 8.881784197001252323389053344720723756399e-16L),
    TEST_c_c (casinh, -0x1p50L, -1.0L, -3.535050620855721078027883819436759661753e1L, -8.881784197001252323389053344720723756399e-16L),
    TEST_c_c (casinh, 1.0L, 0x1p50L, 3.535050620855721078027883819436720218708e1L, 1.570796326794895731052901991514519103193L),
    TEST_c_c (casinh, -1.0L, 0x1p50L, -3.535050620855721078027883819436720218708e1L, 1.570796326794895731052901991514519103193L),
    TEST_c_c (casinh, 1.0L, -0x1p50L, 3.535050620855721078027883819436720218708e1L, -1.570796326794895731052901991514519103193L),
    TEST_c_c (casinh, -1.0L, -0x1p50L, -3.535050620855721078027883819436720218708e1L, -1.570796326794895731052901991514519103193L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1p500L, 1.0L, 3.472667374605326000180332928505464606058e2L, 3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (casinh, 0x1p500L, -1.0L, 3.472667374605326000180332928505464606058e2L, -3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (casinh, -0x1p500L, 1.0L, -3.472667374605326000180332928505464606058e2L, 3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (casinh, -0x1p500L, -1.0L, -3.472667374605326000180332928505464606058e2L, -3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (casinh, 1.0L, 0x1p500L, 3.472667374605326000180332928505464606058e2L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -1.0L, 0x1p500L, -3.472667374605326000180332928505464606058e2L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, -0x1p500L, 3.472667374605326000180332928505464606058e2L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -1.0L, -0x1p500L, -3.472667374605326000180332928505464606058e2L, -1.570796326794896619231321691639751442099L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (casinh, 0x1p5000L, 1.0L, 3.466429049980286492395577839412341016946e3L, 7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (casinh, 0x1p5000L, -1.0L, 3.466429049980286492395577839412341016946e3L, -7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (casinh, -0x1p5000L, 1.0L, -3.466429049980286492395577839412341016946e3L, 7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (casinh, -0x1p5000L, -1.0L, -3.466429049980286492395577839412341016946e3L, -7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (casinh, 1.0L, 0x1p5000L, 3.466429049980286492395577839412341016946e3L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -1.0L, 0x1p5000L, -3.466429049980286492395577839412341016946e3L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, -0x1p5000L, 3.466429049980286492395577839412341016946e3L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -1.0L, -0x1p5000L, -3.466429049980286492395577839412341016946e3L, -1.570796326794896619231321691639751442099L),
#endif

    TEST_c_c (casinh, 0x1.fp127L, 0x1.fp127L, 8.973081118419833726837456344608533993585e1L, 7.853981633974483096156608458198757210493e-1L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp1023L, 0x1.fp1023L, 7.107906849659093345062145442726115449315e2L, 7.853981633974483096156608458198757210493e-1L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (casinh, 0x1.fp16383L, 0x1.fp16383L, 1.135753137836666928715489992987020363057e4L, 7.853981633974483096156608458198757210493e-1L),
#endif

    TEST_c_c (casinh, 0x1.fp-129L, 1.5L, 9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (casinh, 0x1.fp-129L, -1.5L, 9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (casinh, -0x1.fp-129L, 1.5L, -9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (casinh, -0x1.fp-129L, -1.5L, -9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (casinh, 1.5L, 0x1.fp-129L, 1.194763217287109304111930828519090523536L, 1.579176199917649005841160751101628985741e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -1.5L, 0x1.fp-129L, -1.194763217287109304111930828519090523536L, 1.579176199917649005841160751101628985741e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 1.5L, -0x1.fp-129L, 1.194763217287109304111930828519090523536L, -1.579176199917649005841160751101628985741e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -1.5L, -0x1.fp-129L, -1.194763217287109304111930828519090523536L, -1.579176199917649005841160751101628985741e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1025L, 1.5L, 9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-1025L, -1.5L, 9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, 1.5L, -9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, -1.5L, -9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.5L, 0x1.fp-1025L, 1.194763217287109304111930828519090523536L, 2.989196569048182929051881765490354365918e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -1.5L, 0x1.fp-1025L, -1.194763217287109304111930828519090523536L, 2.989196569048182929051881765490354365918e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 1.5L, -0x1.fp-1025L, 1.194763217287109304111930828519090523536L, -2.989196569048182929051881765490354365918e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -1.5L, -0x1.fp-1025L, -1.194763217287109304111930828519090523536L, -2.989196569048182929051881765490354365918e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 1.5L, 9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-16385L, -1.5L, 9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, 1.5L, -9.624236501192068949955178268487368462704e-1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, -1.5L, -9.624236501192068949955178268487368462704e-1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.5L, 0x1.fp-16385L, 1.194763217287109304111930828519090523536L, 4.516698239814521372306784062043266700598e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -1.5L, 0x1.fp-16385L, -1.194763217287109304111930828519090523536L, 4.516698239814521372306784062043266700598e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 1.5L, -0x1.fp-16385L, 1.194763217287109304111930828519090523536L, -4.516698239814521372306784062043266700598e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -1.5L, -0x1.fp-16385L, -1.194763217287109304111930828519090523536L, -4.516698239814521372306784062043266700598e-4933L, UNDERFLOW_EXCEPTION),
#endif

    TEST_c_c (casinh, 0.5L, 1.0L, 7.328576759736452608886724437653071523305e-1L, 8.959074812088902390666567243275770102229e-1L),
    TEST_c_c (casinh, 0.5L, -1.0L, 7.328576759736452608886724437653071523305e-1L, -8.959074812088902390666567243275770102229e-1L),
    TEST_c_c (casinh, -0.5L, 1.0L, -7.328576759736452608886724437653071523305e-1L, 8.959074812088902390666567243275770102229e-1L),
    TEST_c_c (casinh, -0.5L, -1.0L, -7.328576759736452608886724437653071523305e-1L, -8.959074812088902390666567243275770102229e-1L),
    TEST_c_c (casinh, 1.0L, 0.5L, 9.261330313501824245501244453057873152694e-1L, 3.494390628572132936274118260753699527325e-1L),
    TEST_c_c (casinh, -1.0L, 0.5L, -9.261330313501824245501244453057873152694e-1L, 3.494390628572132936274118260753699527325e-1L),
    TEST_c_c (casinh, 1.0L, -0.5L, 9.261330313501824245501244453057873152694e-1L, -3.494390628572132936274118260753699527325e-1L),
    TEST_c_c (casinh, -1.0L, -0.5L, -9.261330313501824245501244453057873152694e-1L, -3.494390628572132936274118260753699527325e-1L),
    TEST_c_c (casinh, 0.25L, 1.0L, 5.097911466811016354623559941115413499164e-1L, 1.081751996523816326311037318425097434186L),
    TEST_c_c (casinh, 0.25L, -1.0L, 5.097911466811016354623559941115413499164e-1L, -1.081751996523816326311037318425097434186L),
    TEST_c_c (casinh, -0.25L, 1.0L, -5.097911466811016354623559941115413499164e-1L, 1.081751996523816326311037318425097434186L),
    TEST_c_c (casinh, -0.25L, -1.0L, -5.097911466811016354623559941115413499164e-1L, -1.081751996523816326311037318425097434186L),
    TEST_c_c (casinh, 1.0L, 0.25L, 8.924633639033482359562124741744951972772e-1L, 1.763024327769669304186785666360901026468e-1L),
    TEST_c_c (casinh, -1.0L, 0.25L, -8.924633639033482359562124741744951972772e-1L, 1.763024327769669304186785666360901026468e-1L),
    TEST_c_c (casinh, 1.0L, -0.25L, 8.924633639033482359562124741744951972772e-1L, -1.763024327769669304186785666360901026468e-1L),
    TEST_c_c (casinh, -1.0L, -0.25L, -8.924633639033482359562124741744951972772e-1L, -1.763024327769669304186785666360901026468e-1L),
    TEST_c_c (casinh, 0x1.fp-10L, 1.0L, 4.350501469856803800217957402220976497152e-2L, 1.527305029163877791518741192097931722508L),
    TEST_c_c (casinh, 0x1.fp-10L, -1.0L, 4.350501469856803800217957402220976497152e-2L, -1.527305029163877791518741192097931722508L),
    TEST_c_c (casinh, -0x1.fp-10L, 1.0L, -4.350501469856803800217957402220976497152e-2L, 1.527305029163877791518741192097931722508L),
    TEST_c_c (casinh, -0x1.fp-10L, -1.0L, -4.350501469856803800217957402220976497152e-2L, -1.527305029163877791518741192097931722508L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-10L, 8.813742198809567991336704287826445879025e-1L, 1.337909359557740912558349531052239112857e-3L),
    TEST_c_c (casinh, -1.0L, 0x1.fp-10L, -8.813742198809567991336704287826445879025e-1L, 1.337909359557740912558349531052239112857e-3L),
    TEST_c_c (casinh, 1.0L, -0x1.fp-10L, 8.813742198809567991336704287826445879025e-1L, -1.337909359557740912558349531052239112857e-3L),
    TEST_c_c (casinh, -1.0L, -0x1.fp-10L, -8.813742198809567991336704287826445879025e-1L, -1.337909359557740912558349531052239112857e-3L),
    TEST_c_c (casinh, 0x1.fp-30L, 1.0L, 4.247867098745151888768727039216644758847e-5L, 1.570753848123921942730162693731872690232L),
    TEST_c_c (casinh, 0x1.fp-30L, -1.0L, 4.247867098745151888768727039216644758847e-5L, -1.570753848123921942730162693731872690232L),
    TEST_c_c (casinh, -0x1.fp-30L, 1.0L, -4.247867098745151888768727039216644758847e-5L, 1.570753848123921942730162693731872690232L),
    TEST_c_c (casinh, -0x1.fp-30L, -1.0L, -4.247867098745151888768727039216644758847e-5L, -1.570753848123921942730162693731872690232L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-30L, 8.813735870195430258081932989769495326854e-1L, 1.275929984216518540252717485007112529021e-9L),
    TEST_c_c (casinh, -1.0L, 0x1.fp-30L, -8.813735870195430258081932989769495326854e-1L, 1.275929984216518540252717485007112529021e-9L),
    TEST_c_c (casinh, 1.0L, -0x1.fp-30L, 8.813735870195430258081932989769495326854e-1L, -1.275929984216518540252717485007112529021e-9L),
    TEST_c_c (casinh, -1.0L, -0x1.fp-30L, -8.813735870195430258081932989769495326854e-1L, -1.275929984216518540252717485007112529021e-9L),
    TEST_c_c (casinh, 0x1.fp-100L, 1.0L, 1.236292038260260888664514866457202186027e-15L, 1.570796326794895382939283431378862777584L),
    TEST_c_c (casinh, 0x1.fp-100L, -1.0L, 1.236292038260260888664514866457202186027e-15L, -1.570796326794895382939283431378862777584L),
    TEST_c_c (casinh, -0x1.fp-100L, 1.0L, -1.236292038260260888664514866457202186027e-15L, 1.570796326794895382939283431378862777584L),
    TEST_c_c (casinh, -0x1.fp-100L, -1.0L, -1.236292038260260888664514866457202186027e-15L, -1.570796326794895382939283431378862777584L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-100L, 8.813735870195430252326093249797923090282e-1L, 1.080754735021050612990719608916167354321e-30L),
    TEST_c_c (casinh, -1.0L, 0x1.fp-100L, -8.813735870195430252326093249797923090282e-1L, 1.080754735021050612990719608916167354321e-30L),
    TEST_c_c (casinh, 1.0L, -0x1.fp-100L, 8.813735870195430252326093249797923090282e-1L, -1.080754735021050612990719608916167354321e-30L),
    TEST_c_c (casinh, -1.0L, -0x1.fp-100L, -8.813735870195430252326093249797923090282e-1L, -1.080754735021050612990719608916167354321e-30L),
    TEST_c_c (casinh, 0x1.fp-129L, 1.0L, 5.335635276982233498398987585285818977933e-20L, 1.570796326794896619177965338869929107115L),
    TEST_c_c (casinh, 0x1.fp-129L, -1.0L, 5.335635276982233498398987585285818977933e-20L, -1.570796326794896619177965338869929107115L),
    TEST_c_c (casinh, -0x1.fp-129L, 1.0L, -5.335635276982233498398987585285818977933e-20L, 1.570796326794896619177965338869929107115L),
    TEST_c_c (casinh, -0x1.fp-129L, -1.0L, -5.335635276982233498398987585285818977933e-20L, -1.570796326794896619177965338869929107115L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-129L, 8.813735870195430252326093249797923090282e-1L, 2.013062564695348242280482517399205554874e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -1.0L, 0x1.fp-129L, -8.813735870195430252326093249797923090282e-1L, 2.013062564695348242280482517399205554874e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 1.0L, -0x1.fp-129L, 8.813735870195430252326093249797923090282e-1L, -2.013062564695348242280482517399205554874e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -1.0L, -0x1.fp-129L, -8.813735870195430252326093249797923090282e-1L, -2.013062564695348242280482517399205554874e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1000L, 1.0L, 4.252291453851660175550490409247739011867e-151L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-1000L, -1.0L, 4.252291453851660175550490409247739011867e-151L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1000L, 1.0L, -4.252291453851660175550490409247739011867e-151L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1000L, -1.0L, -4.252291453851660175550490409247739011867e-151L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-1000L, 8.813735870195430252326093249797923090282e-1L, 1.278589251976747242280879285935084814093e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -1.0L, 0x1.fp-1000L, -8.813735870195430252326093249797923090282e-1L, 1.278589251976747242280879285935084814093e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 1.0L, -0x1.fp-1000L, 8.813735870195430252326093249797923090282e-1L, -1.278589251976747242280879285935084814093e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -1.0L, -0x1.fp-1000L, -8.813735870195430252326093249797923090282e-1L, -1.278589251976747242280879285935084814093e-301L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 0x1.fp-1025L, 1.0L, 7.340879205566679497036857179189356754017e-155L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-1025L, -1.0L, 7.340879205566679497036857179189356754017e-155L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, 1.0L, -7.340879205566679497036857179189356754017e-155L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, -1.0L, -7.340879205566679497036857179189356754017e-155L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-1025L, 8.813735870195430252326093249797923090282e-1L, 3.810492908885321743133304375216617626230e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -1.0L, 0x1.fp-1025L, -8.813735870195430252326093249797923090282e-1L, 3.810492908885321743133304375216617626230e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 1.0L, -0x1.fp-1025L, 8.813735870195430252326093249797923090282e-1L, -3.810492908885321743133304375216617626230e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -1.0L, -0x1.fp-1025L, -8.813735870195430252326093249797923090282e-1L, -3.810492908885321743133304375216617626230e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-10000L, 1.0L, 9.854680208706673586644342922051388714633e-1506L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-10000L, -1.0L, 9.854680208706673586644342922051388714633e-1506L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-10000L, 1.0L, -9.854680208706673586644342922051388714633e-1506L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-10000L, -1.0L, -9.854680208706673586644342922051388714633e-1506L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-10000L, 8.813735870195430252326093249797923090282e-1L, 6.867047849047171855399183659351043150871e-3011L),
    TEST_c_c (casinh, -1.0L, 0x1.fp-10000L, -8.813735870195430252326093249797923090282e-1L, 6.867047849047171855399183659351043150871e-3011L),
    TEST_c_c (casinh, 1.0L, -0x1.fp-10000L, 8.813735870195430252326093249797923090282e-1L, -6.867047849047171855399183659351043150871e-3011L),
    TEST_c_c (casinh, -1.0L, -0x1.fp-10000L, -8.813735870195430252326093249797923090282e-1L, -6.867047849047171855399183659351043150871e-3011L),
    TEST_c_c (casinh, 0x1.fp-16385L, 1.0L, 9.023632056840860275214893047597614177639e-2467L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-16385L, -1.0L, 9.023632056840860275214893047597614177639e-2467L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, 1.0L, -9.023632056840860275214893047597614177639e-2467L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, -1.0L, -9.023632056840860275214893047597614177639e-2467L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 1.0L, 0x1.fp-16385L, 8.813735870195430252326093249797923090282e-1L, 5.757683115456107044131264955348448954458e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -1.0L, 0x1.fp-16385L, -8.813735870195430252326093249797923090282e-1L, 5.757683115456107044131264955348448954458e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 1.0L, -0x1.fp-16385L, 8.813735870195430252326093249797923090282e-1L, -5.757683115456107044131264955348448954458e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -1.0L, -0x1.fp-16385L, -8.813735870195430252326093249797923090282e-1L, -5.757683115456107044131264955348448954458e-4933L, UNDERFLOW_EXCEPTION),
#endif

    TEST_c_c (casinh, 0x1p-23L, 0x1.000002p0L, 5.364668491573609633134147164031476452679e-4L, 1.570574114956455764594481054234834678422L),
    TEST_c_c (casinh, 0x1p-23L, -0x1.000002p0L, 5.364668491573609633134147164031476452679e-4L, -1.570574114956455764594481054234834678422L),
    TEST_c_c (casinh, -0x1p-23L, 0x1.000002p0L, -5.364668491573609633134147164031476452679e-4L, 1.570574114956455764594481054234834678422L),
    TEST_c_c (casinh, -0x1p-23L, -0x1.000002p0L, -5.364668491573609633134147164031476452679e-4L, -1.570574114956455764594481054234834678422L),
    TEST_c_c (casinh, 0x1.000002p0L, 0x1p-23L, 8.813736713132400470205730751186547909968e-1L, 8.429369199749229560964789467980644296420e-8L),
    TEST_c_c (casinh, -0x1.000002p0L, 0x1p-23L, -8.813736713132400470205730751186547909968e-1L, 8.429369199749229560964789467980644296420e-8L),
    TEST_c_c (casinh, 0x1.000002p0L, -0x1p-23L, 8.813736713132400470205730751186547909968e-1L, -8.429369199749229560964789467980644296420e-8L),
    TEST_c_c (casinh, -0x1.000002p0L, -0x1p-23L, -8.813736713132400470205730751186547909968e-1L, -8.429369199749229560964789467980644296420e-8L),
    TEST_c_c (casinh, 0x1.fp-129L, 0x1.000002p0L, 4.882812451493617206486388134172712975070e-4L, 1.570796326794896619231321691639751436268L),
    TEST_c_c (casinh, 0x1.fp-129L, -0x1.000002p0L, 4.882812451493617206486388134172712975070e-4L, -1.570796326794896619231321691639751436268L),
    TEST_c_c (casinh, -0x1.fp-129L, 0x1.000002p0L, -4.882812451493617206486388134172712975070e-4L, 1.570796326794896619231321691639751436268L),
    TEST_c_c (casinh, -0x1.fp-129L, -0x1.000002p0L, -4.882812451493617206486388134172712975070e-4L, -1.570796326794896619231321691639751436268L),
    TEST_c_c (casinh, 0x1.000002p0L, 0x1.fp-129L, 8.813736713132375348727889167749389235161e-1L, 2.013062444707472738895109955455676357057e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.000002p0L, 0x1.fp-129L, -8.813736713132375348727889167749389235161e-1L, 2.013062444707472738895109955455676357057e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1.000002p0L, -0x1.fp-129L, 8.813736713132375348727889167749389235161e-1L, -2.013062444707472738895109955455676357057e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.000002p0L, -0x1.fp-129L, -8.813736713132375348727889167749389235161e-1L, -2.013062444707472738895109955455676357057e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0.0L, 0x1.000002p0L, 4.882812451493617206486388134172712975070e-4L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0.0L, -0x1.000002p0L, 4.882812451493617206486388134172712975070e-4L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, 0x1.000002p0L, -4.882812451493617206486388134172712975070e-4L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, -0x1.000002p0L, -4.882812451493617206486388134172712975070e-4L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.000002p0L, 0.0L, 8.813736713132375348727889167749389235161e-1L, 0.0L),
    TEST_c_c (casinh, -0x1.000002p0L, 0.0L, -8.813736713132375348727889167749389235161e-1L, 0.0L),
    TEST_c_c (casinh, 0x1.000002p0L, -0.0L, 8.813736713132375348727889167749389235161e-1L, -0.0L),
    TEST_c_c (casinh, -0x1.000002p0L, -0.0L, -8.813736713132375348727889167749389235161e-1L, -0.0L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1p-52L, 0x1.0000000000001p0L, 2.315303644582684770975188768022139415020e-8L, 1.570796317204594913251280305810847349436L),
    TEST_c_c (casinh, 0x1p-52L, -0x1.0000000000001p0L, 2.315303644582684770975188768022139415020e-8L, -1.570796317204594913251280305810847349436L),
    TEST_c_c (casinh, -0x1p-52L, 0x1.0000000000001p0L, -2.315303644582684770975188768022139415020e-8L, 1.570796317204594913251280305810847349436L),
    TEST_c_c (casinh, -0x1p-52L, -0x1.0000000000001p0L, -2.315303644582684770975188768022139415020e-8L, -1.570796317204594913251280305810847349436L),
    TEST_c_c (casinh, 0x1.0000000000001p0L, 0x1p-52L, 8.813735870195431822418551933572982483664e-1L, 1.570092458683774885078102529858632363236e-16L),
    TEST_c_c (casinh, -0x1.0000000000001p0L, 0x1p-52L, -8.813735870195431822418551933572982483664e-1L, 1.570092458683774885078102529858632363236e-16L),
    TEST_c_c (casinh, 0x1.0000000000001p0L, -0x1p-52L, 8.813735870195431822418551933572982483664e-1L, -1.570092458683774885078102529858632363236e-16L),
    TEST_c_c (casinh, -0x1.0000000000001p0L, -0x1p-52L, -8.813735870195431822418551933572982483664e-1L, -1.570092458683774885078102529858632363236e-16L),
    TEST_c_c (casinh, 0x1.fp-1025L, 0x1.0000000000001p0L, 2.107342425544701550354780375182800088393e-8L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x1.0000000000001p0L, 2.107342425544701550354780375182800088393e-8L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x1.0000000000001p0L, -2.107342425544701550354780375182800088393e-8L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x1.0000000000001p0L, -2.107342425544701550354780375182800088393e-8L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000001p0L, 0x1.fp-1025L, 8.813735870195431822418551933572895326024e-1L, 3.810492908885321320083608113679347200012e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.0000000000001p0L, 0x1.fp-1025L, -8.813735870195431822418551933572895326024e-1L, 3.810492908885321320083608113679347200012e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.0000000000001p0L, -0x1.fp-1025L, 8.813735870195431822418551933572895326024e-1L, -3.810492908885321320083608113679347200012e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.0000000000001p0L, -0x1.fp-1025L, -8.813735870195431822418551933572895326024e-1L, -3.810492908885321320083608113679347200012e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.0L, 0x1.0000000000001p0L, 2.107342425544701550354780375182800088393e-8L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0.0L, -0x1.0000000000001p0L, 2.107342425544701550354780375182800088393e-8L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, 0x1.0000000000001p0L, -2.107342425544701550354780375182800088393e-8L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, -0x1.0000000000001p0L, -2.107342425544701550354780375182800088393e-8L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000001p0L, 0.0L, 8.813735870195431822418551933572895326024e-1L, 0.0L),
    TEST_c_c (casinh, -0x1.0000000000001p0L, 0.0L, -8.813735870195431822418551933572895326024e-1L, 0.0L),
    TEST_c_c (casinh, 0x1.0000000000001p0L, -0.0L, 8.813735870195431822418551933572895326024e-1L, -0.0L),
    TEST_c_c (casinh, -0x1.0000000000001p0L, -0.0L, -8.813735870195431822418551933572895326024e-1L, -0.0L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (casinh, 0x1p-63L, 0x1.0000000000000002p0L, 5.116146586219826555037807251857670783420e-10L, 1.570796326582978888921215348380499122131L),
    TEST_c_c (casinh, 0x1p-63L, -0x1.0000000000000002p0L, 5.116146586219826555037807251857670783420e-10L, -1.570796326582978888921215348380499122131L),
    TEST_c_c (casinh, -0x1p-63L, 0x1.0000000000000002p0L, -5.116146586219826555037807251857670783420e-10L, 1.570796326582978888921215348380499122131L),
    TEST_c_c (casinh, -0x1p-63L, -0x1.0000000000000002p0L, -5.116146586219826555037807251857670783420e-10L, -1.570796326582978888921215348380499122131L),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, 0x1p-63L, 8.813735870195430253092739958139610131001e-1L, 7.666467083416870406778649849746878368519e-20L),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, 0x1p-63L, -8.813735870195430253092739958139610131001e-1L, 7.666467083416870406778649849746878368519e-20L),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, -0x1p-63L, 8.813735870195430253092739958139610131001e-1L, -7.666467083416870406778649849746878368519e-20L),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, -0x1p-63L, -8.813735870195430253092739958139610131001e-1L, -7.666467083416870406778649849746878368519e-20L),
# if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0x1.0000000000000002p0L, 4.656612873077392578082927418388212703712e-10L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x1.0000000000000002p0L, 4.656612873077392578082927418388212703712e-10L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x1.0000000000000002p0L, -4.656612873077392578082927418388212703712e-10L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x1.0000000000000002p0L, -4.656612873077392578082927418388212703712e-10L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, 0x1.fp-16385L, 8.813735870195430253092739958139610130980e-1L, 5.757683115456107043819140328235418018963e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, 0x1.fp-16385L, -8.813735870195430253092739958139610130980e-1L, 5.757683115456107043819140328235418018963e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, -0x1.fp-16385L, 8.813735870195430253092739958139610130980e-1L, -5.757683115456107043819140328235418018963e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, -0x1.fp-16385L, -8.813735870195430253092739958139610130980e-1L, -5.757683115456107043819140328235418018963e-4933L, UNDERFLOW_EXCEPTION),
# endif
    TEST_c_c (casinh, 0.0L, 0x1.0000000000000002p0L, 4.656612873077392578082927418388212703712e-10L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0.0L, -0x1.0000000000000002p0L, 4.656612873077392578082927418388212703712e-10L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, 0x1.0000000000000002p0L, -4.656612873077392578082927418388212703712e-10L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, -0x1.0000000000000002p0L, -4.656612873077392578082927418388212703712e-10L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, 0.0L, 8.813735870195430253092739958139610130980e-1L, 0.0L),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, 0.0L, -8.813735870195430253092739958139610130980e-1L, 0.0L),
    TEST_c_c (casinh, 0x1.0000000000000002p0L, -0.0L, 8.813735870195430253092739958139610130980e-1L, -0.0L),
    TEST_c_c (casinh, -0x1.0000000000000002p0L, -0.0L, -8.813735870195430253092739958139610130980e-1L, -0.0L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (casinh, 0x1p-106L, 0x1.000000000000000000000000008p0L, 2.285028863093221674154232933662774454211e-16L, 1.570796326794896565289107467733682961928L),
    TEST_c_c (casinh, 0x1p-106L, -0x1.000000000000000000000000008p0L, 2.285028863093221674154232933662774454211e-16L, -1.570796326794896565289107467733682961928L),
    TEST_c_c (casinh, -0x1p-106L, 0x1.000000000000000000000000008p0L, -2.285028863093221674154232933662774454211e-16L, 1.570796326794896565289107467733682961928L),
    TEST_c_c (casinh, -0x1p-106L, -0x1.000000000000000000000000008p0L, -2.285028863093221674154232933662774454211e-16L, -1.570796326794896565289107467733682961928L),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, 0x1p-106L, 8.813735870195430252326093249798097405561e-1L, 8.715763992105246878957416200936726072500e-33L),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, 0x1p-106L, -8.813735870195430252326093249798097405561e-1L, 8.715763992105246878957416200936726072500e-33L),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, -0x1p-106L, 8.813735870195430252326093249798097405561e-1L, -8.715763992105246878957416200936726072500e-33L),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, -0x1p-106L, -8.813735870195430252326093249798097405561e-1L, -8.715763992105246878957416200936726072500e-33L),
    TEST_c_c (casinh, 0x1.fp-1025L, 0x1.000000000000000000000000008p0L, 2.220446049250313080847263336181636063482e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x1.000000000000000000000000008p0L, 2.220446049250313080847263336181636063482e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x1.000000000000000000000000008p0L, -2.220446049250313080847263336181636063482e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x1.000000000000000000000000008p0L, -2.220446049250313080847263336181636063482e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, 0x1.fp-1025L, 8.813735870195430252326093249798097405561e-1L, 3.810492908885321743133304375216570658278e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, 0x1.fp-1025L, -8.813735870195430252326093249798097405561e-1L, 3.810492908885321743133304375216570658278e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, -0x1.fp-1025L, 8.813735870195430252326093249798097405561e-1L, -3.810492908885321743133304375216570658278e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, -0x1.fp-1025L, -8.813735870195430252326093249798097405561e-1L, -3.810492908885321743133304375216570658278e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.0L, 0x1.000000000000000000000000008p0L, 2.220446049250313080847263336181636063482e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0.0L, -0x1.000000000000000000000000008p0L, 2.220446049250313080847263336181636063482e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, 0x1.000000000000000000000000008p0L, -2.220446049250313080847263336181636063482e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, -0x1.000000000000000000000000008p0L, -2.220446049250313080847263336181636063482e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, 0.0L, 8.813735870195430252326093249798097405561e-1L, 0.0L),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, 0.0L, -8.813735870195430252326093249798097405561e-1L, 0.0L),
    TEST_c_c (casinh, 0x1.000000000000000000000000008p0L, -0.0L, 8.813735870195430252326093249798097405561e-1L, -0.0L),
    TEST_c_c (casinh, -0x1.000000000000000000000000008p0L, -0.0L, -8.813735870195430252326093249798097405561e-1L, -0.0L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (casinh, 0x1p-113L, 0x1.0000000000000000000000000001p0L, 2.019699255375255198156433710951064632386e-17L, 1.570796326794896614463458507897073960405L),
    TEST_c_c (casinh, 0x1p-113L, -0x1.0000000000000000000000000001p0L, 2.019699255375255198156433710951064632386e-17L, -1.570796326794896614463458507897073960405L),
    TEST_c_c (casinh, -0x1p-113L, 0x1.0000000000000000000000000001p0L, -2.019699255375255198156433710951064632386e-17L, 1.570796326794896614463458507897073960405L),
    TEST_c_c (casinh, -0x1p-113L, -0x1.0000000000000000000000000001p0L, -2.019699255375255198156433710951064632386e-17L, -1.570796326794896614463458507897073960405L),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, 0x1p-113L, 8.813735870195430252326093249797924452120e-1L, 6.809190618832224124185481406981900518193e-35L),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, 0x1p-113L, -8.813735870195430252326093249797924452120e-1L, 6.809190618832224124185481406981900518193e-35L),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, -0x1p-113L, 8.813735870195430252326093249797924452120e-1L, -6.809190618832224124185481406981900518193e-35L),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, -0x1p-113L, -8.813735870195430252326093249797924452120e-1L, -6.809190618832224124185481406981900518193e-35L),
    TEST_c_c (casinh, 0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, 1.962615573354718824241727964954454332780e-17L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, 1.962615573354718824241727964954454332780e-17L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x1.0000000000000000000000000001p0L, -1.962615573354718824241727964954454332780e-17L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x1.0000000000000000000000000001p0L, -1.962615573354718824241727964954454332780e-17L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, 8.813735870195430252326093249797924452120e-1L, 5.757683115456107044131264955348448400014e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, 0x1.fp-16385L, -8.813735870195430252326093249797924452120e-1L, 5.757683115456107044131264955348448400014e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, 8.813735870195430252326093249797924452120e-1L, -5.757683115456107044131264955348448400014e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, -0x1.fp-16385L, -8.813735870195430252326093249797924452120e-1L, -5.757683115456107044131264955348448400014e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0.0L, 0x1.0000000000000000000000000001p0L, 1.962615573354718824241727964954454332780e-17L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0.0L, -0x1.0000000000000000000000000001p0L, 1.962615573354718824241727964954454332780e-17L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, 0x1.0000000000000000000000000001p0L, -1.962615573354718824241727964954454332780e-17L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, -0.0L, -0x1.0000000000000000000000000001p0L, -1.962615573354718824241727964954454332780e-17L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, 0.0L, 8.813735870195430252326093249797924452120e-1L, 0.0L),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, 0.0L, -8.813735870195430252326093249797924452120e-1L, 0.0L),
    TEST_c_c (casinh, 0x1.0000000000000000000000000001p0L, -0.0L, 8.813735870195430252326093249797924452120e-1L, -0.0L),
    TEST_c_c (casinh, -0x1.0000000000000000000000000001p0L, -0.0L, -8.813735870195430252326093249797924452120e-1L, -0.0L),
#endif

    TEST_c_c (casinh, 0x1p-23L, 0x0.ffffffp0L, 2.714321200917194650737217746780928423385e-4L, 1.570357140408705570520377172865353621698L),
    TEST_c_c (casinh, 0x1p-23L, -0x0.ffffffp0L, 2.714321200917194650737217746780928423385e-4L, -1.570357140408705570520377172865353621698L),
    TEST_c_c (casinh, -0x1p-23L, 0x0.ffffffp0L, -2.714321200917194650737217746780928423385e-4L, 1.570357140408705570520377172865353621698L),
    TEST_c_c (casinh, -0x1p-23L, -0x0.ffffffp0L, -2.714321200917194650737217746780928423385e-4L, -1.570357140408705570520377172865353621698L),
    TEST_c_c (casinh, 0x0.ffffffp0L, 0x1p-23L, 8.813735448726963984495965873956465777250e-1L, 8.429369953393598499000760938112968755257e-8L),
    TEST_c_c (casinh, -0x0.ffffffp0L, 0x1p-23L, -8.813735448726963984495965873956465777250e-1L, 8.429369953393598499000760938112968755257e-8L),
    TEST_c_c (casinh, 0x0.ffffffp0L, -0x1p-23L, 8.813735448726963984495965873956465777250e-1L, -8.429369953393598499000760938112968755257e-8L),
    TEST_c_c (casinh, -0x0.ffffffp0L, -0x1p-23L, -8.813735448726963984495965873956465777250e-1L, -8.429369953393598499000760938112968755257e-8L),
    TEST_c_c (casinh, 0x1.fp-129L, 0x0.ffffffp0L, 8.245504387859737323891843603996428534945e-36L, 1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, 0x1.fp-129L, -0x0.ffffffp0L, 8.245504387859737323891843603996428534945e-36L, -1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, -0x1.fp-129L, 0x0.ffffffp0L, -8.245504387859737323891843603996428534945e-36L, 1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, -0x1.fp-129L, -0x0.ffffffp0L, -8.245504387859737323891843603996428534945e-36L, -1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, 0x0.ffffffp0L, 0x1.fp-129L, 8.813735448726938863015878255140556727969e-1L, 2.013062624689288675911339929224634593425e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x0.ffffffp0L, 0x1.fp-129L, -8.813735448726938863015878255140556727969e-1L, 2.013062624689288675911339929224634593425e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x0.ffffffp0L, -0x1.fp-129L, 8.813735448726938863015878255140556727969e-1L, -2.013062624689288675911339929224634593425e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x0.ffffffp0L, -0x1.fp-129L, -8.813735448726938863015878255140556727969e-1L, -2.013062624689288675911339929224634593425e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0.0L, 0x0.ffffffp0L, 0.0L, 1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, 0.0L, -0x0.ffffffp0L, 0.0L, -1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, -0.0L, 0x0.ffffffp0L, -0.0L, 1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, -0.0L, -0x0.ffffffp0L, -0.0L, -1.570451059810180415643718442157112705601L),
    TEST_c_c (casinh, 0x0.ffffffp0L, 0.0L, 8.813735448726938863015878255140556727969e-1L, 0.0L),
    TEST_c_c (casinh, -0x0.ffffffp0L, 0.0L, -8.813735448726938863015878255140556727969e-1L, 0.0L),
    TEST_c_c (casinh, 0x0.ffffffp0L, -0.0L, 8.813735448726938863015878255140556727969e-1L, -0.0L),
    TEST_c_c (casinh, -0x0.ffffffp0L, -0.0L, -8.813735448726938863015878255140556727969e-1L, -0.0L),
    TEST_c_c (casinh, 0x1p-23L, 0.5L, 1.376510308240943236356088341381173571841e-7L, 5.235987755982934033165770729745469098254e-1L),
    TEST_c_c (casinh, 0x1p-23L, -0.5L, 1.376510308240943236356088341381173571841e-7L, -5.235987755982934033165770729745469098254e-1L),
    TEST_c_c (casinh, -0x1p-23L, 0.5L, -1.376510308240943236356088341381173571841e-7L, 5.235987755982934033165770729745469098254e-1L),
    TEST_c_c (casinh, -0x1p-23L, -0.5L, -1.376510308240943236356088341381173571841e-7L, -5.235987755982934033165770729745469098254e-1L),
    TEST_c_c (casinh, 0.5L, 0x1p-23L, 4.812118250596059896127318386463676808798e-1L, 1.066240299940009780479664302917201696886e-7L),
    TEST_c_c (casinh, -0.5L, 0x1p-23L, -4.812118250596059896127318386463676808798e-1L, 1.066240299940009780479664302917201696886e-7L),
    TEST_c_c (casinh, 0.5L, -0x1p-23L, 4.812118250596059896127318386463676808798e-1L, -1.066240299940009780479664302917201696886e-7L),
    TEST_c_c (casinh, -0.5L, -0x1p-23L, -4.812118250596059896127318386463676808798e-1L, -1.066240299940009780479664302917201696886e-7L),
    TEST_c_c (casinh, 0x1.fp-129L, 0.5L, 3.287317402534702257036015056278368274737e-39L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1.fp-129L, -0.5L, 3.287317402534702257036015056278368274737e-39L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, 0.5L, -3.287317402534702257036015056278368274737e-39L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, -0.5L, -3.287317402534702257036015056278368274737e-39L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0.5L, 0x1.fp-129L, 4.812118250596034474977589134243684231352e-1L, 2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0.5L, 0x1.fp-129L, -4.812118250596034474977589134243684231352e-1L, 2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0.5L, -0x1.fp-129L, 4.812118250596034474977589134243684231352e-1L, -2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0.5L, -0x1.fp-129L, -4.812118250596034474977589134243684231352e-1L, -2.546345110742945032959687790021055102355e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1p-23L, 0x1p-23L, 1.192092895507818146886315028596704749235e-7L, 1.192092895507806853113684971258850504935e-7L),
    TEST_c_c (casinh, 0x1p-23L, -0x1p-23L, 1.192092895507818146886315028596704749235e-7L, -1.192092895507806853113684971258850504935e-7L),
    TEST_c_c (casinh, -0x1p-23L, 0x1p-23L, -1.192092895507818146886315028596704749235e-7L, 1.192092895507806853113684971258850504935e-7L),
    TEST_c_c (casinh, -0x1p-23L, -0x1p-23L, -1.192092895507818146886315028596704749235e-7L, -1.192092895507806853113684971258850504935e-7L),
    TEST_c_c (casinh, 0x1.fp-129L, 0x1p-23L, 2.846900380897747786805634596726756660388e-39L, 1.192092895507815323443157514352519154304e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1.fp-129L, -0x1p-23L, 2.846900380897747786805634596726756660388e-39L, -1.192092895507815323443157514352519154304e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, 0x1p-23L, -2.846900380897747786805634596726756660388e-39L, 1.192092895507815323443157514352519154304e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, -0x1p-23L, -2.846900380897747786805634596726756660388e-39L, -1.192092895507815323443157514352519154304e-7L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1p-23L, 0x1.fp-129L, 1.192092895507809676556842485683592032154e-7L, 2.846900380897707329917933005874691056564e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1p-23L, 0x1.fp-129L, -1.192092895507809676556842485683592032154e-7L, 2.846900380897707329917933005874691056564e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1p-23L, -0x1.fp-129L, 1.192092895507809676556842485683592032154e-7L, -2.846900380897707329917933005874691056564e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1p-23L, -0x1.fp-129L, -1.192092895507809676556842485683592032154e-7L, -2.846900380897707329917933005874691056564e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0.0L, 0x1p-23L, 0.0L, 1.192092895507815323443157514352519154304e-7L),
    TEST_c_c (casinh, 0.0L, -0x1p-23L, 0.0L, -1.192092895507815323443157514352519154304e-7L),
    TEST_c_c (casinh, -0.0L, 0x1p-23L, -0.0L, 1.192092895507815323443157514352519154304e-7L),
    TEST_c_c (casinh, -0.0L, -0x1p-23L, -0.0L, -1.192092895507815323443157514352519154304e-7L),
    TEST_c_c (casinh, 0x1p-23L, 0.0L, 1.192092895507809676556842485683592032154e-7L, 0.0L),
    TEST_c_c (casinh, -0x1p-23L, 0.0L, -1.192092895507809676556842485683592032154e-7L, 0.0L),
    TEST_c_c (casinh, 0x1p-23L, -0.0L, 1.192092895507809676556842485683592032154e-7L, -0.0L),
    TEST_c_c (casinh, -0x1p-23L, -0.0L, -1.192092895507809676556842485683592032154e-7L, -0.0L),
    TEST_c_c (casinh, 0x1.fp-129L, 0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, 0x1.fp-129L, -0x1.fp-129L, 2.846900380897727558361783801085126250967e-39L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, 0x1.fp-129L, -2.846900380897727558361783801085126250967e-39L, 2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (casinh, -0x1.fp-129L, -0x1.fp-129L, -2.846900380897727558361783801085126250967e-39L, -2.846900380897727558361783801085126250967e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1p-52L, 0x0.fffffffffffff8p0L, 1.171456840272878582596796205397918831268e-8L, 1.570796307840326780080580569360500172043L),
    TEST_c_c (casinh, 0x1p-52L, -0x0.fffffffffffff8p0L, 1.171456840272878582596796205397918831268e-8L, -1.570796307840326780080580569360500172043L),
    TEST_c_c (casinh, -0x1p-52L, 0x0.fffffffffffff8p0L, -1.171456840272878582596796205397918831268e-8L, 1.570796307840326780080580569360500172043L),
    TEST_c_c (casinh, -0x1p-52L, -0x0.fffffffffffff8p0L, -1.171456840272878582596796205397918831268e-8L, -1.570796307840326780080580569360500172043L),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, 0x1p-52L, 8.813735870195429467279863907910458761820e-1L, 1.570092458683775146551022293016031474627e-16L),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, 0x1p-52L, -8.813735870195429467279863907910458761820e-1L, 1.570092458683775146551022293016031474627e-16L),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, -0x1p-52L, 8.813735870195429467279863907910458761820e-1L, -1.570092458683775146551022293016031474627e-16L),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, -0x1p-52L, -8.813735870195429467279863907910458761820e-1L, -1.570092458683775146551022293016031474627e-16L),
    TEST_c_c (casinh, 0x1.fp-1025L, 0x0.fffffffffffff8p0L, 3.616396521699973256461764099945789620604e-301L, 1.570796311893735425383665303776316016594L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x0.fffffffffffff8p0L, 3.616396521699973256461764099945789620604e-301L, -1.570796311893735425383665303776316016594L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x0.fffffffffffff8p0L, -3.616396521699973256461764099945789620604e-301L, 1.570796311893735425383665303776316016594L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x0.fffffffffffff8p0L, -3.616396521699973256461764099945789620604e-301L, -1.570796311893735425383665303776316016594L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, 0x1.fp-1025L, 8.813735870195429467279863907910371604180e-1L, 3.810492908885321954658152505985270452320e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, 0x1.fp-1025L, -8.813735870195429467279863907910371604180e-1L, 3.810492908885321954658152505985270452320e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, -0x1.fp-1025L, 8.813735870195429467279863907910371604180e-1L, -3.810492908885321954658152505985270452320e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, -0x1.fp-1025L, -8.813735870195429467279863907910371604180e-1L, -3.810492908885321954658152505985270452320e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.0L, 0x0.fffffffffffff8p0L, 0.0L, 1.570796311893735425383665303776316016594L),
    TEST_c_c (casinh, 0.0L, -0x0.fffffffffffff8p0L, 0.0L, -1.570796311893735425383665303776316016594L),
    TEST_c_c (casinh, -0.0L, 0x0.fffffffffffff8p0L, -0.0L, 1.570796311893735425383665303776316016594L),
    TEST_c_c (casinh, -0.0L, -0x0.fffffffffffff8p0L, -0.0L, -1.570796311893735425383665303776316016594L),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, 0.0L, 8.813735870195429467279863907910371604180e-1L, 0.0L),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, 0.0L, -8.813735870195429467279863907910371604180e-1L, 0.0L),
    TEST_c_c (casinh, 0x0.fffffffffffff8p0L, -0.0L, 8.813735870195429467279863907910371604180e-1L, -0.0L),
    TEST_c_c (casinh, -0x0.fffffffffffff8p0L, -0.0L, -8.813735870195429467279863907910371604180e-1L, -0.0L),
#endif
    TEST_c_c (casinh, 0x1p-52L, 0.5L, 2.563950248511418570403591756798643804971e-16L, 5.235987755982988730771072305465648369889e-1L),
    TEST_c_c (casinh, 0x1p-52L, -0.5L, 2.563950248511418570403591756798643804971e-16L, -5.235987755982988730771072305465648369889e-1L),
    TEST_c_c (casinh, -0x1p-52L, 0.5L, -2.563950248511418570403591756798643804971e-16L, 5.235987755982988730771072305465648369889e-1L),
    TEST_c_c (casinh, -0x1p-52L, -0.5L, -2.563950248511418570403591756798643804971e-16L, -5.235987755982988730771072305465648369889e-1L),
    TEST_c_c (casinh, 0.5L, 0x1p-52L, 4.812118250596034474977589134243772428682e-1L, 1.986027322597818397661816253158024383707e-16L),
    TEST_c_c (casinh, -0.5L, 0x1p-52L, -4.812118250596034474977589134243772428682e-1L, 1.986027322597818397661816253158024383707e-16L),
    TEST_c_c (casinh, 0.5L, -0x1p-52L, 4.812118250596034474977589134243772428682e-1L, -1.986027322597818397661816253158024383707e-16L),
    TEST_c_c (casinh, -0.5L, -0x1p-52L, -4.812118250596034474977589134243772428682e-1L, -1.986027322597818397661816253158024383707e-16L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1025L, 0.5L, 6.222508863508420569166420770843207333493e-309L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.fp-1025L, -0.5L, 6.222508863508420569166420770843207333493e-309L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, 0.5L, -6.222508863508420569166420770843207333493e-309L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, -0.5L, -6.222508863508420569166420770843207333493e-309L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.5L, 0x1.fp-1025L, 4.812118250596034474977589134243684231352e-1L, 4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0.5L, 0x1.fp-1025L, -4.812118250596034474977589134243684231352e-1L, 4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.5L, -0x1.fp-1025L, 4.812118250596034474977589134243684231352e-1L, -4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0.5L, -0x1.fp-1025L, -4.812118250596034474977589134243684231352e-1L, -4.819934639999230680322935210539402497827e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casinh, 0x1p-52L, 0x1p-52L, 2.220446049250313080847263336181677117148e-16L, 2.220446049250313080847263336181604132852e-16L),
    TEST_c_c (casinh, 0x1p-52L, -0x1p-52L, 2.220446049250313080847263336181677117148e-16L, -2.220446049250313080847263336181604132852e-16L),
    TEST_c_c (casinh, -0x1p-52L, 0x1p-52L, -2.220446049250313080847263336181677117148e-16L, 2.220446049250313080847263336181604132852e-16L),
    TEST_c_c (casinh, -0x1p-52L, -0x1p-52L, -2.220446049250313080847263336181677117148e-16L, -2.220446049250313080847263336181604132852e-16L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1025L, 0x1p-52L, 5.388850751072128349671657362289548938458e-309L, 2.220446049250313080847263336181658871074e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x1p-52L, 5.388850751072128349671657362289548938458e-309L, -2.220446049250313080847263336181658871074e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x1p-52L, -5.388850751072128349671657362289548938458e-309L, 2.220446049250313080847263336181658871074e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x1p-52L, -5.388850751072128349671657362289548938458e-309L, -2.220446049250313080847263336181658871074e-16L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1p-52L, 0x1.fp-1025L, 2.220446049250313080847263336181622378926e-16L, 5.388850751072128349671657362289283247603e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1p-52L, 0x1.fp-1025L, -2.220446049250313080847263336181622378926e-16L, 5.388850751072128349671657362289283247603e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1p-52L, -0x1.fp-1025L, 2.220446049250313080847263336181622378926e-16L, -5.388850751072128349671657362289283247603e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1p-52L, -0x1.fp-1025L, -2.220446049250313080847263336181622378926e-16L, -5.388850751072128349671657362289283247603e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casinh, 0.0L, 0x1p-52L, 0.0L, 2.220446049250313080847263336181658871074e-16L),
    TEST_c_c (casinh, 0.0L, -0x1p-52L, 0.0L, -2.220446049250313080847263336181658871074e-16L),
    TEST_c_c (casinh, -0.0L, 0x1p-52L, -0.0L, 2.220446049250313080847263336181658871074e-16L),
    TEST_c_c (casinh, -0.0L, -0x1p-52L, -0.0L, -2.220446049250313080847263336181658871074e-16L),
    TEST_c_c (casinh, 0x1p-52L, 0.0L, 2.220446049250313080847263336181622378926e-16L, 0.0L),
    TEST_c_c (casinh, -0x1p-52L, 0.0L, -2.220446049250313080847263336181622378926e-16L, 0.0L),
    TEST_c_c (casinh, 0x1p-52L, -0.0L, 2.220446049250313080847263336181622378926e-16L, -0.0L),
    TEST_c_c (casinh, -0x1p-52L, -0.0L, -2.220446049250313080847263336181622378926e-16L, -0.0L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1025L, 0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x1.fp-1025L, 5.388850751072128349671657362289416093031e-309L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x1.fp-1025L, -5.388850751072128349671657362289416093031e-309L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x1.fp-1025L, -5.388850751072128349671657362289416093031e-309L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 64
    TEST_c_c (casinh, 0x1p-63L, 0x0.ffffffffffffffffp0L, 2.588578361325995866221775673638805081337e-10L, 1.570796326376055842114524927965556247908L),
    TEST_c_c (casinh, 0x1p-63L, -0x0.ffffffffffffffffp0L, 2.588578361325995866221775673638805081337e-10L, -1.570796326376055842114524927965556247908L),
    TEST_c_c (casinh, -0x1p-63L, 0x0.ffffffffffffffffp0L, -2.588578361325995866221775673638805081337e-10L, 1.570796326376055842114524927965556247908L),
    TEST_c_c (casinh, -0x1p-63L, -0x0.ffffffffffffffffp0L, -2.588578361325995866221775673638805081337e-10L, -1.570796326376055842114524927965556247908L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, 0x1p-63L, 8.813735870195430251942769895627079569937e-1L, 7.666467083416870407402049869781566856442e-20L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, 0x1p-63L, -8.813735870195430251942769895627079569937e-1L, 7.666467083416870407402049869781566856442e-20L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, -0x1p-63L, 8.813735870195430251942769895627079569937e-1L, -7.666467083416870407402049869781566856442e-20L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, -0x1p-63L, -8.813735870195430251942769895627079569937e-1L, -7.666467083416870407402049869781566856442e-20L),
# if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0x0.ffffffffffffffffp0L, 2.472906068161537187835415298076415423459e-4923L, 1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x0.ffffffffffffffffp0L, 2.472906068161537187835415298076415423459e-4923L, -1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x0.ffffffffffffffffp0L, -2.472906068161537187835415298076415423459e-4923L, 1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x0.ffffffffffffffffp0L, -2.472906068161537187835415298076415423459e-4923L, -1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, 0x1.fp-16385L, 8.813735870195430251942769895627079569917e-1L, 5.757683115456107044287327268904964422212e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, 0x1.fp-16385L, -8.813735870195430251942769895627079569917e-1L, 5.757683115456107044287327268904964422212e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, -0x1.fp-16385L, 8.813735870195430251942769895627079569917e-1L, -5.757683115456107044287327268904964422212e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, -0x1.fp-16385L, -8.813735870195430251942769895627079569917e-1L, -5.757683115456107044287327268904964422212e-4933L, UNDERFLOW_EXCEPTION),
# endif
    TEST_c_c (casinh, 0.0L, 0x0.ffffffffffffffffp0L, 0.0L, 1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, 0.0L, -0x0.ffffffffffffffffp0L, 0.0L, -1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, -0.0L, 0x0.ffffffffffffffffp0L, -0.0L, 1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, -0.0L, -0x0.ffffffffffffffffp0L, -0.0L, -1.570796326465624365239962068302568888898L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, 0.0L, 8.813735870195430251942769895627079569917e-1L, 0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, 0.0L, -8.813735870195430251942769895627079569917e-1L, 0.0L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffp0L, -0.0L, 8.813735870195430251942769895627079569917e-1L, -0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffp0L, -0.0L, -8.813735870195430251942769895627079569917e-1L, -0.0L),
#endif
    TEST_c_c (casinh, 0x1p-63L, 0.5L, 1.251928832280966098829878787499365228659e-19L, 5.235987755982988730771072305465838140283e-1L),
    TEST_c_c (casinh, 0x1p-63L, -0.5L, 1.251928832280966098829878787499365228659e-19L, -5.235987755982988730771072305465838140283e-1L),
    TEST_c_c (casinh, -0x1p-63L, 0.5L, -1.251928832280966098829878787499365228659e-19L, 5.235987755982988730771072305465838140283e-1L),
    TEST_c_c (casinh, -0x1p-63L, -0.5L, -1.251928832280966098829878787499365228659e-19L, -5.235987755982988730771072305465838140283e-1L),
    TEST_c_c (casinh, 0.5L, 0x1p-63L, 4.812118250596034474977589134243684231373e-1L, 9.697399036122160144833087173623140436413e-20L),
    TEST_c_c (casinh, -0.5L, 0x1p-63L, -4.812118250596034474977589134243684231373e-1L, 9.697399036122160144833087173623140436413e-20L),
    TEST_c_c (casinh, 0.5L, -0x1p-63L, 4.812118250596034474977589134243684231373e-1L, -9.697399036122160144833087173623140436413e-20L),
    TEST_c_c (casinh, -0.5L, -0x1p-63L, -4.812118250596034474977589134243684231373e-1L, -9.697399036122160144833087173623140436413e-20L),
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0.5L, 9.402257155670418112159334875199821342890e-4933L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.fp-16385L, -0.5L, 9.402257155670418112159334875199821342890e-4933L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, 0.5L, -9.402257155670418112159334875199821342890e-4933L, 5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, -0.5L, -9.402257155670418112159334875199821342890e-4933L, -5.235987755982988730771072305465838140329e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0.5L, 0x1.fp-16385L, 4.812118250596034474977589134243684231352e-1L, 7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0.5L, 0x1.fp-16385L, -4.812118250596034474977589134243684231352e-1L, 7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0.5L, -0x1.fp-16385L, 4.812118250596034474977589134243684231352e-1L, -7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0.5L, -0x1.fp-16385L, -4.812118250596034474977589134243684231352e-1L, -7.282957076134209141226696333885150260319e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casinh, 0x1p-63L, 0x1p-63L, 1.084202172485504434007452800869941711430e-19L, 1.084202172485504434007452800869941711422e-19L),
    TEST_c_c (casinh, 0x1p-63L, -0x1p-63L, 1.084202172485504434007452800869941711430e-19L, -1.084202172485504434007452800869941711422e-19L),
    TEST_c_c (casinh, -0x1p-63L, 0x1p-63L, -1.084202172485504434007452800869941711430e-19L, 1.084202172485504434007452800869941711422e-19L),
    TEST_c_c (casinh, -0x1p-63L, -0x1p-63L, -1.084202172485504434007452800869941711430e-19L, -1.084202172485504434007452800869941711422e-19L),
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0x1p-63L, 8.142593549724601460479922838826119584465e-4933L, 1.084202172485504434007452800869941711428e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x1p-63L, 8.142593549724601460479922838826119584465e-4933L, -1.084202172485504434007452800869941711428e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x1p-63L, -8.142593549724601460479922838826119584465e-4933L, 1.084202172485504434007452800869941711428e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x1p-63L, -8.142593549724601460479922838826119584465e-4933L, -1.084202172485504434007452800869941711428e-19L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1p-63L, 0x1.fp-16385L, 1.084202172485504434007452800869941711424e-19L, 8.142593549724601460479922838826119584369e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1p-63L, 0x1.fp-16385L, -1.084202172485504434007452800869941711424e-19L, 8.142593549724601460479922838826119584369e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1p-63L, -0x1.fp-16385L, 1.084202172485504434007452800869941711424e-19L, -8.142593549724601460479922838826119584369e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1p-63L, -0x1.fp-16385L, -1.084202172485504434007452800869941711424e-19L, -8.142593549724601460479922838826119584369e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casinh, 0.0L, 0x1p-63L, 0.0L, 1.084202172485504434007452800869941711428e-19L),
    TEST_c_c (casinh, 0.0L, -0x1p-63L, 0.0L, -1.084202172485504434007452800869941711428e-19L),
    TEST_c_c (casinh, -0.0L, 0x1p-63L, -0.0L, 1.084202172485504434007452800869941711428e-19L),
    TEST_c_c (casinh, -0.0L, -0x1p-63L, -0.0L, -1.084202172485504434007452800869941711428e-19L),
    TEST_c_c (casinh, 0x1p-63L, 0.0L, 1.084202172485504434007452800869941711424e-19L, 0.0L),
    TEST_c_c (casinh, -0x1p-63L, 0.0L, -1.084202172485504434007452800869941711424e-19L, 0.0L),
    TEST_c_c (casinh, 0x1p-63L, -0.0L, 1.084202172485504434007452800869941711424e-19L, -0.0L),
    TEST_c_c (casinh, -0x1p-63L, -0.0L, -1.084202172485504434007452800869941711424e-19L, -0.0L),
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x1.fp-16385L, 8.142593549724601460479922838826119584417e-4933L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x1.fp-16385L, -8.142593549724601460479922838826119584417e-4933L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x1.fp-16385L, -8.142593549724601460479922838826119584417e-4933L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
#if MANT_DIG >= 106
    TEST_c_c (casinh, 0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, 1.234330349600789959989661887846005659983e-16L, 1.570796326794896419512475791677915736035L),
    TEST_c_c (casinh, 0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, 1.234330349600789959989661887846005659983e-16L, -1.570796326794896419512475791677915736035L),
    TEST_c_c (casinh, -0x1p-105L, 0x0.ffffffffffffffffffffffffffcp0L, -1.234330349600789959989661887846005659983e-16L, 1.570796326794896419512475791677915736035L),
    TEST_c_c (casinh, -0x1p-105L, -0x0.ffffffffffffffffffffffffffcp0L, -1.234330349600789959989661887846005659983e-16L, -1.570796326794896419512475791677915736035L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, 8.813735870195430252326093249797835932642e-1L, 1.743152798421049375791483240187377443526e-32L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-105L, -8.813735870195430252326093249797835932642e-1L, 1.743152798421049375791483240187377443526e-32L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, 8.813735870195430252326093249797835932642e-1L, -1.743152798421049375791483240187377443526e-32L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-105L, -8.813735870195430252326093249797835932642e-1L, -1.743152798421049375791483240187377443526e-32L),
    TEST_c_c (casinh, 0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, 3.432186888910770473784103112009399360940e-293L, 1.570796326794896462222075823262245502760L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, 3.432186888910770473784103112009399360940e-293L, -1.570796326794896462222075823262245502760L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x0.ffffffffffffffffffffffffffcp0L, -3.432186888910770473784103112009399360940e-293L, 1.570796326794896462222075823262245502760L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x0.ffffffffffffffffffffffffffcp0L, -3.432186888910770473784103112009399360940e-293L, -1.570796326794896462222075823262245502760L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, 8.813735870195430252326093249797835932642e-1L, 3.810492908885321743133304375216641110205e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1.fp-1025L, -8.813735870195430252326093249797835932642e-1L, 3.810492908885321743133304375216641110205e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, 8.813735870195430252326093249797835932642e-1L, -3.810492908885321743133304375216641110205e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1.fp-1025L, -8.813735870195430252326093249797835932642e-1L, -3.810492908885321743133304375216641110205e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0.0L, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 1.570796326794896462222075823262245502760L),
    TEST_c_c (casinh, 0.0L, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, -1.570796326794896462222075823262245502760L),
    TEST_c_c (casinh, -0.0L, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 1.570796326794896462222075823262245502760L),
    TEST_c_c (casinh, -0.0L, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, -1.570796326794896462222075823262245502760L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, 0.0L, 8.813735870195430252326093249797835932642e-1L, 0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, 0.0L, -8.813735870195430252326093249797835932642e-1L, 0.0L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffcp0L, -0.0L, 8.813735870195430252326093249797835932642e-1L, -0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffcp0L, -0.0L, -8.813735870195430252326093249797835932642e-1L, -0.0L),
#endif
    TEST_c_c (casinh, 0x1p-105L, 0.5L, 2.846556599890768890940941645338546615370e-32L, 5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, 0x1p-105L, -0.5L, 2.846556599890768890940941645338546615370e-32L, -5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, -0x1p-105L, 0.5L, -2.846556599890768890940941645338546615370e-32L, 5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, -0x1p-105L, -0.5L, -2.846556599890768890940941645338546615370e-32L, -5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, 0.5L, 0x1p-105L, 4.812118250596034474977589134243684231352e-1L, 2.204933261082751447436996198490110245167e-32L),
    TEST_c_c (casinh, -0.5L, 0x1p-105L, -4.812118250596034474977589134243684231352e-1L, 2.204933261082751447436996198490110245167e-32L),
    TEST_c_c (casinh, 0.5L, -0x1p-105L, 4.812118250596034474977589134243684231352e-1L, -2.204933261082751447436996198490110245167e-32L),
    TEST_c_c (casinh, -0.5L, -0x1p-105L, -4.812118250596034474977589134243684231352e-1L, -2.204933261082751447436996198490110245167e-32L),
    TEST_c_c (casinh, 0x1p-105L, 0x1p-105L, 2.465190328815661891911651766508706967729e-32L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, 0x1p-105L, -0x1p-105L, 2.465190328815661891911651766508706967729e-32L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, -0x1p-105L, 0x1p-105L, -2.465190328815661891911651766508706967729e-32L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, -0x1p-105L, -0x1p-105L, -2.465190328815661891911651766508706967729e-32L, -2.465190328815661891911651766508706967729e-32L),
#if !TEST_COND_binary32
    TEST_c_c (casinh, 0x1.fp-1025L, 0x1p-105L, 5.388850751072128349671657362289416093031e-309L, 2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1.fp-1025L, -0x1p-105L, 5.388850751072128349671657362289416093031e-309L, -2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, 0x1p-105L, -5.388850751072128349671657362289416093031e-309L, 2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1.fp-1025L, -0x1p-105L, -5.388850751072128349671657362289416093031e-309L, -2.465190328815661891911651766508706967729e-32L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1p-105L, 0x1.fp-1025L, 2.465190328815661891911651766508706967729e-32L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1p-105L, 0x1.fp-1025L, -2.465190328815661891911651766508706967729e-32L, 5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, 0x1p-105L, -0x1.fp-1025L, 2.465190328815661891911651766508706967729e-32L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (casinh, -0x1p-105L, -0x1.fp-1025L, -2.465190328815661891911651766508706967729e-32L, -5.388850751072128349671657362289416093031e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
    TEST_c_c (casinh, 0.0L, 0x1p-105L, 0.0L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, 0.0L, -0x1p-105L, 0.0L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, -0.0L, 0x1p-105L, -0.0L, 2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, -0.0L, -0x1p-105L, -0.0L, -2.465190328815661891911651766508706967729e-32L),
    TEST_c_c (casinh, 0x1p-105L, 0.0L, 2.465190328815661891911651766508706967729e-32L, 0.0L),
    TEST_c_c (casinh, -0x1p-105L, 0.0L, -2.465190328815661891911651766508706967729e-32L, 0.0L),
    TEST_c_c (casinh, 0x1p-105L, -0.0L, 2.465190328815661891911651766508706967729e-32L, -0.0L),
    TEST_c_c (casinh, -0x1p-105L, -0.0L, -2.465190328815661891911651766508706967729e-32L, -0.0L),
#if MANT_DIG >= 113
    TEST_c_c (casinh, 0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.091004200533850618345330909857103457781e-17L, 1.570796326794896601578502908312986698624L),
    TEST_c_c (casinh, 0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.091004200533850618345330909857103457781e-17L, -1.570796326794896601578502908312986698624L),
    TEST_c_c (casinh, -0x1p-112L, 0x0.ffffffffffffffffffffffffffff8p0L, -1.091004200533850618345330909857103457781e-17L, 1.570796326794896601578502908312986698624L),
    TEST_c_c (casinh, -0x1p-112L, -0x0.ffffffffffffffffffffffffffff8p0L, -1.091004200533850618345330909857103457781e-17L, -1.570796326794896601578502908312986698624L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, 8.813735870195430252326093249797922409363e-1L, 1.361838123766444824837096281396380300349e-34L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-112L, -8.813735870195430252326093249797922409363e-1L, 1.361838123766444824837096281396380300349e-34L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, 8.813735870195430252326093249797922409363e-1L, -1.361838123766444824837096281396380300349e-34L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-112L, -8.813735870195430252326093249797922409363e-1L, -1.361838123766444824837096281396380300349e-34L),
    TEST_c_c (casinh, 0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, 5.867357004219059108790368268900490184505e-4916L, 1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, 5.867357004219059108790368268900490184505e-4916L, -1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x0.ffffffffffffffffffffffffffff8p0L, -5.867357004219059108790368268900490184505e-4916L, 1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x0.ffffffffffffffffffffffffffff8p0L, -5.867357004219059108790368268900490184505e-4916L, -1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, 8.813735870195430252326093249797922409363e-1L, 5.757683115456107044131264955348449231681e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1.fp-16385L, -8.813735870195430252326093249797922409363e-1L, 5.757683115456107044131264955348449231681e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, 8.813735870195430252326093249797922409363e-1L, -5.757683115456107044131264955348449231681e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1.fp-16385L, -8.813735870195430252326093249797922409363e-1L, -5.757683115456107044131264955348449231681e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, 0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, -1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, -0.0L, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, -0.0L, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, -1.570796326794896605353533883825294686803L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, 8.813735870195430252326093249797922409363e-1L, 0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, 0.0L, -8.813735870195430252326093249797922409363e-1L, 0.0L),
    TEST_c_c (casinh, 0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, 8.813735870195430252326093249797922409363e-1L, -0.0L),
    TEST_c_c (casinh, -0x0.ffffffffffffffffffffffffffff8p0L, -0.0L, -8.813735870195430252326093249797922409363e-1L, -0.0L),
#endif
    TEST_c_c (casinh, 0x1p-112L, 0.5L, 2.223872343664663196047610660420739543258e-34L, 5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, 0x1p-112L, -0.5L, 2.223872343664663196047610660420739543258e-34L, -5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, -0x1p-112L, 0.5L, -2.223872343664663196047610660420739543258e-34L, 5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, -0x1p-112L, -0.5L, -2.223872343664663196047610660420739543258e-34L, -5.235987755982988730771072305465838140329e-1L),
    TEST_c_c (casinh, 0.5L, 0x1p-112L, 4.812118250596034474977589134243684231352e-1L, 1.722604110220899568310153280070398629037e-34L),
    TEST_c_c (casinh, -0.5L, 0x1p-112L, -4.812118250596034474977589134243684231352e-1L, 1.722604110220899568310153280070398629037e-34L),
    TEST_c_c (casinh, 0.5L, -0x1p-112L, 4.812118250596034474977589134243684231352e-1L, -1.722604110220899568310153280070398629037e-34L),
    TEST_c_c (casinh, -0.5L, -0x1p-112L, -4.812118250596034474977589134243684231352e-1L, -1.722604110220899568310153280070398629037e-34L),
    TEST_c_c (casinh, 0x1p-112L, 0x1p-112L, 1.925929944387235853055977942584927318538e-34L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, 0x1p-112L, -0x1p-112L, 1.925929944387235853055977942584927318538e-34L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, -0x1p-112L, 0x1p-112L, -1.925929944387235853055977942584927318538e-34L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, -0x1p-112L, -0x1p-112L, -1.925929944387235853055977942584927318538e-34L, -1.925929944387235853055977942584927318538e-34L),
#if MIN_EXP <= -16381
    TEST_c_c (casinh, 0x1.fp-16385L, 0x1p-112L, 8.142593549724601460479922838826119584417e-4933L, 1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1.fp-16385L, -0x1p-112L, 8.142593549724601460479922838826119584417e-4933L, -1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, 0x1p-112L, -8.142593549724601460479922838826119584417e-4933L, 1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1.fp-16385L, -0x1p-112L, -8.142593549724601460479922838826119584417e-4933L, -1.925929944387235853055977942584927318538e-34L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1p-112L, 0x1.fp-16385L, 1.925929944387235853055977942584927318538e-34L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1p-112L, 0x1.fp-16385L, -1.925929944387235853055977942584927318538e-34L, 8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, 0x1p-112L, -0x1.fp-16385L, 1.925929944387235853055977942584927318538e-34L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (casinh, -0x1p-112L, -0x1.fp-16385L, -1.925929944387235853055977942584927318538e-34L, -8.142593549724601460479922838826119584417e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (casinh, 0.0L, 0x1p-112L, 0.0L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, 0.0L, -0x1p-112L, 0.0L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, -0.0L, 0x1p-112L, -0.0L, 1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, -0.0L, -0x1p-112L, -0.0L, -1.925929944387235853055977942584927318538e-34L),
    TEST_c_c (casinh, 0x1p-112L, 0.0L, 1.925929944387235853055977942584927318538e-34L, 0.0L),
    TEST_c_c (casinh, -0x1p-112L, 0.0L, -1.925929944387235853055977942584927318538e-34L, 0.0L),
    TEST_c_c (casinh, 0x1p-112L, -0.0L, 1.925929944387235853055977942584927318538e-34L, -0.0L),
    TEST_c_c (casinh, -0x1p-112L, -0.0L, -1.925929944387235853055977942584927318538e-34L, -0.0L),

    TEST_c_c (casinh, 0.75L, 1.25L, 1.03171853444778027336364058631006594L, 0.911738290968487636358489564316731207L),
    TEST_c_c (casinh, -2, -3, -1.9686379257930962917886650952454982L, -0.96465850440760279204541105949953237L),
  };

static void
casinh_test (void)
{
  ALL_RM_TEST (casinh, 0, casinh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data catan_test_data[] =
  {
    TEST_c_c (catan, 0, 0, 0, 0),
    TEST_c_c (catan, minus_zero, 0, minus_zero, 0),
    TEST_c_c (catan, 0, minus_zero, 0, minus_zero),
    TEST_c_c (catan, minus_zero, minus_zero, minus_zero, minus_zero),

    TEST_c_c (catan, 1.0L, plus_zero, lit_pi_4_d, plus_zero),
    TEST_c_c (catan, 1.0L, minus_zero, lit_pi_4_d, minus_zero),
    TEST_c_c (catan, -1.0L, plus_zero, -lit_pi_4_d, plus_zero),
    TEST_c_c (catan, -1.0L, minus_zero, -lit_pi_4_d, minus_zero),
    TEST_c_c (catan, plus_zero, 1.0L, plus_zero, plus_infty, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catan, minus_zero, 1.0L, minus_zero, plus_infty, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catan, plus_zero, -1.0L, plus_zero, minus_infty, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catan, minus_zero, -1.0L, minus_zero, minus_infty, DIVIDE_BY_ZERO_EXCEPTION),

    TEST_c_c (catan, plus_infty, plus_infty, lit_pi_2_d, 0),
    TEST_c_c (catan, plus_infty, minus_infty, lit_pi_2_d, minus_zero),
    TEST_c_c (catan, minus_infty, plus_infty, -lit_pi_2_d, 0),
    TEST_c_c (catan, minus_infty, minus_infty, -lit_pi_2_d, minus_zero),


    TEST_c_c (catan, plus_infty, -10.0, lit_pi_2_d, minus_zero),
    TEST_c_c (catan, minus_infty, -10.0, -lit_pi_2_d, minus_zero),
    TEST_c_c (catan, plus_infty, minus_zero, lit_pi_2_d, minus_zero),
    TEST_c_c (catan, minus_infty, minus_zero, -lit_pi_2_d, minus_zero),
    TEST_c_c (catan, plus_infty, 0.0, lit_pi_2_d, 0),
    TEST_c_c (catan, minus_infty, 0.0, -lit_pi_2_d, 0),
    TEST_c_c (catan, plus_infty, 0.1L, lit_pi_2_d, 0),
    TEST_c_c (catan, minus_infty, 0.1L, -lit_pi_2_d, 0),

    TEST_c_c (catan, 0.0, minus_infty, lit_pi_2_d, minus_zero),
    TEST_c_c (catan, minus_zero, minus_infty, -lit_pi_2_d, minus_zero),
    TEST_c_c (catan, 100.0, minus_infty, lit_pi_2_d, minus_zero),
    TEST_c_c (catan, -100.0, minus_infty, -lit_pi_2_d, minus_zero),

    TEST_c_c (catan, 0.0, plus_infty, lit_pi_2_d, 0),
    TEST_c_c (catan, minus_zero, plus_infty, -lit_pi_2_d, 0),
    TEST_c_c (catan, 0.5, plus_infty, lit_pi_2_d, 0),
    TEST_c_c (catan, -0.5, plus_infty, -lit_pi_2_d, 0),

    TEST_c_c (catan, qnan_value, 0.0, qnan_value, 0),
    TEST_c_c (catan, qnan_value, minus_zero, qnan_value, minus_zero),

    TEST_c_c (catan, qnan_value, plus_infty, qnan_value, 0),
    TEST_c_c (catan, qnan_value, minus_infty, qnan_value, minus_zero),

    TEST_c_c (catan, 0.0, qnan_value, qnan_value, qnan_value),
    TEST_c_c (catan, minus_zero, qnan_value, qnan_value, qnan_value),

    TEST_c_c (catan, plus_infty, qnan_value, lit_pi_2_d, 0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (catan, minus_infty, qnan_value, -lit_pi_2_d, 0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (catan, qnan_value, 10.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (catan, qnan_value, -10.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (catan, 0.75, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (catan, -0.75, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (catan, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (catan, 0x1p50L, 0.0L, 1.570796326794895731052901991514519103193L, 0.0L),
    TEST_c_c (catan, 0x1p50L, -0.0L, 1.570796326794895731052901991514519103193L, -0.0L),
    TEST_c_c (catan, -0x1p50L, 0.0L, -1.570796326794895731052901991514519103193L, 0.0L),
    TEST_c_c (catan, -0x1p50L, -0.0L, -1.570796326794895731052901991514519103193L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p50L, 1.570796326794896619231321691639751442099L, 8.881784197001252323389053344728897997441e-16L),
    TEST_c_c (catan, -0.0L, 0x1p50L, -1.570796326794896619231321691639751442099L, 8.881784197001252323389053344728897997441e-16L),
    TEST_c_c (catan, 0.0L, -0x1p50L, 1.570796326794896619231321691639751442099L, -8.881784197001252323389053344728897997441e-16L),
    TEST_c_c (catan, -0.0L, -0x1p50L, -1.570796326794896619231321691639751442099L, -8.881784197001252323389053344728897997441e-16L),
#if !TEST_COND_binary32
    TEST_c_c (catan, 0x1p500L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1p500L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1p500L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1p500L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p500L, 1.570796326794896619231321691639751442099L, 3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (catan, -0.0L, 0x1p500L, -1.570796326794896619231321691639751442099L, 3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (catan, 0.0L, -0x1p500L, 1.570796326794896619231321691639751442099L, -3.054936363499604682051979393213617699789e-151L),
    TEST_c_c (catan, -0.0L, -0x1p500L, -1.570796326794896619231321691639751442099L, -3.054936363499604682051979393213617699789e-151L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catan, 0x1p5000L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1p5000L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1p5000L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1p5000L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p5000L, 1.570796326794896619231321691639751442099L, 7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (catan, -0.0L, 0x1p5000L, -1.570796326794896619231321691639751442099L, 7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (catan, 0.0L, -0x1p5000L, 1.570796326794896619231321691639751442099L, -7.079811261048172892385615158694057552948e-1506L),
    TEST_c_c (catan, -0.0L, -0x1p5000L, -1.570796326794896619231321691639751442099L, -7.079811261048172892385615158694057552948e-1506L),
#endif
    TEST_c_c (catan, 0x1p63L, 0.5L, 1.570796326794896619122901474391200998698L, 5.877471754111437539843682686111228389007e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1p63L, -0.5L, 1.570796326794896619122901474391200998698L, -5.877471754111437539843682686111228389007e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1p63L, 0.5L, -1.570796326794896619122901474391200998698L, 5.877471754111437539843682686111228389007e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1p63L, -0.5L, -1.570796326794896619122901474391200998698L, -5.877471754111437539843682686111228389007e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0.5L, 0x1p63L, 1.570796326794896619231321691639751442093L, 1.084202172485504434007452800869941711427e-19L),
    TEST_c_c (catan, -0.5L, 0x1p63L, -1.570796326794896619231321691639751442093L, 1.084202172485504434007452800869941711427e-19L),
    TEST_c_c (catan, 0.5L, -0x1p63L, 1.570796326794896619231321691639751442093L, -1.084202172485504434007452800869941711427e-19L),
    TEST_c_c (catan, -0.5L, -0x1p63L, -1.570796326794896619231321691639751442093L, -1.084202172485504434007452800869941711427e-19L),
#if !TEST_COND_binary32
    TEST_c_c (catan, 0x1p511L, 0.5L, 1.570796326794896619231321691639751442099L, 1.112536929253600691545116358666202032110e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p511L, -0.5L, 1.570796326794896619231321691639751442099L, -1.112536929253600691545116358666202032110e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1p511L, 0.5L, -1.570796326794896619231321691639751442099L, 1.112536929253600691545116358666202032110e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1p511L, -0.5L, -1.570796326794896619231321691639751442099L, -1.112536929253600691545116358666202032110e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0.5L, 0x1p511L, 1.570796326794896619231321691639751442099L, 1.491668146240041348658193063092586767475e-154L),
    TEST_c_c (catan, -0.5L, 0x1p511L, -1.570796326794896619231321691639751442099L, 1.491668146240041348658193063092586767475e-154L),
    TEST_c_c (catan, 0.5L, -0x1p511L, 1.570796326794896619231321691639751442099L, -1.491668146240041348658193063092586767475e-154L),
    TEST_c_c (catan, -0.5L, -0x1p511L, -1.570796326794896619231321691639751442099L, -1.491668146240041348658193063092586767475e-154L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catan, 0x1p8191L, 0.5L, 1.570796326794896619231321691639751442099L, 1.681051571556046753131338908660876301299e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1p8191L, -0.5L, 1.570796326794896619231321691639751442099L, -1.681051571556046753131338908660876301299e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1p8191L, 0.5L, -1.570796326794896619231321691639751442099L, 1.681051571556046753131338908660876301299e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1p8191L, -0.5L, -1.570796326794896619231321691639751442099L, -1.681051571556046753131338908660876301299e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0.5L, 0x1p8191L, 1.570796326794896619231321691639751442099L, 1.833603867554847165621412392048483165956e-2466L),
    TEST_c_c (catan, -0.5L, 0x1p8191L, -1.570796326794896619231321691639751442099L, 1.833603867554847165621412392048483165956e-2466L),
    TEST_c_c (catan, 0.5L, -0x1p8191L, 1.570796326794896619231321691639751442099L, -1.833603867554847165621412392048483165956e-2466L),
    TEST_c_c (catan, -0.5L, -0x1p8191L, -1.570796326794896619231321691639751442099L, -1.833603867554847165621412392048483165956e-2466L),
#endif
    TEST_c_c (catan, 0x1p100L, 0.0L, 1.570796326794896619231321691638962581193L, 0.0L),
    TEST_c_c (catan, 0x1p100L, -0.0L, 1.570796326794896619231321691638962581193L, -0.0L),
    TEST_c_c (catan, -0x1p100L, 0.0L, -1.570796326794896619231321691638962581193L, 0.0L),
    TEST_c_c (catan, -0x1p100L, -0.0L, -1.570796326794896619231321691638962581193L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p100L, 1.570796326794896619231321691639751442099L, 7.888609052210118054117285652827862296732e-31L),
    TEST_c_c (catan, -0.0L, 0x1p100L, -1.570796326794896619231321691639751442099L, 7.888609052210118054117285652827862296732e-31L),
    TEST_c_c (catan, 0.0L, -0x1p100L, 1.570796326794896619231321691639751442099L, -7.888609052210118054117285652827862296732e-31L),
    TEST_c_c (catan, -0.0L, -0x1p100L, -1.570796326794896619231321691639751442099L, -7.888609052210118054117285652827862296732e-31L),
    TEST_c_c (catan, 0x1.fp127L, 0.0L, 1.570796326794896619231321691639751442096L, 0.0L),
    TEST_c_c (catan, 0x1.fp127L, -0.0L, 1.570796326794896619231321691639751442096L, -0.0L),
    TEST_c_c (catan, -0x1.fp127L, 0.0L, -1.570796326794896619231321691639751442096L, 0.0L),
    TEST_c_c (catan, -0x1.fp127L, -0.0L, -1.570796326794896619231321691639751442096L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1.fp127L, 1.570796326794896619231321691639751442099L, 3.033533808573645181854803967025150136306e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0.0L, 0x1.fp127L, -1.570796326794896619231321691639751442099L, 3.033533808573645181854803967025150136306e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0.0L, -0x1.fp127L, 1.570796326794896619231321691639751442099L, -3.033533808573645181854803967025150136306e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0.0L, -0x1.fp127L, -1.570796326794896619231321691639751442099L, -3.033533808573645181854803967025150136306e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1.fp127L, 0x1.fp127L, 1.570796326794896619231321691639751442097L, 1.516766904286822590927401983512575068153e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1.fp127L, -0x1.fp127L, 1.570796326794896619231321691639751442097L, -1.516766904286822590927401983512575068153e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1.fp127L, 0x1.fp127L, -1.570796326794896619231321691639751442097L, 1.516766904286822590927401983512575068153e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1.fp127L, -0x1.fp127L, -1.570796326794896619231321691639751442097L, -1.516766904286822590927401983512575068153e-39L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (catan, 0x1p900L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1p900L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1p900L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1p900L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p900L, 1.570796326794896619231321691639751442099L, 1.183052186166774710972751597518026531652e-271L),
    TEST_c_c (catan, -0.0L, 0x1p900L, -1.570796326794896619231321691639751442099L, 1.183052186166774710972751597518026531652e-271L),
    TEST_c_c (catan, 0.0L, -0x1p900L, 1.570796326794896619231321691639751442099L, -1.183052186166774710972751597518026531652e-271L),
    TEST_c_c (catan, -0.0L, -0x1p900L, -1.570796326794896619231321691639751442099L, -1.183052186166774710972751597518026531652e-271L),
    TEST_c_c (catan, 0x1.fp1023L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1.fp1023L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1.fp1023L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1.fp1023L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1.fp1023L, 1.570796326794896619231321691639751442099L, 5.742126086470197117652213464083623391533e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0.0L, 0x1.fp1023L, -1.570796326794896619231321691639751442099L, 5.742126086470197117652213464083623391533e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0.0L, -0x1.fp1023L, 1.570796326794896619231321691639751442099L, -5.742126086470197117652213464083623391533e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0.0L, -0x1.fp1023L, -1.570796326794896619231321691639751442099L, -5.742126086470197117652213464083623391533e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1.fp1023L, 0x1.fp1023L, 1.570796326794896619231321691639751442099L, 2.871063043235098558826106732041811695767e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1.fp1023L, -0x1.fp1023L, 1.570796326794896619231321691639751442099L, -2.871063043235098558826106732041811695767e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.fp1023L, 0x1.fp1023L, -1.570796326794896619231321691639751442099L, 2.871063043235098558826106732041811695767e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.fp1023L, -0x1.fp1023L, -1.570796326794896619231321691639751442099L, -2.871063043235098558826106732041811695767e-309L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catan, 0x1p10000L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1p10000L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1p10000L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1p10000L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1p10000L, 1.570796326794896619231321691639751442099L, 5.012372749206452009297555933742977749322e-3011L),
    TEST_c_c (catan, -0.0L, 0x1p10000L, -1.570796326794896619231321691639751442099L, 5.012372749206452009297555933742977749322e-3011L),
    TEST_c_c (catan, 0.0L, -0x1p10000L, 1.570796326794896619231321691639751442099L, -5.012372749206452009297555933742977749322e-3011L),
    TEST_c_c (catan, -0.0L, -0x1p10000L, -1.570796326794896619231321691639751442099L, -5.012372749206452009297555933742977749322e-3011L),
    TEST_c_c (catan, 0x1.fp16383L, 0.0L, 1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, 0x1.fp16383L, -0.0L, 1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, -0x1.fp16383L, 0.0L, -1.570796326794896619231321691639751442099L, 0.0L),
    TEST_c_c (catan, -0x1.fp16383L, -0.0L, -1.570796326794896619231321691639751442099L, -0.0L),
    TEST_c_c (catan, 0.0L, 0x1.fp16383L, 1.570796326794896619231321691639751442099L, 8.676395208031209048419813722120651877672e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0.0L, 0x1.fp16383L, -1.570796326794896619231321691639751442099L, 8.676395208031209048419813722120651877672e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0.0L, -0x1.fp16383L, 1.570796326794896619231321691639751442099L, -8.676395208031209048419813722120651877672e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0.0L, -0x1.fp16383L, -1.570796326794896619231321691639751442099L, -8.676395208031209048419813722120651877672e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1.fp16383L, 0x1.fp16383L, 1.570796326794896619231321691639751442099L, 4.338197604015604524209906861060325938836e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1.fp16383L, -0x1.fp16383L, 1.570796326794896619231321691639751442099L, -4.338197604015604524209906861060325938836e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.fp16383L, 0x1.fp16383L, -1.570796326794896619231321691639751442099L, 4.338197604015604524209906861060325938836e-4933L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.fp16383L, -0x1.fp16383L, -1.570796326794896619231321691639751442099L, -4.338197604015604524209906861060325938836e-4933L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (catan, 0x1p-13L, 1.0L, 7.854286809755354140031716771044626356262e-1L, 4.852030264850939738801379894163661227127L),
    TEST_c_c (catan, 0x1p-13L, -1.0L, 7.854286809755354140031716771044626356262e-1L, -4.852030264850939738801379894163661227127L),
    TEST_c_c (catan, -0x1p-13L, 1.0L, -7.854286809755354140031716771044626356262e-1L, 4.852030264850939738801379894163661227127L),
    TEST_c_c (catan, -0x1p-13L, -1.0L, -7.854286809755354140031716771044626356262e-1L, -4.852030264850939738801379894163661227127L),
    TEST_c_c (catan, 1.0L, 0x1p-13L, 7.853981671227386080775748393881580082970e-1L, 6.103515609841754902688560615027452023669e-5L),
    TEST_c_c (catan, -1.0L, 0x1p-13L, -7.853981671227386080775748393881580082970e-1L, 6.103515609841754902688560615027452023669e-5L),
    TEST_c_c (catan, 1.0L, -0x1p-13L, 7.853981671227386080775748393881580082970e-1L, -6.103515609841754902688560615027452023669e-5L),
    TEST_c_c (catan, -1.0L, -0x1p-13L, -7.853981671227386080775748393881580082970e-1L, -6.103515609841754902688560615027452023669e-5L),
    TEST_c_c (catan, 0x1p-27L, 1.0L, 7.853981652600934588466178684534110069553e-1L, 9.704060527839234335310696652368086117807L),
    TEST_c_c (catan, 0x1p-27L, -1.0L, 7.853981652600934588466178684534110069553e-1L, -9.704060527839234335310696652368086117807L),
    TEST_c_c (catan, -0x1p-27L, 1.0L, -7.853981652600934588466178684534110069553e-1L, 9.704060527839234335310696652368086117807L),
    TEST_c_c (catan, -0x1p-27L, -1.0L, -7.853981652600934588466178684534110069553e-1L, -9.704060527839234335310696652368086117807L),
    TEST_c_c (catan, 1.0L, 0x1p-27L, 7.853981633974483234934486536343324763447e-1L, 3.725290298461914028034141143623846306386e-9L),
    TEST_c_c (catan, -1.0L, 0x1p-27L, -7.853981633974483234934486536343324763447e-1L, 3.725290298461914028034141143623846306386e-9L),
    TEST_c_c (catan, 1.0L, -0x1p-27L, 7.853981633974483234934486536343324763447e-1L, -3.725290298461914028034141143623846306386e-9L),
    TEST_c_c (catan, -1.0L, -0x1p-27L, -7.853981633974483234934486536343324763447e-1L, -3.725290298461914028034141143623846306386e-9L),
    TEST_c_c (catan, 0x1p-33L, 1.0L, 7.853981634265521400723945494331241018449e-1L, 1.178350206951907026009379309773625595762e1L),
    TEST_c_c (catan, 0x1p-33L, -1.0L, 7.853981634265521400723945494331241018449e-1L, -1.178350206951907026009379309773625595762e1L),
    TEST_c_c (catan, -0x1p-33L, 1.0L, -7.853981634265521400723945494331241018449e-1L, 1.178350206951907026009379309773625595762e1L),
    TEST_c_c (catan, -0x1p-33L, -1.0L, -7.853981634265521400723945494331241018449e-1L, -1.178350206951907026009379309773625595762e1L),
    TEST_c_c (catan, 1.0L, 0x1p-33L, 7.853981633974483096190489776088929224056e-1L, 5.820766091346740722643102318246316469910e-11L),
    TEST_c_c (catan, -1.0L, 0x1p-33L, -7.853981633974483096190489776088929224056e-1L, 5.820766091346740722643102318246316469910e-11L),
    TEST_c_c (catan, 1.0L, -0x1p-33L, 7.853981633974483096190489776088929224056e-1L, -5.820766091346740722643102318246316469910e-11L),
    TEST_c_c (catan, -1.0L, -0x1p-33L, -7.853981633974483096190489776088929224056e-1L, -5.820766091346740722643102318246316469910e-11L),
    TEST_c_c (catan, 0x1p-54L, 1.0L, 7.853981633974483234934486536343324763447e-1L, 1.906154746539849600897388334009985581467e1L),
    TEST_c_c (catan, 0x1p-54L, -1.0L, 7.853981633974483234934486536343324763447e-1L, -1.906154746539849600897388334009985581467e1L),
    TEST_c_c (catan, -0x1p-54L, 1.0L, -7.853981633974483234934486536343324763447e-1L, 1.906154746539849600897388334009985581467e1L),
    TEST_c_c (catan, -0x1p-54L, -1.0L, -7.853981633974483234934486536343324763447e-1L, -1.906154746539849600897388334009985581467e1L),
    TEST_c_c (catan, 1.0L, 0x1p-54L, 7.853981633974483096156608458198764914213e-1L, 2.775557561562891351059079170227049355775e-17L),
    TEST_c_c (catan, -1.0L, 0x1p-54L, -7.853981633974483096156608458198764914213e-1L, 2.775557561562891351059079170227049355775e-17L),
    TEST_c_c (catan, 1.0L, -0x1p-54L, 7.853981633974483096156608458198764914213e-1L, -2.775557561562891351059079170227049355775e-17L),
    TEST_c_c (catan, -1.0L, -0x1p-54L, -7.853981633974483096156608458198764914213e-1L, -2.775557561562891351059079170227049355775e-17L),
    TEST_c_c (catan, 0x1p-57L, 1.0L, 7.853981633974483113503843217966828154612e-1L, 2.010126823623841397309973152228712047720e1L),
    TEST_c_c (catan, 0x1p-57L, -1.0L, 7.853981633974483113503843217966828154612e-1L, -2.010126823623841397309973152228712047720e1L),
    TEST_c_c (catan, -0x1p-57L, 1.0L, -7.853981633974483113503843217966828154612e-1L, 2.010126823623841397309973152228712047720e1L),
    TEST_c_c (catan, -0x1p-57L, -1.0L, -7.853981633974483113503843217966828154612e-1L, -2.010126823623841397309973152228712047720e1L),
    TEST_c_c (catan, 1.0L, 0x1p-57L, 7.853981633974483096156608458198757330864e-1L, 3.469446951953614188823848962783813448721e-18L),
    TEST_c_c (catan, -1.0L, 0x1p-57L, -7.853981633974483096156608458198757330864e-1L, 3.469446951953614188823848962783813448721e-18L),
    TEST_c_c (catan, 1.0L, -0x1p-57L, 7.853981633974483096156608458198757330864e-1L, -3.469446951953614188823848962783813448721e-18L),
    TEST_c_c (catan, -1.0L, -0x1p-57L, -7.853981633974483096156608458198757330864e-1L, -3.469446951953614188823848962783813448721e-18L),
    TEST_c_c (catan, 0x1p-13L, 0x1.000002p0L, 7.859169620684960844300240092596908675974e-1L, 4.852030056234795712498957387213592193975L),
    TEST_c_c (catan, 0x1p-13L, -0x1.000002p0L, 7.859169620684960844300240092596908675974e-1L, -4.852030056234795712498957387213592193975L),
    TEST_c_c (catan, -0x1p-13L, 0x1.000002p0L, -7.859169620684960844300240092596908675974e-1L, 4.852030056234795712498957387213592193975L),
    TEST_c_c (catan, -0x1p-13L, -0x1.000002p0L, -7.859169620684960844300240092596908675974e-1L, -4.852030056234795712498957387213592193975L),
    TEST_c_c (catan, 0x1.000002p0L, 0x1p-13L, 7.853982267273793866654490522673596014524e-1L, 6.103514882246036852433556327261700380577e-5L),
    TEST_c_c (catan, -0x1.000002p0L, 0x1p-13L, -7.853982267273793866654490522673596014524e-1L, 6.103514882246036852433556327261700380577e-5L),
    TEST_c_c (catan, 0x1.000002p0L, -0x1p-13L, 7.853982267273793866654490522673596014524e-1L, -6.103514882246036852433556327261700380577e-5L),
    TEST_c_c (catan, -0x1.000002p0L, -0x1p-13L, -7.853982267273793866654490522673596014524e-1L, -6.103514882246036852433556327261700380577e-5L),
    TEST_c_c (catan, 0x1p-13L, 0x0.ffffffp0L, 7.851845403708474595909269086711426246675e-1L, 4.852030190345140708455871037447717761868L),
    TEST_c_c (catan, 0x1p-13L, -0x0.ffffffp0L, 7.851845403708474595909269086711426246675e-1L, -4.852030190345140708455871037447717761868L),
    TEST_c_c (catan, -0x1p-13L, 0x0.ffffffp0L, -7.851845403708474595909269086711426246675e-1L, 4.852030190345140708455871037447717761868L),
    TEST_c_c (catan, -0x1p-13L, -0x0.ffffffp0L, -7.851845403708474595909269086711426246675e-1L, -4.852030190345140708455871037447717761868L),
    TEST_c_c (catan, 0x0.ffffffp0L, 0x1p-13L, 7.853981373204155542484315721351697277336e-1L, 6.103515973639646453881721999956617260502e-5L),
    TEST_c_c (catan, -0x0.ffffffp0L, 0x1p-13L, -7.853981373204155542484315721351697277336e-1L, 6.103515973639646453881721999956617260502e-5L),
    TEST_c_c (catan, 0x0.ffffffp0L, -0x1p-13L, 7.853981373204155542484315721351697277336e-1L, -6.103515973639646453881721999956617260502e-5L),
    TEST_c_c (catan, -0x0.ffffffp0L, -0x1p-13L, -7.853981373204155542484315721351697277336e-1L, -6.103515973639646453881721999956617260502e-5L),
#if !TEST_COND_binary32
    TEST_c_c (catan, 0x1p-27L, 0x1.0000000000001p0L, 7.853981801612546526942695000283242525531e-1L, 9.704060527839234168777242958594699810015L),
    TEST_c_c (catan, 0x1p-27L, -0x1.0000000000001p0L, 7.853981801612546526942695000283242525531e-1L, -9.704060527839234168777242958594699810015L),
    TEST_c_c (catan, -0x1p-27L, 0x1.0000000000001p0L, -7.853981801612546526942695000283242525531e-1L, 9.704060527839234168777242958594699810015L),
    TEST_c_c (catan, -0x1p-27L, -0x1.0000000000001p0L, -7.853981801612546526942695000283242525531e-1L, -9.704060527839234168777242958594699810015L),
    TEST_c_c (catan, 0x1.0000000000001p0L, 0x1p-27L, 7.853981633974484345157511161499711112683e-1L, 3.725290298461913200853528590596263270474e-9L),
    TEST_c_c (catan, -0x1.0000000000001p0L, 0x1p-27L, -7.853981633974484345157511161499711112683e-1L, 3.725290298461913200853528590596263270474e-9L),
    TEST_c_c (catan, 0x1.0000000000001p0L, -0x1p-27L, 7.853981633974484345157511161499711112683e-1L, -3.725290298461913200853528590596263270474e-9L),
    TEST_c_c (catan, -0x1.0000000000001p0L, -0x1p-27L, -7.853981633974484345157511161499711112683e-1L, -3.725290298461913200853528590596263270474e-9L),
    TEST_c_c (catan, 0x1p-27L, 0x0.fffffffffffff8p0L, 7.853981578095128619227903983047292781021e-1L, 9.704060527839234252043969805481351363824L),
    TEST_c_c (catan, 0x1p-27L, -0x0.fffffffffffff8p0L, 7.853981578095128619227903983047292781021e-1L, -9.704060527839234252043969805481351363824L),
    TEST_c_c (catan, -0x1p-27L, 0x0.fffffffffffff8p0L, -7.853981578095128619227903983047292781021e-1L, 9.704060527839234252043969805481351363824L),
    TEST_c_c (catan, -0x1p-27L, -0x0.fffffffffffff8p0L, -7.853981578095128619227903983047292781021e-1L, -9.704060527839234252043969805481351363824L),
    TEST_c_c (catan, 0x0.fffffffffffff8p0L, 0x1p-27L, 7.853981633974482679822974223765039144191e-1L, 3.725290298461914441624447420137706700965e-9L),
    TEST_c_c (catan, -0x0.fffffffffffff8p0L, 0x1p-27L, -7.853981633974482679822974223765039144191e-1L, 3.725290298461914441624447420137706700965e-9L),
    TEST_c_c (catan, 0x0.fffffffffffff8p0L, -0x1p-27L, 7.853981633974482679822974223765039144191e-1L, -3.725290298461914441624447420137706700965e-9L),
    TEST_c_c (catan, -0x0.fffffffffffff8p0L, -0x1p-27L, -7.853981633974482679822974223765039144191e-1L, -3.725290298461914441624447420137706700965e-9L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (catan, 0x1p-33L, 0x1.0000000000000002p0L, 7.853981638922134273801338071094141188767e-1L, 1.178350206951907025990405771755129268176e1L),
    TEST_c_c (catan, 0x1p-33L, -0x1.0000000000000002p0L, 7.853981638922134273801338071094141188767e-1L, -1.178350206951907025990405771755129268176e1L),
    TEST_c_c (catan, -0x1p-33L, 0x1.0000000000000002p0L, -7.853981638922134273801338071094141188767e-1L, 1.178350206951907025990405771755129268176e1L),
    TEST_c_c (catan, -0x1p-33L, -0x1.0000000000000002p0L, -7.853981638922134273801338071094141188767e-1L, -1.178350206951907025990405771755129268176e1L),
    TEST_c_c (catan, 0x1.0000000000000002p0L, 0x1p-33L, 7.853981633974483096732590862331681441026e-1L, 5.820766091346740722012013594069507025615e-11L),
    TEST_c_c (catan, -0x1.0000000000000002p0L, 0x1p-33L, -7.853981633974483096732590862331681441026e-1L, 5.820766091346740722012013594069507025615e-11L),
    TEST_c_c (catan, 0x1.0000000000000002p0L, -0x1p-33L, 7.853981633974483096732590862331681441026e-1L, -5.820766091346740722012013594069507025615e-11L),
    TEST_c_c (catan, -0x1.0000000000000002p0L, -0x1p-33L, -7.853981633974483096732590862331681441026e-1L, -5.820766091346740722012013594069507025615e-11L),
    TEST_c_c (catan, 0x1p-33L, 0x0.ffffffffffffffffp0L, 7.853981631937214964185249205444919953948e-1L, 1.178350206951907026002603046195591193050e1L),
    TEST_c_c (catan, 0x1p-33L, -0x0.ffffffffffffffffp0L, 7.853981631937214964185249205444919953948e-1L, -1.178350206951907026002603046195591193050e1L),
    TEST_c_c (catan, -0x1p-33L, 0x0.ffffffffffffffffp0L, -7.853981631937214964185249205444919953948e-1L, 1.178350206951907026002603046195591193050e1L),
    TEST_c_c (catan, -0x1p-33L, -0x0.ffffffffffffffffp0L, -7.853981631937214964185249205444919953948e-1L, -1.178350206951907026002603046195591193050e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, 0x1p-33L, 7.853981633974483095919439232967553115548e-1L, 5.820766091346740722958646680334721192083e-11L),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, 0x1p-33L, -7.853981633974483095919439232967553115548e-1L, 5.820766091346740722958646680334721192083e-11L),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, -0x1p-33L, 7.853981633974483095919439232967553115548e-1L, -5.820766091346740722958646680334721192083e-11L),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, -0x1p-33L, -7.853981633974483095919439232967553115548e-1L, -5.820766091346740722958646680334721192083e-11L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (catan, 0x1p-54L, 0x1.000000000000000000000000008p0L, 7.853981633974485455380535786656405610710e-1L, 1.906154746539849600897388334009981267384e1L),
    TEST_c_c (catan, 0x1p-54L, -0x1.000000000000000000000000008p0L, 7.853981633974485455380535786656405610710e-1L, -1.906154746539849600897388334009981267384e1L),
    TEST_c_c (catan, -0x1p-54L, 0x1.000000000000000000000000008p0L, -7.853981633974485455380535786656405610710e-1L, 1.906154746539849600897388334009981267384e1L),
    TEST_c_c (catan, -0x1p-54L, -0x1.000000000000000000000000008p0L, -7.853981633974485455380535786656405610710e-1L, -1.906154746539849600897388334009981267384e1L),
    TEST_c_c (catan, 0x1.000000000000000000000000008p0L, 0x1p-54L, 7.853981633974483096156608458198888173729e-1L, 2.775557561562891351059079170226980932999e-17L),
    TEST_c_c (catan, -0x1.000000000000000000000000008p0L, 0x1p-54L, -7.853981633974483096156608458198888173729e-1L, 2.775557561562891351059079170226980932999e-17L),
    TEST_c_c (catan, 0x1.000000000000000000000000008p0L, -0x1p-54L, 7.853981633974483096156608458198888173729e-1L, -2.775557561562891351059079170226980932999e-17L),
    TEST_c_c (catan, -0x1.000000000000000000000000008p0L, -0x1p-54L, -7.853981633974483096156608458198888173729e-1L, -2.775557561562891351059079170226980932999e-17L),
    TEST_c_c (catan, 0x1p-54L, 0x0.ffffffffffffffffffffffffffcp0L, 7.853981633974482124711461911186784339815e-1L, 1.906154746539849600897388334009984040723e1L),
    TEST_c_c (catan, 0x1p-54L, -0x0.ffffffffffffffffffffffffffcp0L, 7.853981633974482124711461911186784339815e-1L, -1.906154746539849600897388334009984040723e1L),
    TEST_c_c (catan, -0x1p-54L, 0x0.ffffffffffffffffffffffffffcp0L, -7.853981633974482124711461911186784339815e-1L, 1.906154746539849600897388334009984040723e1L),
    TEST_c_c (catan, -0x1p-54L, -0x0.ffffffffffffffffffffffffffcp0L, -7.853981633974482124711461911186784339815e-1L, -1.906154746539849600897388334009984040723e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-54L, 7.853981633974483096156608458198703284454e-1L, 2.775557561562891351059079170227083567164e-17L),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-54L, -7.853981633974483096156608458198703284454e-1L, 2.775557561562891351059079170227083567164e-17L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-54L, 7.853981633974483096156608458198703284454e-1L, -2.775557561562891351059079170227083567164e-17L),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-54L, -7.853981633974483096156608458198703284454e-1L, -2.775557561562891351059079170227083567164e-17L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (catan, 0x1p-57L, 0x1.0000000000000000000000000001p0L, 7.853981633974483252281721296111395707566e-1L, 2.010126823623841397309973152228712033275e1L),
    TEST_c_c (catan, 0x1p-57L, -0x1.0000000000000000000000000001p0L, 7.853981633974483252281721296111395707566e-1L, -2.010126823623841397309973152228712033275e1L),
    TEST_c_c (catan, -0x1p-57L, 0x1.0000000000000000000000000001p0L, -7.853981633974483252281721296111395707566e-1L, 2.010126823623841397309973152228712033275e1L),
    TEST_c_c (catan, -0x1p-57L, -0x1.0000000000000000000000000001p0L, -7.853981633974483252281721296111395707566e-1L, -2.010126823623841397309973152228712033275e1L),
    TEST_c_c (catan, 0x1.0000000000000000000000000001p0L, 0x1p-57L, 7.853981633974483096156608458198758293829e-1L, 3.469446951953614188823848962783812780530e-18L),
    TEST_c_c (catan, -0x1.0000000000000000000000000001p0L, 0x1p-57L, -7.853981633974483096156608458198758293829e-1L, 3.469446951953614188823848962783812780530e-18L),
    TEST_c_c (catan, 0x1.0000000000000000000000000001p0L, -0x1p-57L, 7.853981633974483096156608458198758293829e-1L, -3.469446951953614188823848962783812780530e-18L),
    TEST_c_c (catan, -0x1.0000000000000000000000000001p0L, -0x1p-57L, -7.853981633974483096156608458198758293829e-1L, -3.469446951953614188823848962783812780530e-18L),
    TEST_c_c (catan, 0x1p-57L, 0x0.ffffffffffffffffffffffffffff8p0L, 7.853981633974483044114904178894544378135e-1L, 2.010126823623841397309973152228712040498e1L),
    TEST_c_c (catan, 0x1p-57L, -0x0.ffffffffffffffffffffffffffff8p0L, 7.853981633974483044114904178894544378135e-1L, -2.010126823623841397309973152228712040498e1L),
    TEST_c_c (catan, -0x1p-57L, 0x0.ffffffffffffffffffffffffffff8p0L, -7.853981633974483044114904178894544378135e-1L, 2.010126823623841397309973152228712040498e1L),
    TEST_c_c (catan, -0x1p-57L, -0x0.ffffffffffffffffffffffffffff8p0L, -7.853981633974483044114904178894544378135e-1L, -2.010126823623841397309973152228712040498e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-57L, 7.853981633974483096156608458198756849381e-1L, 3.469446951953614188823848962783813782817e-18L),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-57L, -7.853981633974483096156608458198756849381e-1L, 3.469446951953614188823848962783813782817e-18L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-57L, 7.853981633974483096156608458198756849381e-1L, -3.469446951953614188823848962783813782817e-18L),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-57L, -7.853981633974483096156608458198756849381e-1L, -3.469446951953614188823848962783813782817e-18L),
#endif
    TEST_c_c (catan, 1.0L, 0x1p-64L, 7.853981633974483096156608458198757210500e-1L, 2.710505431213761085018632002174854278563e-20L),
    TEST_c_c (catan, 1.0L, -0x1p-64L, 7.853981633974483096156608458198757210500e-1L, -2.710505431213761085018632002174854278563e-20L),
    TEST_c_c (catan, -1.0L, 0x1p-64L, -7.853981633974483096156608458198757210500e-1L, 2.710505431213761085018632002174854278563e-20L),
    TEST_c_c (catan, -1.0L, -0x1p-64L, -7.853981633974483096156608458198757210500e-1L, -2.710505431213761085018632002174854278563e-20L),
    TEST_c_c (catan, 0x1p-64L, 1.0L, 7.853981633974483096292133729759445264744e-1L, 2.252728336819822255606004394739073846245e1L),
    TEST_c_c (catan, -0x1p-64L, 1.0L, -7.853981633974483096292133729759445264744e-1L, 2.252728336819822255606004394739073846245e1L),
    TEST_c_c (catan, 0x1p-64L, -1.0L, 7.853981633974483096292133729759445264744e-1L, -2.252728336819822255606004394739073846245e1L),
    TEST_c_c (catan, -0x1p-64L, -1.0L, -7.853981633974483096292133729759445264744e-1L, -2.252728336819822255606004394739073846245e1L),
    TEST_c_c (catan, 1.0L, 0x1.3p-73L, 7.853981633974483096156608458198757210493e-1L, 6.286572655403010329022706059731717687100e-23L),
    TEST_c_c (catan, 1.0L, -0x1.3p-73L, 7.853981633974483096156608458198757210493e-1L, -6.286572655403010329022706059731717687100e-23L),
    TEST_c_c (catan, -1.0L, 0x1.3p-73L, -7.853981633974483096156608458198757210493e-1L, 6.286572655403010329022706059731717687100e-23L),
    TEST_c_c (catan, -1.0L, -0x1.3p-73L, -7.853981633974483096156608458198757210493e-1L, -6.286572655403010329022706059731717687100e-23L),
    TEST_c_c (catan, 0x1.3p-73L, 1.0L, 7.853981633974483096156922786831527361009e-1L, 2.556052055225464683726753902092495938633e1L),
    TEST_c_c (catan, -0x1.3p-73L, 1.0L, -7.853981633974483096156922786831527361009e-1L, 2.556052055225464683726753902092495938633e1L),
    TEST_c_c (catan, 0x1.3p-73L, -1.0L, 7.853981633974483096156922786831527361009e-1L, -2.556052055225464683726753902092495938633e1L),
    TEST_c_c (catan, -0x1.3p-73L, -1.0L, -7.853981633974483096156922786831527361009e-1L, -2.556052055225464683726753902092495938633e1L),
    TEST_c_c (catan, 1.0L, 0x1p-124L, 7.853981633974483096156608458198757210493e-1L, 2.350988701644575015937473074444491355637e-38L),
    TEST_c_c (catan, 1.0L, -0x1p-124L, 7.853981633974483096156608458198757210493e-1L, -2.350988701644575015937473074444491355637e-38L),
    TEST_c_c (catan, -1.0L, 0x1p-124L, -7.853981633974483096156608458198757210493e-1L, 2.350988701644575015937473074444491355637e-38L),
    TEST_c_c (catan, -1.0L, -0x1p-124L, -7.853981633974483096156608458198757210493e-1L, -2.350988701644575015937473074444491355637e-38L),
    TEST_c_c (catan, 0x1p-124L, 1.0L, 7.853981633974483096156608458198757210610e-1L, 4.332169878499658183857700759113603550472e1L),
    TEST_c_c (catan, -0x1p-124L, 1.0L, -7.853981633974483096156608458198757210610e-1L, 4.332169878499658183857700759113603550472e1L),
    TEST_c_c (catan, 0x1p-124L, -1.0L, 7.853981633974483096156608458198757210610e-1L, -4.332169878499658183857700759113603550472e1L),
    TEST_c_c (catan, -0x1p-124L, -1.0L, -7.853981633974483096156608458198757210610e-1L, -4.332169878499658183857700759113603550472e1L),
    TEST_c_c (catan, 0x1.000002p0L, 0x1p-126L, 7.853982230020895322927482174768220815493e-1L, 5.877471053462247139383742414506671233336e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1.000002p0L, -0x1p-126L, 7.853982230020895322927482174768220815493e-1L, -5.877471053462247139383742414506671233336e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1.000002p0L, 0x1p-126L, -7.853982230020895322927482174768220815493e-1L, 5.877471053462247139383742414506671233336e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x1.000002p0L, -0x1p-126L, -7.853982230020895322927482174768220815493e-1L, -5.877471053462247139383742414506671233336e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1p-126L, 0x1.000002p0L, 1.570796326794896619231321691639702138295L, 8.317766196521665212523713550410777685446L),
    TEST_c_c (catan, -0x1p-126L, 0x1.000002p0L, -1.570796326794896619231321691639702138295L, 8.317766196521665212523713550410777685446L),
    TEST_c_c (catan, 0x1p-126L, -0x1.000002p0L, 1.570796326794896619231321691639702138295L, -8.317766196521665212523713550410777685446L),
    TEST_c_c (catan, -0x1p-126L, -0x1.000002p0L, -1.570796326794896619231321691639702138295L, -8.317766196521665212523713550410777685446L),
    TEST_c_c (catan, 0x0.ffffffp0L, 0x1p-126L, 7.853981335951250337419109991749089175724e-1L, 5.877472104436064061535099214664320916184e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x0.ffffffp0L, -0x1p-126L, 7.853981335951250337419109991749089175724e-1L, -5.877472104436064061535099214664320916184e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x0.ffffffp0L, 0x1p-126L, -7.853981335951250337419109991749089175724e-1L, 5.877472104436064061535099214664320916184e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, -0x0.ffffffp0L, -0x1p-126L, -7.853981335951250337419109991749089175724e-1L, -5.877472104436064061535099214664320916184e-39L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catan, 0x1p-126L, 0x0.ffffffp0L, 9.860761609136244031334147100512154160097e-32L, 8.664339742098154951823135931565866792454L),
    TEST_c_c (catan, -0x1p-126L, 0x0.ffffffp0L, -9.860761609136244031334147100512154160097e-32L, 8.664339742098154951823135931565866792454L),
    TEST_c_c (catan, 0x1p-126L, -0x0.ffffffp0L, 9.860761609136244031334147100512154160097e-32L, -8.664339742098154951823135931565866792454L),
    TEST_c_c (catan, -0x1p-126L, -0x0.ffffffp0L, -9.860761609136244031334147100512154160097e-32L, -8.664339742098154951823135931565866792454L),
#if !TEST_COND_binary32
    TEST_c_c (catan, 1.0L, 0x1p-512L, 7.853981633974483096156608458198757210493e-1L, 3.729170365600103371645482657731466918688e-155L),
    TEST_c_c (catan, 1.0L, -0x1p-512L, 7.853981633974483096156608458198757210493e-1L, -3.729170365600103371645482657731466918688e-155L),
    TEST_c_c (catan, -1.0L, 0x1p-512L, -7.853981633974483096156608458198757210493e-1L, 3.729170365600103371645482657731466918688e-155L),
    TEST_c_c (catan, -1.0L, -0x1p-512L, -7.853981633974483096156608458198757210493e-1L, -3.729170365600103371645482657731466918688e-155L),
    TEST_c_c (catan, 0x1p-512L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 1.777922518136259718655200391540222897114e2L),
    TEST_c_c (catan, -0x1p-512L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 1.777922518136259718655200391540222897114e2L),
    TEST_c_c (catan, 0x1p-512L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -1.777922518136259718655200391540222897114e2L),
    TEST_c_c (catan, -0x1p-512L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -1.777922518136259718655200391540222897114e2L),
    TEST_c_c (catan, 1.0L, 0x1.3p-536L, 7.853981633974483096156608458198757210493e-1L, 2.639526015013529511588222179446290115084e-162L),
    TEST_c_c (catan, 1.0L, -0x1.3p-536L, 7.853981633974483096156608458198757210493e-1L, -2.639526015013529511588222179446290115084e-162L),
    TEST_c_c (catan, -1.0L, 0x1.3p-536L, -7.853981633974483096156608458198757210493e-1L, 2.639526015013529511588222179446290115084e-162L),
    TEST_c_c (catan, -1.0L, -0x1.3p-536L, -7.853981633974483096156608458198757210493e-1L, -2.639526015013529511588222179446290115084e-162L),
    TEST_c_c (catan, 0x1.3p-536L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 1.860240928518819859673567751384928348958e2L),
    TEST_c_c (catan, -0x1.3p-536L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 1.860240928518819859673567751384928348958e2L),
    TEST_c_c (catan, 0x1.3p-536L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -1.860240928518819859673567751384928348958e2L),
    TEST_c_c (catan, -0x1.3p-536L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -1.860240928518819859673567751384928348958e2L),
    TEST_c_c (catan, 1.0L, 0x1p-1020L, 7.853981633974483096156608458198757210493e-1L, 4.450147717014402766180465434664808128438e-308L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, 1.0L, -0x1p-1020L, 7.853981633974483096156608458198757210493e-1L, -4.450147717014402766180465434664808128438e-308L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, -1.0L, 0x1p-1020L, -7.853981633974483096156608458198757210493e-1L, 4.450147717014402766180465434664808128438e-308L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, -1.0L, -0x1p-1020L, -7.853981633974483096156608458198757210493e-1L, -4.450147717014402766180465434664808128438e-308L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, 0x1p-1020L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 3.538516356758520804574969980043991380025e2L),
    TEST_c_c (catan, -0x1p-1020L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 3.538516356758520804574969980043991380025e2L),
    TEST_c_c (catan, 0x1p-1020L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -3.538516356758520804574969980043991380025e2L),
    TEST_c_c (catan, -0x1p-1020L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -3.538516356758520804574969980043991380025e2L),
    TEST_c_c (catan, 0x1.0000000000001p0L, 0x1p-1022L, 7.853981633974484206379633083355174374608e-1L, 1.112536929253600444512293438042957369978e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1.0000000000001p0L, -0x1p-1022L, 7.853981633974484206379633083355174374608e-1L, -1.112536929253600444512293438042957369978e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.0000000000001p0L, 0x1p-1022L, -7.853981633974484206379633083355174374608e-1L, 1.112536929253600444512293438042957369978e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.0000000000001p0L, -0x1p-1022L, -7.853981633974484206379633083355174374608e-1L, -1.112536929253600444512293438042957369978e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, 1.836840028483855075506780244989950299369e1L),
    TEST_c_c (catan, -0x1p-1022L, 0x1.0000000000001p0L, -1.570796326794896619231321691639751442099L, 1.836840028483855075506780244989950299369e1L),
    TEST_c_c (catan, 0x1p-1022L, -0x1.0000000000001p0L, 1.570796326794896619231321691639751442099L, -1.836840028483855075506780244989950299369e1L),
    TEST_c_c (catan, -0x1p-1022L, -0x1.0000000000001p0L, -1.570796326794896619231321691639751442099L, -1.836840028483855075506780244989950299369e1L),
    TEST_c_c (catan, 0x0.fffffffffffff8p0L, 0x1p-1022L, 7.853981633974482541045096145620456183798e-1L, 1.112536929253600815061527818977844932790e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x0.fffffffffffff8p0L, -0x1p-1022L, 7.853981633974482541045096145620456183798e-1L, -1.112536929253600815061527818977844932790e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.fffffffffffff8p0L, 0x1p-1022L, -7.853981633974482541045096145620456183798e-1L, 1.112536929253600815061527818977844932790e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.fffffffffffff8p0L, -0x1p-1022L, -7.853981633974482541045096145620456183798e-1L, -1.112536929253600815061527818977844932790e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x0.fffffffffffff8p0L, 1.002084180004486444624900488355118689113e-292L, 1.871497387511852332650969166374185305708e1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, -0x1p-1022L, 0x0.fffffffffffff8p0L, -1.002084180004486444624900488355118689113e-292L, 1.871497387511852332650969166374185305708e1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, 0x1p-1022L, -0x0.fffffffffffff8p0L, 1.002084180004486444624900488355118689113e-292L, -1.871497387511852332650969166374185305708e1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catan, -0x1p-1022L, -0x0.fffffffffffff8p0L, -1.002084180004486444624900488355118689113e-292L, -1.871497387511852332650969166374185305708e1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
#endif
#if MANT_DIG >= 64
    TEST_c_c (catan, 0x1.0000000000000002p0L, 0x1p-1022L, 7.853981633974483096698709544441509427467e-1L, 1.112536929253600691424494863099491450042e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1.0000000000000002p0L, -0x1p-1022L, 7.853981633974483096698709544441509427467e-1L, -1.112536929253600691424494863099491450042e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.0000000000000002p0L, 0x1p-1022L, -7.853981633974483096698709544441509427467e-1L, 1.112536929253600691424494863099491450042e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.0000000000000002p0L, -0x1p-1022L, -7.853981633974483096698709544441509427467e-1L, -1.112536929253600691424494863099491450042e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, -0x1p-1022L, 0x1.0000000000000002p0L, -1.570796326794896619231321691639751442099L, 2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, 0x1p-1022L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, -0x1p-1022L, -0x1.0000000000000002p0L, -1.570796326794896619231321691639751442099L, -2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, 0x1p-1022L, 7.853981633974483095885557915077381101984e-1L, 1.112536929253600691605427106449557323148e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, -0x1p-1022L, 7.853981633974483095885557915077381101984e-1L, -1.112536929253600691605427106449557323148e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, 0x1p-1022L, -7.853981633974483095885557915077381101984e-1L, 1.112536929253600691605427106449557323148e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, -0x1p-1022L, -7.853981633974483095885557915077381101984e-1L, -1.112536929253600691605427106449557323148e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x0.ffffffffffffffffp0L, 2.052268400649188124723641491045245971623e-289L, 2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, -0x1p-1022L, 0x0.ffffffffffffffffp0L, -2.052268400649188124723641491045245971623e-289L, 2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, 0x1p-1022L, -0x0.ffffffffffffffffp0L, 2.052268400649188124723641491045245971623e-289L, -2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, -0x1p-1022L, -0x0.ffffffffffffffffp0L, -2.052268400649188124723641491045245971623e-289L, -2.252728336819822255604649142023466965703e1L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (catan, 0x1.000000000000000000000000008p0L, 0x1p-1022L, 7.853981633974483096156608458198880470009e-1L, 1.112536929253600691545116358666174605957e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1.000000000000000000000000008p0L, -0x1p-1022L, 7.853981633974483096156608458198880470009e-1L, -1.112536929253600691545116358666174605957e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.000000000000000000000000008p0L, 0x1p-1022L, -7.853981633974483096156608458198880470009e-1L, 1.112536929253600691545116358666174605957e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x1.000000000000000000000000008p0L, -0x1p-1022L, -7.853981633974483096156608458198880470009e-1L, -1.112536929253600691545116358666174605957e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, 3.673680056967710139911330243728336427098e1L),
    TEST_c_c (catan, -0x1p-1022L, 0x1.000000000000000000000000008p0L, -1.570796326794896619231321691639751442099L, 3.673680056967710139911330243728336427098e1L),
    TEST_c_c (catan, 0x1p-1022L, -0x1.000000000000000000000000008p0L, 1.570796326794896619231321691639751442099L, -3.673680056967710139911330243728336427098e1L),
    TEST_c_c (catan, -0x1p-1022L, -0x1.000000000000000000000000008p0L, -1.570796326794896619231321691639751442099L, -3.673680056967710139911330243728336427098e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-1022L, 7.853981633974483096156608458198695580735e-1L, 1.112536929253600691545116358666215745186e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-1022L, 7.853981633974483096156608458198695580735e-1L, -1.112536929253600691545116358666215745186e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-1022L, -7.853981633974483096156608458198695580735e-1L, 1.112536929253600691545116358666215745186e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-1022L, -7.853981633974483096156608458198695580735e-1L, -1.112536929253600691545116358666215745186e-308L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catan, 0x1p-1022L, 0x0.ffffffffffffffffffffffffffcp0L, 9.025971879324147880346310405868788320726e-277L, 3.708337415995707405382191849801244331055e1L),
    TEST_c_c (catan, -0x1p-1022L, 0x0.ffffffffffffffffffffffffffcp0L, -9.025971879324147880346310405868788320726e-277L, 3.708337415995707405382191849801244331055e1L),
    TEST_c_c (catan, 0x1p-1022L, -0x0.ffffffffffffffffffffffffffcp0L, 9.025971879324147880346310405868788320726e-277L, -3.708337415995707405382191849801244331055e1L),
    TEST_c_c (catan, -0x1p-1022L, -0x0.ffffffffffffffffffffffffffcp0L, -9.025971879324147880346310405868788320726e-277L, -3.708337415995707405382191849801244331055e1L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catan, 1.0L, 0x1p-8192L, 7.853981633974483096156608458198757210493e-1L, 4.584009668887117914053530980121207914891e-2467L),
    TEST_c_c (catan, 1.0L, -0x1p-8192L, 7.853981633974483096156608458198757210493e-1L, -4.584009668887117914053530980121207914891e-2467L),
    TEST_c_c (catan, -1.0L, 0x1p-8192L, -7.853981633974483096156608458198757210493e-1L, 4.584009668887117914053530980121207914891e-2467L),
    TEST_c_c (catan, -1.0L, -0x1p-8192L, -7.853981633974483096156608458198757210493e-1L, -4.584009668887117914053530980121207914891e-2467L),
    TEST_c_c (catan, 0x1p-8192L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 2.839477425163815960027691385553420311121e3L),
    TEST_c_c (catan, -0x1p-8192L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 2.839477425163815960027691385553420311121e3L),
    TEST_c_c (catan, 0x1p-8192L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -2.839477425163815960027691385553420311121e3L),
    TEST_c_c (catan, -0x1p-8192L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -2.839477425163815960027691385553420311121e3L),
    TEST_c_c (catan, 1.0L, 0x1.3p-8221L, 7.853981633974483096156608458198757210493e-1L, 1.013933025636421986471594877335044443409e-2475L),
    TEST_c_c (catan, 1.0L, -0x1.3p-8221L, 7.853981633974483096156608458198757210493e-1L, -1.013933025636421986471594877335044443409e-2475L),
    TEST_c_c (catan, -1.0L, 0x1.3p-8221L, -7.853981633974483096156608458198757210493e-1L, 1.013933025636421986471594877335044443409e-2475L),
    TEST_c_c (catan, -1.0L, -0x1.3p-8221L, -7.853981633974483096156608458198757210493e-1L, -1.013933025636421986471594877335044443409e-2475L),
    TEST_c_c (catan, 0x1.3p-8221L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 2.849442134153471837403071201841536297726e3L),
    TEST_c_c (catan, -0x1.3p-8221L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 2.849442134153471837403071201841536297726e3L),
    TEST_c_c (catan, 0x1.3p-8221L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -2.849442134153471837403071201841536297726e3L),
    TEST_c_c (catan, -0x1.3p-8221L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -2.849442134153471837403071201841536297726e3L),
    TEST_c_c (catan, 1.0L, 0x1.3p-8246L, 7.853981633974483096156608458198757210493e-1L, 3.021755890954798419688924781486524472858e-2483L),
    TEST_c_c (catan, 1.0L, -0x1.3p-8246L, 7.853981633974483096156608458198757210493e-1L, -3.021755890954798419688924781486524472858e-2483L),
    TEST_c_c (catan, -1.0L, 0x1.3p-8246L, -7.853981633974483096156608458198757210493e-1L, 3.021755890954798419688924781486524472858e-2483L),
    TEST_c_c (catan, -1.0L, -0x1.3p-8246L, -7.853981633974483096156608458198757210493e-1L, -3.021755890954798419688924781486524472858e-2483L),
    TEST_c_c (catan, 0x1.3p-8246L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 2.858106473910471153770786603359763504827e3L),
    TEST_c_c (catan, -0x1.3p-8246L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 2.858106473910471153770786603359763504827e3L),
    TEST_c_c (catan, 0x1.3p-8246L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -2.858106473910471153770786603359763504827e3L),
    TEST_c_c (catan, -0x1.3p-8246L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -2.858106473910471153770786603359763504827e3L),
    TEST_c_c (catan, 1.0L, 0x1p-16380L, 7.853981633974483096156608458198757210493e-1L, 6.724206286224187012525355634643505205196e-4932L),
    TEST_c_c (catan, 1.0L, -0x1p-16380L, 7.853981633974483096156608458198757210493e-1L, -6.724206286224187012525355634643505205196e-4932L),
    TEST_c_c (catan, -1.0L, 0x1p-16380L, -7.853981633974483096156608458198757210493e-1L, 6.724206286224187012525355634643505205196e-4932L),
    TEST_c_c (catan, -1.0L, -0x1p-16380L, -7.853981633974483096156608458198757210493e-1L, -6.724206286224187012525355634643505205196e-4932L),
    TEST_c_c (catan, 0x1p-16380L, 1.0L, 7.853981633974483096156608458198757210493e-1L, 5.677221982376232056781839690803195180822e3L),
    TEST_c_c (catan, -0x1p-16380L, 1.0L, -7.853981633974483096156608458198757210493e-1L, 5.677221982376232056781839690803195180822e3L),
    TEST_c_c (catan, 0x1p-16380L, -1.0L, 7.853981633974483096156608458198757210493e-1L, -5.677221982376232056781839690803195180822e3L),
    TEST_c_c (catan, -0x1p-16380L, -1.0L, -7.853981633974483096156608458198757210493e-1L, -5.677221982376232056781839690803195180822e3L),
    TEST_c_c (catan, 0x1.0000000000000002p0L, 0x1p-16382L, 7.853981633974483096698709544441509427467e-1L, 1.681051571556046752949078932066752571182e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1.0000000000000002p0L, -0x1p-16382L, 7.853981633974483096698709544441509427467e-1L, -1.681051571556046752949078932066752571182e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.0000000000000002p0L, 0x1p-16382L, -7.853981633974483096698709544441509427467e-1L, 1.681051571556046752949078932066752571182e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.0000000000000002p0L, -0x1p-16382L, -7.853981633974483096698709544441509427467e-1L, -1.681051571556046752949078932066752571182e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1p-16382L, 0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, 2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, -0x1p-16382L, 0x1.0000000000000002p0L, -1.570796326794896619231321691639751442099L, 2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, 0x1p-16382L, -0x1.0000000000000002p0L, 1.570796326794896619231321691639751442099L, -2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, -0x1p-16382L, -0x1.0000000000000002p0L, -1.570796326794896619231321691639751442099L, -2.218070977791824990137853294097378778927e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, 0x1p-16382L, 7.853981633974483095885557915077381101984e-1L, 1.681051571556046753222468896957938166365e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x0.ffffffffffffffffp0L, -0x1p-16382L, 7.853981633974483095885557915077381101984e-1L, -1.681051571556046753222468896957938166365e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, 0x1p-16382L, -7.853981633974483095885557915077381101984e-1L, 1.681051571556046753222468896957938166365e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x0.ffffffffffffffffp0L, -0x1p-16382L, -7.853981633974483095885557915077381101984e-1L, -1.681051571556046753222468896957938166365e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1p-16382L, 0x0.ffffffffffffffffp0L, 3.100992811520163369065387859792822623745e-4913L, 2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, -0x1p-16382L, 0x0.ffffffffffffffffp0L, -3.100992811520163369065387859792822623745e-4913L, 2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, 0x1p-16382L, -0x0.ffffffffffffffffp0L, 3.100992811520163369065387859792822623745e-4913L, -2.252728336819822255604649142023466965703e1L),
    TEST_c_c (catan, -0x1p-16382L, -0x0.ffffffffffffffffp0L, -3.100992811520163369065387859792822623745e-4913L, -2.252728336819822255604649142023466965703e1L),
# if MANT_DIG >= 113
    TEST_c_c (catan, 0x1.0000000000000000000000000001p0L, 0x1p-16382L, 7.853981633974483096156608458198758173458e-1L, 1.681051571556046753131338908660875977540e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1.0000000000000000000000000001p0L, -0x1p-16382L, 7.853981633974483096156608458198758173458e-1L, -1.681051571556046753131338908660875977540e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.0000000000000000000000000001p0L, 0x1p-16382L, -7.853981633974483096156608458198758173458e-1L, 1.681051571556046753131338908660875977540e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x1.0000000000000000000000000001p0L, -0x1p-16382L, -7.853981633974483096156608458198758173458e-1L, -1.681051571556046753131338908660875977540e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1p-16382L, 0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, 3.916281570163690998207361486238697614441e1L),
    TEST_c_c (catan, -0x1p-16382L, 0x1.0000000000000000000000000001p0L, -1.570796326794896619231321691639751442099L, 3.916281570163690998207361486238697614441e1L),
    TEST_c_c (catan, 0x1p-16382L, -0x1.0000000000000000000000000001p0L, 1.570796326794896619231321691639751442099L, -3.916281570163690998207361486238697614441e1L),
    TEST_c_c (catan, -0x1p-16382L, -0x1.0000000000000000000000000001p0L, -1.570796326794896619231321691639751442099L, -3.916281570163690998207361486238697614441e1L),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-16382L, 7.853981633974483096156608458198756729010e-1L, 1.681051571556046753131338908660876463178e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-16382L, 7.853981633974483096156608458198756729010e-1L, -1.681051571556046753131338908660876463178e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-16382L, -7.853981633974483096156608458198756729010e-1L, 1.681051571556046753131338908660876463178e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-16382L, -7.853981633974483096156608458198756729010e-1L, -1.681051571556046753131338908660876463178e-4932L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catan, 0x1p-16382L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.745703758805099310527547423749501866998e-4898L, 3.950938929191688263678223092311606435623e1L),
    TEST_c_c (catan, -0x1p-16382L, 0x0.ffffffffffffffffffffffffffff8p0L, -1.745703758805099310527547423749501866998e-4898L, 3.950938929191688263678223092311606435623e1L),
    TEST_c_c (catan, 0x1p-16382L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.745703758805099310527547423749501866998e-4898L, -3.950938929191688263678223092311606435623e1L),
    TEST_c_c (catan, -0x1p-16382L, -0x0.ffffffffffffffffffffffffffff8p0L, -1.745703758805099310527547423749501866998e-4898L, -3.950938929191688263678223092311606435623e1L),
# endif
#endif

    TEST_c_c (catan, 0.75L, 1.25L, 1.10714871779409050301706546017853704L, 0.549306144334054845697622618461262852L),
    TEST_c_c (catan, -2, -3, -1.4099210495965755225306193844604208L, -0.22907268296853876629588180294200276L),
  };

static void
catan_test (void)
{
  ALL_RM_TEST (catan, 0, catan_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}

static const struct test_c_c_data catanh_test_data[] =
  {
    TEST_c_c (catanh, 0, 0, 0.0, 0.0),
    TEST_c_c (catanh, minus_zero, 0, minus_zero, 0.0),
    TEST_c_c (catanh, 0, minus_zero, 0.0, minus_zero),
    TEST_c_c (catanh, minus_zero, minus_zero, minus_zero, minus_zero),

    TEST_c_c (catanh, 1.0L, plus_zero, plus_infty, plus_zero, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catanh, 1.0L, minus_zero, plus_infty, minus_zero, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catanh, -1.0L, plus_zero, minus_infty, plus_zero, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catanh, -1.0L, minus_zero, minus_infty, minus_zero, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (catanh, plus_zero, 1.0L, plus_zero, lit_pi_4_d),
    TEST_c_c (catanh, minus_zero, 1.0L, minus_zero, lit_pi_4_d),
    TEST_c_c (catanh, plus_zero, -1.0L, plus_zero, -lit_pi_4_d),
    TEST_c_c (catanh, minus_zero, -1.0L, minus_zero, -lit_pi_4_d),

    TEST_c_c (catanh, plus_infty, plus_infty, 0.0, lit_pi_2_d),
    TEST_c_c (catanh, plus_infty, minus_infty, 0.0, -lit_pi_2_d),
    TEST_c_c (catanh, minus_infty, plus_infty, minus_zero, lit_pi_2_d),
    TEST_c_c (catanh, minus_infty, minus_infty, minus_zero, -lit_pi_2_d),

    TEST_c_c (catanh, -10.0, plus_infty, minus_zero, lit_pi_2_d),
    TEST_c_c (catanh, -10.0, minus_infty, minus_zero, -lit_pi_2_d),
    TEST_c_c (catanh, minus_zero, plus_infty, minus_zero, lit_pi_2_d),
    TEST_c_c (catanh, minus_zero, minus_infty, minus_zero, -lit_pi_2_d),
    TEST_c_c (catanh, 0, plus_infty, 0.0, lit_pi_2_d),
    TEST_c_c (catanh, 0, minus_infty, 0.0, -lit_pi_2_d),
    TEST_c_c (catanh, 0.1L, plus_infty, 0.0, lit_pi_2_d),
    TEST_c_c (catanh, 0.1L, minus_infty, 0.0, -lit_pi_2_d),

    TEST_c_c (catanh, minus_infty, 0, minus_zero, lit_pi_2_d),
    TEST_c_c (catanh, minus_infty, minus_zero, minus_zero, -lit_pi_2_d),
    TEST_c_c (catanh, minus_infty, 100, minus_zero, lit_pi_2_d),
    TEST_c_c (catanh, minus_infty, -100, minus_zero, -lit_pi_2_d),

    TEST_c_c (catanh, plus_infty, 0, 0.0, lit_pi_2_d),
    TEST_c_c (catanh, plus_infty, minus_zero, 0.0, -lit_pi_2_d),
    TEST_c_c (catanh, plus_infty, 0.5, 0.0, lit_pi_2_d),
    TEST_c_c (catanh, plus_infty, -0.5, 0.0, -lit_pi_2_d),

    TEST_c_c (catanh, 0, qnan_value, 0.0, qnan_value),
    TEST_c_c (catanh, minus_zero, qnan_value, minus_zero, qnan_value),

    TEST_c_c (catanh, plus_infty, qnan_value, 0.0, qnan_value),
    TEST_c_c (catanh, minus_infty, qnan_value, minus_zero, qnan_value),

    TEST_c_c (catanh, qnan_value, 0, qnan_value, qnan_value),
    TEST_c_c (catanh, qnan_value, minus_zero, qnan_value, qnan_value),

    TEST_c_c (catanh, qnan_value, plus_infty, 0.0, lit_pi_2_d, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (catanh, qnan_value, minus_infty, 0.0, -lit_pi_2_d, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (catanh, 10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (catanh, -10.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (catanh, qnan_value, 0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (catanh, qnan_value, -0.75, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (catanh, qnan_value, qnan_value, qnan_value, qnan_value),

    TEST_c_c (catanh, 0x1p50L, 0.0L, 8.881784197001252323389053344728897997441e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p50L, -0.0L, 8.881784197001252323389053344728897997441e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p50L, 0.0L, -8.881784197001252323389053344728897997441e-16L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p50L, -0.0L, -8.881784197001252323389053344728897997441e-16L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p50L, 0.0L, 1.570796326794895731052901991514519103193L),
    TEST_c_c (catanh, -0.0L, 0x1p50L, -0.0L, 1.570796326794895731052901991514519103193L),
    TEST_c_c (catanh, 0.0L, -0x1p50L, 0.0L, -1.570796326794895731052901991514519103193L),
    TEST_c_c (catanh, -0.0L, -0x1p50L, -0.0L, -1.570796326794895731052901991514519103193L),
#if !TEST_COND_binary32
    TEST_c_c (catanh, 0x1p500L, 0.0L, 3.054936363499604682051979393213617699789e-151L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p500L, -0.0L, 3.054936363499604682051979393213617699789e-151L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p500L, 0.0L, -3.054936363499604682051979393213617699789e-151L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p500L, -0.0L, -3.054936363499604682051979393213617699789e-151L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p500L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1p500L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1p500L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1p500L, -0.0L, -1.570796326794896619231321691639751442099L),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catanh, 0x1p5000L, 0.0L, 7.079811261048172892385615158694057552948e-1506L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p5000L, -0.0L, 7.079811261048172892385615158694057552948e-1506L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p5000L, 0.0L, -7.079811261048172892385615158694057552948e-1506L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p5000L, -0.0L, -7.079811261048172892385615158694057552948e-1506L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p5000L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1p5000L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1p5000L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1p5000L, -0.0L, -1.570796326794896619231321691639751442099L),
#endif
    TEST_c_c (catanh, 0x1p63L, 0.5L, 1.084202172485504434007452800869941711427e-19L, 1.570796326794896619231321691639751442093L),
    TEST_c_c (catanh, 0x1p63L, -0.5L, 1.084202172485504434007452800869941711427e-19L, -1.570796326794896619231321691639751442093L),
    TEST_c_c (catanh, -0x1p63L, 0.5L, -1.084202172485504434007452800869941711427e-19L, 1.570796326794896619231321691639751442093L),
    TEST_c_c (catanh, -0x1p63L, -0.5L, -1.084202172485504434007452800869941711427e-19L, -1.570796326794896619231321691639751442093L),
    TEST_c_c (catanh, 0.5L, 0x1p63L, 5.877471754111437539843682686111228389007e-39L, 1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0.5L, 0x1p63L, -5.877471754111437539843682686111228389007e-39L, 1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0.5L, -0x1p63L, 5.877471754111437539843682686111228389007e-39L, -1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0.5L, -0x1p63L, -5.877471754111437539843682686111228389007e-39L, -1.570796326794896619122901474391200998698L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (catanh, 0x1p511L, 0.5L, 1.491668146240041348658193063092586767475e-154L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p511L, -0.5L, 1.491668146240041348658193063092586767475e-154L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p511L, 0.5L, -1.491668146240041348658193063092586767475e-154L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p511L, -0.5L, -1.491668146240041348658193063092586767475e-154L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.5L, 0x1p511L, 1.112536929253600691545116358666202032110e-308L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0.5L, 0x1p511L, -1.112536929253600691545116358666202032110e-308L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0.5L, -0x1p511L, 1.112536929253600691545116358666202032110e-308L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0.5L, -0x1p511L, -1.112536929253600691545116358666202032110e-308L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catanh, 0x1p8191L, 0.5L, 1.833603867554847165621412392048483165956e-2466L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p8191L, -0.5L, 1.833603867554847165621412392048483165956e-2466L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p8191L, 0.5L, -1.833603867554847165621412392048483165956e-2466L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p8191L, -0.5L, -1.833603867554847165621412392048483165956e-2466L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.5L, 0x1p8191L, 1.681051571556046753131338908660876301299e-4932L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0.5L, 0x1p8191L, -1.681051571556046753131338908660876301299e-4932L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0.5L, -0x1p8191L, 1.681051571556046753131338908660876301299e-4932L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0.5L, -0x1p8191L, -1.681051571556046753131338908660876301299e-4932L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (catanh, 0x1p100L, 0.0L, 7.888609052210118054117285652827862296732e-31L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p100L, -0.0L, 7.888609052210118054117285652827862296732e-31L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p100L, 0.0L, -7.888609052210118054117285652827862296732e-31L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p100L, -0.0L, -7.888609052210118054117285652827862296732e-31L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p100L, 0.0L, 1.570796326794896619231321691638962581193L),
    TEST_c_c (catanh, -0.0L, 0x1p100L, -0.0L, 1.570796326794896619231321691638962581193L),
    TEST_c_c (catanh, 0.0L, -0x1p100L, 0.0L, -1.570796326794896619231321691638962581193L),
    TEST_c_c (catanh, -0.0L, -0x1p100L, -0.0L, -1.570796326794896619231321691638962581193L),
    TEST_c_c (catanh, 0x1.fp127L, 0.0L, 3.033533808573645181854803967025150136306e-39L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0x1.fp127L, -0.0L, 3.033533808573645181854803967025150136306e-39L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1.fp127L, 0.0L, -3.033533808573645181854803967025150136306e-39L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1.fp127L, -0.0L, -3.033533808573645181854803967025150136306e-39L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0.0L, 0x1.fp127L, 0.0L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (catanh, -0.0L, 0x1.fp127L, -0.0L, 1.570796326794896619231321691639751442096L),
    TEST_c_c (catanh, 0.0L, -0x1.fp127L, 0.0L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (catanh, -0.0L, -0x1.fp127L, -0.0L, -1.570796326794896619231321691639751442096L),
    TEST_c_c (catanh, 0x1.fp127L, 0x1.fp127L, 1.516766904286822590927401983512575068153e-39L, 1.570796326794896619231321691639751442097L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0x1.fp127L, -0x1.fp127L, 1.516766904286822590927401983512575068153e-39L, -1.570796326794896619231321691639751442097L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1.fp127L, 0x1.fp127L, -1.516766904286822590927401983512575068153e-39L, 1.570796326794896619231321691639751442097L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1.fp127L, -0x1.fp127L, -1.516766904286822590927401983512575068153e-39L, -1.570796326794896619231321691639751442097L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (catanh, 0x1p900L, 0.0L, 1.183052186166774710972751597518026531652e-271L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p900L, -0.0L, 1.183052186166774710972751597518026531652e-271L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p900L, 0.0L, -1.183052186166774710972751597518026531652e-271L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p900L, -0.0L, -1.183052186166774710972751597518026531652e-271L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p900L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1p900L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1p900L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1p900L, -0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.fp1023L, 0.0L, 5.742126086470197117652213464083623391533e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1.fp1023L, -0.0L, 5.742126086470197117652213464083623391533e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1.fp1023L, 0.0L, -5.742126086470197117652213464083623391533e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1.fp1023L, -0.0L, -5.742126086470197117652213464083623391533e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0.0L, 0x1.fp1023L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1.fp1023L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1.fp1023L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1.fp1023L, -0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.fp1023L, 0x1.fp1023L, 2.871063043235098558826106732041811695767e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1.fp1023L, -0x1.fp1023L, 2.871063043235098558826106732041811695767e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1.fp1023L, 0x1.fp1023L, -2.871063043235098558826106732041811695767e-309L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1.fp1023L, -0x1.fp1023L, -2.871063043235098558826106732041811695767e-309L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catanh, 0x1p10000L, 0.0L, 5.012372749206452009297555933742977749322e-3011L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p10000L, -0.0L, 5.012372749206452009297555933742977749322e-3011L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p10000L, 0.0L, -5.012372749206452009297555933742977749322e-3011L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1p10000L, -0.0L, -5.012372749206452009297555933742977749322e-3011L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, 0x1p10000L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1p10000L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1p10000L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1p10000L, -0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.fp16383L, 0.0L, 8.676395208031209048419813722120651877672e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1.fp16383L, -0.0L, 8.676395208031209048419813722120651877672e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1.fp16383L, 0.0L, -8.676395208031209048419813722120651877672e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1.fp16383L, -0.0L, -8.676395208031209048419813722120651877672e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0.0L, 0x1.fp16383L, 0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, 0x1.fp16383L, -0.0L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0.0L, -0x1.fp16383L, 0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0.0L, -0x1.fp16383L, -0.0L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.fp16383L, 0x1.fp16383L, 4.338197604015604524209906861060325938836e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1.fp16383L, -0x1.fp16383L, 4.338197604015604524209906861060325938836e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1.fp16383L, 0x1.fp16383L, -4.338197604015604524209906861060325938836e-4933L, 1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1.fp16383L, -0x1.fp16383L, -4.338197604015604524209906861060325938836e-4933L, -1.570796326794896619231321691639751442099L, UNDERFLOW_EXCEPTION),
#endif
    TEST_c_c (catanh, 0x1p-13L, 1.0L, 6.103515609841754902688560615027452023669e-5L, 7.853981671227386080775748393881580082970e-1L),
    TEST_c_c (catanh, 0x1p-13L, -1.0L, 6.103515609841754902688560615027452023669e-5L, -7.853981671227386080775748393881580082970e-1L),
    TEST_c_c (catanh, -0x1p-13L, 1.0L, -6.103515609841754902688560615027452023669e-5L, 7.853981671227386080775748393881580082970e-1L),
    TEST_c_c (catanh, -0x1p-13L, -1.0L, -6.103515609841754902688560615027452023669e-5L, -7.853981671227386080775748393881580082970e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-13L, 4.852030264850939738801379894163661227127L, 7.854286809755354140031716771044626356262e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-13L, -4.852030264850939738801379894163661227127L, 7.854286809755354140031716771044626356262e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-13L, 4.852030264850939738801379894163661227127L, -7.854286809755354140031716771044626356262e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-13L, -4.852030264850939738801379894163661227127L, -7.854286809755354140031716771044626356262e-1L),
    TEST_c_c (catanh, 0x1p-27L, 1.0L, 3.725290298461914028034141143623846306386e-9L, 7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, 0x1p-27L, -1.0L, 3.725290298461914028034141143623846306386e-9L, -7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, -0x1p-27L, 1.0L, -3.725290298461914028034141143623846306386e-9L, 7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, -0x1p-27L, -1.0L, -3.725290298461914028034141143623846306386e-9L, -7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-27L, 9.704060527839234335310696652368086117807L, 7.853981652600934588466178684534110069553e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-27L, -9.704060527839234335310696652368086117807L, 7.853981652600934588466178684534110069553e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-27L, 9.704060527839234335310696652368086117807L, -7.853981652600934588466178684534110069553e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-27L, -9.704060527839234335310696652368086117807L, -7.853981652600934588466178684534110069553e-1L),
    TEST_c_c (catanh, 0x1p-33L, 1.0L, 5.820766091346740722643102318246316469910e-11L, 7.853981633974483096190489776088929224056e-1L),
    TEST_c_c (catanh, 0x1p-33L, -1.0L, 5.820766091346740722643102318246316469910e-11L, -7.853981633974483096190489776088929224056e-1L),
    TEST_c_c (catanh, -0x1p-33L, 1.0L, -5.820766091346740722643102318246316469910e-11L, 7.853981633974483096190489776088929224056e-1L),
    TEST_c_c (catanh, -0x1p-33L, -1.0L, -5.820766091346740722643102318246316469910e-11L, -7.853981633974483096190489776088929224056e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-33L, 1.178350206951907026009379309773625595762e1L, 7.853981634265521400723945494331241018449e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-33L, -1.178350206951907026009379309773625595762e1L, 7.853981634265521400723945494331241018449e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-33L, 1.178350206951907026009379309773625595762e1L, -7.853981634265521400723945494331241018449e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-33L, -1.178350206951907026009379309773625595762e1L, -7.853981634265521400723945494331241018449e-1L),
    TEST_c_c (catanh, 0x1p-54L, 1.0L, 2.775557561562891351059079170227049355775e-17L, 7.853981633974483096156608458198764914213e-1L),
    TEST_c_c (catanh, 0x1p-54L, -1.0L, 2.775557561562891351059079170227049355775e-17L, -7.853981633974483096156608458198764914213e-1L),
    TEST_c_c (catanh, -0x1p-54L, 1.0L, -2.775557561562891351059079170227049355775e-17L, 7.853981633974483096156608458198764914213e-1L),
    TEST_c_c (catanh, -0x1p-54L, -1.0L, -2.775557561562891351059079170227049355775e-17L, -7.853981633974483096156608458198764914213e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-54L, 1.906154746539849600897388334009985581467e1L, 7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-54L, -1.906154746539849600897388334009985581467e1L, 7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-54L, 1.906154746539849600897388334009985581467e1L, -7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-54L, -1.906154746539849600897388334009985581467e1L, -7.853981633974483234934486536343324763447e-1L),
    TEST_c_c (catanh, 0x1p-57L, 1.0L, 3.469446951953614188823848962783813448721e-18L, 7.853981633974483096156608458198757330864e-1L),
    TEST_c_c (catanh, 0x1p-57L, -1.0L, 3.469446951953614188823848962783813448721e-18L, -7.853981633974483096156608458198757330864e-1L),
    TEST_c_c (catanh, -0x1p-57L, 1.0L, -3.469446951953614188823848962783813448721e-18L, 7.853981633974483096156608458198757330864e-1L),
    TEST_c_c (catanh, -0x1p-57L, -1.0L, -3.469446951953614188823848962783813448721e-18L, -7.853981633974483096156608458198757330864e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-57L, 2.010126823623841397309973152228712047720e1L, 7.853981633974483113503843217966828154612e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-57L, -2.010126823623841397309973152228712047720e1L, 7.853981633974483113503843217966828154612e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-57L, 2.010126823623841397309973152228712047720e1L, -7.853981633974483113503843217966828154612e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-57L, -2.010126823623841397309973152228712047720e1L, -7.853981633974483113503843217966828154612e-1L),
    TEST_c_c (catanh, 0x1p-13L, 0x1.000002p0L, 6.103514882246036852433556327261700380577e-5L, 7.853982267273793866654490522673596014524e-1L),
    TEST_c_c (catanh, 0x1p-13L, -0x1.000002p0L, 6.103514882246036852433556327261700380577e-5L, -7.853982267273793866654490522673596014524e-1L),
    TEST_c_c (catanh, -0x1p-13L, 0x1.000002p0L, -6.103514882246036852433556327261700380577e-5L, 7.853982267273793866654490522673596014524e-1L),
    TEST_c_c (catanh, -0x1p-13L, -0x1.000002p0L, -6.103514882246036852433556327261700380577e-5L, -7.853982267273793866654490522673596014524e-1L),
    TEST_c_c (catanh, 0x1.000002p0L, 0x1p-13L, 4.852030056234795712498957387213592193975L, 7.859169620684960844300240092596908675974e-1L),
    TEST_c_c (catanh, -0x1.000002p0L, 0x1p-13L, -4.852030056234795712498957387213592193975L, 7.859169620684960844300240092596908675974e-1L),
    TEST_c_c (catanh, 0x1.000002p0L, -0x1p-13L, 4.852030056234795712498957387213592193975L, -7.859169620684960844300240092596908675974e-1L),
    TEST_c_c (catanh, -0x1.000002p0L, -0x1p-13L, -4.852030056234795712498957387213592193975L, -7.859169620684960844300240092596908675974e-1L),
    TEST_c_c (catanh, 0x1p-13L, 0x0.ffffffp0L, 6.103515973639646453881721999956617260502e-5L, 7.853981373204155542484315721351697277336e-1L),
    TEST_c_c (catanh, 0x1p-13L, -0x0.ffffffp0L, 6.103515973639646453881721999956617260502e-5L, -7.853981373204155542484315721351697277336e-1L),
    TEST_c_c (catanh, -0x1p-13L, 0x0.ffffffp0L, -6.103515973639646453881721999956617260502e-5L, 7.853981373204155542484315721351697277336e-1L),
    TEST_c_c (catanh, -0x1p-13L, -0x0.ffffffp0L, -6.103515973639646453881721999956617260502e-5L, -7.853981373204155542484315721351697277336e-1L),
    TEST_c_c (catanh, 0x0.ffffffp0L, 0x1p-13L, 4.852030190345140708455871037447717761868L, 7.851845403708474595909269086711426246675e-1L),
    TEST_c_c (catanh, -0x0.ffffffp0L, 0x1p-13L, -4.852030190345140708455871037447717761868L, 7.851845403708474595909269086711426246675e-1L),
    TEST_c_c (catanh, 0x0.ffffffp0L, -0x1p-13L, 4.852030190345140708455871037447717761868L, -7.851845403708474595909269086711426246675e-1L),
    TEST_c_c (catanh, -0x0.ffffffp0L, -0x1p-13L, -4.852030190345140708455871037447717761868L, -7.851845403708474595909269086711426246675e-1L),
#if !TEST_COND_binary32
    TEST_c_c (catanh, 0x1p-27L, 0x1.0000000000001p0L, 3.725290298461913200853528590596263270474e-9L, 7.853981633974484345157511161499711112683e-1L),
    TEST_c_c (catanh, 0x1p-27L, -0x1.0000000000001p0L, 3.725290298461913200853528590596263270474e-9L, -7.853981633974484345157511161499711112683e-1L),
    TEST_c_c (catanh, -0x1p-27L, 0x1.0000000000001p0L, -3.725290298461913200853528590596263270474e-9L, 7.853981633974484345157511161499711112683e-1L),
    TEST_c_c (catanh, -0x1p-27L, -0x1.0000000000001p0L, -3.725290298461913200853528590596263270474e-9L, -7.853981633974484345157511161499711112683e-1L),
    TEST_c_c (catanh, 0x1.0000000000001p0L, 0x1p-27L, 9.704060527839234168777242958594699810015L, 7.853981801612546526942695000283242525531e-1L),
    TEST_c_c (catanh, -0x1.0000000000001p0L, 0x1p-27L, -9.704060527839234168777242958594699810015L, 7.853981801612546526942695000283242525531e-1L),
    TEST_c_c (catanh, 0x1.0000000000001p0L, -0x1p-27L, 9.704060527839234168777242958594699810015L, -7.853981801612546526942695000283242525531e-1L),
    TEST_c_c (catanh, -0x1.0000000000001p0L, -0x1p-27L, -9.704060527839234168777242958594699810015L, -7.853981801612546526942695000283242525531e-1L),
    TEST_c_c (catanh, 0x1p-27L, 0x0.fffffffffffff8p0L, 3.725290298461914441624447420137706700965e-9L, 7.853981633974482679822974223765039144191e-1L),
    TEST_c_c (catanh, 0x1p-27L, -0x0.fffffffffffff8p0L, 3.725290298461914441624447420137706700965e-9L, -7.853981633974482679822974223765039144191e-1L),
    TEST_c_c (catanh, -0x1p-27L, 0x0.fffffffffffff8p0L, -3.725290298461914441624447420137706700965e-9L, 7.853981633974482679822974223765039144191e-1L),
    TEST_c_c (catanh, -0x1p-27L, -0x0.fffffffffffff8p0L, -3.725290298461914441624447420137706700965e-9L, -7.853981633974482679822974223765039144191e-1L),
    TEST_c_c (catanh, 0x0.fffffffffffff8p0L, 0x1p-27L, 9.704060527839234252043969805481351363824L, 7.853981578095128619227903983047292781021e-1L),
    TEST_c_c (catanh, -0x0.fffffffffffff8p0L, 0x1p-27L, -9.704060527839234252043969805481351363824L, 7.853981578095128619227903983047292781021e-1L),
    TEST_c_c (catanh, 0x0.fffffffffffff8p0L, -0x1p-27L, 9.704060527839234252043969805481351363824L, -7.853981578095128619227903983047292781021e-1L),
    TEST_c_c (catanh, -0x0.fffffffffffff8p0L, -0x1p-27L, -9.704060527839234252043969805481351363824L, -7.853981578095128619227903983047292781021e-1L),
#endif
#if MANT_DIG >= 64
    TEST_c_c (catanh, 0x1p-33L, 0x1.0000000000000002p0L, 5.820766091346740722012013594069507025615e-11L, 7.853981633974483096732590862331681441026e-1L),
    TEST_c_c (catanh, 0x1p-33L, -0x1.0000000000000002p0L, 5.820766091346740722012013594069507025615e-11L, -7.853981633974483096732590862331681441026e-1L),
    TEST_c_c (catanh, -0x1p-33L, 0x1.0000000000000002p0L, -5.820766091346740722012013594069507025615e-11L, 7.853981633974483096732590862331681441026e-1L),
    TEST_c_c (catanh, -0x1p-33L, -0x1.0000000000000002p0L, -5.820766091346740722012013594069507025615e-11L, -7.853981633974483096732590862331681441026e-1L),
    TEST_c_c (catanh, 0x1.0000000000000002p0L, 0x1p-33L, 1.178350206951907025990405771755129268176e1L, 7.853981638922134273801338071094141188767e-1L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, 0x1p-33L, -1.178350206951907025990405771755129268176e1L, 7.853981638922134273801338071094141188767e-1L),
    TEST_c_c (catanh, 0x1.0000000000000002p0L, -0x1p-33L, 1.178350206951907025990405771755129268176e1L, -7.853981638922134273801338071094141188767e-1L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, -0x1p-33L, -1.178350206951907025990405771755129268176e1L, -7.853981638922134273801338071094141188767e-1L),
    TEST_c_c (catanh, 0x1p-33L, 0x0.ffffffffffffffffp0L, 5.820766091346740722958646680334721192083e-11L, 7.853981633974483095919439232967553115548e-1L),
    TEST_c_c (catanh, 0x1p-33L, -0x0.ffffffffffffffffp0L, 5.820766091346740722958646680334721192083e-11L, -7.853981633974483095919439232967553115548e-1L),
    TEST_c_c (catanh, -0x1p-33L, 0x0.ffffffffffffffffp0L, -5.820766091346740722958646680334721192083e-11L, 7.853981633974483095919439232967553115548e-1L),
    TEST_c_c (catanh, -0x1p-33L, -0x0.ffffffffffffffffp0L, -5.820766091346740722958646680334721192083e-11L, -7.853981633974483095919439232967553115548e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, 0x1p-33L, 1.178350206951907026002603046195591193050e1L, 7.853981631937214964185249205444919953948e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, 0x1p-33L, -1.178350206951907026002603046195591193050e1L, 7.853981631937214964185249205444919953948e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, -0x1p-33L, 1.178350206951907026002603046195591193050e1L, -7.853981631937214964185249205444919953948e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, -0x1p-33L, -1.178350206951907026002603046195591193050e1L, -7.853981631937214964185249205444919953948e-1L),
#endif
#if MANT_DIG >= 106
    TEST_c_c (catanh, 0x1p-54L, 0x1.000000000000000000000000008p0L, 2.775557561562891351059079170226980932999e-17L, 7.853981633974483096156608458198888173729e-1L),
    TEST_c_c (catanh, 0x1p-54L, -0x1.000000000000000000000000008p0L, 2.775557561562891351059079170226980932999e-17L, -7.853981633974483096156608458198888173729e-1L),
    TEST_c_c (catanh, -0x1p-54L, 0x1.000000000000000000000000008p0L, -2.775557561562891351059079170226980932999e-17L, 7.853981633974483096156608458198888173729e-1L),
    TEST_c_c (catanh, -0x1p-54L, -0x1.000000000000000000000000008p0L, -2.775557561562891351059079170226980932999e-17L, -7.853981633974483096156608458198888173729e-1L),
    TEST_c_c (catanh, 0x1.000000000000000000000000008p0L, 0x1p-54L, 1.906154746539849600897388334009981267384e1L, 7.853981633974485455380535786656405610710e-1L),
    TEST_c_c (catanh, -0x1.000000000000000000000000008p0L, 0x1p-54L, -1.906154746539849600897388334009981267384e1L, 7.853981633974485455380535786656405610710e-1L),
    TEST_c_c (catanh, 0x1.000000000000000000000000008p0L, -0x1p-54L, 1.906154746539849600897388334009981267384e1L, -7.853981633974485455380535786656405610710e-1L),
    TEST_c_c (catanh, -0x1.000000000000000000000000008p0L, -0x1p-54L, -1.906154746539849600897388334009981267384e1L, -7.853981633974485455380535786656405610710e-1L),
    TEST_c_c (catanh, 0x1p-54L, 0x0.ffffffffffffffffffffffffffcp0L, 2.775557561562891351059079170227083567164e-17L, 7.853981633974483096156608458198703284454e-1L),
    TEST_c_c (catanh, 0x1p-54L, -0x0.ffffffffffffffffffffffffffcp0L, 2.775557561562891351059079170227083567164e-17L, -7.853981633974483096156608458198703284454e-1L),
    TEST_c_c (catanh, -0x1p-54L, 0x0.ffffffffffffffffffffffffffcp0L, -2.775557561562891351059079170227083567164e-17L, 7.853981633974483096156608458198703284454e-1L),
    TEST_c_c (catanh, -0x1p-54L, -0x0.ffffffffffffffffffffffffffcp0L, -2.775557561562891351059079170227083567164e-17L, -7.853981633974483096156608458198703284454e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-54L, 1.906154746539849600897388334009984040723e1L, 7.853981633974482124711461911186784339815e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-54L, -1.906154746539849600897388334009984040723e1L, 7.853981633974482124711461911186784339815e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-54L, 1.906154746539849600897388334009984040723e1L, -7.853981633974482124711461911186784339815e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-54L, -1.906154746539849600897388334009984040723e1L, -7.853981633974482124711461911186784339815e-1L),
#endif
#if MANT_DIG >= 113
    TEST_c_c (catanh, 0x1p-57L, 0x1.0000000000000000000000000001p0L, 3.469446951953614188823848962783812780530e-18L, 7.853981633974483096156608458198758293829e-1L),
    TEST_c_c (catanh, 0x1p-57L, -0x1.0000000000000000000000000001p0L, 3.469446951953614188823848962783812780530e-18L, -7.853981633974483096156608458198758293829e-1L),
    TEST_c_c (catanh, -0x1p-57L, 0x1.0000000000000000000000000001p0L, -3.469446951953614188823848962783812780530e-18L, 7.853981633974483096156608458198758293829e-1L),
    TEST_c_c (catanh, -0x1p-57L, -0x1.0000000000000000000000000001p0L, -3.469446951953614188823848962783812780530e-18L, -7.853981633974483096156608458198758293829e-1L),
    TEST_c_c (catanh, 0x1.0000000000000000000000000001p0L, 0x1p-57L, 2.010126823623841397309973152228712033275e1L, 7.853981633974483252281721296111395707566e-1L),
    TEST_c_c (catanh, -0x1.0000000000000000000000000001p0L, 0x1p-57L, -2.010126823623841397309973152228712033275e1L, 7.853981633974483252281721296111395707566e-1L),
    TEST_c_c (catanh, 0x1.0000000000000000000000000001p0L, -0x1p-57L, 2.010126823623841397309973152228712033275e1L, -7.853981633974483252281721296111395707566e-1L),
    TEST_c_c (catanh, -0x1.0000000000000000000000000001p0L, -0x1p-57L, -2.010126823623841397309973152228712033275e1L, -7.853981633974483252281721296111395707566e-1L),
    TEST_c_c (catanh, 0x1p-57L, 0x0.ffffffffffffffffffffffffffff8p0L, 3.469446951953614188823848962783813782817e-18L, 7.853981633974483096156608458198756849381e-1L),
    TEST_c_c (catanh, 0x1p-57L, -0x0.ffffffffffffffffffffffffffff8p0L, 3.469446951953614188823848962783813782817e-18L, -7.853981633974483096156608458198756849381e-1L),
    TEST_c_c (catanh, -0x1p-57L, 0x0.ffffffffffffffffffffffffffff8p0L, -3.469446951953614188823848962783813782817e-18L, 7.853981633974483096156608458198756849381e-1L),
    TEST_c_c (catanh, -0x1p-57L, -0x0.ffffffffffffffffffffffffffff8p0L, -3.469446951953614188823848962783813782817e-18L, -7.853981633974483096156608458198756849381e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-57L, 2.010126823623841397309973152228712040498e1L, 7.853981633974483044114904178894544378135e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-57L, -2.010126823623841397309973152228712040498e1L, 7.853981633974483044114904178894544378135e-1L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-57L, 2.010126823623841397309973152228712040498e1L, -7.853981633974483044114904178894544378135e-1L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-57L, -2.010126823623841397309973152228712040498e1L, -7.853981633974483044114904178894544378135e-1L),
#endif
    TEST_c_c (catanh, 1.0L, 0x1p-64L, 2.252728336819822255606004394739073846245e1L, 7.853981633974483096292133729759445264744e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-64L, 2.252728336819822255606004394739073846245e1L, -7.853981633974483096292133729759445264744e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-64L, -2.252728336819822255606004394739073846245e1L, 7.853981633974483096292133729759445264744e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-64L, -2.252728336819822255606004394739073846245e1L, -7.853981633974483096292133729759445264744e-1L),
    TEST_c_c (catanh, 0x1p-64L, 1.0L, 2.710505431213761085018632002174854278563e-20L, 7.853981633974483096156608458198757210500e-1L),
    TEST_c_c (catanh, -0x1p-64L, 1.0L, -2.710505431213761085018632002174854278563e-20L, 7.853981633974483096156608458198757210500e-1L),
    TEST_c_c (catanh, 0x1p-64L, -1.0L, 2.710505431213761085018632002174854278563e-20L, -7.853981633974483096156608458198757210500e-1L),
    TEST_c_c (catanh, -0x1p-64L, -1.0L, -2.710505431213761085018632002174854278563e-20L, -7.853981633974483096156608458198757210500e-1L),
    TEST_c_c (catanh, 1.0L, 0x1.3p-73L, 2.556052055225464683726753902092495938633e1L, 7.853981633974483096156922786831527361009e-1L),
    TEST_c_c (catanh, 1.0L, -0x1.3p-73L, 2.556052055225464683726753902092495938633e1L, -7.853981633974483096156922786831527361009e-1L),
    TEST_c_c (catanh, -1.0L, 0x1.3p-73L, -2.556052055225464683726753902092495938633e1L, 7.853981633974483096156922786831527361009e-1L),
    TEST_c_c (catanh, -1.0L, -0x1.3p-73L, -2.556052055225464683726753902092495938633e1L, -7.853981633974483096156922786831527361009e-1L),
    TEST_c_c (catanh, 0x1.3p-73L, 1.0L, 6.286572655403010329022706059731717687100e-23L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-73L, 1.0L, -6.286572655403010329022706059731717687100e-23L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-73L, -1.0L, 6.286572655403010329022706059731717687100e-23L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-73L, -1.0L, -6.286572655403010329022706059731717687100e-23L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-124L, 4.332169878499658183857700759113603550472e1L, 7.853981633974483096156608458198757210610e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-124L, 4.332169878499658183857700759113603550472e1L, -7.853981633974483096156608458198757210610e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-124L, -4.332169878499658183857700759113603550472e1L, 7.853981633974483096156608458198757210610e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-124L, -4.332169878499658183857700759113603550472e1L, -7.853981633974483096156608458198757210610e-1L),
    TEST_c_c (catanh, 0x1p-124L, 1.0L, 2.350988701644575015937473074444491355637e-38L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-124L, 1.0L, -2.350988701644575015937473074444491355637e-38L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-124L, -1.0L, 2.350988701644575015937473074444491355637e-38L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-124L, -1.0L, -2.350988701644575015937473074444491355637e-38L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.000002p0L, 0x1p-126L, 8.317766196521665212523713550410777685446L, 1.570796326794896619231321691639702138295L),
    TEST_c_c (catanh, 0x1.000002p0L, -0x1p-126L, 8.317766196521665212523713550410777685446L, -1.570796326794896619231321691639702138295L),
    TEST_c_c (catanh, -0x1.000002p0L, 0x1p-126L, -8.317766196521665212523713550410777685446L, 1.570796326794896619231321691639702138295L),
    TEST_c_c (catanh, -0x1.000002p0L, -0x1p-126L, -8.317766196521665212523713550410777685446L, -1.570796326794896619231321691639702138295L),
    TEST_c_c (catanh, 0x1p-126L, 0x1.000002p0L, 5.877471053462247139383742414506671233336e-39L, 7.853982230020895322927482174768220815493e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1p-126L, 0x1.000002p0L, -5.877471053462247139383742414506671233336e-39L, 7.853982230020895322927482174768220815493e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0x1p-126L, -0x1.000002p0L, 5.877471053462247139383742414506671233336e-39L, -7.853982230020895322927482174768220815493e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1p-126L, -0x1.000002p0L, -5.877471053462247139383742414506671233336e-39L, -7.853982230020895322927482174768220815493e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0x0.ffffffp0L, 0x1p-126L, 8.664339742098154951823135931565866792454L, 9.860761609136244031334147100512154160097e-32L),
    TEST_c_c (catanh, 0x0.ffffffp0L, -0x1p-126L, 8.664339742098154951823135931565866792454L, -9.860761609136244031334147100512154160097e-32L),
    TEST_c_c (catanh, -0x0.ffffffp0L, 0x1p-126L, -8.664339742098154951823135931565866792454L, 9.860761609136244031334147100512154160097e-32L),
    TEST_c_c (catanh, -0x0.ffffffp0L, -0x1p-126L, -8.664339742098154951823135931565866792454L, -9.860761609136244031334147100512154160097e-32L),
    TEST_c_c (catanh, 0x1p-126L, 0x0.ffffffp0L, 5.877472104436064061535099214664320916184e-39L, 7.853981335951250337419109991749089175724e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1p-126L, 0x0.ffffffp0L, -5.877472104436064061535099214664320916184e-39L, 7.853981335951250337419109991749089175724e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, 0x1p-126L, -0x0.ffffffp0L, 5.877472104436064061535099214664320916184e-39L, -7.853981335951250337419109991749089175724e-1L, UNDERFLOW_EXCEPTION_FLOAT),
    TEST_c_c (catanh, -0x1p-126L, -0x0.ffffffp0L, -5.877472104436064061535099214664320916184e-39L, -7.853981335951250337419109991749089175724e-1L, UNDERFLOW_EXCEPTION_FLOAT),
#if !TEST_COND_binary32
    TEST_c_c (catanh, 1.0L, 0x1p-512L, 1.777922518136259718655200391540222897114e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-512L, 1.777922518136259718655200391540222897114e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-512L, -1.777922518136259718655200391540222897114e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-512L, -1.777922518136259718655200391540222897114e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-512L, 1.0L, 3.729170365600103371645482657731466918688e-155L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-512L, 1.0L, -3.729170365600103371645482657731466918688e-155L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-512L, -1.0L, 3.729170365600103371645482657731466918688e-155L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-512L, -1.0L, -3.729170365600103371645482657731466918688e-155L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1.3p-536L, 1.860240928518819859673567751384928348958e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1.3p-536L, 1.860240928518819859673567751384928348958e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1.3p-536L, -1.860240928518819859673567751384928348958e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1.3p-536L, -1.860240928518819859673567751384928348958e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-536L, 1.0L, 2.639526015013529511588222179446290115084e-162L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-536L, 1.0L, -2.639526015013529511588222179446290115084e-162L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-536L, -1.0L, 2.639526015013529511588222179446290115084e-162L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-536L, -1.0L, -2.639526015013529511588222179446290115084e-162L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-1020L, 3.538516356758520804574969980043991380025e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-1020L, 3.538516356758520804574969980043991380025e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-1020L, -3.538516356758520804574969980043991380025e2L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-1020L, -3.538516356758520804574969980043991380025e2L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-1020L, 1.0L, 4.450147717014402766180465434664808128438e-308L, 7.853981633974483096156608458198757210493e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, -0x1p-1020L, 1.0L, -4.450147717014402766180465434664808128438e-308L, 7.853981633974483096156608458198757210493e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, 0x1p-1020L, -1.0L, 4.450147717014402766180465434664808128438e-308L, -7.853981633974483096156608458198757210493e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, -0x1p-1020L, -1.0L, -4.450147717014402766180465434664808128438e-308L, -7.853981633974483096156608458198757210493e-1L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, 0x1.0000000000001p0L, 0x1p-1022L, 1.836840028483855075506780244989950299369e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.0000000000001p0L, -0x1p-1022L, 1.836840028483855075506780244989950299369e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000001p0L, 0x1p-1022L, -1.836840028483855075506780244989950299369e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000001p0L, -0x1p-1022L, -1.836840028483855075506780244989950299369e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p-1022L, 0x1.0000000000001p0L, 1.112536929253600444512293438042957369978e-308L, 7.853981633974484206379633083355174374608e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x1.0000000000001p0L, -1.112536929253600444512293438042957369978e-308L, 7.853981633974484206379633083355174374608e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x1.0000000000001p0L, 1.112536929253600444512293438042957369978e-308L, -7.853981633974484206379633083355174374608e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x1.0000000000001p0L, -1.112536929253600444512293438042957369978e-308L, -7.853981633974484206379633083355174374608e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x0.fffffffffffff8p0L, 0x1p-1022L, 1.871497387511852332650969166374185305708e1L, 1.002084180004486444624900488355118689113e-292L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, 0x0.fffffffffffff8p0L, -0x1p-1022L, 1.871497387511852332650969166374185305708e1L, -1.002084180004486444624900488355118689113e-292L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, -0x0.fffffffffffff8p0L, 0x1p-1022L, -1.871497387511852332650969166374185305708e1L, 1.002084180004486444624900488355118689113e-292L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, -0x0.fffffffffffff8p0L, -0x1p-1022L, -1.871497387511852332650969166374185305708e1L, -1.002084180004486444624900488355118689113e-292L, UNDERFLOW_EXCEPTION_LDOUBLE_IBM),
    TEST_c_c (catanh, 0x1p-1022L, 0x0.fffffffffffff8p0L, 1.112536929253600815061527818977844932790e-308L, 7.853981633974482541045096145620456183798e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x0.fffffffffffff8p0L, -1.112536929253600815061527818977844932790e-308L, 7.853981633974482541045096145620456183798e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x0.fffffffffffff8p0L, 1.112536929253600815061527818977844932790e-308L, -7.853981633974482541045096145620456183798e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x0.fffffffffffff8p0L, -1.112536929253600815061527818977844932790e-308L, -7.853981633974482541045096145620456183798e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 64
    TEST_c_c (catanh, 0x1.0000000000000002p0L, 0x1p-1022L, 2.218070977791824990137853294097378778927e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.0000000000000002p0L, -0x1p-1022L, 2.218070977791824990137853294097378778927e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, 0x1p-1022L, -2.218070977791824990137853294097378778927e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, -0x1p-1022L, -2.218070977791824990137853294097378778927e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p-1022L, 0x1.0000000000000002p0L, 1.112536929253600691424494863099491450042e-308L, 7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x1.0000000000000002p0L, -1.112536929253600691424494863099491450042e-308L, 7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x1.0000000000000002p0L, 1.112536929253600691424494863099491450042e-308L, -7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x1.0000000000000002p0L, -1.112536929253600691424494863099491450042e-308L, -7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, 0x1p-1022L, 2.252728336819822255604649142023466965703e1L, 2.052268400649188124723641491045245971623e-289L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, -0x1p-1022L, 2.252728336819822255604649142023466965703e1L, -2.052268400649188124723641491045245971623e-289L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, 0x1p-1022L, -2.252728336819822255604649142023466965703e1L, 2.052268400649188124723641491045245971623e-289L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, -0x1p-1022L, -2.252728336819822255604649142023466965703e1L, -2.052268400649188124723641491045245971623e-289L),
    TEST_c_c (catanh, 0x1p-1022L, 0x0.ffffffffffffffffp0L, 1.112536929253600691605427106449557323148e-308L, 7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x0.ffffffffffffffffp0L, -1.112536929253600691605427106449557323148e-308L, 7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x0.ffffffffffffffffp0L, 1.112536929253600691605427106449557323148e-308L, -7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x0.ffffffffffffffffp0L, -1.112536929253600691605427106449557323148e-308L, -7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MANT_DIG >= 106
    TEST_c_c (catanh, 0x1.000000000000000000000000008p0L, 0x1p-1022L, 3.673680056967710139911330243728336427098e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.000000000000000000000000008p0L, -0x1p-1022L, 3.673680056967710139911330243728336427098e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.000000000000000000000000008p0L, 0x1p-1022L, -3.673680056967710139911330243728336427098e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.000000000000000000000000008p0L, -0x1p-1022L, -3.673680056967710139911330243728336427098e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p-1022L, 0x1.000000000000000000000000008p0L, 1.112536929253600691545116358666174605957e-308L, 7.853981633974483096156608458198880470009e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x1.000000000000000000000000008p0L, -1.112536929253600691545116358666174605957e-308L, 7.853981633974483096156608458198880470009e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x1.000000000000000000000000008p0L, 1.112536929253600691545116358666174605957e-308L, -7.853981633974483096156608458198880470009e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x1.000000000000000000000000008p0L, -1.112536929253600691545116358666174605957e-308L, -7.853981633974483096156608458198880470009e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffcp0L, 0x1p-1022L, 3.708337415995707405382191849801244331055e1L, 9.025971879324147880346310405868788320726e-277L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffcp0L, -0x1p-1022L, 3.708337415995707405382191849801244331055e1L, -9.025971879324147880346310405868788320726e-277L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffcp0L, 0x1p-1022L, -3.708337415995707405382191849801244331055e1L, 9.025971879324147880346310405868788320726e-277L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffcp0L, -0x1p-1022L, -3.708337415995707405382191849801244331055e1L, -9.025971879324147880346310405868788320726e-277L),
    TEST_c_c (catanh, 0x1p-1022L, 0x0.ffffffffffffffffffffffffffcp0L, 1.112536929253600691545116358666215745186e-308L, 7.853981633974483096156608458198695580735e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, 0x0.ffffffffffffffffffffffffffcp0L, -1.112536929253600691545116358666215745186e-308L, 7.853981633974483096156608458198695580735e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, 0x1p-1022L, -0x0.ffffffffffffffffffffffffffcp0L, 1.112536929253600691545116358666215745186e-308L, -7.853981633974483096156608458198695580735e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
    TEST_c_c (catanh, -0x1p-1022L, -0x0.ffffffffffffffffffffffffffcp0L, -1.112536929253600691545116358666215745186e-308L, -7.853981633974483096156608458198695580735e-1L, UNDERFLOW_EXCEPTION_DOUBLE),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (catanh, 1.0L, 0x1p-8192L, 2.839477425163815960027691385553420311121e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-8192L, 2.839477425163815960027691385553420311121e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-8192L, -2.839477425163815960027691385553420311121e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-8192L, -2.839477425163815960027691385553420311121e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-8192L, 1.0L, 4.584009668887117914053530980121207914891e-2467L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-8192L, 1.0L, -4.584009668887117914053530980121207914891e-2467L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-8192L, -1.0L, 4.584009668887117914053530980121207914891e-2467L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-8192L, -1.0L, -4.584009668887117914053530980121207914891e-2467L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1.3p-8221L, 2.849442134153471837403071201841536297726e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1.3p-8221L, 2.849442134153471837403071201841536297726e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1.3p-8221L, -2.849442134153471837403071201841536297726e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1.3p-8221L, -2.849442134153471837403071201841536297726e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-8221L, 1.0L, 1.013933025636421986471594877335044443409e-2475L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-8221L, 1.0L, -1.013933025636421986471594877335044443409e-2475L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-8221L, -1.0L, 1.013933025636421986471594877335044443409e-2475L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-8221L, -1.0L, -1.013933025636421986471594877335044443409e-2475L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1.3p-8246L, 2.858106473910471153770786603359763504827e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1.3p-8246L, 2.858106473910471153770786603359763504827e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1.3p-8246L, -2.858106473910471153770786603359763504827e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1.3p-8246L, -2.858106473910471153770786603359763504827e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-8246L, 1.0L, 3.021755890954798419688924781486524472858e-2483L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-8246L, 1.0L, -3.021755890954798419688924781486524472858e-2483L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.3p-8246L, -1.0L, 3.021755890954798419688924781486524472858e-2483L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1.3p-8246L, -1.0L, -3.021755890954798419688924781486524472858e-2483L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, 0x1p-16380L, 5.677221982376232056781839690803195180822e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 1.0L, -0x1p-16380L, 5.677221982376232056781839690803195180822e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, 0x1p-16380L, -5.677221982376232056781839690803195180822e3L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -1.0L, -0x1p-16380L, -5.677221982376232056781839690803195180822e3L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-16380L, 1.0L, 6.724206286224187012525355634643505205196e-4932L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-16380L, 1.0L, -6.724206286224187012525355634643505205196e-4932L, 7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1p-16380L, -1.0L, 6.724206286224187012525355634643505205196e-4932L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, -0x1p-16380L, -1.0L, -6.724206286224187012525355634643505205196e-4932L, -7.853981633974483096156608458198757210493e-1L),
    TEST_c_c (catanh, 0x1.0000000000000002p0L, 0x1p-16382L, 2.218070977791824990137853294097378778927e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.0000000000000002p0L, -0x1p-16382L, 2.218070977791824990137853294097378778927e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, 0x1p-16382L, -2.218070977791824990137853294097378778927e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000002p0L, -0x1p-16382L, -2.218070977791824990137853294097378778927e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p-16382L, 0x1.0000000000000002p0L, 1.681051571556046752949078932066752571182e-4932L, 7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, 0x1.0000000000000002p0L, -1.681051571556046752949078932066752571182e-4932L, 7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1p-16382L, -0x1.0000000000000002p0L, 1.681051571556046752949078932066752571182e-4932L, -7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, -0x1.0000000000000002p0L, -1.681051571556046752949078932066752571182e-4932L, -7.853981633974483096698709544441509427467e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, 0x1p-16382L, 2.252728336819822255604649142023466965703e1L, 3.100992811520163369065387859792822623745e-4913L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffp0L, -0x1p-16382L, 2.252728336819822255604649142023466965703e1L, -3.100992811520163369065387859792822623745e-4913L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, 0x1p-16382L, -2.252728336819822255604649142023466965703e1L, 3.100992811520163369065387859792822623745e-4913L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffp0L, -0x1p-16382L, -2.252728336819822255604649142023466965703e1L, -3.100992811520163369065387859792822623745e-4913L),
    TEST_c_c (catanh, 0x1p-16382L, 0x0.ffffffffffffffffp0L, 1.681051571556046753222468896957938166365e-4932L, 7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, 0x0.ffffffffffffffffp0L, -1.681051571556046753222468896957938166365e-4932L, 7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1p-16382L, -0x0.ffffffffffffffffp0L, 1.681051571556046753222468896957938166365e-4932L, -7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, -0x0.ffffffffffffffffp0L, -1.681051571556046753222468896957938166365e-4932L, -7.853981633974483095885557915077381101984e-1L, UNDERFLOW_EXCEPTION),
# if MANT_DIG >= 113
    TEST_c_c (catanh, 0x1.0000000000000000000000000001p0L, 0x1p-16382L, 3.916281570163690998207361486238697614441e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1.0000000000000000000000000001p0L, -0x1p-16382L, 3.916281570163690998207361486238697614441e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000000000000000001p0L, 0x1p-16382L, -3.916281570163690998207361486238697614441e1L, 1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, -0x1.0000000000000000000000000001p0L, -0x1p-16382L, -3.916281570163690998207361486238697614441e1L, -1.570796326794896619231321691639751442099L),
    TEST_c_c (catanh, 0x1p-16382L, 0x1.0000000000000000000000000001p0L, 1.681051571556046753131338908660875977540e-4932L, 7.853981633974483096156608458198758173458e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, 0x1.0000000000000000000000000001p0L, -1.681051571556046753131338908660875977540e-4932L, 7.853981633974483096156608458198758173458e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1p-16382L, -0x1.0000000000000000000000000001p0L, 1.681051571556046753131338908660875977540e-4932L, -7.853981633974483096156608458198758173458e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, -0x1.0000000000000000000000000001p0L, -1.681051571556046753131338908660875977540e-4932L, -7.853981633974483096156608458198758173458e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-16382L, 3.950938929191688263678223092311606435623e1L, 1.745703758805099310527547423749501866998e-4898L),
    TEST_c_c (catanh, 0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-16382L, 3.950938929191688263678223092311606435623e1L, -1.745703758805099310527547423749501866998e-4898L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffff8p0L, 0x1p-16382L, -3.950938929191688263678223092311606435623e1L, 1.745703758805099310527547423749501866998e-4898L),
    TEST_c_c (catanh, -0x0.ffffffffffffffffffffffffffff8p0L, -0x1p-16382L, -3.950938929191688263678223092311606435623e1L, -1.745703758805099310527547423749501866998e-4898L),
    TEST_c_c (catanh, 0x1p-16382L, 0x0.ffffffffffffffffffffffffffff8p0L, 1.681051571556046753131338908660876463178e-4932L, 7.853981633974483096156608458198756729010e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, 0x0.ffffffffffffffffffffffffffff8p0L, -1.681051571556046753131338908660876463178e-4932L, 7.853981633974483096156608458198756729010e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, 0x1p-16382L, -0x0.ffffffffffffffffffffffffffff8p0L, 1.681051571556046753131338908660876463178e-4932L, -7.853981633974483096156608458198756729010e-1L, UNDERFLOW_EXCEPTION),
    TEST_c_c (catanh, -0x1p-16382L, -0x0.ffffffffffffffffffffffffffff8p0L, -1.681051571556046753131338908660876463178e-4932L, -7.853981633974483096156608458198756729010e-1L, UNDERFLOW_EXCEPTION),
# endif
#endif

    TEST_c_c (catanh, 0.75L, 1.25L, 0.261492138795671927078652057366532140L, 0.996825126463918666098902241310446708L),
    TEST_c_c (catanh, -2, -3, -0.14694666622552975204743278515471595L, -1.3389725222944935611241935759091443L),
  };

static void
catanh_test (void)
{
  ALL_RM_TEST (catanh, 0, catanh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}

static const struct test_f_f_data cbrt_test_data[] =
  {
    TEST_f_f (cbrt, plus_infty, plus_infty, ERRNO_UNCHANGED),
    TEST_f_f (cbrt, minus_infty, minus_infty, ERRNO_UNCHANGED),
    TEST_f_f (cbrt, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cbrt, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cbrt, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (cbrt, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (cbrt),
  };

static void
cbrt_test (void)
{
  ALL_RM_TEST (cbrt, 0, cbrt_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_c_c_data ccos_test_data[] =
  {
    TEST_c_c (ccos, plus_infty, 0.0, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccos, plus_infty, minus_zero, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccos, minus_infty, 0.0, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccos, minus_infty, minus_zero, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccos, 0.0, plus_infty, plus_infty, minus_zero),
    TEST_c_c (ccos, 0.0, minus_infty, plus_infty, 0.0),
    TEST_c_c (ccos, minus_zero, plus_infty, plus_infty, 0.0),
    TEST_c_c (ccos, minus_zero, minus_infty, plus_infty, minus_zero),

    TEST_c_c (ccos, plus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, minus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, plus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, minus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ccos, 4.625, plus_infty, minus_infty, plus_infty),
    TEST_c_c (ccos, 4.625, minus_infty, minus_infty, minus_infty),
    TEST_c_c (ccos, -4.625, plus_infty, minus_infty, minus_infty),
    TEST_c_c (ccos, -4.625, minus_infty, minus_infty, plus_infty),

    TEST_c_c (ccos, plus_infty, 6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, plus_infty, -6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, minus_infty, 6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccos, minus_infty, -6.75, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ccos, qnan_value, 0.0, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccos, qnan_value, minus_zero, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccos, qnan_value, plus_infty, plus_infty, qnan_value),
    TEST_c_c (ccos, qnan_value, minus_infty, plus_infty, qnan_value),

    TEST_c_c (ccos, qnan_value, 9.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccos, qnan_value, -9.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccos, 0.0, qnan_value, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccos, minus_zero, qnan_value, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccos, 10.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccos, -10.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccos, plus_infty, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccos, minus_infty, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccos, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (ccos),
  };

static void
ccos_test (void)
{
  ALL_RM_TEST (ccos, 0, ccos_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data ccosh_test_data[] =
  {
    TEST_c_c (ccosh, 0.0, plus_infty, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccosh, minus_zero, plus_infty, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccosh, 0.0, minus_infty, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccosh, minus_zero, minus_infty, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccosh, plus_infty, 0.0, plus_infty, 0.0),
    TEST_c_c (ccosh, minus_infty, 0.0, plus_infty, minus_zero),
    TEST_c_c (ccosh, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (ccosh, minus_infty, minus_zero, plus_infty, 0.0),

    TEST_c_c (ccosh, plus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, minus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, plus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, minus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ccosh, plus_infty, 4.625, minus_infty, minus_infty),
    TEST_c_c (ccosh, minus_infty, 4.625, minus_infty, plus_infty),
    TEST_c_c (ccosh, plus_infty, -4.625, minus_infty, plus_infty),
    TEST_c_c (ccosh, minus_infty, -4.625, minus_infty, minus_infty),

    TEST_c_c (ccosh, 6.75, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, -6.75, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, 6.75, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ccosh, -6.75, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ccosh, 0.0, qnan_value, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccosh, minus_zero, qnan_value, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccosh, plus_infty, qnan_value, plus_infty, qnan_value),
    TEST_c_c (ccosh, minus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (ccosh, 9.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccosh, -9.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccosh, qnan_value, 0.0, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ccosh, qnan_value, minus_zero, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ccosh, qnan_value, 10.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccosh, qnan_value, -10.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccosh, qnan_value, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ccosh, qnan_value, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ccosh, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (ccosh),
  };

static void
ccosh_test (void)
{
  ALL_RM_TEST (ccosh, 0, ccosh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_f_f_data ceil_test_data[] =
  {
    TEST_f_f (ceil, 0.0, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_f_f (ceil, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_f_f (ceil, lit_pi, 4.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -lit_pi, -3.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, min_subnorm_value, 1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, min_value, 1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0.1, 1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0.25, 1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0.625, 1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 1, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 2, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p23, 0x1p23, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p24, 0x1p24, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p25, 0x1p25, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p52, 0x1p52, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p53, 0x1p53, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p54, 0x1p54, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p63, 0x1p63, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p64, 0x1p64, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p65, 0x1p65, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p105, 0x1p105, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p106, 0x1p106, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p107, 0x1p107, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p112, 0x1p112, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p113, 0x1p113, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 0x1p114, 0x1p114, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0.1, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0.25, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0.625, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p23, -0x1p23, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p24, -0x1p24, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p25, -0x1p25, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p52, -0x1p52, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p53, -0x1p53, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p54, -0x1p54, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p63, -0x1p63, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p64, -0x1p64, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p65, -0x1p65, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p105, -0x1p105, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p106, -0x1p106, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p107, -0x1p107, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p112, -0x1p112, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p113, -0x1p113, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -0x1p114, -0x1p114, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

#if MANT_DIG >= 64
    /* The result can only be represented in long double.  */
    TEST_f_f (ceil, 4503599627370495.5L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370496.25L, 4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370496.5L, 4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370496.75L, 4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370497.5L, 4503599627370498.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (ceil, -4503599627370495.5L, -4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370496.25L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370496.5L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370496.75L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370497.5L, -4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

# if MANT_DIG > 100
    TEST_f_f (ceil, 4503599627370494.5000000000001L, 4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370495.5000000000001L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 4503599627370496.5000000000001L, 4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370494.5000000000001L, -4503599627370494.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370495.5000000000001L, -4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -4503599627370496.5000000000001L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (ceil, 9007199254740991.5L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740992.25L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740992.5L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740992.75L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740993.5L, 9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (ceil, -9007199254740991.5L, -9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740992.25L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740992.5L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740992.75L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740993.5L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

# if MANT_DIG > 100
    TEST_f_f (ceil, 9007199254740991.0000000000001L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740992.0000000000001L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740993.0000000000001L, 9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740991.5000000000001L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740992.5000000000001L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 9007199254740993.5000000000001L, 9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (ceil, -9007199254740991.0000000000001L, -9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740992.0000000000001L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740993.0000000000001L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740991.5000000000001L, -9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740992.5000000000001L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -9007199254740993.5000000000001L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (ceil, 72057594037927935.5L, 72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 72057594037927936.25L, 72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 72057594037927936.5L, 72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 72057594037927936.75L, 72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 72057594037927937.5L, 72057594037927938.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (ceil, -72057594037927935.5L, -72057594037927935.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -72057594037927936.25L, -72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -72057594037927936.5L, -72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -72057594037927936.75L, -72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -72057594037927937.5L, -72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    /* Check cases where first double is a exact integer higher than 2^52 and
       the precision is determined by second long double for IBM long double.  */
    TEST_f_f (ceil,  34503599627370498.515625L, 34503599627370499.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -34503599627370498.515625L, -34503599627370498.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# if MANT_DIG >= 106
    TEST_f_f (ceil,  1192568192774434123539907640624.484375L, 1192568192774434123539907640625.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, -1192568192774434123539907640624.484375L, -1192568192774434123539907640624.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (ceil, 10141204801825835211973625643007.5L, 10141204801825835211973625643008.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 10141204801825835211973625643008.25L, 10141204801825835211973625643009.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 10141204801825835211973625643008.5L, 10141204801825835211973625643009.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 10141204801825835211973625643008.75L, 10141204801825835211973625643009.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (ceil, 10141204801825835211973625643009.5L, 10141204801825835211973625643010.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
  };

static void
ceil_test (void)
{
  ALL_RM_TEST (ceil, 1, ceil_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_c_c_data cexp_test_data[] =
  {
    TEST_c_c (cexp, plus_infty, plus_zero, plus_infty, 0.0),
    TEST_c_c (cexp, plus_infty, minus_zero, plus_infty, minus_zero),

    TEST_c_c (cexp, minus_infty, plus_zero, 0.0, 0.0),
    TEST_c_c (cexp, minus_infty, minus_zero, 0.0, minus_zero),

    TEST_c_c (cexp, 0.0, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (cexp, minus_zero, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (cexp, 0.0, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (cexp, minus_zero, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (cexp, 100.0, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (cexp, -100.0, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (cexp, 100.0, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (cexp, -100.0, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (cexp, minus_infty, 2.0, minus_zero, 0.0),
    TEST_c_c (cexp, minus_infty, 4.0, minus_zero, minus_zero),
    TEST_c_c (cexp, plus_infty, 2.0, minus_infty, plus_infty),
    TEST_c_c (cexp, plus_infty, 4.0, minus_infty, minus_infty),

    TEST_c_c (cexp, plus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (cexp, plus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (cexp, minus_infty, plus_infty, 0.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (cexp, minus_infty, minus_infty, 0.0, minus_zero, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (cexp, minus_infty, qnan_value, 0, 0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (cexp, plus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (cexp, qnan_value, 0.0, qnan_value, 0.0),
    TEST_c_c (cexp, qnan_value, minus_zero, qnan_value, minus_zero),
    TEST_c_c (cexp, qnan_value, 1.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (cexp, qnan_value, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cexp, 0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cexp, 1, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (cexp, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (cexp),
  };

static void
cexp_test (void)
{
  ALL_RM_TEST (cexp, 0, cexp_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_f_data cimag_test_data[] =
  {
    TEST_c_f (cimag, 1.0, 0.0, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 1.0, minus_zero, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 1.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 1.0, snan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, qnan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, snan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 1.0, plus_infty, plus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 1.0, minus_infty, minus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_f (cimag, 2.0, 3.0, 3.0, NO_INEXACT_EXCEPTION),
  };

static void
cimag_test (void)
{
  ALL_RM_TEST (cimag, 1, cimag_test_data, RUN_TEST_LOOP_c_f, END);
}

static const struct test_c_c_data clog_test_data[] =
  {
    TEST_c_c (clog, minus_zero, 0, minus_infty, lit_pi, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (clog, minus_zero, minus_zero, minus_infty, -lit_pi, DIVIDE_BY_ZERO_EXCEPTION),

    TEST_c_c (clog, 0, 0, minus_infty, 0.0, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (clog, 0, minus_zero, minus_infty, minus_zero, DIVIDE_BY_ZERO_EXCEPTION),

    TEST_c_c (clog, minus_infty, plus_infty, plus_infty, lit_pi_3_m_4_d),
    TEST_c_c (clog, minus_infty, minus_infty, plus_infty, -lit_pi_3_m_4_d),

    TEST_c_c (clog, plus_infty, plus_infty, plus_infty, lit_pi_4_d),
    TEST_c_c (clog, plus_infty, minus_infty, plus_infty, -lit_pi_4_d),

    TEST_c_c (clog, 0, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (clog, 3, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (clog, minus_zero, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (clog, -3, plus_infty, plus_infty, lit_pi_2_d),
    TEST_c_c (clog, 0, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (clog, 3, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (clog, minus_zero, minus_infty, plus_infty, -lit_pi_2_d),
    TEST_c_c (clog, -3, minus_infty, plus_infty, -lit_pi_2_d),

    TEST_c_c (clog, minus_infty, 0, plus_infty, lit_pi),
    TEST_c_c (clog, minus_infty, 1, plus_infty, lit_pi),
    TEST_c_c (clog, minus_infty, minus_zero, plus_infty, -lit_pi),
    TEST_c_c (clog, minus_infty, -1, plus_infty, -lit_pi),

    TEST_c_c (clog, plus_infty, 0, plus_infty, 0.0),
    TEST_c_c (clog, plus_infty, 1, plus_infty, 0.0),
    TEST_c_c (clog, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (clog, plus_infty, -1, plus_infty, minus_zero),

    TEST_c_c (clog, plus_infty, qnan_value, plus_infty, qnan_value),
    TEST_c_c (clog, minus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (clog, qnan_value, plus_infty, plus_infty, qnan_value),
    TEST_c_c (clog, qnan_value, minus_infty, plus_infty, qnan_value),

    TEST_c_c (clog, 0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, 3, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, minus_zero, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, -3, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (clog, qnan_value, 0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, qnan_value, 5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, qnan_value, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog, qnan_value, -5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (clog, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (clog),
  };

static void
clog_test (void)
{
  ALL_RM_TEST (clog, 0, clog_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data clog10_test_data[] =
  {
    TEST_c_c (clog10, minus_zero, 0, minus_infty, lit_pi_ln10_d, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (clog10, minus_zero, minus_zero, minus_infty, -lit_pi_ln10_d, DIVIDE_BY_ZERO_EXCEPTION),

    TEST_c_c (clog10, 0, 0, minus_infty, 0.0, DIVIDE_BY_ZERO_EXCEPTION),
    TEST_c_c (clog10, 0, minus_zero, minus_infty, minus_zero, DIVIDE_BY_ZERO_EXCEPTION),

    TEST_c_c (clog10, minus_infty, plus_infty, plus_infty, lit_pi_3_m_4_ln10_m_d),

    TEST_c_c (clog10, plus_infty, plus_infty, plus_infty, lit_pi_4_ln10_m_d),
    TEST_c_c (clog10, plus_infty, minus_infty, plus_infty, -lit_pi_4_ln10_m_d),

    TEST_c_c (clog10, 0, plus_infty, plus_infty, lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, 3, plus_infty, plus_infty, lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, minus_zero, plus_infty, plus_infty, lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, -3, plus_infty, plus_infty, lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, 0, minus_infty, plus_infty, -lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, 3, minus_infty, plus_infty, -lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, minus_zero, minus_infty, plus_infty, -lit_pi_2_ln10_m_d),
    TEST_c_c (clog10, -3, minus_infty, plus_infty, -lit_pi_2_ln10_m_d),

    TEST_c_c (clog10, minus_infty, 0, plus_infty, lit_pi_ln10_d),
    TEST_c_c (clog10, minus_infty, 1, plus_infty, lit_pi_ln10_d),
    TEST_c_c (clog10, minus_infty, minus_zero, plus_infty, -lit_pi_ln10_d),
    TEST_c_c (clog10, minus_infty, -1, plus_infty, -lit_pi_ln10_d),

    TEST_c_c (clog10, plus_infty, 0, plus_infty, 0.0),
    TEST_c_c (clog10, plus_infty, 1, plus_infty, 0.0),
    TEST_c_c (clog10, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (clog10, plus_infty, -1, plus_infty, minus_zero),

    TEST_c_c (clog10, plus_infty, qnan_value, plus_infty, qnan_value),
    TEST_c_c (clog10, minus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (clog10, qnan_value, plus_infty, plus_infty, qnan_value),
    TEST_c_c (clog10, qnan_value, minus_infty, plus_infty, qnan_value),

    TEST_c_c (clog10, 0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, 3, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, minus_zero, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, -3, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (clog10, qnan_value, 0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, qnan_value, 5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, qnan_value, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (clog10, qnan_value, -5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (clog10, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (clog10),
  };

static void
clog10_test (void)
{
  ALL_RM_TEST (clog10, 0, clog10_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data conj_test_data[] =
  {
    TEST_c_c (conj, 0.0, 0.0, 0.0, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, 0.0, minus_zero, 0.0, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, qnan_value, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, qnan_value, snan_value, qnan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, snan_value, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, snan_value, snan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, plus_infty, minus_infty, plus_infty, plus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, plus_infty, plus_infty, plus_infty, minus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, 1.0, 2.0, 1.0, -2.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (conj, 3.0, -4.0, 3.0, 4.0, NO_INEXACT_EXCEPTION),
  };

static void
conj_test (void)
{
  ALL_RM_TEST (conj, 1, conj_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_ff_f_data copysign_test_data[] =
  {
    TEST_ff_f (copysign, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 0, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, 0, 4, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 0, -4, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, 4, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, -4, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, 2, 0, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 2, minus_zero, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -2, 0, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -2, minus_zero, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, plus_infty, 0, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, plus_infty, minus_zero, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_infty, 0, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_infty, minus_zero, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, 0, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 0, minus_infty, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, minus_infty, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, 0, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 0, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, qnan_value, minus_zero, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, minus_zero, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, qnan_value, -qnan_value, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, -qnan_value, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, 0, snan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, 0, -snan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, snan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, minus_zero, -snan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, snan_value, 0, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, snan_value, minus_zero, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, 0, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, minus_zero, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, qnan_value, -snan_value, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -qnan_value, -snan_value, -qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, snan_value, qnan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, snan_value, -qnan_value, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, qnan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, -qnan_value, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, snan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, snan_value, -snan_value, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_ff_f (copysign, -snan_value, -snan_value, -snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_ff_f (copysign, min_value, min_subnorm_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, min_value, -min_subnorm_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -min_value, min_subnorm_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -min_value, -min_subnorm_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, min_subnorm_value, max_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, min_subnorm_value, -max_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -min_subnorm_value, max_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -min_subnorm_value, -max_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (copysign, max_value, min_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, max_value, -min_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -max_value, min_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (copysign, -max_value, -min_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
  };

static void
copysign_test (void)
{
  ALL_RM_TEST (copysign, 1, copysign_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_f_f_data cos_test_data[] =
  {
    TEST_f_f (cos, plus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (cos, minus_infty, qnan_value, INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_f_f (cos, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cos, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cos, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (cos, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (cos),
  };

static void
cos_test (void)
{
  ALL_RM_TEST (cos, 0, cos_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data cosh_test_data[] =
  {
    TEST_f_f (cosh, plus_infty, plus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (cosh, minus_infty, plus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (cosh, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cosh, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (cosh, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (cosh, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (cosh),
  };

static void
cosh_test (void)
{
  ALL_RM_TEST (cosh, 0, cosh_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_cc_c_data cpow_test_data[] =
  {
    TEST_cc_c (cpow, qnan_value, qnan_value, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_cc_c (cpow),
  };

static void
cpow_test (void)
{
  ALL_RM_TEST (cpow, 0, cpow_test_data, RUN_TEST_LOOP_cc_c, END_COMPLEX);
}


static const struct test_c_c_data cproj_test_data[] =
  {
    TEST_c_c (cproj, 0.0, 0.0, 0.0, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_zero, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, 0.0, minus_zero, 0.0, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_zero, 0.0, minus_zero, 0.0, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, qnan_value, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_zero, qnan_value, plus_zero, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_zero, qnan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, qnan_value, plus_zero, qnan_value, plus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, qnan_value, minus_zero, qnan_value, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, 1.0, qnan_value, 1.0, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, -1.0, qnan_value, -1.0, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, qnan_value, 1.0, qnan_value, 1.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, qnan_value, -1.0, qnan_value, -1.0, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_infty, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, plus_infty, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_infty, plus_zero, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, plus_infty, minus_zero, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, plus_zero, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, minus_zero, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_zero, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, plus_zero, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_zero, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_zero, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_infty, 1.0, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, plus_infty, -1.0, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, 1.0, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, -1.0, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, 1.0, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, 1.0, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, -1.0, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, -1.0, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, plus_infty, qnan_value, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, plus_infty, -qnan_value, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, qnan_value, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, minus_infty, -qnan_value, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, qnan_value, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, qnan_value, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, -qnan_value, plus_infty, plus_infty, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, -qnan_value, minus_infty, plus_infty, minus_zero, NO_INEXACT_EXCEPTION),

    TEST_c_c (cproj, 1.0, 0.0, 1.0, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_c (cproj, 2.0, 3.0, 2.0, 3.0, NO_INEXACT_EXCEPTION),
  };

static void
cproj_test (void)
{
  ALL_RM_TEST (cproj, 1, cproj_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_f_data creal_test_data[] =
  {
    TEST_c_f (creal, 0.0, 1.0, 0.0, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, minus_zero, 1.0, minus_zero, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, qnan_value, 1.0, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, snan_value, 1.0, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, snan_value, qnan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, snan_value, snan_value, snan_value, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, plus_infty, 1.0, plus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, minus_infty, 1.0, minus_infty, NO_INEXACT_EXCEPTION),
    TEST_c_f (creal, 2.0, 3.0, 2.0, NO_INEXACT_EXCEPTION),
  };

static void
creal_test (void)
{
  ALL_RM_TEST (creal, 1, creal_test_data, RUN_TEST_LOOP_c_f, END);
}

static const struct test_c_c_data csin_test_data[] =
  {
    TEST_c_c (csin, 0.0, plus_infty, 0.0, plus_infty),
    TEST_c_c (csin, minus_zero, plus_infty, minus_zero, plus_infty),
    TEST_c_c (csin, 0.0, minus_infty, 0.0, minus_infty),
    TEST_c_c (csin, minus_zero, minus_infty, minus_zero, minus_infty),

    TEST_c_c (csin, plus_infty, 0.0, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, minus_infty, 0.0, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, plus_infty, minus_zero, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, minus_infty, minus_zero, qnan_value, 0.0, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csin, plus_infty, plus_infty, qnan_value, plus_infty, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, minus_infty, plus_infty, qnan_value, plus_infty, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, plus_infty, minus_infty, qnan_value, plus_infty, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, minus_infty, minus_infty, qnan_value, plus_infty, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csin, plus_infty, 6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csin, plus_infty, -6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csin, minus_infty, 6.75, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csin, minus_infty, -6.75,  qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (csin, 4.625, plus_infty, minus_infty, minus_infty),
    TEST_c_c (csin, 4.625, minus_infty, minus_infty, plus_infty),
    TEST_c_c (csin, -4.625, plus_infty, plus_infty, minus_infty),
    TEST_c_c (csin, -4.625, minus_infty, plus_infty, plus_infty),

    TEST_c_c (csin, qnan_value, 0.0, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, qnan_value, minus_zero, qnan_value, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csin, qnan_value, plus_infty, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csin, qnan_value, minus_infty, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csin, qnan_value, 9.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csin, qnan_value, -9.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csin, 0.0, qnan_value, 0.0, qnan_value),
    TEST_c_c (csin, minus_zero, qnan_value, minus_zero, qnan_value),

    TEST_c_c (csin, 10.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csin, qnan_value, -10.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csin, plus_infty, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csin, minus_infty, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csin, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (csin),

    TEST_c_c (csin, min_subnorm_value, min_value, min_subnorm_value, min_value, UNDERFLOW_EXCEPTION),
    TEST_c_c (csin, min_value, min_subnorm_value, min_value, min_subnorm_value, UNDERFLOW_EXCEPTION),
  };

static void
csin_test (void)
{
  ALL_RM_TEST (csin, 0, csin_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data csinh_test_data[] =
  {
    TEST_c_c (csinh, 0.0, plus_infty, 0.0, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_zero, plus_infty, 0.0, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, 0.0, minus_infty, 0.0, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_zero, minus_infty, 0.0, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csinh, plus_infty, 0.0, plus_infty, 0.0),
    TEST_c_c (csinh, minus_infty, 0.0, minus_infty, 0.0),
    TEST_c_c (csinh, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (csinh, minus_infty, minus_zero, minus_infty, minus_zero),

    TEST_c_c (csinh, plus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_infty, plus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, plus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_infty, minus_infty, plus_infty, qnan_value, INVALID_EXCEPTION|IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csinh, plus_infty, 4.625, minus_infty, minus_infty),
    TEST_c_c (csinh, minus_infty, 4.625, plus_infty, minus_infty),
    TEST_c_c (csinh, plus_infty, -4.625, minus_infty, plus_infty),
    TEST_c_c (csinh, minus_infty, -4.625, plus_infty, plus_infty),

    TEST_c_c (csinh, 6.75, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csinh, -6.75, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csinh, 6.75, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (csinh, -6.75, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (csinh, 0.0, qnan_value, 0.0, qnan_value, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_zero, qnan_value, 0.0, qnan_value, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csinh, plus_infty, qnan_value, plus_infty, qnan_value, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (csinh, minus_infty, qnan_value, plus_infty, qnan_value, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csinh, 9.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csinh, -9.0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csinh, qnan_value, 0.0, qnan_value, 0.0),
    TEST_c_c (csinh, qnan_value, minus_zero, qnan_value, minus_zero),

    TEST_c_c (csinh, qnan_value, 10.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csinh, qnan_value, -10.0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csinh, qnan_value, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csinh, qnan_value, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csinh, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (csinh),

    TEST_c_c (csinh, min_subnorm_value, min_value, min_subnorm_value, min_value, UNDERFLOW_EXCEPTION),
    TEST_c_c (csinh, min_value, min_subnorm_value, min_value, min_subnorm_value, UNDERFLOW_EXCEPTION),
  };

static void
csinh_test (void)
{
  ALL_RM_TEST (csinh, 0, csinh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data csqrt_test_data[] =
  {
    TEST_c_c (csqrt, minus_infty, 0, 0.0, plus_infty),
    TEST_c_c (csqrt, minus_infty, 6, 0.0, plus_infty),
    TEST_c_c (csqrt, minus_infty, minus_zero, 0.0, minus_infty),
    TEST_c_c (csqrt, minus_infty, -6, 0.0, minus_infty),

    TEST_c_c (csqrt, plus_infty, 0, plus_infty, 0.0),
    TEST_c_c (csqrt, plus_infty, 6, plus_infty, 0.0),
    TEST_c_c (csqrt, plus_infty, minus_zero, plus_infty, minus_zero),
    TEST_c_c (csqrt, plus_infty, -6, plus_infty, minus_zero),

    TEST_c_c (csqrt, 0, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, 4, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, plus_infty, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, minus_zero, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, -4, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, minus_infty, plus_infty, plus_infty, plus_infty),
    TEST_c_c (csqrt, 0, minus_infty, plus_infty, minus_infty),
    TEST_c_c (csqrt, 4, minus_infty, plus_infty, minus_infty),
    TEST_c_c (csqrt, plus_infty, minus_infty, plus_infty, minus_infty),
    TEST_c_c (csqrt, minus_zero, minus_infty, plus_infty, minus_infty),
    TEST_c_c (csqrt, -4, minus_infty, plus_infty, minus_infty),
    TEST_c_c (csqrt, minus_infty, minus_infty, plus_infty, minus_infty),

    TEST_c_c (csqrt, minus_infty, qnan_value, qnan_value, plus_infty, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (csqrt, plus_infty, qnan_value, plus_infty, qnan_value),

    TEST_c_c (csqrt, 0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, 1, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, minus_zero, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, -1, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csqrt, qnan_value, 0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, qnan_value, 8, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, qnan_value, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (csqrt, qnan_value, -8, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (csqrt, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (csqrt),
  };

static void
csqrt_test (void)
{
  ALL_RM_TEST (csqrt, 0, csqrt_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}

static const struct test_c_c_data ctan_test_data[] =
  {
    TEST_c_c (ctan, 0, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, 1, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, 2, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, 3, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, 4, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, min_value, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, min_subnorm_value, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, 0x1p127, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, 0x1.11p127, plus_infty, minus_zero, 1.0),
#if !TEST_COND_binary32
    TEST_c_c (ctan, 0x1p1023, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, 0x1.5p1023, plus_infty, 0.0, 1.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctan, 0x1p16383L, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, 0x1.2p16383L, plus_infty, minus_zero, 1.0),
#endif
    TEST_c_c (ctan, minus_zero, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -1, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -2, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, -3, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, -4, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -min_value, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -min_subnorm_value, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -0x1p127, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -0x1.11p127, plus_infty, 0.0, 1.0),
#if !TEST_COND_binary32
    TEST_c_c (ctan, -0x1p1023, plus_infty, 0.0, 1.0),
    TEST_c_c (ctan, -0x1.5p1023, plus_infty, minus_zero, 1.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctan, -0x1p16383L, plus_infty, minus_zero, 1.0),
    TEST_c_c (ctan, -0x1.2p16383L, plus_infty, 0.0, 1.0),
#endif

    TEST_c_c (ctan, 0, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, 1, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, 2, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, 3, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, 4, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, min_value, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, min_subnorm_value, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, 0x1p127, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, 0x1.11p127, minus_infty, minus_zero, -1.0),
#if !TEST_COND_binary32
    TEST_c_c (ctan, 0x1p1023, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, 0x1.5p1023, minus_infty, 0.0, -1.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctan, 0x1p16383L, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, 0x1.2p16383L, minus_infty, minus_zero, -1.0),
#endif
    TEST_c_c (ctan, minus_zero, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -1, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -2, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, -3, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, -4, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -min_value, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -min_subnorm_value, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -0x1p127, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -0x1.11p127, minus_infty, 0.0, -1.0),
#if !TEST_COND_binary32
    TEST_c_c (ctan, -0x1p1023, minus_infty, 0.0, -1.0),
    TEST_c_c (ctan, -0x1.5p1023, minus_infty, minus_zero, -1.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctan, -0x1p16383L, minus_infty, minus_zero, -1.0),
    TEST_c_c (ctan, -0x1.2p16383L, minus_infty, 0.0, -1.0),
#endif

    TEST_c_c (ctan, plus_infty, 0, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, plus_infty, 2, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, minus_infty, 0, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, minus_infty, 2, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, plus_infty, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, plus_infty, -2, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, minus_infty, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctan, minus_infty, -2, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ctan, qnan_value, plus_infty, 0.0, 1.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctan, plus_infty, plus_infty, 0.0, 1.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctan, minus_infty, plus_infty, 0.0, 1.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctan, qnan_value, minus_infty, 0.0, -1.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctan, plus_infty, minus_infty, 0.0, -1.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctan, minus_infty, minus_infty, 0.0, -1.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ctan, 0, qnan_value, 0.0, qnan_value),
    TEST_c_c (ctan, minus_zero, qnan_value, minus_zero, qnan_value),

    TEST_c_c (ctan, 0.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctan, -4.5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ctan, qnan_value, 0, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctan, qnan_value, 5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctan, qnan_value, minus_zero, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctan, qnan_value, -0.25, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ctan, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (ctan),
  };

static void
ctan_test (void)
{
  ALL_RM_TEST (ctan, 0, ctan_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_c_c_data ctanh_test_data[] =
  {
    TEST_c_c (ctanh, plus_infty, 0, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, 1, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, 2, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, 3, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, 4, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, min_value, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, min_subnorm_value, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, 0x1p127, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, 0x1.11p127, 1.0, minus_zero),
#if !TEST_COND_binary32
    TEST_c_c (ctanh, plus_infty, 0x1p1023, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, 0x1.5p1023, 1.0, 0.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctanh, plus_infty, 0x1p16383L, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, 0x1.2p16383L, 1.0, minus_zero),
#endif
    TEST_c_c (ctanh, plus_infty, minus_zero, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -1, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -2, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, -3, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, -4, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -min_value, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -min_subnorm_value, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -0x1p127, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -0x1.11p127, 1.0, 0.0),
#if !TEST_COND_binary32
    TEST_c_c (ctanh, plus_infty, -0x1p1023, 1.0, 0.0),
    TEST_c_c (ctanh, plus_infty, -0x1.5p1023, 1.0, minus_zero),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctanh, plus_infty, -0x1p16383L, 1.0, minus_zero),
    TEST_c_c (ctanh, plus_infty, -0x1.2p16383L, 1.0, 0.0),
#endif
    TEST_c_c (ctanh, minus_infty, 0, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, 1, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, 2, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, 3, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, 4, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, min_value, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, min_subnorm_value, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, 0x1p127, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, 0x1.11p127, -1.0, minus_zero),
#if !TEST_COND_binary32
    TEST_c_c (ctanh, minus_infty, 0x1p1023, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, 0x1.5p1023, -1.0, 0.0),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctanh, minus_infty, 0x1p16383L, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, 0x1.2p16383L, -1.0, minus_zero),
#endif
    TEST_c_c (ctanh, minus_infty, minus_zero, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -1, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -2, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, -3, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, -4, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -min_value, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -min_subnorm_value, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -0x1p127, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -0x1.11p127, -1.0, 0.0),
#if !TEST_COND_binary32
    TEST_c_c (ctanh, minus_infty, -0x1p1023, -1.0, 0.0),
    TEST_c_c (ctanh, minus_infty, -0x1.5p1023, -1.0, minus_zero),
#endif
#if MAX_EXP >= 16384
    TEST_c_c (ctanh, minus_infty, -0x1p16383L, -1.0, minus_zero),
    TEST_c_c (ctanh, minus_infty, -0x1.2p16383L, -1.0, 0.0),
#endif

    TEST_c_c (ctanh, 0, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, 2, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, 0, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, 2, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, minus_zero, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, -2, plus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, minus_zero, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),
    TEST_c_c (ctanh, -2, minus_infty, qnan_value, qnan_value, INVALID_EXCEPTION),

    TEST_c_c (ctanh, plus_infty, qnan_value, 1.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctanh, plus_infty, plus_infty, 1.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctanh, plus_infty, minus_infty, 1.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctanh, minus_infty, qnan_value, -1.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctanh, minus_infty, plus_infty, -1.0, 0.0, IGNORE_ZERO_INF_SIGN),
    TEST_c_c (ctanh, minus_infty, minus_infty, -1.0, 0.0, IGNORE_ZERO_INF_SIGN),

    TEST_c_c (ctanh, qnan_value, 0, qnan_value, 0.0),
    TEST_c_c (ctanh, qnan_value, minus_zero, qnan_value, minus_zero),

    TEST_c_c (ctanh, qnan_value, 0.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctanh, qnan_value, -4.5, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ctanh, 0, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctanh, 5, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctanh, minus_zero, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),
    TEST_c_c (ctanh, -0.25, qnan_value, qnan_value, qnan_value, INVALID_EXCEPTION_OK),

    TEST_c_c (ctanh, qnan_value, qnan_value, qnan_value, qnan_value),

    AUTO_TESTS_c_c (ctanh),
  };

static void
ctanh_test (void)
{
  ALL_RM_TEST (ctanh, 0, ctanh_test_data, RUN_TEST_LOOP_c_c, END_COMPLEX);
}


static const struct test_f_f_data erf_test_data[] =
  {
    TEST_f_f (erf, plus_infty, 1, ERRNO_UNCHANGED),
    TEST_f_f (erf, minus_infty, -1, ERRNO_UNCHANGED),
    TEST_f_f (erf, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (erf, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (erf, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (erf, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (erf),
  };

static void
erf_test (void)
{
  ALL_RM_TEST (erf, 0, erf_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data erfc_test_data[] =
  {
    TEST_f_f (erfc, plus_infty, 0.0, ERRNO_UNCHANGED),
    TEST_f_f (erfc, minus_infty, 2.0, ERRNO_UNCHANGED),
    TEST_f_f (erfc, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (erfc, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    AUTO_TESTS_f_f (erfc),
  };

static void
erfc_test (void)
{
  ALL_RM_TEST (erfc, 0, erfc_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data exp_test_data[] =
  {
    TEST_f_f (exp, plus_infty, plus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (exp, minus_infty, 0, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (exp, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (exp, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (exp),
  };

static void
exp_test (void)
{
  ALL_RM_TEST (exp, 0, exp_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data exp10_test_data[] =
  {
    TEST_f_f (exp10, plus_infty, plus_infty, ERRNO_UNCHANGED),
    TEST_f_f (exp10, minus_infty, 0, ERRNO_UNCHANGED),
    TEST_f_f (exp10, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp10, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp10, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (exp10, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (exp10),
  };

static void
exp10_test (void)
{
  ALL_RM_TEST (exp10, 0, exp10_test_data, RUN_TEST_LOOP_f_f, END);
}

static void
pow10_test (void)
{
  /* pow10 uses the same test data as exp10.  */
  ALL_RM_TEST (pow10, 0, exp10_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data exp2_test_data[] =
  {
    TEST_f_f (exp2, plus_infty, plus_infty, ERRNO_UNCHANGED),
    TEST_f_f (exp2, minus_infty, 0, ERRNO_UNCHANGED),
    TEST_f_f (exp2, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp2, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (exp2, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (exp2, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (exp2),
  };

static void
exp2_test (void)
{
  ALL_RM_TEST (exp2, 0, exp2_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data expm1_test_data[] =
  {
    TEST_f_f (expm1, plus_infty, plus_infty, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (expm1, minus_infty, -1, ERRNO_UNCHANGED|NO_TEST_INLINE),
    TEST_f_f (expm1, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (expm1, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (expm1, snan_value, qnan_value, INVALID_EXCEPTION),
    TEST_f_f (expm1, -snan_value, qnan_value, INVALID_EXCEPTION),

    AUTO_TESTS_f_f (expm1),
  };

static void
expm1_test (void)
{
  ALL_RM_TEST (expm1, 0, expm1_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_f_f_data fabs_test_data[] =
  {
    TEST_f_f (fabs, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, minus_zero, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (fabs, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, minus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_f_f (fabs, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_f_f (fabs, snan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),
    TEST_f_f (fabs, -snan_value, snan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|TEST_NAN_SIGN),

    TEST_f_f (fabs, min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, -min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, -min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, -max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, 38.0, 38.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (fabs, -lit_e, lit_e, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
  };

static void
fabs_test (void)
{
  ALL_RM_TEST (fabs, 1, fabs_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_ff_f_data fdim_test_data[] =
  {
    TEST_ff_f (fdim, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 9, 0, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 0, 9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -9, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 0, -9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if TEST_COND_binary64
    TEST_ff_f (fdim, 1.0, -0x1.0000000000001p-53, 1.0, INEXACT_EXCEPTION|ERRNO_UNCHANGED, 0x1.0000000000001p0, INEXACT_EXCEPTION|ERRNO_UNCHANGED, 1.0, INEXACT_EXCEPTION|ERRNO_UNCHANGED, 0x1.0000000000001p0, INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
    TEST_ff_f (fdim, min_subnorm_value, min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -min_subnorm_value, -min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, min_subnorm_value, -min_subnorm_value, 2 * min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, min_value, min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -min_value, -min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, max_value, max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -max_value, -max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -max_value, max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, max_value, -max_value, plus_oflow, INEXACT_EXCEPTION|OVERFLOW_EXCEPTION|ERRNO_PLUS_OFLOW|XFAIL_ROUNDING_IBM128_LIBGCC),

    TEST_ff_f (fdim, plus_infty, 9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, plus_infty, -9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, 9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, -9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 9, minus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -9, minus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 9, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -9, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fdim, 0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 9, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -9, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 9, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -9, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, 9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, 9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, qnan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, qnan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, plus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, plus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fdim, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_ff_f (fdim, plus_infty, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, plus_infty, minus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fdim, minus_infty, minus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
  };

static void
fdim_test (void)
{
  ALL_RM_TEST (fdim, 1, fdim_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_f_f_data floor_test_data[] =
  {
    TEST_f_f (floor, 0.0, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_f_f (floor, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_f_f (floor, lit_pi, 3.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -lit_pi, -4.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, min_subnorm_value, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, min_value, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0.1, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0.25, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0.625, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 1, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 2, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p23, 0x1p23, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p24, 0x1p24, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p25, 0x1p25, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p52, 0x1p52, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p53, 0x1p53, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p54, 0x1p54, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p63, 0x1p63, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p64, 0x1p64, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p65, 0x1p65, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p105, 0x1p105, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p106, 0x1p106, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p107, 0x1p107, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p112, 0x1p112, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p113, 0x1p113, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 0x1p114, 0x1p114, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -min_subnorm_value, -1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -min_value, -1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0.1, -1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0.25, -1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0.625, -1.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p23, -0x1p23, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p24, -0x1p24, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p25, -0x1p25, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p52, -0x1p52, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p53, -0x1p53, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p54, -0x1p54, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p63, -0x1p63, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p64, -0x1p64, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p65, -0x1p65, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p105, -0x1p105, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p106, -0x1p106, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p107, -0x1p107, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p112, -0x1p112, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p113, -0x1p113, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x1p114, -0x1p114, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

#if MANT_DIG >= 64
    /* The result can only be represented in long double.  */
    TEST_f_f (floor, 4503599627370495.5L, 4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370496.25L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370496.5L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370496.75L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370497.5L, 4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# if MANT_DIG > 100
    TEST_f_f (floor, 4503599627370494.5000000000001L, 4503599627370494.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370495.5000000000001L, 4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 4503599627370496.5000000000001L, 4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (floor, -4503599627370495.5L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370496.25L, -4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370496.5L, -4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370496.75L, -4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370497.5L, -4503599627370498.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# if MANT_DIG > 100
    TEST_f_f (floor, -4503599627370494.5000000000001L, -4503599627370495.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370495.5000000000001L, -4503599627370496.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -4503599627370496.5000000000001L, -4503599627370497.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (floor, 9007199254740991.5L, 9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740992.25L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740992.5L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740992.75L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740993.5L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

# if MANT_DIG > 100
    TEST_f_f (floor, 9007199254740991.0000000000001L, 9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740992.0000000000001L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740993.0000000000001L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740991.5000000000001L, 9007199254740991.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740992.5000000000001L, 9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 9007199254740993.5000000000001L, 9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (floor, -9007199254740991.5L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740992.25L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740992.5L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740992.75L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740993.5L, -9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

# if MANT_DIG > 100
    TEST_f_f (floor, -9007199254740991.0000000000001L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740992.0000000000001L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740993.0000000000001L, -9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740991.5000000000001L, -9007199254740992.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740992.5000000000001L, -9007199254740993.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -9007199254740993.5000000000001L, -9007199254740994.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif

    TEST_f_f (floor, 72057594037927935.5L, 72057594037927935.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 72057594037927936.25L, 72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 72057594037927936.5L, 72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 72057594037927936.75L, 72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 72057594037927937.5L, 72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (floor, -72057594037927935.5L, -72057594037927936.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -72057594037927936.25L, -72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -72057594037927936.5L, -72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -72057594037927936.75L, -72057594037927937.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -72057594037927937.5L, -72057594037927938.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (floor, 10141204801825835211973625643007.5L, 10141204801825835211973625643007.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 10141204801825835211973625643008.25L, 10141204801825835211973625643008.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 10141204801825835211973625643008.5L, 10141204801825835211973625643008.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 10141204801825835211973625643008.75L, 10141204801825835211973625643008.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, 10141204801825835211973625643009.5L, 10141204801825835211973625643009.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_f_f (floor, 0xf.ffffffffffffff8p+47L, 0xf.fffffffffffep+47L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_f (floor, -0x8.000000000000004p+48L, -0x8.000000000001p+48L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
  };

static void
floor_test (void)
{
  ALL_RM_TEST (floor, 1, floor_test_data, RUN_TEST_LOOP_f_f, END);
}


static const struct test_fff_f_data fma_test_data[] =
  {
    TEST_fff_f (fma, qnan_value, 2.0, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, 2.0, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, snan_value, 2.0, 3.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, -snan_value, 2.0, 3.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 1.0, qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, -qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, snan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 1.0, -snan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 1.0, 2.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, 2.0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, 2.0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 1.0, 2.0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, max_value, max_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, max_value, max_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, min_value, min_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, min_value, min_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, min_subnorm_value, min_subnorm_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, min_subnorm_value, min_subnorm_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, max_value, max_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, max_value, max_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, min_value, min_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, min_value, min_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, min_subnorm_value, min_subnorm_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, min_subnorm_value, min_subnorm_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, qnan_value, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, qnan_value, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, qnan_value, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, qnan_value, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, snan_value, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, qnan_value, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, qnan_value, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, snan_value, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 1.0, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 1.0, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, qnan_value, 2.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, qnan_value, 2.0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, 2.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, 2.0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, snan_value, 2.0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, qnan_value, qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, qnan_value, -qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -qnan_value, -qnan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, snan_value, snan_value, 3.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, plus_infty, 0.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, plus_infty, 0.0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, minus_infty, 0.0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, minus_infty, 0.0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, plus_infty, 0.0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, plus_infty, 0.0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, 0.0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, 0.0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, plus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, 0.0, plus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, 0.0, minus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, 0.0, minus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION_OK),
    TEST_fff_f (fma, 0.0, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    /* Bug 6801: errno setting may be missing.  */
    TEST_fff_f (fma, plus_infty, 0.0, 1.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, 0.0, 1.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, plus_infty, 1.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, 0.0, minus_infty, 1.0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_fff_f (fma, plus_infty, plus_infty, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, plus_infty, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, plus_infty, minus_infty, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, minus_infty, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, plus_infty, 3.5L, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, -7.5L, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, -13.5L, plus_infty, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fff_f (fma, minus_infty, 7.5L, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_fff_f (fma, -max_value, -max_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|XFAIL_ROUNDING_IBM128_LIBGCC),
    TEST_fff_f (fma, max_value / 2, max_value / 2, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, -max_value, max_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|XFAIL_ROUNDING_IBM128_LIBGCC),
    TEST_fff_f (fma, max_value / 2, -max_value / 4, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, plus_infty, 4, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, 2, minus_infty, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, plus_infty, plus_infty, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, minus_infty, minus_infty, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, plus_infty, minus_infty, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fff_f (fma, minus_infty, plus_infty, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    AUTO_TESTS_fff_f (fma),
  };

static void
fma_test (void)
{
  ALL_RM_TEST (fma, 1, fma_test_data, RUN_TEST_LOOP_fff_f, END);
}


static const struct test_ff_f_data fmax_test_data[] =
  {
    TEST_ff_f (fmax, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, minus_zero, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmax, minus_zero, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmax, min_subnorm_value, -min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -min_subnorm_value, min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, min_value, -min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -min_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, max_value, -max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -max_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 9, 0, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -9, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, -9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmax, plus_infty, 9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -9, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, plus_infty, -9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmax, minus_infty, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_infty, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 9, minus_infty, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -9, minus_infty, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmax, 0, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, -qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_zero, qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_zero, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 9, qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 9, -qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -9, qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -9, -qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, minus_zero, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, minus_zero, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, 9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, 9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, plus_infty, qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, plus_infty, -qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_infty, qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, minus_infty, -qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmax, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmax, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
  };

static void
fmax_test (void)
{
  ALL_RM_TEST (fmax, 1, fmax_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_ff_f_data fmaxmag_test_data[] =
  {
    TEST_ff_f (fmaxmag, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, minus_zero, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmaxmag, minus_zero, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmaxmag, min_subnorm_value, min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, min_subnorm_value, -min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_subnorm_value, min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_subnorm_value, -min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, min_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, min_value, -min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, max_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, max_value, -max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -max_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -max_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, max_value, min_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, max_value, -min_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -max_value, min_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -max_value, -min_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, min_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, min_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -min_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 9, 0, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -9, 0, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmaxmag, plus_infty, 9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -9, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, plus_infty, -9, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmaxmag, minus_infty, 9, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_infty, -9, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 9, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -9, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmaxmag, 0, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, -qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_zero, qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_zero, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 9, qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 9, -qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -9, qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -9, -qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, minus_zero, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, minus_zero, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, 9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, 9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, plus_infty, qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, plus_infty, -qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_infty, qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, minus_infty, -qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmaxmag, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmaxmag, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
  };

static void
fmaxmag_test (void)
{
  ALL_RM_TEST (fmaxmag, 1, fmaxmag_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_ff_f_data fmin_test_data[] =
  {
    TEST_ff_f (fmin, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 0, minus_zero, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmin, minus_zero, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fmin, min_subnorm_value, -min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -min_subnorm_value, min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, min_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -min_value, min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, max_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -max_value, max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 9, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 0, 9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -9, 0, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 0, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmin, plus_infty, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 9, plus_infty, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, plus_infty, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -9, plus_infty, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_infty, 9, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_infty, -9, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 9, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -9, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmin, 0, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 0, -qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_zero, qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_zero, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 9, qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 9, -qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -9, qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -9, -qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, minus_zero, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, minus_zero, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, 9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, 9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, plus_infty, qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, plus_infty, -qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_infty, qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, minus_infty, -qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmin, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmin, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
  };

static void
fmin_test (void)
{
  ALL_RM_TEST (fmin, 1, fmin_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_ff_f_data fminmag_test_data[] =
  {
    TEST_ff_f (fminmag, 0, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_zero, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 0, minus_zero, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fminmag, minus_zero, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED|IGNORE_ZERO_INF_SIGN),
    TEST_ff_f (fminmag, min_subnorm_value, min_subnorm_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, min_subnorm_value, -min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_subnorm_value, min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_subnorm_value, -min_subnorm_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, min_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, min_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_value, min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, max_value, max_value, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, max_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -max_value, max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -max_value, -max_value, -max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, max_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, max_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -max_value, min_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -max_value, -min_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, min_value, max_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, min_value, -max_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_value, max_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -min_value, -max_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 9, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 0, 9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -9, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 0, -9, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fminmag, plus_infty, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 9, plus_infty, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, plus_infty, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -9, plus_infty, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_infty, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_infty, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 9, minus_infty, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -9, minus_infty, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fminmag, 0, qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 0, -qnan_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_zero, qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_zero, -qnan_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 9, qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 9, -qnan_value, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -9, qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -9, -qnan_value, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, minus_zero, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, minus_zero, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, 9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, 9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -9, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -9, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, 0, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, minus_zero, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, 9, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, -9, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, 9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, -9, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, plus_infty, qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, plus_infty, -qnan_value, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_infty, qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, minus_infty, -qnan_value, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, plus_infty, plus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, minus_infty, minus_infty, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fminmag, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fminmag, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
  };

static void
fminmag_test (void)
{
  ALL_RM_TEST (fminmag, 1, fminmag_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_ff_f_data fmod_test_data[] =
  {
    /* fmod (+0, y) == +0 for y != 0.  */
    TEST_ff_f (fmod, 0, 3, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, -min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, -min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, -max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    /* fmod (-0, y) == -0 for y != 0.  */
    TEST_ff_f (fmod, minus_zero, 3, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, -min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, -min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, max_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, -max_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    /* fmod (+inf, y) == qNaN plus invalid exception.  */
    TEST_ff_f (fmod, plus_infty, 3, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, -1.1L, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, min_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, max_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, plus_infty, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    /* fmod (-inf, y) == qNaN plus invalid exception.  */
    TEST_ff_f (fmod, minus_infty, 3, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, -1.1L, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, min_subnorm_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, min_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, max_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_infty, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    /* fmod (x, +0) == qNaN plus invalid exception.  */
    TEST_ff_f (fmod, 3, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, -1.1L, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, 0, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_zero, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, min_subnorm_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, min_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, max_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    /* fmod (x, -0) == qNaN plus invalid exception.  */
    TEST_ff_f (fmod, 3, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, -1.1L, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, 0, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, minus_zero, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, min_subnorm_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, min_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_ff_f (fmod, max_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),

    /* fmod (x, +inf) == x for x not infinite.  */
    TEST_ff_f (fmod, 0, plus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, plus_infty, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, plus_infty, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, plus_infty, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, plus_infty, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 3.0, plus_infty, 3.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    /* fmod (x, -inf) == x for x not infinite.  */
    TEST_ff_f (fmod, 0, minus_infty, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, minus_infty, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, minus_infty, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, minus_infty, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, minus_infty, max_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 3.0, minus_infty, 3.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmod, 0, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_zero, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 1, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 1, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, plus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, plus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_infty, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, minus_infty, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, 0, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, minus_zero, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, minus_zero, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, 1, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, 1, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, plus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, plus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, minus_infty, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, minus_infty, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, 0, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, minus_zero, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, 1, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, plus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, minus_infty, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -qnan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -qnan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -qnan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, -qnan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_ff_f (fmod, -snan_value, -snan_value, qnan_value, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_ff_f (fmod, 6.5, 2.25L, 2.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -6.5, 2.25L, -2.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 6.5, -2.25L, 2.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -6.5, -2.25L, -2.0L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmod, max_value, max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, -max_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, -min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, max_value, -min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, max_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, -max_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, -min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -max_value, -min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmod, min_value, max_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, -max_value, min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, -min_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_value, -min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, max_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, -max_value, -min_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, -min_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_value, -min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmod, min_subnorm_value, max_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, -max_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, min_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, -min_value, min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, min_subnorm_value, -min_subnorm_value, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, max_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, -max_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, min_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, -min_value, -min_subnorm_value, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -min_subnorm_value, -min_subnorm_value, minus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_ff_f (fmod, 0x1p127L, 0x3p-149L, 0x1p-149L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p127L, -0x3p-149L, 0x1p-149L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p127L, 0x3p-148L, 0x1p-147L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p127L, -0x3p-148L, 0x1p-147L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p127L, 0x3p-126L, 0x1p-125L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p127L, -0x3p-126L, 0x1p-125L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, 0x3p-149L, -0x1p-149L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, -0x3p-149L, -0x1p-149L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, 0x3p-148L, -0x1p-147L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, -0x3p-148L, -0x1p-147L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, 0x3p-126L, -0x1p-125L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p127L, -0x3p-126L, -0x1p-125L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if !TEST_COND_binary32
    TEST_ff_f (fmod, 0x1p1023L, 0x3p-1074L, 0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p1023L, -0x3p-1074L, 0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p1023L, 0x3p-1073L, 0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p1023L, -0x3p-1073L, 0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p1023L, 0x3p-1022L, 0x1p-1021L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p1023L, -0x3p-1022L, 0x1p-1021L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, 0x3p-1074L, -0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, -0x3p-1074L, -0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, 0x3p-1073L, -0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, -0x3p-1073L, -0x1p-1073L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, 0x3p-1022L, -0x1p-1021L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p1023L, -0x3p-1022L, -0x1p-1021L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
#if MIN_EXP <= -16381
    TEST_ff_f (fmod, 0x1p16383L, 0x3p-16445L, 0x1p-16445L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, -0x3p-16445L, 0x1p-16445L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, 0x3p-16444L, 0x1p-16443L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, -0x3p-16444L, 0x1p-16443L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, 0x3p-16382L, 0x1p-16381L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, -0x3p-16382L, 0x1p-16381L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, 0x3p-16445L, -0x1p-16445L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, -0x3p-16445L, -0x1p-16445L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, 0x3p-16444L, -0x1p-16443L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, -0x3p-16444L, -0x1p-16443L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, 0x3p-16382L, -0x1p-16381L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, -0x3p-16382L, -0x1p-16381L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# if MANT_DIG >= 113
    TEST_ff_f (fmod, 0x1p16383L, 0x3p-16494L, 0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, -0x3p-16494L, 0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, 0x3p-16493L, 0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1p16383L, -0x3p-16493L, 0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, 0x3p-16494L, -0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, -0x3p-16494L, -0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, 0x3p-16493L, -0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1p16383L, -0x3p-16493L, -0x1p-16493L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif
#endif

    TEST_ff_f (fmod, 0x0.fffffep-126L, 0x1p-149L, plus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if !TEST_COND_binary32
    TEST_ff_f (fmod, 0x0.fffffffffffffp-1022L, 0x1p-1074L, plus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
#if MIN_EXP <= -16381
    TEST_ff_f (fmod, 0x0.fffffffffffffffep-16382L, 0x1p-16445L, plus_zero, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
#if MANT_DIG >= 56
    TEST_ff_f (fmod, -0x1.00000000000004p+0L, 0x1.fffffffffffff8p-1L, -0x1p-53L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1.fffffffffffffap-1L, 0x1.fffffffffffff8p-1L, 0x1p-56L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1.fffffffffffffap-1L, 0x1.fffffffffffff8p-1L, -0x1p-56L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, 0x1.fffffffffffffap-1L, -0x1.fffffffffffff8p-1L, 0x1p-56L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_ff_f (fmod, -0x1.fffffffffffffap-1L, -0x1.fffffffffffff8p-1L, -0x1p-56L, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
  };

static void
fmod_test (void)
{
  ALL_RM_TEST (fmod, 1, fmod_test_data, RUN_TEST_LOOP_ff_f, END);
}


static const struct test_f_i_data fpclassify_test_data[] =
  {
    TEST_f_i (fpclassify, qnan_value, FP_NAN, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, -qnan_value, FP_NAN, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, snan_value, FP_NAN, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, -snan_value, FP_NAN, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, plus_infty, FP_INFINITE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, minus_infty, FP_INFINITE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, plus_zero, FP_ZERO, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, minus_zero, FP_ZERO, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, 1000, FP_NORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, max_value, FP_NORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, -max_value, FP_NORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, min_value, FP_NORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, -min_value, FP_NORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, min_subnorm_value, FP_SUBNORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_f_i (fpclassify, -min_subnorm_value, FP_SUBNORMAL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
  };

static void
fpclassify_test (void)
{
  ALL_RM_TEST (fpclassify, 1, fpclassify_test_data, RUN_TEST_LOOP_f_i_tg, END);
}


static const struct test_f_f1_data frexp_test_data[] =
  {
    TEST_fI_f1 (frexp, plus_infty, plus_infty, IGNORE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, minus_infty, minus_infty, IGNORE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, qnan_value, qnan_value, IGNORE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -qnan_value, qnan_value, IGNORE, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, snan_value, qnan_value, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),
    TEST_fI_f1 (frexp, -snan_value, qnan_value, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION),

    TEST_fI_f1 (frexp, 0.0, 0.0, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, minus_zero, minus_zero, 0.0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    TEST_fI_f1 (frexp, 12.8L, 0.8L, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -27.34L, -0.854375L, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0.25L, 0.5L, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0.5L, 0.5L, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 1.0L, 0.5L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -1.0L, -0.5L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 2.0L, 0.5L, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 4.0L, 0.5L, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p127L, 0.5L, 128, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p127L, -0.5L, 128, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-126L, 0.5L, -125, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-127L, 0.5L, -126, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-149L, 0.5L, -148, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p-149L, -0.5L, -148, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if !TEST_COND_binary32
    TEST_fI_f1 (frexp, 0x1p1023L, 0.5L, 1024, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p1023L, -0.5L, 1024, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-1022L, 0.5L, -1021, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-1023L, 0.5L, -1022, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-1074L, 0.5L, -1073, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p-1074L, -0.5L, -1073, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
#if MIN_EXP <= -16381
    TEST_fI_f1 (frexp, 0x1p16383L, 0.5L, 16384, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p16383L, -0.5L, 16384, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-16382L, 0.5L, -16381, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-16383L, 0.5L, -16382, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 0x1p-16445L, 0.5L, -16444, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p-16445L, -0.5L, -16444, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# if MANT_DIG >= 113
    TEST_fI_f1 (frexp, 0x1p-16494L, 0.5L, -16493, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -0x1p-16494L, -0.5L, -16493, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
# endif
#endif

#if MANT_DIG >= 106
    TEST_fI_f1 (frexp, 1.0L - 0x1p-106L, 1.0L - 0x1p-106L, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 1.0L, 0.5L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, 1.0L + 0x1p-105L, 0.5L + 0x1p-106L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -1.0L + 0x1p-106L, -1.0L + 0x1p-106L, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -1.0L, -0.5L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fI_f1 (frexp, -1.0L - 0x1p-105L, -0.5L - 0x1p-106L, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
  };

static void
frexp_test (void)
{
  int x;

  ALL_RM_TEST (frexp, 1, frexp_test_data, RUN_TEST_LOOP_fI_f1, END, x);
}


static const struct test_fiu_M_data fromfp_test_data[] =
  {
    /* Infinity, NaN and values that are definitely out of range
       (large values of width count as the width of the return
       type).  */
    TEST_fiu_M (fromfp, plus_infty, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_infty, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_infty, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, qnan_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -qnan_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, snan_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -snan_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, max_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -max_value, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1p64, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1p64, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.000002p63, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
#if MANT_DIG >= 53
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1.0000000000001p63, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
#endif
#if MANT_DIG >= 64
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_UPWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_UPWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_DOWNWARD, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_DOWNWARD, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TOWARDZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TOWARDZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEARESTFROMZERO, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEARESTFROMZERO, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEAREST, 64, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000001p0, FP_INT_TONEAREST, UINT_MAX, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
#endif

    /* Zero (OK unless the width specified is zero).  */
    TEST_fiu_M (fromfp, plus_zero, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_UPWARD, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_UPWARD, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_UPWARD, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_DOWNWARD, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_DOWNWARD, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_DOWNWARD, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TOWARDZERO, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TOWARDZERO, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TOWARDZERO, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEARESTFROMZERO, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEARESTFROMZERO, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEAREST, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEAREST, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, plus_zero, FP_INT_TONEAREST, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_UPWARD, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_UPWARD, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_UPWARD, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_DOWNWARD, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_DOWNWARD, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_DOWNWARD, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TOWARDZERO, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TOWARDZERO, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TOWARDZERO, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEARESTFROMZERO, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEARESTFROMZERO, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEAREST, 1, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEAREST, 64, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, minus_zero, FP_INT_TONEAREST, UINT_MAX, 0, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),

    /* Tests below here are generated with gen-fromfp-tests.py.  */

    /* Integer inputs within range of uintmax_t.  */
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, 2, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, 3, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, 64, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_UPWARD, UINT_MAX, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, 2, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, 3, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, 64, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_DOWNWARD, UINT_MAX, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, 2, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, 3, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, 64, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TOWARDZERO, UINT_MAX, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, 2, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, 3, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, 64, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEARESTFROMZERO, UINT_MAX, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, 2, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, 3, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, 64, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 1, FP_INT_TONEAREST, UINT_MAX, 1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -1, FP_INT_UPWARD, 1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_UPWARD, 2, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_UPWARD, 64, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_UPWARD, UINT_MAX, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -1, FP_INT_DOWNWARD, 1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_DOWNWARD, 2, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_DOWNWARD, 64, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_DOWNWARD, UINT_MAX, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -1, FP_INT_TOWARDZERO, 1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TOWARDZERO, 2, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TOWARDZERO, 64, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TOWARDZERO, UINT_MAX, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEARESTFROMZERO, 1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEARESTFROMZERO, 2, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEARESTFROMZERO, 64, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEARESTFROMZERO, UINT_MAX, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEAREST, 1, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEAREST, 2, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEAREST, 64, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -1, FP_INT_TONEAREST, UINT_MAX, -1, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 3, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 4, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, 64, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_UPWARD, UINT_MAX, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 3, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 4, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, 64, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_DOWNWARD, UINT_MAX, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 3, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 4, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, 64, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TOWARDZERO, UINT_MAX, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 3, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 4, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, 64, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEARESTFROMZERO, UINT_MAX, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 3, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 4, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, 64, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 2, FP_INT_TONEAREST, UINT_MAX, 2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, 2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, 3, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, 64, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_UPWARD, UINT_MAX, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, 2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, 3, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, 64, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_DOWNWARD, UINT_MAX, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, 2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, 3, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, 64, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TOWARDZERO, UINT_MAX, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, 2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, 3, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, 64, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEARESTFROMZERO, UINT_MAX, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, 2, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, 3, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, 64, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -2, FP_INT_TONEAREST, UINT_MAX, -2, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 3, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 4, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, 64, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_UPWARD, UINT_MAX, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 3, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 4, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, 64, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_DOWNWARD, UINT_MAX, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 3, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 4, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, 64, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TOWARDZERO, UINT_MAX, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 3, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 4, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, 64, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEARESTFROMZERO, UINT_MAX, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 3, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 4, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, 64, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 3, FP_INT_TONEAREST, UINT_MAX, 3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 3, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 4, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, 64, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_UPWARD, UINT_MAX, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 3, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 4, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, 64, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_DOWNWARD, UINT_MAX, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 3, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 4, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, 64, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TOWARDZERO, UINT_MAX, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 3, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 4, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, 64, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEARESTFROMZERO, UINT_MAX, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 3, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 4, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, 64, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -3, FP_INT_TONEAREST, UINT_MAX, -3, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 4, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 5, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, 64, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_UPWARD, UINT_MAX, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 4, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 5, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, 64, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_DOWNWARD, UINT_MAX, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 4, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 5, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, 64, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TOWARDZERO, UINT_MAX, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 4, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 5, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, 64, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEARESTFROMZERO, UINT_MAX, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 4, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 5, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, 64, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 4, FP_INT_TONEAREST, UINT_MAX, 4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 3, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 4, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, 64, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_UPWARD, UINT_MAX, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 3, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 4, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, 64, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_DOWNWARD, UINT_MAX, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 3, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 4, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, 64, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TOWARDZERO, UINT_MAX, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 3, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 4, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, 64, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEARESTFROMZERO, UINT_MAX, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 2, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 3, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 4, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, 64, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -4, FP_INT_TONEAREST, UINT_MAX, -4, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 4, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 5, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, 64, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_UPWARD, UINT_MAX, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 4, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 5, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, 64, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_DOWNWARD, UINT_MAX, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 4, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 5, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, 64, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TOWARDZERO, UINT_MAX, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 4, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 5, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, 64, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEARESTFROMZERO, UINT_MAX, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 4, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 5, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, 64, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 5, FP_INT_TONEAREST, UINT_MAX, 5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 4, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 5, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, 64, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_UPWARD, UINT_MAX, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 4, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 5, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, 64, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_DOWNWARD, UINT_MAX, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 4, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 5, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, 64, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TOWARDZERO, UINT_MAX, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 4, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 5, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, 64, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEARESTFROMZERO, UINT_MAX, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 4, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 5, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, 64, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -5, FP_INT_TONEAREST, UINT_MAX, -5, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 4, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 5, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, 64, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_UPWARD, UINT_MAX, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 4, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 5, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, 64, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_DOWNWARD, UINT_MAX, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 4, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 5, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, 64, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TOWARDZERO, UINT_MAX, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 4, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 5, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, 64, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEARESTFROMZERO, UINT_MAX, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 4, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 5, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, 64, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 6, FP_INT_TONEAREST, UINT_MAX, 6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 4, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 5, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, 64, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_UPWARD, UINT_MAX, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 4, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 5, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, 64, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_DOWNWARD, UINT_MAX, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 4, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 5, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, 64, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TOWARDZERO, UINT_MAX, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 4, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 5, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, 64, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEARESTFROMZERO, UINT_MAX, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 4, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 5, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, 64, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -6, FP_INT_TONEAREST, UINT_MAX, -6, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 4, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 5, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, 64, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_UPWARD, UINT_MAX, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 4, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 5, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, 64, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_DOWNWARD, UINT_MAX, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 4, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 5, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, 64, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TOWARDZERO, UINT_MAX, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 4, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 5, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, 64, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEARESTFROMZERO, UINT_MAX, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 4, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 5, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, 64, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 7, FP_INT_TONEAREST, UINT_MAX, 7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 4, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 5, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, 64, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_UPWARD, UINT_MAX, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 4, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 5, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, 64, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_DOWNWARD, UINT_MAX, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 4, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 5, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, 64, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TOWARDZERO, UINT_MAX, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 4, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 5, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, 64, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEARESTFROMZERO, UINT_MAX, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 4, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 5, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, 64, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -7, FP_INT_TONEAREST, UINT_MAX, -7, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 5, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 6, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, 64, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_UPWARD, UINT_MAX, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 5, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 6, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, 64, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_DOWNWARD, UINT_MAX, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 5, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 6, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, 64, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TOWARDZERO, UINT_MAX, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 5, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 6, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, 64, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEARESTFROMZERO, UINT_MAX, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 5, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 6, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, 64, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 8, FP_INT_TONEAREST, UINT_MAX, 8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 4, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 5, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, 64, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_UPWARD, UINT_MAX, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 4, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 5, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, 64, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_DOWNWARD, UINT_MAX, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 4, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 5, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, 64, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TOWARDZERO, UINT_MAX, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 4, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 5, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, 64, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEARESTFROMZERO, UINT_MAX, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 3, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 4, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 5, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, 64, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -8, FP_INT_TONEAREST, UINT_MAX, -8, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 5, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 6, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, 64, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_UPWARD, UINT_MAX, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 5, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 6, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, 64, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_DOWNWARD, UINT_MAX, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 5, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 6, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, 64, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TOWARDZERO, UINT_MAX, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 5, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 6, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, 64, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEARESTFROMZERO, UINT_MAX, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 5, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 6, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, 64, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 9, FP_INT_TONEAREST, UINT_MAX, 9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 5, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 6, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, 64, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_UPWARD, UINT_MAX, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 5, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 6, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, 64, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_DOWNWARD, UINT_MAX, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 5, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 6, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, 64, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TOWARDZERO, UINT_MAX, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 5, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 6, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, 64, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEARESTFROMZERO, UINT_MAX, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 5, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 6, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, 64, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -9, FP_INT_TONEAREST, UINT_MAX, -9, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 5, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 6, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, 64, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_UPWARD, UINT_MAX, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 5, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 6, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, 64, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_DOWNWARD, UINT_MAX, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 5, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 6, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, 64, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TOWARDZERO, UINT_MAX, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 5, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 6, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, 64, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEARESTFROMZERO, UINT_MAX, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 5, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 6, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, 64, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 15, FP_INT_TONEAREST, UINT_MAX, 15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 5, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 6, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, 64, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_UPWARD, UINT_MAX, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 5, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 6, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, 64, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_DOWNWARD, UINT_MAX, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 5, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 6, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, 64, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TOWARDZERO, UINT_MAX, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 5, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 6, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, 64, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEARESTFROMZERO, UINT_MAX, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 5, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 6, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, 64, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -15, FP_INT_TONEAREST, UINT_MAX, -15, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 6, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 7, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, 64, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_UPWARD, UINT_MAX, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 6, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 7, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, 64, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_DOWNWARD, UINT_MAX, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 6, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 7, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, 64, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TOWARDZERO, UINT_MAX, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 6, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 7, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, 64, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEARESTFROMZERO, UINT_MAX, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 6, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 7, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, 64, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 16, FP_INT_TONEAREST, UINT_MAX, 16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 5, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 6, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, 64, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_UPWARD, UINT_MAX, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 5, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 6, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, 64, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_DOWNWARD, UINT_MAX, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 5, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 6, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, 64, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TOWARDZERO, UINT_MAX, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 5, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 6, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, 64, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEARESTFROMZERO, UINT_MAX, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 4, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 5, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 6, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, 64, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -16, FP_INT_TONEAREST, UINT_MAX, -16, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 6, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 7, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, 64, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_UPWARD, UINT_MAX, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 6, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 7, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, 64, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_DOWNWARD, UINT_MAX, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 6, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 7, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, 64, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TOWARDZERO, UINT_MAX, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 6, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 7, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, 64, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEARESTFROMZERO, UINT_MAX, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 6, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 7, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, 64, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 17, FP_INT_TONEAREST, UINT_MAX, 17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 6, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 7, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, 64, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_UPWARD, UINT_MAX, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 6, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 7, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, 64, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_DOWNWARD, UINT_MAX, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 6, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 7, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, 64, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TOWARDZERO, UINT_MAX, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 6, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 7, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, 64, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEARESTFROMZERO, UINT_MAX, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 5, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 6, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 7, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, 64, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -17, FP_INT_TONEAREST, UINT_MAX, -17, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 23, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 24, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, 64, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_UPWARD, UINT_MAX, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 23, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 24, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, 64, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_DOWNWARD, UINT_MAX, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 23, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 24, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, 64, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TOWARDZERO, UINT_MAX, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 23, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 24, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, 64, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 23, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 24, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, 64, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffff, FP_INT_TONEAREST, UINT_MAX, 0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 23, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 24, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, 64, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_UPWARD, UINT_MAX, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 23, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 24, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, 64, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_DOWNWARD, UINT_MAX, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 23, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 24, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, 64, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TOWARDZERO, UINT_MAX, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 23, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 24, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, 64, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 23, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 24, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, 64, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffff, FP_INT_TONEAREST, UINT_MAX, -0x3fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 24, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 25, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, 64, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_UPWARD, UINT_MAX, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 24, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 25, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, 64, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_DOWNWARD, UINT_MAX, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 24, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 25, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, 64, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TOWARDZERO, UINT_MAX, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 24, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 25, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, 64, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 24, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 25, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, 64, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400000, FP_INT_TONEAREST, UINT_MAX, 0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 23, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 24, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, 64, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_UPWARD, UINT_MAX, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 23, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 24, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, 64, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_DOWNWARD, UINT_MAX, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 23, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 24, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, 64, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TOWARDZERO, UINT_MAX, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 23, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 24, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, 64, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 22, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 23, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 24, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, 64, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400000, FP_INT_TONEAREST, UINT_MAX, -0x400000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 24, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 25, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, 64, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_UPWARD, UINT_MAX, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 24, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 25, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, 64, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_DOWNWARD, UINT_MAX, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 24, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 25, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, 64, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TOWARDZERO, UINT_MAX, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 24, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 25, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, 64, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 24, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 25, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, 64, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x400001, FP_INT_TONEAREST, UINT_MAX, 0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 24, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 25, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, 64, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_UPWARD, UINT_MAX, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 24, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 25, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, 64, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_DOWNWARD, UINT_MAX, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 24, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 25, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, 64, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TOWARDZERO, UINT_MAX, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 24, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 25, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, 64, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 24, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 25, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, 64, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x400001, FP_INT_TONEAREST, UINT_MAX, -0x400001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 24, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 25, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, 64, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_UPWARD, UINT_MAX, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 24, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 25, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, 64, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_DOWNWARD, UINT_MAX, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 24, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 25, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, 64, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TOWARDZERO, UINT_MAX, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 24, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 25, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, 64, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 24, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 25, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, 64, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7fffff, FP_INT_TONEAREST, UINT_MAX, 0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 24, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 25, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, 64, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_UPWARD, UINT_MAX, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 24, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 25, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, 64, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_DOWNWARD, UINT_MAX, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 24, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 25, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, 64, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TOWARDZERO, UINT_MAX, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 24, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 25, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, 64, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 24, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 25, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, 64, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7fffff, FP_INT_TONEAREST, UINT_MAX, -0x7fffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 25, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 26, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, 64, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_UPWARD, UINT_MAX, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 25, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 26, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, 64, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_DOWNWARD, UINT_MAX, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 25, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 26, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, 64, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TOWARDZERO, UINT_MAX, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 25, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 26, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, 64, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 25, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 26, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, 64, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800000, FP_INT_TONEAREST, UINT_MAX, 0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 24, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 25, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, 64, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_UPWARD, UINT_MAX, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 24, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 25, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, 64, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_DOWNWARD, UINT_MAX, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 24, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 25, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, 64, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TOWARDZERO, UINT_MAX, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 24, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 25, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, 64, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 23, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 24, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 25, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, 64, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800000, FP_INT_TONEAREST, UINT_MAX, -0x800000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 25, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 26, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, 64, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_UPWARD, UINT_MAX, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 25, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 26, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, 64, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_DOWNWARD, UINT_MAX, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 25, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 26, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, 64, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TOWARDZERO, UINT_MAX, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 25, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 26, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, 64, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 25, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 26, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, 64, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x800001, FP_INT_TONEAREST, UINT_MAX, 0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 25, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 26, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, 64, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_UPWARD, UINT_MAX, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 25, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 26, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, 64, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_DOWNWARD, UINT_MAX, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 25, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 26, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, 64, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TOWARDZERO, UINT_MAX, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 25, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 26, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, 64, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 25, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 26, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, 64, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x800001, FP_INT_TONEAREST, UINT_MAX, -0x800001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 25, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 26, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, 64, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_UPWARD, UINT_MAX, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 25, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 26, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, 64, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_DOWNWARD, UINT_MAX, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 25, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 26, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, 64, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TOWARDZERO, UINT_MAX, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 25, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 26, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, 64, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 25, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 26, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, 64, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xffffff, FP_INT_TONEAREST, UINT_MAX, 0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 25, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 26, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, 64, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_UPWARD, UINT_MAX, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 25, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 26, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, 64, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_DOWNWARD, UINT_MAX, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 25, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 26, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, 64, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TOWARDZERO, UINT_MAX, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 25, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 26, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, 64, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 25, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 26, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, 64, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xffffff, FP_INT_TONEAREST, UINT_MAX, -0xffffff, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 26, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 27, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, 64, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_UPWARD, UINT_MAX, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 26, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 27, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, 64, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_DOWNWARD, UINT_MAX, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 26, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 27, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, 64, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TOWARDZERO, UINT_MAX, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 26, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 27, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, 64, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 26, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 27, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, 64, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000000, FP_INT_TONEAREST, UINT_MAX, 0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 25, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 26, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, 64, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_UPWARD, UINT_MAX, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 25, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 26, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, 64, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_DOWNWARD, UINT_MAX, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 25, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 26, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, 64, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TOWARDZERO, UINT_MAX, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 25, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 26, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, 64, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 24, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 25, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 26, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, 64, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000000, FP_INT_TONEAREST, UINT_MAX, -0x1000000, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if MANT_DIG >= 53
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 26, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 27, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, 64, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_UPWARD, UINT_MAX, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 26, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 27, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, 64, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_DOWNWARD, UINT_MAX, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 26, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 27, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, 64, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TOWARDZERO, UINT_MAX, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 26, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 27, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, 64, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 26, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 27, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, 64, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1000001, FP_INT_TONEAREST, UINT_MAX, 0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 26, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 27, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, 64, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_UPWARD, UINT_MAX, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 26, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 27, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, 64, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_DOWNWARD, UINT_MAX, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 26, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 27, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, 64, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TOWARDZERO, UINT_MAX, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 26, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 27, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, 64, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 25, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 26, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 27, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, 64, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1000001, FP_INT_TONEAREST, UINT_MAX, -0x1000001, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 52, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 53, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, 64, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_UPWARD, UINT_MAX, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 52, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 53, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, 64, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 52, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 53, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, 64, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 52, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 52, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 53, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, 64, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x7ffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, 0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 52, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 53, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, 64, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_UPWARD, UINT_MAX, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 52, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 53, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, 64, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 52, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 53, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, 64, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 52, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 52, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 53, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, 64, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x7ffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, -0x7ffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 53, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 54, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, 64, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_UPWARD, UINT_MAX, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 53, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 54, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, 64, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_DOWNWARD, UINT_MAX, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 53, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 54, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, 64, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 53, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 54, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 64, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 53, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 54, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, 64, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000000LL, FP_INT_TONEAREST, UINT_MAX, 0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 52, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 53, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, 64, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_UPWARD, UINT_MAX, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 52, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 53, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, 64, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_DOWNWARD, UINT_MAX, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 52, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 53, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, 64, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 52, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 53, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, 64, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 51, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 52, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 53, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, 64, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000000LL, FP_INT_TONEAREST, UINT_MAX, -0x8000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if MANT_DIG >= 53
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 53, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 54, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, 64, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_UPWARD, UINT_MAX, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 53, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 54, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, 64, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_DOWNWARD, UINT_MAX, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 53, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 54, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, 64, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 53, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 54, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 64, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 53, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 54, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, 64, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x8000000000001LL, FP_INT_TONEAREST, UINT_MAX, 0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 53, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 54, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, 64, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_UPWARD, UINT_MAX, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 53, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 54, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, 64, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_DOWNWARD, UINT_MAX, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 53, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 54, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, 64, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 53, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 54, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, 64, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 53, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 54, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, 64, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x8000000000001LL, FP_INT_TONEAREST, UINT_MAX, -0x8000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 53, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 54, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, 64, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_UPWARD, UINT_MAX, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 53, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 54, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, 64, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 53, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 54, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, 64, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 54, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 53, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 54, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, 64, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0xfffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, 0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 53, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 54, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, 64, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_UPWARD, UINT_MAX, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 53, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 54, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, 64, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 53, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 54, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, 64, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 54, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 53, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 54, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, 64, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0xfffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, -0xfffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 54, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 55, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, 64, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_UPWARD, UINT_MAX, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 54, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 55, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, 64, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_DOWNWARD, UINT_MAX, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 54, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 55, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, 64, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 54, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 55, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 64, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 54, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 55, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, 64, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000000LL, FP_INT_TONEAREST, UINT_MAX, 0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 53, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 54, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, 64, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_UPWARD, UINT_MAX, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 53, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 54, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, 64, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_DOWNWARD, UINT_MAX, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 53, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 54, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, 64, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 53, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 54, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, 64, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 52, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 53, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 54, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, 64, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000000LL, FP_INT_TONEAREST, UINT_MAX, -0x10000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if MANT_DIG >= 53
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 54, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 55, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, 64, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_UPWARD, UINT_MAX, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 54, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 55, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, 64, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_DOWNWARD, UINT_MAX, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 54, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 55, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, 64, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 54, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 55, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 64, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 54, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 55, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, 64, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x10000000000001LL, FP_INT_TONEAREST, UINT_MAX, 0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 54, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 55, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, 64, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_UPWARD, UINT_MAX, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 54, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 55, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, 64, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_DOWNWARD, UINT_MAX, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 54, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 55, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, 64, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 54, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 55, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, 64, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 54, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 55, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, 64, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x10000000000001LL, FP_INT_TONEAREST, UINT_MAX, -0x10000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 54, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 55, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, 64, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_UPWARD, UINT_MAX, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 54, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 55, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, 64, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 54, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 55, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, 64, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 54, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 55, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 54, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 55, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, 64, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x1fffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, 0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 54, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 55, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, 64, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_UPWARD, UINT_MAX, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 54, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 55, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, 64, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 54, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 55, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, 64, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 54, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 55, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 54, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 55, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, 64, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x1fffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, -0x1fffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 55, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 56, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, 64, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_UPWARD, UINT_MAX, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 55, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 56, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, 64, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_DOWNWARD, UINT_MAX, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 55, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 56, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, 64, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 55, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 56, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 64, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 55, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 56, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, 64, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000000LL, FP_INT_TONEAREST, UINT_MAX, 0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 54, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 55, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, 64, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_UPWARD, UINT_MAX, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 54, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 55, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, 64, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_DOWNWARD, UINT_MAX, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 54, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 55, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, 64, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 54, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 55, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, 64, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 53, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 54, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 55, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, 64, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000000LL, FP_INT_TONEAREST, UINT_MAX, -0x20000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if MANT_DIG >= 64
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 55, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 56, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, 64, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_UPWARD, UINT_MAX, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 55, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 56, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, 64, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_DOWNWARD, UINT_MAX, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 55, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 56, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, 64, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 55, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 56, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 64, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 55, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 56, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, 64, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x20000000000001LL, FP_INT_TONEAREST, UINT_MAX, 0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 55, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 56, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, 64, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_UPWARD, UINT_MAX, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 55, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 56, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, 64, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_DOWNWARD, UINT_MAX, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 55, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 56, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, 64, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 55, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 56, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, 64, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 54, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 55, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 56, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, 64, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x20000000000001LL, FP_INT_TONEAREST, UINT_MAX, -0x20000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, 63, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, 64, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_UPWARD, UINT_MAX, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, 63, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, 64, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 63, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 64, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 63, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, 63, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, 64, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x3fffffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, 0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, 63, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, 64, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_UPWARD, UINT_MAX, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, 63, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, 64, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_DOWNWARD, UINT_MAX, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 63, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, 64, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TOWARDZERO, UINT_MAX, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 63, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, 64, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, 63, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, 64, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x3fffffffffffffffLL, FP_INT_TONEAREST, UINT_MAX, -0x3fffffffffffffffLL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#endif
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_UPWARD, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_UPWARD, 64, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_UPWARD, UINT_MAX, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_DOWNWARD, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_DOWNWARD, 64, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_DOWNWARD, UINT_MAX, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TOWARDZERO, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TOWARDZERO, 64, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 64, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEAREST, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEAREST, 64, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000000LL, FP_INT_TONEAREST, UINT_MAX, 0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, 63, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, 64, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_UPWARD, UINT_MAX, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, 63, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, 64, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_DOWNWARD, UINT_MAX, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, 63, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, 64, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TOWARDZERO, UINT_MAX, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 63, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, 64, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEARESTFROMZERO, UINT_MAX, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, 62, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, 63, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, 64, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, -0x4000000000000000LL, FP_INT_TONEAREST, UINT_MAX, -0x4000000000000000LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
#if MANT_DIG >= 64
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_UPWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_UPWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_UPWARD, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_UPWARD, 64, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_UPWARD, UINT_MAX, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_DOWNWARD, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_DOWNWARD, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_DOWNWARD, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_DOWNWARD, 64, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_DOWNWARD, UINT_MAX, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TOWARDZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TOWARDZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TOWARDZERO, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TOWARDZERO, 64, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TOWARDZERO, UINT_MAX, 0x4000000000000001LL, NO_INEXACT_EXCEPTION|ERRNO_UNCHANGED),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TONEARESTFROMZERO, 0, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TONEARESTFROMZERO, 1, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO_EDOM),
    TEST_fiu_M (fromfp, 0x4000000000000001LL, FP_INT_TONEARESTFROMZERO, 63, IGNORE, NO_INEXACT_EXCEPTION|INVALID_EXCEPTION|ERRNO