/* -*- c -*- */
/*****************************************************************************/
/*  LibreDWG - free implementation of the DWG file format                    */
/*                                                                           */
/*  Copyright (C) 2018-2020 Free Software Foundation, Inc.                   */
/*                                                                           */
/*  This library is free software, licensed under the terms of the GNU       */
/*  General Public License as published by the Free Software Foundation,     */
/*  either version 3 of the License, or (at your option) any later version.  */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program.  If not, see <http://www.gnu.org/licenses/>.    */
/*****************************************************************************/

/*
 * classes.inc: the dynamic class dispatcher
 * for all types > 500
 * written by Reini Urban
 */

/**
 * Variable-typed classes and its stability
 */

#ifndef ACTION
# error ACTION define missing: decode, encode, dxf, indxf, print, free, ...
#endif

#define _XSTR(s) _STR(s)
#define _STR(s)  #s

#define WARN_UNHANDLED_CLASS \
      LOG_WARN ("Unhandled Class %s %d %s (0x%x%s) %u/%lX", is_entity ? "entity" : "object",\
               klass->number, klass->dxfname, klass->proxyflag,\
               klass->is_zombie ? "is_zombie" : "",\
               obj->index, obj->handle.value)
#define WARN_UNSTABLE_CLASS \
      LOG_WARN ("Unstable Class %s %d %s (0x%x%s) %u/%lX", is_entity ? "entity" : "object",\
               klass->number, klass->dxfname, klass->proxyflag,\
               klass->is_zombie ? "is_zombie" : "",\
               obj->index, obj->handle.value)
#if defined(IS_FREE) || defined(IS_PRINT)
# undef WARN_UNSTABLE_CLASS
# undef WARN_UNHANDLED_CLASS
# define WARN_UNSTABLE_CLASS
# define WARN_UNHANDLED_CLASS
#endif

#define DISALLOW_DXF(action, _name) \
  (strEQc (_STR(action), "dxf")) ? DWG_ERR_NOTYETSUPPORTED :
#define ALLOW_DXF(action, _name)

#ifdef DEBUG_CLASSES
# define DEBUGGING_DXF(action, name)                 ALLOW_DXF(action, name)
# define DEBUGGING_CLASS(action, name)               UNSTABLE_CLASS(action, name)
# define DEBUGGING_CLASS_DXF(action, name, _dxfname) UNSTABLE_CLASS_DXF(action, name, _dxfname)
# define DEBUGGING_CLASS_CPP(action, name, _cppname) UNSTABLE_CLASS_CPP(action, name, _cppname)
#else
# define DEBUGGING_DXF(action, name)                 DISALLOW_DXF(action, name)
# define DEBUGGING_CLASS(action, name)               UNHANDLED_CLASS(action, name)
# define DEBUGGING_CLASS_DXF(action, name, _dxfname) UNHANDLED_CLASS_DXF(action, name, _dxfname)
# define DEBUGGING_CLASS_CPP(action, name, _cppname) UNHANDLED_CLASS_CPP(action, name, _cppname)
#endif

#define STABLE_CLASS(action, _name) \
  if (klass->dxfname && strEQc (klass->dxfname, #_name)) \
    { \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define STABLE_CLASS_DXF(action, _name, _dxfname) \
  if (klass->dxfname && strEQc (klass->dxfname, #_dxfname)) \
    { \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_dxfname; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define STABLE_CLASS_CPP(action, _name, _cppname) \
  if (klass->cppname && strEQc (klass->cppname, #_name)) \
    { \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define UNSTABLE_CLASS(action, _name) \
  if (klass->dxfname && strEQc (klass->dxfname, #_name)) \
    { \
      WARN_UNSTABLE_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DEBUGGING_DXF(action, name) DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define UNSTABLE_CLASS_DXF(action, _name, _dxfname) \
  if (klass->dxfname && strEQc (klass->dxfname, #_dxfname)) \
    { \
      WARN_UNSTABLE_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_dxfname; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DEBUGGING_DXF(action, name) DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define UNSTABLE_CLASS_CPP(action, _name, _cppname) \
  if (klass->cppname && strEQc (klass->cppname, #_cppname)) \
    { \
      WARN_UNSTABLE_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->fixedtype = DWG_TYPE_##_name; \
        obj->klass = klass; \
      } \
      return DEBUGGING_DXF(action, name) DWG_FUNC_N(action,_name) (dat, obj); \
    }
#define UNHANDLED_CLASS(action, _name) \
  if (klass->dxfname && strEQc (klass->dxfname, #_name)) \
    { \
      WARN_UNHANDLED_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->klass = klass; \
      } \
      return DWG_ERR_UNHANDLEDCLASS; \
    }
#define UNHANDLED_CLASS_DXF(action, _name, _dxfname) \
  if (klass->dxfname && strEQc (klass->dxfname, #_dxfname)) \
    { \
      WARN_UNHANDLED_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_dxfname; \
        obj->klass = klass; \
      } \
      return DWG_ERR_UNHANDLEDCLASS; \
    }
#define UNHANDLED_CLASS_CPP(action, _name, _cppname) \
  if (klass->cppname && strEQc (klass->cppname, #_cppname)) \
    { \
      WARN_UNHANDLED_CLASS; \
      if (strEQc (_STR(action), "decode") || !memcmp (_STR(action), "in", 2)) { \
        obj->name = (char*) #_name; \
        obj->dxfname = (char*) #_name; \
        obj->klass = klass; \
      } \
      return DWG_ERR_UNHANDLEDCLASS; \
    }

  /* Entities */
  STABLE_CLASS    (ACTION, CAMERA)    /* ent, not persistent in a DWG */
  STABLE_CLASS    (ACTION, IMAGE)     /* ent */
  STABLE_CLASS    (ACTION, LIGHT)     /* ent */
  STABLE_CLASS    (ACTION, LWPOLYLINE)
  STABLE_CLASS    (ACTION, HATCH)
  STABLE_CLASS    (ACTION, MESH)      /* ent */
  STABLE_CLASS    (ACTION, MULTILEADER) /* ent */
  STABLE_CLASS    (ACTION, OLE2FRAME) /* ent (also fixed) */
  STABLE_CLASS    (ACTION, SECTIONOBJECT) /* ent */
  STABLE_CLASS_DXF(ACTION, UNDERLAY, PDFUNDERLAY)
  STABLE_CLASS_DXF(ACTION, UNDERLAY, DGNUNDERLAY)
  STABLE_CLASS_DXF(ACTION, UNDERLAY, DWFUNDERLAY)

  /* Objects */
  STABLE_CLASS    (ACTION, ACSH_BOOLEAN_CLASS)
  STABLE_CLASS    (ACTION, ACSH_BOX_CLASS)
  STABLE_CLASS    (ACTION, ACSH_CYLINDER_CLASS)
  STABLE_CLASS    (ACTION, ACSH_FILLET_CLASS)
  STABLE_CLASS    (ACTION, ACSH_HISTORY_CLASS)
  STABLE_CLASS    (ACTION, ACSH_SPHERE_CLASS)
  STABLE_CLASS    (ACTION, ACSH_WEDGE_CLASS) // the same as box
  STABLE_CLASS    (ACTION, BLOCKGRIPLOCATIONCOMPONENT) // AcDbBlockGripExpr
  STABLE_CLASS    (ACTION, BLOCKBASEPOINTPARAMETER)
  STABLE_CLASS    (ACTION, BLOCKFLIPACTION)
  STABLE_CLASS    (ACTION, BLOCKFLIPPARAMETER)
  STABLE_CLASS    (ACTION, BLOCKFLIPGRIP)
  STABLE_CLASS    (ACTION, BLOCKLINEARGRIP)
  STABLE_CLASS    (ACTION, BLOCKMOVEACTION)
  STABLE_CLASS    (ACTION, BLOCKROTATEACTION)
  STABLE_CLASS    (ACTION, BLOCKSCALEACTION)
  STABLE_CLASS    (ACTION, BLOCKVISIBILITYGRIP)
  STABLE_CLASS    (ACTION, CELLSTYLEMAP)
  STABLE_CLASS    (ACTION, DICTIONARYVAR)
  STABLE_CLASS_DXF(ACTION, DICTIONARYWDFLT, ACDBDICTIONARYWDFLT)
  STABLE_CLASS_DXF(ACTION, DYNAMICBLOCKPURGEPREVENTER, ACDB_DYNAMICBLOCKPURGEPREVENTER_VERSION)
  STABLE_CLASS    (ACTION, FIELD)
  STABLE_CLASS    (ACTION, FIELDLIST)
  STABLE_CLASS    (ACTION, GEODATA)
  STABLE_CLASS    (ACTION, GROUP)
  STABLE_CLASS    (ACTION, IDBUFFER)
  STABLE_CLASS    (ACTION, INDEX)
  STABLE_CLASS    (ACTION, IMAGEDEF)
  STABLE_CLASS    (ACTION, IMAGEDEF_REACTOR)
  STABLE_CLASS    (ACTION, LAYER_INDEX)
  STABLE_CLASS    (ACTION, LAYERFILTER)
  STABLE_CLASS    (ACTION, LAYOUT)       // includes PLOTSETTINGS
  STABLE_CLASS    (ACTION, MLEADERSTYLE) // TODO block_rotation nan
  STABLE_CLASS_DXF(ACTION, PLACEHOLDER, ACDBPLACEHOLDER)
  STABLE_CLASS    (ACTION, PLOTSETTINGS)
  STABLE_CLASS    (ACTION, RASTERVARIABLES)
  STABLE_CLASS    (ACTION, SCALE)
  STABLE_CLASS    (ACTION, SECTION_MANAGER)
  STABLE_CLASS_DXF(ACTION, DETAILVIEWSTYLE, ACDBDETAILVIEWSTYLE) // excellent coverage
  STABLE_CLASS_DXF(ACTION, SECTIONVIEWSTYLE, ACDBSECTIONVIEWSTYLE)
  STABLE_CLASS    (ACTION, SORTENTSTABLE)
  STABLE_CLASS    (ACTION, SPATIAL_FILTER)
  STABLE_CLASS    (ACTION, TABLEGEOMETRY)
  STABLE_CLASS    (ACTION, UNDERLAYDEFINITION)    // only tested pdf, but very simple
  STABLE_CLASS_DXF(ACTION, UNDERLAYDEFINITION, PDFDEFINITION)
  STABLE_CLASS_DXF(ACTION, UNDERLAYDEFINITION, DGNDEFINITION)
  STABLE_CLASS_DXF(ACTION, UNDERLAYDEFINITION, DWFDEFINITION)
  STABLE_CLASS    (ACTION, VBA_PROJECT)
  STABLE_CLASS    (ACTION, VISUALSTYLE)
  STABLE_CLASS    (ACTION, WIPEOUTVARIABLES)
  STABLE_CLASS    (ACTION, XRECORD)

  // unstable/undertested.
  // not enough coverage, but assumed mostly ok. field names may change.
  // DXF output skipped
  UNSTABLE_CLASS     (ACTION, ACSH_BREP_CLASS)
  UNSTABLE_CLASS     (ACTION, ACSH_CHAMFER_CLASS)
  UNSTABLE_CLASS     (ACTION, ACSH_CONE_CLASS)
  UNSTABLE_CLASS     (ACTION, ACSH_PYRAMID_CLASS)
  UNSTABLE_CLASS     (ACTION, ACSH_TORUS_CLASS)
  UNSTABLE_CLASS     (ACTION, ARC_DIMENSION)     /* ent */
  UNSTABLE_CLASS_DXF (ACTION, ASSOCACTION, ACDBASSOCACTION)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCNETWORK, ACDBASSOCNETWORK)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCDEPENDENCY, ACDBASSOCDEPENDENCY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCGEOMDEPENDENCY, ACDBASSOCGEOMDEPENDENCY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCVALUEDEPENDENCY, ACDBASSOCVALUEDEPENDENCY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCBLENDSURFACEACTIONBODY, ACDBASSOCBLENDSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCEXTENDSURFACEACTIONBODY, ACDBASSOCEXTENDSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCEXTRUDEDSURFACEACTIONBODY, ACDBASSOCEXTRUDEDSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCFILLETSURFACEACTIONBODY, ACDBASSOCFILLETSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCLOFTEDSURFACEACTIONBODY, ACDBASSOCLOFTEDSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCNETWORKSURFACEACTIONBODY, ACDBASSOCNETWORKSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCOFFSETSURFACEACTIONBODY, ACDBASSOCOFFSETSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCPATCHSURFACEACTIONBODY, ACDBASSOCPATCHSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCPLANESURFACEACTIONBODY, ACDBASSOCPLANESURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCREVOLVEDSURFACEACTIONBODY, ACDBASSOCREVOLVEDSURFACEACTIONBODY)
  UNSTABLE_CLASS_DXF (ACTION, ASSOCTRIMSURFACEACTIONBODY, ACDBASSOCTRIMSURFACEACTIONBODY)
  UNSTABLE_CLASS     (ACTION, DBCOLOR)
  UNSTABLE_CLASS     (ACTION, HELIX)     /* ent, looks very stable */
  UNSTABLE_CLASS     (ACTION, WIPEOUT)   /* ent, stable but causes acad redraw to crash (GH #244) */
  UNSTABLE_CLASS     (ACTION, LARGE_RADIAL_DIMENSION) /* ent */
  UNSTABLE_CLASS     (ACTION, LIGHTLIST)
  UNSTABLE_CLASS     (ACTION, MATERIAL)
  UNSTABLE_CLASS     (ACTION, MENTALRAYRENDERSETTINGS)
  UNSTABLE_CLASS     (ACTION, OBJECT_PTR)
  UNSTABLE_CLASS_CPP (ACTION, OBJECT_PTR, CAseDLPNTableRecord)
  UNSTABLE_CLASS_DXF (ACTION, PROXY_OBJECT, ACAD_PROXY_OBJECT)
  UNSTABLE_CLASS     (ACTION, RAPIDRTRENDERSETTINGS)
  UNSTABLE_CLASS     (ACTION, RENDERSETTINGS)
  UNSTABLE_CLASS     (ACTION, SPATIAL_INDEX)
  UNSTABLE_CLASS     (ACTION, TABLESTYLE)    // stable, but for r2010+ some bits missing at the end
  UNSTABLE_CLASS     (ACTION, SECTION_SETTINGS)
  UNSTABLE_CLASS     (ACTION, SUN)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, SOLID_BACKGROUND)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, SKYLIGHT_BACKGROUND)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, GROUND_PLANE_BACKGROUND)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, GRADIENT_BACKGROUND)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, IMAGE_BACKGROUND)
  UNSTABLE_CLASS_DXF (ACTION, BACKGROUND, IBL_BACKGROUND)
  UNSTABLE_CLASS     (ACTION, BLOCKALIGNMENTGRIP)
  UNSTABLE_CLASS     (ACTION, BLOCKALIGNMENTPARAMETER)
  UNSTABLE_CLASS     (ACTION, BLOCKLINEARPARAMETER)
  UNSTABLE_CLASS     (ACTION, BLOCKROTATIONPARAMETER)
  UNSTABLE_CLASS     (ACTION, BLOCKLOOKUPGRIP)
  UNSTABLE_CLASS     (ACTION, BLOCKROTATIONGRIP)
  UNSTABLE_CLASS     (ACTION, BLOCKXYPARAMETER)
  UNSTABLE_CLASS     (ACTION, BLOCKVISIBILITYPARAMETER)

  // coverage exists, but broken and being worked on. needs -DDEBUG_CLASSES. See also
  // examples/unknown with all the unknown blobs of these.
  // PROXY_ENTITY has a fixed type
  // DXF output skipped, but some INDXF are done.
  DEBUGGING_CLASS_DXF (ACTION, GEOPOSITIONMARKER, POSITIONMARKER) /* ent, yet unsorted. no coverage */
  DEBUGGING_CLASS     (ACTION, NURBSURFACE)       /* ent */
  DEBUGGING_CLASS     (ACTION, PLANESURFACE)      /* ent */
  DEBUGGING_CLASS     (ACTION, EXTRUDEDSURFACE)   /* ent */
  DEBUGGING_CLASS     (ACTION, LOFTEDSURFACE)     /* ent */
  DEBUGGING_CLASS     (ACTION, REVOLVEDSURFACE)   /* ent */
  DEBUGGING_CLASS     (ACTION, SWEPTSURFACE)      /* ent */
  DEBUGGING_CLASS_DXF (ACTION, TABLE, ACAD_TABLE) /* ent, r2010+ needs subclassing */
  DEBUGGING_CLASS     (ACTION, ATEXT)             // ent expresstools
  DEBUGGING_CLASS     (ACTION, RTEXT)             // ent expresstools
  DEBUGGING_CLASS     (ACTION, MPOLYGON)          // hatch-like ent
  DEBUGGING_CLASS_CPP (ACTION, NAVISWORKSMODEL, AcDbNavisworksModel)   /* ent,
                                                     dxfname COORDINATION_MODEL? */
  DEBUGGING_CLASS     (ACTION, DIMASSOC)
  DEBUGGING_CLASS_CPP (ACTION, TABLECONTENT, AcDbTableContent) // dxfname: "TABLE"
  DEBUGGING_CLASS     (ACTION, SUNSTUDY)     // almost
  DEBUGGING_CLASS     (ACTION, ACSH_EXTRUSION_CLASS)
  DEBUGGING_CLASS     (ACTION, ACSH_LOFT_CLASS)
  DEBUGGING_CLASS     (ACTION, ACSH_REVOLVE_CLASS)
  DEBUGGING_CLASS     (ACTION, ACSH_SWEEP_CLASS)
  DEBUGGING_CLASS     (ACTION, GEOMAPIMAGE)  // LiveMap image overlay
  DEBUGGING_CLASS_DXF (ACTION, NAVISWORKSMODELDEF, ACDBNAVISWORKSMODELDEF)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCVARIABLE, ACDBASSOCVARIABLE)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCRESTOREENTITYSTATEACTIONBODY, ACDBASSOCRESTOREENTITYSTATEACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCMLEADERACTIONBODY, ACDBASSOCMLEADERACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOC3POINTANGULARDIMACTIONBODY, ACDBASSOC3POINTANGULARDIMACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCALIGNEDDIMACTIONBODY, ACDBASSOCALIGNEDDIMACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCORDINATEDIMACTIONBODY, ACDBASSOCORDINATEDIMACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCROTATEDDIMACTIONBODY, ACDBASSOCROTATEDDIMACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCSWEPTSURFACEACTIONBODY, ACDBASSOCSWEPTSURFACEACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCEDGECHAMFERACTIONBODY, ACDbAssocEDGECHAMFERACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCEDGEFILLETACTIONBODY, ASSOCEDGEFILLETACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYACTIONBODY, ACDBASSOCARRAYACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYMODIFYACTIONBODY, ACDBASSOCARRAYMODIFYACTIONBODY)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCPERSSUBENTMANAGER, ACDBASSOCPERSSUBENTMANAGER)
  DEBUGGING_CLASS_DXF (ACTION, PERSUBENTMGR, ACDBPERSSUBENTMANAGER)
  DEBUGGING_CLASS_DXF (ACTION, ASSOC2DCONSTRAINTGROUP, ACDBASSOC2DCONSTRAINTGROUP)
  DEBUGGING_CLASS_DXF (ACTION, EVALUATION_GRAPH, ACAD_EVALUATION_GRAPH) /* almost */
  DEBUGGING_CLASS     (ACTION, RENDERENVIRONMENT) // no coverage
  DEBUGGING_CLASS     (ACTION, RENDERENTRY)       // no coverage
  DEBUGGING_CLASS     (ACTION, RENDERGLOBAL)      // no coverage
  DEBUGGING_CLASS     (ACTION, DATALINK)
  DEBUGGING_CLASS     (ACTION, DATATABLE)
  DEBUGGING_CLASS     (ACTION, LAYOUTPRINTCONFIG)
  DEBUGGING_CLASS     (ACTION, CONTEXTDATAMANAGER)
  DEBUGGING_CLASS_DXF (ACTION, ANNOTSCALEOBJECTCONTEXTDATA, ACDB_ANNOTSCALEOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, ALDIMOBJECTCONTEXTDATA, ACDB_ALDIMOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, ANGDIMOBJECTCONTEXTDATA, ACDB_ANGDIMOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, DMDIMOBJECTCONTEXTDATA, ACDB_DMDIMOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, ORDDIMOBJECTCONTEXTDATA, ACDB_ORDDIMOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, RADIMOBJECTCONTEXTDATA, ACDB_RADIMOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, RADIMLGOBJECTCONTEXTDATA, ACDB_RADIMLGOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, BLKREFOBJECTCONTEXTDATA, ACDB_BLKREFOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, FCFOBJECTCONTEXTDATA, ACDB_FCFOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, LEADEROBJECTCONTEXTDATA, ACDB_LEADEROBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, MLEADEROBJECTCONTEXTDATA, ACDB_MLEADEROBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, MTEXTATTRIBUTEOBJECTCONTEXTDATA, ACDB_MTEXTATTRIBUTEOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, MTEXTOBJECTCONTEXTDATA, ACDB_MTEXTOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS_DXF (ACTION, TEXTOBJECTCONTEXTDATA,  ACDB_TEXTOBJECTCONTEXTDATA_CLASS)
  DEBUGGING_CLASS     (ACTION, ACMECOMMANDHISTORY)
  DEBUGGING_CLASS     (ACTION, ACMESCOPE)
  DEBUGGING_CLASS     (ACTION, ACMESTATEMGR)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCOSNAPPOINTREFACTIONPARAM, ACDBASSOCOSNAPPOINTREFACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCPOINTREFACTIONPARAM, ACDBASSOCPointRefActionParam)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCASMBODYACTIONPARAM, ACDBASSOCASMBODYACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCACTIONPARAM, ACDBASSOCACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCCOMPOUNDACTIONPARAM, ACDBASSOCCOMPOUNDACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCEDGEACTIONPARAM, ACDBASSOCEDGEACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCFACEACTIONPARAM, ACDBASSOCFACEACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCOBJECTACTIONPARAM, ACDBASSOCOBJECTACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCPATHACTIONPARAM, ACDBASSOCPATHACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCVERTEXACTIONPARAM, ACDBASSOCVERTEXACTIONPARAM)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYMODIFYPARAMETERS, ACDBASSOCARRAYMODIFYPARAMETERS)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYPATHPARAMETERS, ACDBASSOCARRAYPATHPARAMETERS)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYPOLARPARAMETERS, ACDBASSOCARRAYPOLARPARAMETERS)
  DEBUGGING_CLASS_DXF (ACTION, ASSOCARRAYRECTANGULARPARAMETERS, ACDBASSOCARRAYRECTANGULARPARAMETERS)
  DEBUGGING_CLASS     (ACTION, CSACDOCUMENTOPTIONS)
  DEBUGGING_CLASS_DXF (ACTION, CURVEPATH, ACDBCURVEPATH)
  DEBUGGING_CLASS     (ACTION, MOTIONPATH)
  DEBUGGING_CLASS_DXF (ACTION, POINTPATH, ACDBPOINTPATH)
  DEBUGGING_CLASS     (ACTION, TVDEVICEPROPERTIES)
  DEBUGGING_CLASS     (ACTION, ASSOCDIMDEPENDENCYBODY)
  DEBUGGING_CLASS     (ACTION, BLOCKPARAMDEPENDENCYBODY)
  DEBUGGING_CLASS     (ACTION, ALIGNMENTPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, BASEPOINTPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, FLIPPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, LINEARPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, POINTPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, ROTATIONPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, VISIBILITYPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, VISIBILITYGRIPENTITY)
  DEBUGGING_CLASS     (ACTION, XYPARAMETERENTITY)
  DEBUGGING_CLASS     (ACTION, BLOCKALIGNEDCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKANGULARCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKARRAYACTION)
  DEBUGGING_CLASS     (ACTION, BLOCKDIAMETRICCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKHORIZONTALCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKLINEARCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKLOOKUPACTION)
  DEBUGGING_CLASS     (ACTION, BLOCKLOOKUPPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKPOINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKPOLARGRIP)
  DEBUGGING_CLASS     (ACTION, BLOCKPOLARPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKPOLARSTRETCHACTION)
  DEBUGGING_CLASS     (ACTION, BLOCKPROPERTIESTABLE)
  DEBUGGING_CLASS     (ACTION, BLOCKPROPERTIESTABLEGRIP)
  DEBUGGING_CLASS     (ACTION, BLOCKRADIALCONSTRAINTPARAMETER)
  DEBUGGING_CLASS_DXF (ACTION, BLOCKREPRESENTATION, ACDB_BLOCKREPRESENTATION_DATA)
  DEBUGGING_CLASS     (ACTION, BLOCKSTRETCHACTION)
  DEBUGGING_CLASS     (ACTION, BLOCKUSERPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKVERTICALCONSTRAINTPARAMETER)
  DEBUGGING_CLASS     (ACTION, BLOCKXYGRIP)
  DEBUGGING_CLASS_DXF (ACTION, DYNAMICBLOCKPROXYNODE, ACDB_DYNAMICBLOCKPROXYNODE)
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUD, ACDBPOINTCLOUD)	/* ent */
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDEX, ACDBPOINTCLOUDEX)	/* ent */
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDDEF, ACDBPOINTCLOUDDEF)
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDDEFEX, ACDBPOINTCLOUDDEFEX)
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDDEF_REACTOR, ACDBPOINTCLOUDDEF_REACTOR)
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDDEF_REACTOR_EX, ACDBPOINTCLOUDDEF_REACTOR_EX)
  DEBUGGING_CLASS_DXF (ACTION, POINTCLOUDCOLORMAP, ACDBPOINTCLOUDCOLORMAP)

  // unimplemented, passed through. coverage exists mostly, but documentation not.
  // DXF output skipped
  UNHANDLED_CLASS     (ACTION, PROXY_LWPOLYLINE) 		/* ent */
  UNHANDLED_CLASS     (ACTION, ACDSRECORD)
  UNHANDLED_CLASS     (ACTION, ACDSSCHEMA)
  UNHANDLED_CLASS     (ACTION, NPOCOLLECTION)
  UNHANDLED_CLASS     (ACTION, RAPIDRTRENDERENVIRONMENT)  //no coverage
  UNHANDLED_CLASS_DXF (ACTION, XREFPANELOBJECT, EXACXREFPANELOBJECT)

/* Missing DXF names:
ACAD_PROXY_ENTITY
ACAD_PROXY_OBJECT
ACDBPOINTCLOUDEX
ARRAY
ATTBLOCKREF
ATTDYNBLOCKREF
BLOCKREF
DYNBLOCKREF
XREF
CENTERMARK
CENTERLINE

// Declared are these DXF names, but not yet added:
ACAD_PROXY_OBJECT_WRAPPER
ACAECUSTOBJ
ACAEEEMGROBJ
AC_AM_2D_XREF_MGR
AC_AM_BASIC_VIEW
AC_AM_BASIC_VIEW_DEF
ACAMCOMP
ACAMCOMPDEF
ACAMCOMPDEFMGR
AC_AM_COMPLEX_HIDE_SITUATION
AC_AM_COMP_VIEW_DEF
AC_AM_COMP_VIEW_INST
ACAMCONTEXTMODELER
AC_AM_DIRTY_NODES
ACAMGDIMSTD
ACAMGFILTERDAT
ACAMGHOLECHARTSTDCSN
ACAMGHOLECHARTSTDDIN
ACAMGHOLECHARTSTDISO
ACAMGLAYSTD
ACAMGRCOMPDEF
ACAMGRCOMPDEFSET
ACAMGTITLESTD
AC_AM_HIDE_SITUATION
AC_AM_MAPPER_CACHE
AC_AM_MASTER_VIEW_DEF
ACAMMVDBACKUPOBJECT
AC_AM_MVD_DEP_MGR
AC_AM_OVERRIDE_FILTER
ACAMPROJECT
AC_AM_PROPS_OVERRIDE
ACAMSHAFTCOMPDEF
AC_AM_SHAFT_HIDE_SITUATION
ACAMSTDPCOMPDEF
AC_AM_STDP_VIEW_DEF
AC_AM_TRANSFORM_GHOST
ACAMWBLOCKTEMPENTS
ACARRAYJIGENTITY
ACCMCONTEXT
ACDBASSOCARRAYACTIONBODY
ACDBASSOCARRAYPOLARPARAMETERS
ACDBASSOCARRAYRECTANGULARPARAMETERS
ACDB_BLOCKREPRESENTATION_DATA
ACDBCIRCARCRES
ACDBDIMENSIONRES
ACDBENTITYCACHE
ACDB_HATCHSCALECONTEXTDATA_CLASS
ACDB_HATCHVIEWCONTEXTDATA_CLASS
ACDBLINERES
ACDB_PROXY_ENTITY_DATA
ACDBSTDPARTRES_ARC
ACDBSTDPARTRES_LINE
ACGREFACADMASTER
ACGREFMASTER
ACIMINTSYSVAR
ACIMREALSYSVAR
ACIMSTRSYSVAR
ACIMSYSVARMAN
ACMANOOTATIONVIEWSTANDARDANSI
ACMANOOTATIONVIEWSTANDARDCSN
ACMANOOTATIONVIEWSTANDARDDIN
ACMANOOTATIONVIEWSTANDARDISO
ACMAPLEGENDDBOBJECT
ACMAPLEGENDITEMDBOBJECT
ACMAPMAPVIEWPORTDBOBJECT
ACMAPPRINTLAYOUTELEMENTDBOBJECTCONTAINER
ACMBALLOON
ACMBOM
ACMBOMROW
ACMBOMROWSTRUCT
ACMBOMSTANDARDANSI
ACMBOMSTANDARDCSN
ACMBOMSTANDARDDIN
ACMBOMSTANDARDISO
ACMCENTERLINESTANDARDANSI
ACMCENTERLINESTANDARDCSN
ACMCENTERLINESTANDARDDIN
ACMCENTERLINESTANDARDISO
ACMDATADICTIONARY
ACMDATAENTRY
ACMDATAENTRYBLOCK
ACMDATUMID
ACMDATUMSTANDARDANSI
ACMDATUMSTANDARDCSN
ACMDATUMSTANDARDDIN
ACMDATUMSTANDARDISO
ACMDATUMSTANDARDISO2012
ACMDETAILSTANDARDANSI
ACMDETAILSTANDARDCSN
ACMDETAILSTANDARDDIN
ACMDETAILSTANDARDISO
ACMDETAILTANDARDCUSTOM
ACMDIMBREAKPERSREACTOR
ACME_DATABASE
ACME_DOCUMENT
ACMEDRAWINGMAN
ACMEVIEW
ACMFCFRAME
ACMFCFSTANDARDANSI
ACMFCFSTANDARDCSN
ACMFCFSTANDARDDIN
ACMFCFSTANDARDISO
ACMFCFSTANDARDISO2004
ACMFCFSTANDARDISO2012
ACMIDSTANDARDANSI
ACMIDSTANDARDCSN
ACMIDSTANDARDDIN
ACMIDSTANDARDISO
ACMIDSTANDARDISO2004
ACMIDSTANDARDISO2012
ACMNOTESTANDARDANSI
ACMNOTESTANDARDCSN
ACMNOTESTANDARDDIN
ACMNOTESTANDARDISO
ACMPARTLIST
ACMPICKOBJ
ACMSECTIONSTANDARDANSI
ACMSECTIONSTANDARDCSN2002
ACMSECTIONSTANDARDCUSTOM
ACMSECTIONSTANDARDDIN
ACMSECTIONSTANDARDISO
ACMSECTIONSTANDARDISO2001
ACMSTANDARDANSI
ACMSTANDARDCSN
ACMSTANDARDDIN
ACMSTANDARDISO
ACMSURFSTANDARDANSI
ACMSURFSTANDARDCSN
ACMSURFSTANDARDDIN
ACMSURFSTANDARDISO
ACMSURFSTANDARDISO2002
ACMSURFSYM
ACMTAPERSTANDARDANSI
ACMTAPERSTANDARDCSN
ACMTAPERSTANDARDDIN
ACMTAPERSTANDARDISO
ACMTHREADLINESTANDARDANSI
ACMTHREADLINESTANDARDCSN
ACMTHREADLINESTANDARDDIN
ACMTHREADLINESTANDARDISO
ACMWELDSTANDARDANSI
ACMWELDSTANDARDCSN
ACMWELDSTANDARDDIN
ACMWELDSTANDARDISO
ACMWELDSYM
ACRFATTGENMGR
ACRFINSADJ
ACRFINSADJUSTERMGR
ACRFMCADAPIATTHOLDER
ACRFOBJATTMGR
ACSH_SUBENT_MATERIAL_CLASS
ADAPPL
AEC_AECDBDISPREPBDGELEVLINEPLAN100
AEC_AECDBDISPREPBDGELEVLINEPLAN50
AEC_AECDBDISPREPBDGSECTIONLINEPLAN100
AEC_AECDBDISPREPBDGSECTIONLINEPLAN50
AEC_AECDBDISPREPCEILINGGRIDPLAN100
AEC_AECDBDISPREPCEILINGGRIDPLAN50
AEC_AECDBDISPREPCOLUMNGRIDPLAN100
AEC_AECDBDISPREPCOLUMNGRIDPLAN50
AEC_AECDBDISPREPCURTAINWALLLAYOUTPLAN100
AEC_AECDBDISPREPCURTAINWALLLAYOUTPLAN50
AEC_AECDBDISPREPCURTAINWALLUNITPLAN100
AEC_AECDBDISPREPCURTAINWALLUNITPLAN50
AEC_AECDBDISPREPMVBLOCKREFPLAN100
AEC_AECDBDISPREPMVBLOCKREFPLAN50
AEC_AECDBDISPREPROOFPLAN100
AEC_AECDBDISPREPROOFPLAN50
AEC_AECDBDISPREPROOFSLABPLAN100
AEC_AECDBDISPREPROOFSLABPLAN50
AEC_AECDBDISPREPSLABPLAN100
AEC_AECDBDISPREPSLABPLAN50
AEC_AECDBDISPREPSPACEPLAN100
AEC_AECDBDISPREPSPACEPLAN50
AEC_AECDBDISPREPWALLPLAN100
AEC_AECDBDISPREPWALLPLAN50
AEC_AECDBDISPREPWINDOWASSEMBLYPLAN100
AEC_AECDBDISPREPWINDOWASSEMBLYPLAN50
AEC_AECDBDISPREPZONE100
AEC_AECDBDISPREPZONE50
AEC_AECDBZONESTYLE
AECC_ALIGNMENT_DESIGN_CHECK_SET
AECC_ALIGNMENT_LABEL_SET
AECC_ALIGNMENT_LABEL_SET_EXT
AECC_ALIGNMENT_PARCEL_NODE
AECC_ALIGNMENT_STYLE
AECC_APPURTENANCE_STYLE
AECC_ASSEMBLY_STYLE
AECC_BUILDING_SITE_STYLE
AECC_CANT_DIAGRAM_VIEW_STYLE
AECC_CATCHMENT_STYLE
AECC_CLASS_NODE
AECC_CONTOURVIEW
AECC_CORRIDOR_STYLE
AECC_DISP_REP_ALIGNMENT
AECC_DISP_REP_ALIGNMENT_CANT_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_CSV
AECC_DISP_REP_ALIGNMENT_CURVE_LABEL
AECC_DISP_REP_ALIGNMENT_DESIGNSPEED_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_GEOMPT_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_INDEXED_PI_LABEL
AECC_DISP_REP_ALIGNMENT_MINOR_STATION_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_PI_LABEL
AECC_DISP_REP_ALIGNMENT_SPIRAL_LABEL
AECC_DISP_REP_ALIGNMENT_STAEQU_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_STATION_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_STATION_OFFSET_LABEL
AECC_DISP_REP_ALIGNMENT_SUPERELEVATION_LABEL_GROUP
AECC_DISP_REP_ALIGNMENT_TABLE
AECC_DISP_REP_ALIGNMENT_TANGENT_LABEL
AECC_DISP_REP_ALIGNMENT_VERTICAL_GEOMPT_LABELING
AECC_DISP_REP_APPURTENANCE
AECC_DISP_REP_APPURTENANCE_CSV
AECC_DISP_REP_APPURTENANCE_LABELING
AECC_DISP_REP_APPURTENANCE_PROFILE_LABELING
AECC_DISP_REP_ASSEMBLY
AECC_DISP_REP_AUTO_CORRIDOR_FEATURE_LINE
AECC_DISP_REP_AUTO_CORRIDOR_FEATURE_LINE_PROFILE
AECC_DISP_REP_AUTO_CORRIDOR_FEATURE_LINE_SECTION
AECC_DISP_REP_AUTO_FEATURE_LINE
AECC_DISP_REP_AUTO_FEATURE_LINE_CSV
AECC_DISP_REP_AUTO_FEATURE_LINE_PROFILE
AECC_DISP_REP_AUTO_FEATURE_LINE_SECTION
AECC_DISP_REP_BUILDINGSITE
AECC_DISP_REP_BUILDINGUTIL_CONNECTOR
AECC_DISP_REP_CANT_DIAGRAM_VIEW
AECC_DISP_REP_CATCHMENT_AREA
AECC_DISP_REP_CATCHMENT_AREA_LABEL
AECC_DISP_REP_CORRIDOR
AECC_DISP_REP_CROSSING_PIPE_PROFILE_LABELING
AECC_DISP_REP_CROSSING_PRESSURE_PIPE_PROFILE_LABELING
AECC_DISP_REP_CSVSTATIONSLIDER
AECC_DISP_REP_FACE
AECC_DISP_REP_FEATURE
AECC_DISP_REP_FEATURE_LABEL
AECC_DISP_REP_FEATURE_LINE
AECC_DISP_REP_FEATURE_LINE_CSV
AECC_DISP_REP_FEATURE_LINE_PROFILE
AECC_DISP_REP_FEATURE_LINE_SECTION
AECC_DISP_REP_FITTING
AECC_DISP_REP_FITTING_CSV
AECC_DISP_REP_FITTING_LABELING
AECC_DISP_REP_FITTING_PROFILE_LABELING
AECC_DISP_REP_FLOW_SEGMENT_LABEL
AECC_DISP_REP_GENERAL_SEGMENT_LABEL
AECC_DISP_REP_GRADING
AECC_DISP_REP_GRAPH
AECC_DISP_REP_GRAPHPROFILE_NETWORKPART
AECC_DISP_REP_GRAPHPROFILE_PRESSUREPART
AECC_DISP_REP_GRID_SURFACE
AECC_DISP_REP_GRID_SURFACE_CSV
AECC_DISP_REP_HORGEOMETRY_BAND_LABEL_GROUP
AECC_DISP_REP_HYDRO_REGION
AECC_DISP_REP_INTERFERENCE_CHECK
AECC_DISP_REP_INTERFERENCE_PART
AECC_DISP_REP_INTERFERENCE_PART_SECTION
AECC_DISP_REP_INTERSECTION
AECC_DISP_REP_INTERSECTION_LOCATION_LABELING
AECC_DISP_REP_LABELING
AECC_DISP_REP_LEGEND_TABLE
AECC_DISP_REP_LINE_BETWEEN_POINTS_LABEL
AECC_DISP_REP_LOTLINE_CSV
AECC_DISP_REP_MASSHAULLINE
AECC_DISP_REP_MASS_HAUL_VIEW
AECC_DISP_REP_MATCH_LINE
AECC_DISP_REP_MATCHLINE_LABELING
AECC_DISP_REP_MATERIAL_SECTION
AECC_DISP_REP_NETWORK
AECC_DISP_REP_NOTE_LABEL
AECC_DISP_REP_OFFSET_ELEV_LABEL
AECC_DISP_REP_PARCEL_BOUNDARY
AECC_DISP_REP_PARCEL_FACE_LABEL
AECC_DISP_REP_PARCEL_SEGMENT
AECC_DISP_REP_PARCEL_SEGMENT_LABEL
AECC_DISP_REP_PARCEL_SEGMENT_TABLE
AECC_DISP_REP_PARCEL_TABLE
AECC_DISP_REP_PIPE
AECC_DISP_REP_PIPE_CSV
AECC_DISP_REP_PIPE_LABELING
AECC_DISP_REP_PIPENETWORK_BAND_LABEL_GROUP
AECC_DISP_REP_PIPE_PROFILE_LABELING
AECC_DISP_REP_PIPE_SECTION_LABELING
AECC_DISP_REP_PIPE_TABLE
AECC_DISP_REP_POINT_ENT
AECC_DISP_REP_POINT_GROUP
AECC_DISP_REP_POINT_TABLE
AECC_DISP_REP_PRESSURE_PART_TABLE
AECC_DISP_REP_PRESSURE_PIPE
AECC_DISP_REP_PRESSURE_PIPE_CSV
AECC_DISP_REP_PRESSURE_PIPE_LABELING
AECC_DISP_REP_PRESSUREPIPENETWORK
AECC_DISP_REP_PRESSURE_PIPE_PROFILE_LABELING
AECC_DISP_REP_PRESSURE_PIPE_SECTION_LABELING
AECC_DISP_REP_PROFILE
AECC_DISP_REP_PROFILEDATA_BAND_LABEL_GROUP
AECC_DISP_REP_PROFILE_PROJECTION
AECC_DISP_REP_PROFILE_PROJECTION_LABEL
AECC_DISP_REP_PROFILE_VIEW
AECC_DISP_REP_PROFILE_VIEW_DEPTH_LABEL
AECC_DISP_REP_QUANTITY_TAKEOFF_AGGREGATE_EARTHWORK_TABLE
AECC_DISP_REP_RIGHT_OF_WAY
AECC_DISP_REP_SAMPLE_LINE
AECC_DISP_REP_SAMPLE_LINE_GROUP
AECC_DISP_REP_SAMPLELINE_LABELING
AECC_DISP_REP_SECTION
AECC_DISP_REP_SECTIONALDATA_BAND_LABEL_GROUP
AECC_DISP_REP_SECTION_CORRIDOR
AECC_DISP_REP_SECTION_CORRIDOR_POINT_LABEL_GROUP
AECC_DISP_REP_SECTIONDATA_BAND_LABEL_GROUP
AECC_DISP_REP_SECTION_GRADEBREAK_LABEL_GROUP
AECC_DISP_REP_SECTION_MINOR_OFFSET_LABEL_GROUP
AECC_DISP_REP_SECTION_OFFSET_LABEL_GROUP
AECC_DISP_REP_SECTION_PIPENETWORK
AECC_DISP_REP_SECTION_PRESSUREPIPENETWORK
AECC_DISP_REP_SECTION_PROJECTION
AECC_DISP_REP_SECTION_PROJECTION_LABEL
AECC_DISP_REP_SECTIONSEGMENT_BAND_LABEL_GROUP
AECC_DISP_REP_SECTION_SEGMENT_LABEL_GROUP
AECC_DISP_REP_SECTION_VIEW
AECC_DISP_REP_SECTION_VIEW_DEPTH_LABEL
AECC_DISP_REP_SECTION_VIEW_QUANTITY_TAKEOFF_TABLE
AECC_DISP_REP_SHEET
AECC_DISP_REP_SPANNING_PIPE_LABELING
AECC_DISP_REP_SPANNING_PIPE_PROFILE_LABELING
AECC_DISP_REP_STATION_ELEV_LABEL
AECC_DISP_REP_STRUCTURE
AECC_DISP_REP_STRUCTURE_CSV
AECC_DISP_REP_STRUCTURE_LABELING
AECC_DISP_REP_STRUCTURE_PROFILE_LABELING
AECC_DISP_REP_STRUCTURE_SECTION_LABELING
AECC_DISP_REP_SUBASSEMBLY
AECC_DISP_REP_SUPERELEVATION_BAND_LABEL_GROUP
AECC_DISP_REP_SUPERELEVATION_DIAGRAM_VIEW
AECC_DISP_REP_SURFACE_CONTOUR_LABEL_GROUP
AECC_DISP_REP_SURFACE_ELEVATION_LABEL
AECC_DISP_REP_SURFACE_SLOPE_LABEL
AECC_DISP_REP_SURVEY_FIGURE_LABEL_GROUP
AECC_DISP_REP_SVFIGURE
AECC_DISP_REP_SVFIGURE_CSV
AECC_DISP_REP_SVFIGURE_PROFILE
AECC_DISP_REP_SVFIGURE_SECTION
AECC_DISP_REP_SVFIGURE_SEGMENT_LABEL
AECC_DISP_REP_SVNETWORK
AECC_DISP_REP_TANGENT_INTERSECTION_TABLE
AECC_DISP_REP_TIN_SURFACE
AECC_DISP_REP_TIN_SURFACE_CSV
AECC_DISP_REP_VALIGNMENT_CRESTCURVE_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_CSV
AECC_DISP_REP_VALIGNMENT_HAGEOMPT_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_LINE_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_MINOR_STATION_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_PVI_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_SAGCURVE_LABEL_GROUP
AECC_DISP_REP_VALIGNMENT_STATION_LABEL_GROUP
AECC_DISP_REP_VERTICALGEOMETRY_BAND_LABEL_GROUP
AECC_DISP_REP_VIEW_FRAME
AECC_DISP_REP_VIEWFRAME_LABELING
AECC_FEATURELINE_STYLE
AECC_FEATURE_STYLE
AECC_FITTING_STYLE
AECC_FORMAT_MANAGER_OBJECT
AECC_GRADEVIEW
AECC_GRADING_CRITERIA
AECC_GRADING_CRITERIA_SET
AECC_GRADING_GROUP
AECC_GRADING_STYLE
AECC_IMPORT_STORM_SEWER_DEFAULTS
AECC_INTERFERENCE_STYLE
AECC_INTERSECTION_STYLE
AECC_LABEL_COLLECTOR_STYLE
AECC_LABEL_NODE
AECC_LABEL_RADIAL_LINE_STYLE
AECC_LABEL_TEXT_ITERATOR_CURVE_OR_SPIRAL_STYLE
AECC_LABEL_TEXT_ITERATOR_STYLE
AECC_LABEL_TEXT_STYLE
AECC_LABEL_VECTOR_ARROW_STYLE
AEC_CLASSIFICATION_DEF
AEC_CLASSIFICATION_SYSTEM_DEF
AEC_CLEANUP_GROUP_DEF
AECC_LEGEND_TABLE_STYLE
AECC_MASS_HAUL_LINE_STYLE
AECC_MASS_HAUL_VIEW_STYLE
AECC_MATCHLINE_STYLE
AECC_MATERIAL_STYLE
AECC_NETWORK_PART_CATALOG_DEF_NODE
AECC_NETWORK_PART_FAMILY_ITEM
AECC_NETWORK_PART_LIST
AECC_NETWORK_RULE
AECC_PARCEL_NODE
AECC_PARCEL_STYLE
AECC_PART_SIZE_FILTER
AECC_PIPE_RULES
AECC_PIPE_STYLE
AECC_PIPE_STYLE_EXTENSION
AECC_POINTCLOUD_STYLE
AECC_POINT_STYLE
AECC_POINTVIEW
AECC_PRESSURE_PART_LIST
AECC_PRESSURE_PIPE_STYLE
AECC_PROFILE_DESIGN_CHECK_SET
AECC_PROFILE_LABEL_SET
AECC_PROFILESECTIONENTITY_STYLE
AECC_PROFILE_STYLE
AECC_PROFILE_VIEW_BAND_STYLE_SET
AECC_PROFILE_VIEW_DATA_BAND_STYLE
AECC_PROFILE_VIEW_HORIZONTAL_GEOMETRY_BAND_STYLE
AECC_PROFILE_VIEW_PIPE_NETWORK_BAND_STYLE
AECC_PROFILE_VIEW_SECTIONAL_DATA_BAND_STYLE
AECC_PROFILE_VIEW_STYLE
AECC_PROFILE_VIEW_SUPERELEVATION_DIAGRAM_BAND_STYLE
AECC_PROFILE_VIEW_VERTICAL_GEOMETRY_BAND_STYLE
AECC_QUANTITY_TAKEOFF_CRITERIA
AECC_ROADWAYLINK_STYLE
AECC_ROADWAYMARKER_STYLE
AECC_ROADWAYSHAPE_STYLE
AECC_ROADWAY_STYLE_SET
AECC_ROOT_SETTINGS_NODE
AECC_SAMPLE_LINE_GROUP_STYLE
AECC_SAMPLE_LINE_STYLE
AECC_SECTION_LABEL_SET
AECC_SECTION_STYLE
AECC_SECTION_VIEW_BAND_STYLE_SET
AECC_SECTION_VIEW_DATA_BAND_STYLE
AECC_SECTION_VIEW_ROAD_SURFACE_BAND_STYLE
AECC_SECTION_VIEW_STYLE
AECC_SETTINGS_NODE
AECC_SHEET_STYLE
AECC_SLOPE_PATTERN_STYLE
AECC_STATION_FORMAT_STYLE
AECC_STRUCTURE_RULES
AECC_STUCTURE_STYLE
AECC_SUPERELEVATION_DIAGRAM_VIEW_STYLE
AECC_SURFACE_STYLE
AECC_SVFIGURE_STYLE
AECC_SVNETWORK_STYLE
AECC_TABLE_STYLE
AECC_TAG_MANAGER
AECC_TREE_NODE
AEC_CURTAIN_WALL_LAYOUT_STYLE
AECC_USER_DEFINED_ATTRIBUTE_CLASSIFICATION
AECC_VALIGNMENT_STYLE_EXTENSION
AECC_VIEW_FRAME_STYLE
AEC_CVSECTIONVIEW
AEC_DB_DISP_REP_DIM_GROUP_PLAN
AEC_DB_DISP_REP_DIM_GROUP_PLAN100
AEC_DB_DISP_REP_DIM_GROUP_PLAN50
AEC_DIM_STYLE
AEC_DISP_PROPS_2D_SECTION
AEC_DISP_PROPS_CLIP_VOLUME
AEC_DISP_PROPS_CLIP_VOLUME_RESULT
AEC_DISP_PROPS_DIM
AEC_DISP_PROPS_DISPLAYTHEME
AEC_DISP_PROPS_DOOR
AEC_DISP_PROPS_DOOR_NOMINAL
AEC_DISP_PROPS_DOOR_PLAN_100
AEC_DISP_PROPS_DOOR_PLAN_50
AEC_DISP_PROPS_DOOR_THRESHOLD_PLAN
AEC_DISP_PROPS_DOOR_THRESHOLD_SYMBOL_PLAN
AEC_DISP_PROPS_EDITINPLACEPROFILE_MODEL
AEC_DISP_PROPS_ENT
AEC_DISP_PROPS_ENT_REF
AEC_DISP_PROPS_GRID_ASSEMBLY_MODEL
AEC_DISP_PROPS_GRID_ASSEMBLY_PLAN
AEC_DISP_PROPS_LAYOUT_CURVE
AEC_DISP_PROPS_LAYOUT_GRID2D
AEC_DISP_PROPS_LAYOUT_GRID3D
AEC_DISP_PROPS_MASKBLOCK
AEC_DISP_PROPS_MASS_ELEM_MODEL
AEC_DISP_PROPS_MASS_GROUP
AEC_DISP_PROPS_MATERIAL
AEC_DISP_PROPS_OPENING
AEC_DISP_PROPS_POLYGON_MODEL
AEC_DISP_PROPS_POLYGON_TRUECOLOUR
AEC_DISP_PROPS_RAILING_MODEL
AEC_DISP_PROPS_RAILING_PLAN
AEC_DISP_PROPS_ROOF
AEC_DISP_PROPS_ROOFSLAB
AEC_DISP_PROPS_ROOFSLAB_PLAN
AEC_DISP_PROPS_SCHEDULE_TABLE
AEC_DISP_PROPS_SLAB
AEC_DISP_PROPS_SLAB_PLAN
AEC_DISP_PROPS_SLICE
AEC_DISP_PROPS_SPACE_DECOMPOSED
AEC_DISP_PROPS_SPACE_MODEL
AEC_DISP_PROPS_SPACE_PLAN
AEC_DISP_PROPS_STAIR_MODEL
AEC_DISP_PROPS_STAIR_PLAN
AEC_DISP_PROPS_STAIR_PLAN_OVERLAPPING
AEC_DISP_PROPS_WALL_GRAPH
AEC_DISP_PROPS_WALL_MODEL
AEC_DISP_PROPS_WALL_PLAN
AEC_DISP_PROPS_WALL_SCHEM
AEC_DISP_PROPS_WINDOW
AEC_DISP_PROPS_WINDOW_ASSEMBLY_SILL_PLAN
AEC_DISP_PROPS_WINDOW_NOMINAL
AEC_DISP_PROPS_WINDOW_PLAN_100
AEC_DISP_PROPS_WINDOW_PLAN_50
AEC_DISP_PROPS_WINDOW_SILL_PLAN
AEC_DISP_PROPS_ZONE
AEC_DISP_REP_2D_SECTION
AEC_DISPREPAECDBDISPREPMASSELEMPLAN100
AEC_DISPREPAECDBDISPREPMASSELEMPLAN50
AEC_DISPREPAECDBDISPREPMASSGROUPPLAN100
AEC_DISPREPAECDBDISPREPMASSGROUPPLAN50
AEC_DISPREPAECDBDISPREPOPENINGPLAN100
AEC_DISPREPAECDBDISPREPOPENINGPLAN50
AEC_DISPREPAECDBDISPREPOPENINGPLANREFLECTED
AEC_DISPREPAECDBDISPREPOPENINGSILLPLAN
AEC_DISP_REP_ANCHOR
AEC_DISP_REP_ANCHOR_BUB_TO_GRID
AEC_DISP_REP_ANCHOR_BUB_TO_GRID_MODEL
AEC_DISP_REP_ANCHOR_BUB_TO_GRID_RCP
AEC_DISP_REP_ANCHOR_ENT_TO_NODE
AEC_DISP_REP_ANCHOR_EXT_TAG_TO_ENT
AEC_DISP_REP_ANCHOR_TAG_TO_ENT
AEC_DISP_REP_BDG_ELEVLINE_MODEL
AEC_DISP_REP_BDG_ELEVLINE_PLAN
AEC_DISP_REP_BDG_ELEVLINE_RCP
AEC_DISP_REP_BDG_SECTIONLINE_MODEL
AEC_DISP_REP_BDG_SECTIONLINE_PLAN
AEC_DISP_REP_BDG_SECTIONLINE_RCP
AEC_DISP_REP_BDG_SECTION_MODEL
AEC_DISP_REP_BDG_SECTION_SUBDIV
AEC_DISP_REP_CEILING_GRID
AEC_DISP_REP_CEILING_GRID_MODEL
AEC_DISP_REP_CEILING_GRID_RCP
AEC_DISP_REP_CLIP_VOLUME_MODEL
AEC_DISP_REP_CLIP_VOLUME_PLAN
AEC_DISP_REP_CLIP_VOLUME_RESULT
AEC_DISP_REP_CLIP_VOLUME_RESULT_SUBDIV
AEC_DISP_REP_COL_BLOCK
AEC_DISP_REP_COL_CIRCARC2D
AEC_DISP_REP_COL_CONCOINCIDENT
AEC_DISP_REP_COL_CONCONCENTRIC
AEC_DISP_REP_COL_CONEQUALDISTANCE
AEC_DISP_REP_COL_CONMIDPOINT
AEC_DISP_REP_COL_CONNECTOR
AEC_DISP_REP_COL_CONNORMAL
AEC_DISP_REP_COL_CONPARALLEL
AEC_DISP_REP_COL_CONPERPENDICULAR
AEC_DISP_REP_COL_CONSYMMETRIC
AEC_DISP_REP_COL_CONTANGENT
AEC_DISP_REP_COL_DIMANGLE
AEC_DISP_REP_COL_DIMDIAMETER
AEC_DISP_REP_COL_DIMDISTANCE
AEC_DISP_REP_COL_DIMLENGTH
AEC_DISP_REP_COL_DIMMAJORRADIUS
AEC_DISP_REP_COL_DIMMINORRADIUS
AEC_DISP_REP_COL_ELLIPARC2D
AEC_DISP_REP_COL_LAYOUTDATA
AEC_DISP_REP_COL_LINE2D
AEC_DISP_REP_COL_MODIFIER_ADD
AEC_DISP_REP_COL_MODIFIER_CUTPLANE
AEC_DISP_REP_COL_MODIFIER_EXTRUSION
AEC_DISP_REP_COL_MODIFIER_GROUP
AEC_DISP_REP_COL_MODIFIER_PATH
AEC_DISP_REP_COL_MODIFIER_SUBTRACT
AEC_DISP_REP_COL_MODIFIER_TRANSITION
AEC_DISP_REP_COL_POINT2D
AEC_DISP_REP_COL_PROFILE
AEC_DISP_REP_COLUMN_GRID
AEC_DISP_REP_COLUMN_GRID_MODEL
AEC_DISP_REP_COLUMN_GRID_RCP
AEC_DISP_REP_COL_WORKPLANE
AEC_DISP_REP_COL_WORKPLANE_REF
AEC_DISP_REP_CONFIG
AEC_DISP_REP_CURTAIN_WALL_LAYOUT_MODEL
AEC_DISP_REP_CURTAIN_WALL_LAYOUT_PLAN
AEC_DISP_REP_CURTAIN_WALL_UNIT_MODEL
AEC_DISP_REP_CURTAIN_WALL_UNIT_PLAN
AEC_DISP_REP_DCM_DIMRADIUS
AEC_DISP_REP_DISPLAYTHEME
AEC_DISP_REP_DOOR_ELEV
AEC_DISP_REP_DOOR_MODEL
AEC_DISP_REP_DOOR_NOMINAL
AEC_DISP_REP_DOOR_PLAN
AEC_DISP_REP_DOOR_PLAN_50
AEC_DISP_REP_DOOR_PLAN_HEKTO
AEC_DISP_REP_DOOR_RCP
AEC_DISP_REP_DOOR_THRESHOLD_PLAN
AEC_DISP_REP_DOOR_THRESHOLD_SYMBOL_PLAN
AEC_DISP_REP_EDITINPLACEPROFILE
AEC_DISP_REP_ENT_REF
AEC_DISP_REP_LAYOUT_CURVE
AEC_DISP_REP_LAYOUT_GRID2D
AEC_DISP_REP_LAYOUT_GRID3D
AEC_DISP_REP_MASKBLOCK_REF
AEC_DISP_REP_MASKBLOCK_REF_RCP
AEC_DISP_REP_MASS_ELEM_MODEL
AEC_DISP_REP_MASS_ELEM_RCP
AEC_DISP_REP_MASS_ELEM_SCHEM
AEC_DISP_REP_MASS_GROUP_MODEL
AEC_DISP_REP_MASS_GROUP_PLAN
AEC_DISP_REP_MASS_GROUP_RCP
AEC_DISP_REP_MATERIAL
AEC_DISP_REP_MVBLOCK_REF
AEC_DISP_REP_MVBLOCK_REF_MODEL
AEC_DISP_REP_MVBLOCK_REF_RCP
AEC_DISP_REP_OPENING
AEC_DISP_REP_OPENING_MODEL
AEC_DISP_REP_POLYGON_MODEL
AEC_DISP_REP_POLYGON_TRUECOLOUR
AEC_DISP_REP_RAILING_MODEL
AEC_DISP_REP_RAILING_PLAN
AEC_DISP_REP_RAILING_PLAN_100
AEC_DISP_REP_RAILING_PLAN_50
AEC_DISP_REP_ROOF_MODEL
AEC_DISP_REP_ROOF_PLAN
AEC_DISP_REP_ROOF_RCP
AEC_DISP_REP_ROOFSLAB_MODEL
AEC_DISP_REP_ROOFSLAB_PLAN
AEC_DISP_REP_SCHEDULE_TABLE
AEC_DISP_REP_SET
AEC_DISP_REP_SLAB_MODEL
AEC_DISP_REP_SLAB_PLAN
AEC_DISP_REP_SLICE
AEC_DISP_REP_SPACE_DECOMPOSED
AEC_DISP_REP_SPACE_MODEL
AEC_DISP_REP_SPACE_PLAN
AEC_DISP_REP_SPACE_RCP
AEC_DISP_REP_SPACE_VOLUME
AEC_DISP_REP_STAIR_MODEL
AEC_DISP_REP_STAIR_PLAN
AEC_DISP_REP_STAIR_PLAN_100
AEC_DISP_REP_STAIR_PLAN_50
AEC_DISP_REP_STAIR_PLAN_OVERLAPPING
AEC_DISP_REP_STAIR_RCP
AEC_DISP_REP_WALL_GRAPH
AEC_DISP_REP_WALL_MODEL
AEC_DISP_REP_WALL_PLAN
AEC_DISP_REP_WALL_RCP
AEC_DISP_REP_WALL_SCHEM
AEC_DISP_REP_WINDOW_ASSEMBLY_MODEL
AEC_DISP_REP_WINDOW_ASSEMBLY_PLAN
AEC_DISP_REP_WINDOWASSEMBLY_SILL_PLAN
AEC_DISP_REP_WINDOW_ELEV
AEC_DISP_REP_WINDOW_MODEL
AEC_DISP_REP_WINDOW_NOMINAL
AEC_DISP_REP_WINDOW_PLAN
AEC_DISP_REP_WINDOW_PLAN_100
AEC_DISP_REP_WINDOW_PLAN_50
AEC_DISP_REP_WINDOW_RCP
AEC_DISP_REP_WINDOW_SILL_PLAN
AEC_DISP_REP_ZONE
AEC_DISPROPSMASSELEMPLANCOMMON
AEC_DISPROPSMASSGROUPPLANCOMMON
AEC_DISPROPSOPENINGPLANCOMMON
AEC_DISPROPSOPENINGPLANCOMMONHATCHED
AEC_DISPROPSOPENINGSILLPLAN
AEC_DISP_ROPS_RAILING_PLAN_100
AEC_DISP_ROPS_RAILING_PLAN_50
AEC_DISP_ROPS_STAIR_PLAN_100
AEC_DISP_ROPS_STAIR_PLAN_50
AEC_DOOR_STYLE
AEC_ENDCAP_STYLE
AEC_FRAME_DEF
AEC_LAYERKEY_STYLE
AEC_MASS_ELEM_STYLE
AEC_MATERIAL_DEF
AEC_MVBLOCK_DEF
AEC_MVBLOCK_REF
AEC_NOTIFICATION_TRACKER
AEC_POLYGON
AEC_POLYGON_STYLE
AEC_RAILING_STYLE
AEC_REFEDIT_STATUS_TRACKER
AEC_ROOFSLABEDGE_STYLE
AEC_ROOFSLAB_STYLE
AECS_DISP_PROPS_MEMBER
AECS_DISP_PROPS_MEMBER_LOGICAL
AECS_DISP_PROPS_MEMBER_PLAN
AECS_DISP_PROPS_MEMBER_PLAN_SKETCH
AECS_DISP_PROPS_MEMBER_PROJECTED
AECS_DISP_REP_MEMBER_ELEVATION_DESIGN
AECS_DISP_REP_MEMBER_ELEVATION_DETAIL
AECS_DISP_REP_MEMBER_LOGICAL
AECS_DISP_REP_MEMBER_MODEL_DESIGN
AECS_DISP_REP_MEMBER_MODEL_DETAIL
AECS_DISP_REP_MEMBER_PLAN_DESIGN
AECS_DISP_REP_MEMBER_PLAN_DETAIL
AECS_DISP_REP_MEMBER_PLAN_SKETCH
AEC_SLABEDGE_STYLE
AEC_SLAB_STYLE
AECS_MEMBER_NODE_SHAPE
AECS_MEMBER_STYLE
AEC_SPACE_STYLES
AEC_STAIR_STYLE
AEC_VARS_ARCHBASE
AEC_VARS_DWG_SETUP
AEC_VARS_MUNICH
AEC_VARS_STRUCTUREBASE
AEC_WALL_STYLE
AEC_WINDOW_ASSEMBLY_STYLE
AEC_WINDOW_STYLE
ALIGNMENTGRIPENTITY
ALIGNMENTPARAMETERENTITY
AMCONTEXTMGR
AM_DRAWING_MGR
AMDTADMENUSTATE
AMDTAMMENUSTATE
AMDTBROWSERDBTAB
AMDTDMMENUSTATE
AMDTEDGESTANDARDDIN
AMDTEDGESTANDARDDIN13715
AMDTEDGESTANDARDISO
AMDTEDGESTANDARDISO13715
AMDTFORMULAUPDATEDISPATCHER
AMDTINTERNALREACTOR
AMDTMCOMMENUSTATE
AMDTMENUSTATEMGR
AMDTNOTE
AMDTNOTETEMPLATEDB
AMDTSECTIONSYM
AMDTSECTIONSYMLABEL
AMDTSYSATTR
AM_DWG_DOCUMENT
AM_DWGMGR_NAME
AM_DWG_SHEET
AMGOBJPROPCFG
AMGSETTINGSOBJ
AMIMASTER
AM_VIEWDIMPARMAP
BINRECORD
BREAKDATA
CAMSCATALOGAPPOBJECT
CAMSSTRUCTBTNSTATE
CATALOGSTATE
CBROWSERAPPOBJECT
DEPMGR
DMBASEELEMENT
DMDEFAULTSTYLE
DMLEGEND
DMMAP
DMMAPMANAGER
DM_PLACEHOLDER
DMSTYLECATEGORY
DMSTYLELIBRARY
DMSTYLEREFERENCE
DMSTYLIZEDENTITIESTABLE
DMSURROGATESTYLESETS
EXACTERMXREFMAP
EXACXREFPANELOBJECT
EXPO_NOTIFYBLOCK
EXPO_NOTIFYHALL
EXPO_NOTIFYPILLAR
EXPO_NOTIFYSTAND
EXPO_NOTIFYSTANDNOPOLY
FLIPACTIONENTITY
FLIPGRIPENTITY
GSMANAGER
IRD_DSC_DICT
IRD_DSC_RECORD
IRD_OBJ_RECORD
LINEARGRIPENTITY
MAPFSMRVOBJECT
MAPGWSUNDOOBJECT
MAPIAMMOUDLE
MAPMETADATAOBJECT
MAPRESOURCEMANAGEROBJECT
McDbContainer2
McDbMarker
MOVEACTIONENTITY
NAMEDAPPL
NEWSTDPARTPARLIST
OBJCLONER
PARAMMGR
PARAMSCOPE
PILLAR
ROTATEACTIONENTITY
ROTATIONGRIPENTITY
SCALEACTIONENTITY
STDPART2D
STRETCHACTIONENTITY
TCH_ARROW
TCH_AXIS_LABEL
TCH_BLOCK_INSERT
TCH_COLUMN
TCH_DBCONFIG
TCH_DIMENSION2
TCH_DRAWINGINDEX
TCH_HANDRAIL
TCH_LINESTAIR
TCH_OPENING
TCH_RECTSTAIR
TCH_SLAB
TCH_SPACE
TCH_TEXT
TCH_WALL
TGrupoPuntos
VAACIMAGEINVENTORY
VAACXREFPANELOBJECT
XYGRIPENTITY
*/

/* Teigha does not export:
   ACAD_EVALUATION_GRAPH
   MENTALRAYRENDERSETTINGS
 */
