/*
 *  Some CBOR testcases converted from extras/cbor Makefile
 */

function dec(hexInput, expect) {
    var t1 = CBOR.decode(Duktape.dec('hex', hexInput));
    var t2 = Duktape.enc('jx', t1);
    var t3 = Duktape.enc('jx', expect);
    print(hexInput, t2, t3);
    if (t2 !== t3) {
        print('MISMATCH');
    }
}

function enc(input, hexExpect) {
    var t1 = CBOR.encode(input);
    var t2 = Duktape.enc('hex', t1);
    if (t2 !== hexExpect) {
        print('MISMATCH');
    }
    print(Duktape.enc('jx', input), t2, hexExpect);
}

/*
 *  Decode tests
 */

/*===
f90000 0 0
f98000 -0 -0
===*/

// Half-float zero.
dec('f90000', 0);
dec('f98000', -0);

/*===
f90001 5.960464477539063e-8 5.960464477539063e-8
f98001 -5.960464477539063e-8 -5.960464477539063e-8
f90002 1.1920928955078125e-7 1.1920928955078125e-7
f98002 -1.1920928955078125e-7 -1.1920928955078125e-7
f903ff 0.00006097555160522461 0.00006097555160522461
f983ff -0.00006097555160522461 -0.00006097555160522461
===*/

// Half-float denormals.
dec('f90001', 5.960464477539063e-08);
dec('f98001', -5.960464477539063e-08);
dec('f90002', 1.1920928955078125e-07);
dec('f98002', -1.1920928955078125e-07);
dec('f903ff', 6.097555160522461e-05);
dec('f983ff', -6.097555160522461e-05);

/*===
f90400 0.00006103515625 0.00006103515625
f98400 -0.00006103515625 -0.00006103515625
f907fe 0.00012195110321044922 0.00012195110321044922
f987fe -0.00012195110321044922 -0.00012195110321044922
f90800 0.0001220703125 0.0001220703125
f98800 -0.0001220703125 -0.0001220703125
f97800 32768 32768
f9f800 -32768 -32768
f97bff 65504 65504
f9fbff -65504 -65504
===*/

// Half-float normals.
dec('f90400', 6.103515625e-05);
dec('f98400', -6.103515625e-05);
dec('f907fe', 0.00012195110321044922);
dec('f987fe', -0.00012195110321044922);
dec('f90800', 0.0001220703125);
dec('f98800', -0.0001220703125);
dec('f97800', 32768);
dec('f9f800', -32768);
dec('f97bff', 65504);
dec('f9fbff', -65504);

/*===
f97c00 Infinity Infinity
f9fc00 -Infinity -Infinity
===*/

// Half-float inf.
dec('f97c00', 1/0);
dec('f9fc00', -1/0);

/*===
f97c01 NaN NaN
f9fc01 NaN NaN
f97e00 NaN NaN
f9fe00 NaN NaN
f97f12 NaN NaN
f9ff12 NaN NaN
===*/

// Half-float NaN.
dec('f97c01', 0/0);
dec('f9fc01', 0/0);
dec('f97e00', 0/0);
dec('f9fe00', 0/0);
dec('f97f12', 0/0);
dec('f9ff12', 0/0);

/*===
9f0102039fff0480809f1011ff8404030201ff [1,2,3,[],4,[],[],[16,17],[4,3,2,1]] [1,2,3,[],4,[],[],[16,17],[4,3,2,1]]
===*/

// Array.
dec('9f0102039fff0480809f1011ff8404030201ff', [1, 2, 3, [], 4, [], [], [16, 17], [4, 3, 2, 1]]);

/*===
bf63666f6f6362617263626172a06471757578bfff6362617aa26161616261636164657175757578bf61786179617a6177ffff {foo:"bar",bar:{},quux:{},baz:{a:"b",c:"d"},quuux:{x:"y",z:"w"}} {foo:"bar",bar:{},quux:{},baz:{a:"b",c:"d"},quuux:{x:"y",z:"w"}}
===*/

// Object.
dec('bf63666f6f6362617263626172a06471757578bfff6362617aa26161616261636164657175757578bf61786179617a6177ffff',
    {foo:'bar',bar:{},quux:{},baz:{a:'b',c:'d'},quuux:{x:'y',z:'w'}});

/*===
a163666f6f63626172 {foo:"bar"} {foo:"bar"}
a1a0a0 {"[object Object]":{}} {"[object Object]":{}}
7fff "" ""
7f606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060ff "" ""
7f60606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060ff "" ""
7f63666f6fff "foo" "foo"
7f63666f6f63626172617161756061756178ff "foobarquux" "foobarquux"
5fff || ||
5f4040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040ff || ||
5f404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040ff || ||
5f43666f6fff |666f6f| |666f6f|
5f43666f6f43626172417141754041754178ff |666f6f62617271757578| |666f6f62617271757578|
c063666f6f "foo" "foo"
c0c163666f6f "foo" "foo"
c0d763666f6f "foo" "foo"
c0d8ff63666f6f "foo" "foo"
===*/

// Misc.
dec('a163666f6f63626172', {foo:'bar'});
dec('a1a0a0', { '[object Object]': {} });  // key is {}, value is {}; current behavior string coerces key
dec('7fff', '');
dec('7f606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060ff', '');
dec('7f60606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060606060ff', '');
dec('7f63666f6fff', 'foo');
dec('7f63666f6f63626172617161756061756178ff', 'foobarquux');
dec('5fff', new Uint8Array([]));
dec('5f4040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040ff', new Uint8Array([]));
dec('5f404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040ff', new Uint8Array([]));
dec('5f43666f6fff', new Uint8Array([ 0x66, 0x6f, 0x6f ]));
dec('5f43666f6f43626172417141754041754178ff', new Uint8Array([ 0x66, 0x6f, 0x6f, 0x62, 0x61, 0x72, 0x71, 0x75, 0x75, 0x78 ]));
dec('c063666f6f', 'foo');  // tag ignored
dec('c0c163666f6f', 'foo');
dec('c0d763666f6f', 'foo');
dec('c0d8ff63666f6f', 'foo');

/*
 *  Encode tests
 */

/*===
undefined f7 f7
null f6 f6
true f5 f5
false f4 f4
===*/

// Some primitive types.
enc(void 0, 'f7');
enc(null, 'f6');
enc(true, 'f5')
enc(false, 'f4');

/*===
0 00 00
0.1 fb3fb999999999999a fb3fb999999999999a
1 01 01
23 17 17
24 1818 1818
25 1819 1819
255 18ff 18ff
256 190100 190100
65535 19ffff 19ffff
65536 1a00010000 1a00010000
65536.9 fb40f0000e66666666 fb40f0000e66666666
1048576 1a00100000 1a00100000
4294967295 1affffffff 1affffffff
4294967296 fa4f800000 fa4f800000
4294967297 fb41f0000000100000 fb41f0000000100000
3735928559 1adeadbeef 1adeadbeef
3.141592653589793 fb400921fb54442d18 fb400921fb54442d18
-0 f98000 f98000
-0.1 fbbfb999999999999a fbbfb999999999999a
-1 20 20
-23 36 36
-24 37 37
-25 3818 3818
-255 38fe 38fe
-256 38ff 38ff
-257 390100 390100
-65535 39fffe 39fffe
-65536 39ffff 39ffff
-65536.9 fbc0f0000e66666666 fbc0f0000e66666666
-65537 3a00010000 3a00010000
-1048576 3a000fffff 3a000fffff
-4294967295 3afffffffe 3afffffffe
-4294967296 3affffffff 3affffffff
-4294967297 fbc1f0000000100000 fbc1f0000000100000
-4294967298 fbc1f0000000200000 fbc1f0000000200000
-3735928559 3adeadbeee 3adeadbeee
-3.141592653589793 fbc00921fb54442d18 fbc00921fb54442d18
Infinity f97c00 f97c00
NaN f97e00 f97e00
-Infinity f9fc00 f9fc00
===*/

// Numbers.
enc(0, '00');
enc(0.1, 'fb3fb999999999999a');
enc(1, '01');
enc(23, '17');
enc(24, '1818');
enc(25, '1819');
enc(255, '18ff');
enc(256, '190100');
enc(65535, '19ffff');
enc(65536, '1a00010000');
enc(65536.9, 'fb40f0000e66666666');
enc(1048576, '1a00100000');
enc(4294967295, '1affffffff');
enc(4294967296, 'fa4f800000');  // Duktape uses IEEE single, Python uses 64-bit integer '1b0000000100000000'
enc(4294967297, 'fb41f0000000100000');  // Duktape uses IEEE double, Python uses 64-bit integer '1b0000000100000001'
enc(0xdeadbeef, '1adeadbeef');
enc(3.141592653589793, 'fb400921fb54442d18');
enc(-0, 'f98000');  // half-float is shortest for -0
enc(-0.1, 'fbbfb999999999999a');
enc(-1, '20');
enc(-23, '36');
enc(-24, '37');
enc(-25, '3818');
enc(-255, '38fe');
enc(-256, '38ff');
enc(-257, '390100');
enc(-65535, '39fffe');
enc(-65536, '39ffff');
enc(-65536.9, 'fbc0f0000e66666666');
enc(-65537, '3a00010000');
enc(-1048576, '3a000fffff');
enc(-4294967295, '3afffffffe');
enc(-4294967296, '3affffffff');  // 32-bit negative integer fits here unlike for positive equivalent
enc(-4294967297, 'fbc1f0000000100000');  // Duktape uses IEEE double, Python uses 64-bit integer '3b0000000100000000'
enc(-4294967298, 'fbc1f0000000200000');  // Duktape uses IEEE double, Python uses 64-bit integer '3b0000000100000001'
enc(-0xdeadbeef, '3adeadbeee');
enc(-3.141592653589793, 'fbc00921fb54442d18');
enc(1/0, 'f97c00');  // Shortest encoding is half-float.
enc(0/0, 'f97e00');  // -""-
enc(-1/0, 'f9fc00');  // -""-

/*===
1.5 f93e00 f93e00
0.00006103515625 f90400 f90400
0.000030517578125 fa38000000 fa38000000
-1.5 f9be00 f9be00
-0.00006103515625 f98400 f98400
-0.000030517578125 fab8000000 fab8000000
===*/

// Half floats.
enc(1.5, 'f93e00');
enc(1/16384, 'f90400');
enc(1/32768, 'fa38000000');  // No longer fits.
enc(-1.5, 'f9be00');
enc(-1/16384, 'f98400');
enc(-1/32768, 'fab8000000');

/*===
4294967295 1affffffff 1affffffff
4294967296 fa4f800000 fa4f800000
36811506361428870000 fa5fff6e5f fa5fff6e5f
36811507460940497000 fb43ffedcbf0000000 fb43ffedcbf0000000
-4294967296 3affffffff 3affffffff
-4294967297 fbc1f0000000100000 fbc1f0000000100000
-4295032832 facf800080 facf800080
-36811506361428870000 fadfff6e5f fadfff6e5f
-36811507460940497000 fbc3ffedcbf0000000 fbc3ffedcbf0000000
===*/

// Floats.
enc(0xffffffff, '1affffffff');
enc(0x100000000, 'fa4f800000');  // Duktape uses IEEE single, Python uses 64-bit integer '1b0000000100000000'
enc(0x1fedcbe0000000000, 'fa5fff6e5f');  // Duktape uses IEEE single, Python uses positive bignum byte string 'c24901fedcbe0000000000'
enc(0x1fedcbf0000000000, 'fb43ffedcbf0000000');  // Duktape uses IEEE double, Python uses positive bignum byte string 'c24901fedcbf0000000000'
// Unlike for positive integers, 0x100000000 encodes to a cbor integer (sint).
// Then -0x100000001 doesn't encode to a float because the mantissa is too long.
enc(-0x100000000, '3affffffff');
enc(-0x100000001, 'fbc1f0000000100000');  // Duktape uses IEEE double, Python uses 64-bit integer '3b0000000100000000'
enc(-0x100010000, 'facf800080');  // Duktape uses IEEE single, Python uses 64-bit integer '3b000000010000ffff'
enc(-0x1fedcbe0000000000, 'fadfff6e5f');  // Duktape uses IEEE single, Python uses negative bignum byte string 'c34901fedcbdffffffffff'
enc(-0x1fedcbf0000000000, 'fbc3ffedcbf0000000');  // Duktape uses IEEE double, Python uses negative bignum byte string 'c34901fedcbeffffffffff'

/*===
|| 40 40
|fe| 41fe 41fe
|c080| 42c080 42c080
|314a7a| 43314a7a 43314a7a
|00000000000000000000000000000000000000000000fe| 5700000000000000000000000000000000000000000000fe 5700000000000000000000000000000000000000000000fe
|0000000000000000000000000000000000000000000000fe| 58180000000000000000000000000000000000000000000000fe 58180000000000000000000000000000000000000000000000fe
|0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe| 58ff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe 58ff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe
|000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe| 590100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe 590100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe
|0100020003000400| 480100020003000400 480100020003000400
===*/

// Buffers.
enc(new Uint8Array([]), '40');
enc(new Uint8Array([ 0xfe ]), '41fe');
enc(new Uint8Array([ 0xc0, 0x80 ]), '42c080');
enc(new Uint8Array([ 0x31, 0x4a, 0x7a ]), '43314a7a');
enc(new Uint8Array([ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xfe ]), '5700000000000000000000000000000000000000000000fe');
enc(new Uint8Array([ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xfe ]), '58180000000000000000000000000000000000000000000000fe');
enc(new Uint8Array([
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xfe ]),
    '58ff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe');
enc(new Uint8Array([
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xfe ]),
    '590100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fe');
enc(new Uint16Array([ 1, 2, 3, 4 ]), '480100020003000400');  // Endian specific; not currently tagged

/*===
"" 60 60
"foo" 63666f6f 63666f6f
"foo\u20acbar" 69666f6fe282ac626172 69666f6fe282ac626172
"foo\ud800bar" 49666f6feda080626172 49666f6feda080626172
"foo\udc00bar" 49666f6fedb080626172 49666f6fedb080626172
"\u4321\u4321\u4321\u4321\u4321\u4321\u4321xy" 77e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca17879 77e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca17879
"\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyz" 7818e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a 7818e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a
"\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw" 7819e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a77 7819e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a77
"\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw......................................................................................................................................................................................................................................" 78ffe48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e 78ffe48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e
"\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw......................................................................................................................................................................................................................................!" 790100e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e21 790100e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e21
===*/

// Strings.
enc('', '60');
enc('foo', '63666f6f');
enc('foo\u20acbar', '69666f6fe282ac626172');
enc('foo\ud800bar', '49666f6feda080626172');  // Unpaired surrogate, invalid UTF-8, encodes as byte string
enc('foo\udc00bar', '49666f6fedb080626172');  // Unpaired surrogate, same behavior
enc('\u4321\u4321\u4321\u4321\u4321\u4321\u4321xy', '77e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca17879');
enc('\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyz', '7818e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a');
enc('\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw', '7819e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a77');
enc('\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw......................................................................................................................................................................................................................................', '78ffe48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e');
enc('\u4321\u4321\u4321\u4321\u4321\u4321\u4321xyzw......................................................................................................................................................................................................................................!', '790100e48ca1e48ca1e48ca1e48ca1e48ca1e48ca1e48ca178797a772e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e2e21');

/*===
[] 80 80
["foo","bar","quux"] 8363666f6f636261726471757578 8363666f6f636261726471757578
[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[{foo:123}]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]] 81818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181a163666f6f187b 81818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181a163666f6f187b
===*/

// Array.
enc([], '80');
enc([ 'foo', 'bar', 'quux' ], '8363666f6f636261726471757578');
// XXX: other lengths
enc([[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[{ foo: 123 }]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]], '81818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181a163666f6f187b');

/*===
{} a0 a0
{foo:123} a163666f6f187b a163666f6f187b
{foo:123,bar:234,quux:345} a363666f6f187b6362617218ea6471757578190159 a363666f6f187b6362617218ea6471757578190159
{foo:123,bar:["foo","bar",{baz:true}],quux:345} a363666f6f187b636261728363666f6f63626172a16362617af56471757578190159 a363666f6f187b636261728363666f6f63626172a16362617af56471757578190159
{a:1,b:2,c:3,d:4,e:5,f:6,g:7,h:8,i:9,j:10,k:11,l:12,m:13,n:14,o:15,p:16,q:17,r:18,s:19,t:20,u:21,v:22} b6616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616 b6616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616
{a:1,b:2,c:3,d:4,e:5,f:6,g:7,h:8,i:9,j:10,k:11,l:12,m:13,n:14,o:15,p:16,q:17,r:18,s:19,t:20,u:21,v:22,w:23} b7616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616617717 b7616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616617717
{a:1,b:2,c:3,d:4,e:5,f:6,g:7,h:8,i:9,j:10,k:11,l:12,m:13,n:14,o:15,p:16,q:17,r:18,s:19,t:20,u:21,v:22,w:23,x:24} bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f61701061711161721261731361741461751561761661771761781818ff bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f61701061711161721261731361741461751561761661771761781818ff
{a:1,b:2,c:3,d:4,e:5,f:6,g:7,h:8,i:9,j:10,k:11,l:12,m:13,n:14,o:15,p:16,q:17,r:18,s:19,t:20,u:21,v:22,w:23,x:24,y:25} bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819ff bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819ff
{a:1,b:2,c:3,d:4,e:5,f:6,g:7,h:8,i:9,j:10,k:11,l:12,m:13,n:14,o:15,p:16,q:17,r:18,s:19,t:20,u:21,v:22,w:23,x:24,y:25,z:26} bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819617a181aff bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819617a181aff
{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:123}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}} a163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6f187b a163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6f187b
===*/

// Object.
enc({}, 'a0');
enc({ foo: 123 }, 'a163666f6f187b');
enc({ foo: 123, bar: 234, quux: 345 }, 'a363666f6f187b6362617218ea6471757578190159');
enc({ foo: 123, bar: [ 'foo', 'bar', { baz: true } ], quux: 345 }, 'a363666f6f187b636261728363666f6f63626172a16362617af56471757578190159');
enc({ a:1, b:2, c:3, d:4, e:5, f:6, g:7, h:8, i:9, j:10, k:11, l:12, m:13, n:14, o:15, p:16, q:17, r:18, s:19, t:20, u:21, v:22 }, 'b6616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616');
enc({ a:1, b:2, c:3, d:4, e:5, f:6, g:7, h:8, i:9, j:10, k:11, l:12, m:13, n:14, o:15, p:16, q:17, r:18, s:19, t:20, u:21, v:22, w:23 }, 'b7616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f617010617111617212617313617414617515617616617717');
// After 23 properties Duktape currently switches to indefinite length encoding.
enc({ a:1, b:2, c:3, d:4, e:5, f:6, g:7, h:8, i:9, j:10, k:11, l:12, m:13, n:14, o:15, p:16, q:17, r:18, s:19, t:20, u:21, v:22, w:23, x:24 }, 'bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f61701061711161721261731361741461751561761661771761781818ff');
enc({ a:1, b:2, c:3, d:4, e:5, f:6, g:7, h:8, i:9, j:10, k:11, l:12, m:13, n:14, o:15, p:16, q:17, r:18, s:19, t:20, u:21, v:22, w:23, x:24, y:25 }, 'bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819ff');
enc({ a:1, b:2, c:3, d:4, e:5, f:6, g:7, h:8, i:9, j:10, k:11, l:12, m:13, n:14, o:15, p:16, q:17, r:18, s:19, t:20, u:21, v:22, w:23, x:24, y:25, z:26 }, 'bf616101616202616303616404616505616606616707616808616909616a0a616b0b616c0c616d0d616e0e616f0f6170106171116172126173136174146175156176166177176178181861791819617a181aff');
enc({foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:{foo:123}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}}, 'a163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6fa163666f6f187b');

/*===
{jsonrpc:"2.0",id:"foo-1",method:"Add",params:{a:123,b:234}} a4676a736f6e72706363322e3062696465666f6f2d31666d6574686f646341646466706172616d73a26161187b616218ea a4676a736f6e72706363322e3062696465666f6f2d31666d6574686f646341646466706172616d73a26161187b616218ea
===*/

// Misc.
enc({ jsonrpc: '2.0', id: 'foo-1', method: 'Add', params: { a: 123, b: 234 }}, 'a4676a736f6e72706363322e3062696465666f6f2d31666d6574686f646341646466706172616d73a26161187b616218ea');

/*===
done
===*/

print('done');
