/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alejandro J. Cura <alecu@canonical.com>
 *
 */

using Soup;
using Ubuntuone.Constants;

[DBus (name = "com.ubuntuone.CredentialsManagement")]
interface CredentialsManagement : GLib.Object {
    public signal void credentials_found (HashTable <string, string> info);
    public signal void credentials_not_found ();
    public signal void credentials_error (HashTable <string, string> error_dict);


    [DBus (name = "find_credentials")]
    public abstract void find_credentials () throws Error;
}

namespace Ubuntuone.Webservice
{
    public errordomain PurchaseError
    {
        MISSING_CREDENTIALS_ERROR,
        PURCHASE_ERROR,
        WRONG_PASSWORD_ERROR,
        UNSPECIFIED_ERROR
    }

    public class PurchaseService : GLib.Object
    {
        internal Soup.SessionAsync http_session;
        Soup.SessionAsync http_session_sso;
        CredentialsManagement credentials_management;
        public string nickname { get; private set; default = null; }
        public string email { get; private set; default = null; }
        public string selected_payment_method { get; internal set; default = null; }
        public string consumer_key { get; private set; default = null; }
        public string token { get; private set; default = null; }
        public string open_url { get; private set; default = null; }
        internal HashTable <string, string> _ubuntuone_credentials = null;

        construct {
            http_session = build_http_session ();
            http_session_sso = build_http_session ();

            credentials_management = build_credentials_management ();
        }

        internal Soup.SessionAsync build_http_session ()
        {
            var session = new Soup.SessionAsync ();
            session.user_agent = "%s/%s (libsoup)".printf("UbuntuOneMusicstoreLens", "1.0");
            return session;
        }

        public bool got_credentials () {
            return _ubuntuone_credentials != null;
        }

        internal virtual CredentialsManagement build_credentials_management ()
        {
            try {
                return Bus.get_proxy_sync (BusType.SESSION, "com.ubuntuone.Credentials",
                                           "/credentials", DBusProxyFlags.DO_NOT_AUTO_START);
            } catch (IOError e) {
                error ("Can't connect to DBus: %s", e.message);
            }
        }

        public bool ready_to_purchase {
            get { return selected_payment_method != null; }
        }

        internal Json.Object parse_json (string json_string) throws GLib.Error
        {
            var parser = new Json.Parser();
            parser.load_from_data(json_string, -1);
            return parser.get_root().get_object();
        }

        internal void parse_account_json (string json_string) throws GLib.Error
        {
            var root_object = parse_json (json_string);
            nickname = root_object.get_string_member("nickname");
            email = root_object.get_string_member("email");
        }

        internal void parse_payment_method_json (string json_string) throws GLib.Error, PurchaseError
        {
            var root_object = parse_json (json_string);
            if (root_object.has_member ("selected_payment_method")) {
                selected_payment_method = root_object.get_string_member("selected_payment_method");
            } else {
                open_url = root_object.get_string_member ("open_url");
                var error_message = root_object.get_string_member ("error_message");
                throw new PurchaseError.PURCHASE_ERROR (error_message);
            }
        }

        internal void parse_authentication_json (string json_string) throws GLib.Error
        {
            var root_object = parse_json (json_string);
            consumer_key = root_object.get_string_member ("consumer_key");
            token = root_object.get_string_member ("token");
        }

        internal string parse_purchase_json (string json_string) throws GLib.Error
        {
            var root_object = parse_json (json_string);
            if (root_object.has_member ("open_url")) {
                return root_object.get_string_member("open_url");
            } else {
                return "";
            }
        }

        internal virtual async void fetch_credentials () throws PurchaseError
        {
            PurchaseError error = null;

            ulong found_handler = credentials_management.credentials_found.connect ((credentials) => {
                _ubuntuone_credentials = credentials;
                debug ("got credentials");
                fetch_credentials.callback ();
            });
            ulong not_found_handler = credentials_management.credentials_not_found.connect (() => {
                error = new PurchaseError.MISSING_CREDENTIALS_ERROR ("No Ubuntu One tokens.");
                debug ("not found handler");
                fetch_credentials.callback ();
            });
            ulong error_handler = credentials_management.credentials_error.connect ((error_dict) => {
                error = new PurchaseError.MISSING_CREDENTIALS_ERROR ("Can't get Ubuntu One tokens.");
                debug ("error handler");
                fetch_credentials.callback ();
            });

            try {
                credentials_management.find_credentials ();
                yield;
            } catch (Error e) {
                error = new PurchaseError.MISSING_CREDENTIALS_ERROR ("Can't get Ubuntu One tokens: %s", e.message);
            }

            credentials_management.disconnect (found_handler);
            credentials_management.disconnect (not_found_handler);
            credentials_management.disconnect (error_handler);

            if (error != null) {
                debug ("Can't get Ubuntu One tokens: %s", error.message);
                throw error;
            }
        }

        string oauth_sign (string uri)
        {
            return OAuth.sign_url2(uri, null,
                                   OAuth.Method.PLAINTEXT, "GET",
                                   _ubuntuone_credentials["consumer_key"],
                                   _ubuntuone_credentials["consumer_secret"],
                                   _ubuntuone_credentials["token"],
                                   _ubuntuone_credentials["token_secret"]);
        }

        internal virtual async PurchaseError call_api (string method, string uri, out string response)
        {
            PurchaseError error = null;
            var signed_uri = oauth_sign (uri);
            var message = new Soup.Message (method, signed_uri);
            http_session.queue_message (message, (session, message) => {
                if (message.status_code != Soup.KnownStatusCode.OK) {
                    debug ("Web request failed: HTTP %u %s - %s",
                           message.status_code, message.reason_phrase, uri);
                    error = new PurchaseError.PURCHASE_ERROR (message.reason_phrase);
                }
                call_api.callback ();
            });
            yield;
            message.response_body.flatten ();
            response = (string) message.response_body.data;
            return error;
        }

        internal virtual async void fetch_account () throws PurchaseError
        {
            string response;
            PurchaseError error = yield call_api ("GET", account_uri(), out response);

            if (error != null) {
                debug ("Error while fetching U1 account: %s.", error.message);
                throw error;
            }

            try {
                parse_account_json (response);
                debug ("got account");
            } catch (GLib.Error e) {
                debug ("Error while parsing U1 account: %s.", e.message);
                throw new PurchaseError.PURCHASE_ERROR (e.message);
            }
        }

        internal virtual void fetch_payment_method (string purchase_sku) throws PurchaseError
        {
            var uri = payment_method_uri().printf (purchase_sku);

            var message = send_signed_webservice_call ("GET", uri);
            if (message.status_code != Soup.KnownStatusCode.OK) {
                debug ("Purchase request failed: HTTP %u", message.status_code);
                debug ("Reason: %s", message.reason_phrase);
                try {
                    message.response_body.flatten ();
                    debug ("body: ------\n%s\n------\n", (string) message.response_body.data);
                } catch (Error e) {
                }
                throw new PurchaseError.PURCHASE_ERROR ("Retrieve payment method failed: %s".printf (message.reason_phrase));
            }
            try {
                message.response_body.flatten ();
                var result = (string) message.response_body.data;
                parse_payment_method_json (result);
            } catch (GLib.Error e) {
                throw new PurchaseError.PURCHASE_ERROR (e.message);
            }
        }

        public virtual async void fetch_account_info () throws PurchaseError
        {
            yield fetch_credentials ();
            yield fetch_account ();
        }

        public virtual void fetch_payment_info (string purchase_sku) throws PurchaseError
        {
            fetch_payment_method (purchase_sku);
        }

        internal virtual void _do_sso_webcall (Soup.Message message, string password)
        {
            var handler = http_session_sso.authenticate.connect ((session, message, auth, retrying) => {
                if (!retrying) {
                    auth.authenticate (email, password);
                }
            });
            http_session_sso.send_message (message);
            http_session_sso.disconnect (handler);
        }

        internal virtual string authenticated_sso_webcall (string method, string uri, string operation, string password)
            throws PurchaseError
        {
            var message = new Soup.Message (method, uri);
            message.set_request ("application/x-www-form-urlencoded", Soup.MemoryUse.COPY, operation.data);
            _do_sso_webcall (message, password);
            if (message.status_code != Soup.KnownStatusCode.OK) {
                debug ("Authentication request failed: HTTP %u", message.status_code);
                debug ("Reason: %s", message.reason_phrase);
                if (message.status_code == Soup.KnownStatusCode.UNAUTHORIZED) {
                    throw new PurchaseError.WRONG_PASSWORD_ERROR ("Wrong password");
                }
                try {
                    message.response_body.flatten ();
                    debug ("body: ------\n%s\n------\n", (string) message.response_body.data);
                } catch (Error e) {
                }
                throw new PurchaseError.PURCHASE_ERROR (message.reason_phrase);
            }
            message.response_body.flatten ();
            return (string) message.response_body.data;
        }

        internal virtual string get_purchase_token (string password) throws PurchaseError
        {
            var result = authenticated_sso_webcall ("POST", authentication_uri(), AUTHENTICATE_PARAMS, password);
            try {
                parse_authentication_json (result);
            } catch (GLib.Error e) {
                throw new PurchaseError.PURCHASE_ERROR (e.message);
            }
            return "%s:%s".printf (consumer_key, token);
        }

        internal virtual Soup.Message send_signed_webservice_call (string method, string uri)
        {
            var signed_uri = oauth_sign (uri);
            var message = new Soup.Message (method, signed_uri);
            http_session.send_message (message);
            return message;
        }

        internal virtual void purchase_with_default_payment (string album_id, string purchase_token) throws PurchaseError
        {
            var uri = purchase_with_default_payment_uri().printf (album_id, purchase_token);
            var message = send_signed_webservice_call ("GET", uri);

            if (message.status_code != Soup.KnownStatusCode.OK) {
                debug ("Purchase request failed: HTTP %u", message.status_code);
                debug ("Reason: %s", message.reason_phrase);
                try {
                    message.response_body.flatten ();
                    debug ("body: ------\n%s\n------\n", (string) message.response_body.data);
                } catch (Error e) {
                }
                throw new PurchaseError.PURCHASE_ERROR ("Purchase failed: %s".printf (message.reason_phrase));
            }
            try {
                message.response_body.flatten ();
                var result = (string) message.response_body.data;
                var open_url = parse_purchase_json (result);
                if (open_url != "") {
                    throw new PurchaseError.PURCHASE_ERROR (open_url);
                }
            } catch (GLib.Error e) {
                throw new PurchaseError.PURCHASE_ERROR (e.message);
            }
        }

        public void purchase (string album_id, string password) throws PurchaseError
        {
            var purchase_token = get_purchase_token (password);
            debug ("purchasing...");
            purchase_with_default_payment (album_id, purchase_token);
            debug ("purchase completed.");
        }
    }
}
