; -*- mode: Lisp; fill-column: 400 -*-
;
; Poseidon Library
;
; Copyright (C) 2024 Eric McCarthy (bendyarm on GitHub)
;
;    Licensed under the Apache License, Version 2.0 (the "License");
;    you may not use this file except in compliance with the License.
;    You may obtain a copy of the License at
;
;      http://www.apache.org/licenses/LICENSE-2.0
;
;    Unless required by applicable law or agreed to in writing, software
;    distributed under the License is distributed on an "AS IS" BASIS,
;    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;    See the License for the specific language governing permissions and
;    limitations under the License.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "POSEIDON")

(include-book "std/util/defval" :dir :system)

(include-book "kestrel/crypto/primes/bn-254-group-prime" :dir :system)
(acl2::merge-io-pairs
 dm::primep
 (include-book "kestrel/crypto/primes/bls12-381-prime" :dir :system))

(include-book "kestrel/utilities/strings/chars-codes" :dir :system)
(include-book "kestrel/utilities/digits-any-base/core" :dir :system)
(include-book "std/testing/assert-equal" :dir :system)

(include-book "main-definition")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ poseidon-ingonyama-bn-254
  :parents (poseidon-instantiations)
  :short "Instantiation of Poseidon used by Ingonyama for the BN-254 scalar field."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is an instantiation of the Poseidon permutation used by Ingonyama,
     as described in the URL below.
     It uses the Poseidon permutation but not any sponge absorb or squeeze steps.
     The number of input field elements is constrained to be no more than
     @($t = c + r$), and they are loaded directly into both the capacity and
     rate field elements, left padded with zeros if needed to bring the total
     to @($t$) field elements.
     The output is a single field element taken from @($t[1]$), which is
     equivalent to the first element of the rate @($r[0]$).
     See "
    (xdoc::ahref "https://github.com/ingonyama-zk/poseidon-hash" "the Github repository")
    ".")
   (xdoc::p
    "This particular instantiation has 1 capacity and 2 rate field elements,
     8 full rounds, 57 partial rounds, and an alpha of 5.  The field is
     the prime order of the group defined by BN-254, which is also the base
     field in which the Baby Jubjub curve is defined.")
   (xdoc::p
    "There is no domain separation defined for this instantiation,
     so the caller must make sure to do their own padding and not
     to expect different results for inputs of [1] and [0,1], for example")
   (xdoc::p
    "Note, there is also a test case for this instantiation
     in the source file @('ingonyama-hash-tests.lisp')."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ark and mds matrices for ingonyama-bn-254-hash

; The ark matrix is #rounds rows, and t columns, where t = capacity + rate
; t = 3 and there are 8 + 57 = 65 rounds, so there are 3 * 65 = 195 elements.

(defconst *ingonyama-bn-254-ark*
  '((#x0ee9a592ba9a9518d05986d656f40c2114c4993c11bb29938d21d47304cd8e6e #x00f1445235f2148c5986587169fc1bcd887b08d4d00868df5696fff40956e864 #x08dff3487e8ac99e1f29a058d0fa80b930c728730b7ab36ce879f3890ecf73f5)
    (#x2f27be690fdaee46c3ce28f7532b13c856c35342c84bda6e20966310fadc01d0 #x2b2ae1acf68b7b8d2416bebf3d4f6234b763fe04b8043ee48b8327bebca16cf2 #x0319d062072bef7ecca5eac06f97d4d55952c175ab6b03eae64b44c7dbf11cfa)
    (#x28813dcaebaeaa828a376df87af4a63bc8b7bf27ad49c6298ef7b387bf28526d #x2727673b2ccbc903f181bf38e1c1d40d2033865200c352bc150928adddf9cb78 #x234ec45ca27727c2e74abd2b2a1494cd6efbd43e340587d6b8fb9e31e65cc632)
    (#x15b52534031ae18f7f862cb2cf7cf760ab10a8150a337b1ccd99ff6e8797d428 #x0dc8fad6d9e4b35f5ed9a3d186b79ce38e0e8a8d1b58b132d701d4eecf68d1f6 #x1bcd95ffc211fbca600f705fad3fb567ea4eb378f62e1fec97805518a47e4d9c)
    (#x10520b0ab721cadfe9eff81b016fc34dc76da36c2578937817cb978d069de559 #x1f6d48149b8e7f7d9b257d8ed5fbbaf42932498075fed0ace88a9eb81f5627f6 #x1d9655f652309014d29e00ef35a2089bfff8dc1c816f0dc9ca34bdb5460c8705)
    (#x04df5a56ff95bcafb051f7b1cd43a99ba731ff67e47032058fe3d4185697cc7d #x0672d995f8fff640151b3d290cedaf148690a10a8c8424a7f6ec282b6e4be828 #x099952b414884454b21200d7ffafdd5f0c9a9dcc06f2708e9fc1d8209b5c75b9)
    (#x052cba2255dfd00c7c483143ba8d469448e43586a9b4cd9183fd0e843a6b9fa6 #x0b8badee690adb8eb0bd74712b7999af82de55707251ad7716077cb93c464ddc #x119b1590f13307af5a1ee651020c07c749c15d60683a8050b963d0a8e4b2bdd1)
    (#x03150b7cd6d5d17b2529d36be0f67b832c4acfc884ef4ee5ce15be0bfb4a8d09 #x2cc6182c5e14546e3cf1951f173912355374efb83d80898abe69cb317c9ea565 #x005032551e6378c450cfe129a404b3764218cadedac14e2b92d2cd73111bf0f9)
    (#x233237e3289baa34bb147e972ebcb9516469c399fcc069fb88f9da2cc28276b5 #x05c8f4f4ebd4a6e3c980d31674bfbe6323037f21b34ae5a4e80c2d4c24d60280 #x0a7b1db13042d396ba05d818a319f25252bcf35ef3aeed91ee1f09b2590fc65b)
    (#x2a73b71f9b210cf5b14296572c9d32dbf156e2b086ff47dc5df542365a404ec0 #x1ac9b0417abcc9a1935107e9ffc91dc3ec18f2c4dbe7f22976a760bb5c50c460 #x12c0339ae08374823fabb076707ef479269f3e4d6cb104349015ee046dc93fc0)
    (#x0b7475b102a165ad7f5b18db4e1e704f52900aa3253baac68246682e56e9a28e #x037c2849e191ca3edb1c5e49f6e8b8917c843e379366f2ea32ab3aa88d7f8448 #x05a6811f8556f014e92674661e217e9bd5206c5c93a07dc145fdb176a716346f)
    (#x29a795e7d98028946e947b75d54e9f044076e87a7b2883b47b675ef5f38bd66e #x20439a0c84b322eb45a3857afc18f5826e8c7382c8a1585c507be199981fd22f #x2e0ba8d94d9ecf4a94ec2050c7371ff1bb50f27799a84b6d4a2a6f2a0982c887)
    (#x143fd115ce08fb27ca38eb7cce822b4517822cd2109048d2e6d0ddcca17d71c8 #x0c64cbecb1c734b857968dbbdcf813cdf8611659323dbcbfc84323623be9caf1 #x028a305847c683f646fca925c163ff5ae74f348d62c2b670f1426cef9403da53)
    (#x2e4ef510ff0b6fda5fa940ab4c4380f26a6bcb64d89427b824d6755b5db9e30c #x0081c95bc43384e663d79270c956ce3b8925b4f6d033b078b96384f50579400e #x2ed5f0c91cbd9749187e2fade687e05ee2491b349c039a0bba8a9f4023a0bb38)
    (#x30509991f88da3504bbf374ed5aae2f03448a22c76234c8c990f01f33a735206 #x1c3f20fd55409a53221b7c4d49a356b9f0a1119fb2067b41a7529094424ec6ad #x10b4e7f3ab5df003049514459b6e18eec46bb2213e8e131e170887b47ddcb96c)
    (#x2a1982979c3ff7f43ddd543d891c2abddd80f804c077d775039aa3502e43adef #x1c74ee64f15e1db6feddbead56d6d55dba431ebc396c9af95cad0f1315bd5c91 #x07533ec850ba7f98eab9303cace01b4b9e4f2e8b82708cfa9c2fe45a0ae146a0)
    (#x21576b438e500449a151e4eeaf17b154285c68f42d42c1808a11abf3764c0750 #x2f17c0559b8fe79608ad5ca193d62f10bce8384c815f0906743d6930836d4a9e #x2d477e3862d07708a79e8aae946170bc9775a4201318474ae665b0b1b7e2730e)
    (#x162f5243967064c390e095577984f291afba2266c38f5abcd89be0f5b2747eab #x2b4cb233ede9ba48264ecd2c8ae50d1ad7a8596a87f29f8a7777a70092393311 #x2c8fbcb2dd8573dc1dbaf8f4622854776db2eece6d85c4cf4254e7c35e03b07a)
    (#x1d6f347725e4816af2ff453f0cd56b199e1b61e9f601e9ade5e88db870949da9 #x204b0c397f4ebe71ebc2d8b3df5b913df9e6ac02b68d31324cd49af5c4565529 #x0c4cb9dc3c4fd8174f1149b3c63c3c2f9ecb827cd7dc25534ff8fb75bc79c502)
    (#x174ad61a1448c899a25416474f4930301e5c49475279e0639a616ddc45bc7b54 #x1a96177bcf4d8d89f759df4ec2f3cde2eaaa28c177cc0fa13a9816d49a38d2ef #x066d04b24331d71cd0ef8054bc60c4ff05202c126a233c1a8242ace360b8a30a)
    (#x2a4c4fc6ec0b0cf52195782871c6dd3b381cc65f72e02ad527037a62aa1bd804 #x13ab2d136ccf37d447e9f2e14a7cedc95e727f8446f6d9d7e55afc01219fd649 #x1121552fca26061619d24d843dc82769c1b04fcec26f55194c2e3e869acc6a9a)
    (#x00ef653322b13d6c889bc81715c37d77a6cd267d595c4a8909a5546c7c97cff1 #x0e25483e45a665208b261d8ba74051e6400c776d652595d9845aca35d8a397d3 #x29f536dcb9dd7682245264659e15d88e395ac3d4dde92d8c46448db979eeba89)
    (#x2a56ef9f2c53febadfda33575dbdbd885a124e2780bbea170e456baace0fa5be #x1c8361c78eb5cf5decfb7a2d17b5c409f2ae2999a46762e8ee416240a8cb9af1 #x151aff5f38b20a0fc0473089aaf0206b83e8e68a764507bfd3d0ab4be74319c5)
    (#x04c6187e41ed881dc1b239c88f7f9d43a9f52fc8c8b6cdd1e76e47615b51f100 #x13b37bd80f4d27fb10d84331f6fb6d534b81c61ed15776449e801b7ddc9c2967 #x01a5c536273c2d9df578bfbd32c17b7a2ce3664c2a52032c9321ceb1c4e8a8e4)
    (#x2ab3561834ca73835ad05f5d7acb950b4a9a2c666b9726da832239065b7c3b02 #x1d4d8ec291e720db200fe6d686c0d613acaf6af4e95d3bf69f7ed516a597b646 #x041294d2cc484d228f5784fe7919fd2bb925351240a04b711514c9c80b65af1d)
    (#x154ac98e01708c611c4fa715991f004898f57939d126e392042971dd90e81fc6 #x0b339d8acca7d4f83eedd84093aef51050b3684c88f8b0b04524563bc6ea4da4 #x0955e49e6610c94254a4f84cfbab344598f0e71eaff4a7dd81ed95b50839c82e)
    (#x06746a6156eba54426b9e22206f15abca9a6f41e6f535c6f3525401ea0654626 #x0f18f5a0ecd1423c496f3820c549c27838e5790e2bd0a196ac917c7ff32077fb #x04f6eeca1751f7308ac59eff5beb261e4bb563583ede7bc92a738223d6f76e13)
    (#x2b56973364c4c4f5c1a3ec4da3cdce038811eb116fb3e45bc1768d26fc0b3758 #x123769dd49d5b054dcd76b89804b1bcb8e1392b385716a5d83feb65d437f29ef #x2147b424fc48c80a88ee52b91169aacea989f6446471150994257b2fb01c63e9)
    (#x0fdc1f58548b85701a6c5505ea332a29647e6f34ad4243c2ea54ad897cebe54d #x12373a8251fea004df68abcf0f7786d4bceff28c5dbbe0c3944f685cc0a0b1f2 #x21e4f4ea5f35f85bad7ea52ff742c9e8a642756b6af44203dd8a1f35c1a90035)
    (#x16243916d69d2ca3dfb4722224d4c462b57366492f45e90d8a81934f1bc3b147 #x1efbe46dd7a578b4f66f9adbc88b4378abc21566e1a0453ca13a4159cac04ac2 #x07ea5e8537cf5dd08886020e23a7f387d468d5525be66f853b672cc96a88969a)
    (#x05a8c4f9968b8aa3b7b478a30f9a5b63650f19a75e7ce11ca9fe16c0b76c00bc #x20f057712cc21654fbfe59bd345e8dac3f7818c701b9c7882d9d57b72a32e83f #x04a12ededa9dfd689672f8c67fee31636dcd8e88d01d49019bd90b33eb33db69)
    (#x27e88d8c15f37dcee44f1e5425a51decbd136ce5091a6767e49ec9544ccd101a #x2feed17b84285ed9b8a5c8c5e95a41f66e096619a7703223176c41ee433de4d1 #x1ed7cc76edf45c7c404241420f729cf394e5942911312a0d6972b8bd53aff2b8)
    (#x15742e99b9bfa323157ff8c586f5660eac6783476144cdcadf2874be45466b1a #x1aac285387f65e82c895fc6887ddf40577107454c6ec0317284f033f27d0c785 #x25851c3c845d4790f9ddadbdb6057357832e2e7a49775f71ec75a96554d67c77)
    (#x15a5821565cc2ec2ce78457db197edf353b7ebba2c5523370ddccc3d9f146a67 #x2411d57a4813b9980efa7e31a1db5966dcf64f36044277502f15485f28c71727 #x002e6f8d6520cd4713e335b8c0b6d2e647e9a98e12f4cd2558828b5ef6cb4c9b)
    (#x2ff7bc8f4380cde997da00b616b0fcd1af8f0e91e2fe1ed7398834609e0315d2 #x00b9831b948525595ee02724471bcd182e9521f6b7bb68f1e93be4febb0d3cbe #x0a2f53768b8ebf6a86913b0e57c04e011ca408648a4743a87d77adbf0c9c3512)
    (#x00248156142fd0373a479f91ff239e960f599ff7e94be69b7f2a290305e1198d #x171d5620b87bfb1328cf8c02ab3f0c9a397196aa6a542c2350eb512a2b2bcda9 #x170a4f55536f7dc970087c7c10d6fad760c952172dd54dd99d1045e4ec34a808)
    (#x29aba33f799fe66c2ef3134aea04336ecc37e38c1cd211ba482eca17e2dbfae1 #x1e9bc179a4fdd758fdd1bb1945088d47e70d114a03f6a0e8b5ba650369e64973 #x1dd269799b660fad58f7f4892dfb0b5afeaad869a9c4b44f9c9e1c43bdaf8f09)
    (#x22cdbc8b70117ad1401181d02e15459e7ccd426fe869c7c95d1dd2cb0f24af38 #x0ef042e454771c533a9f57a55c503fcefd3150f52ed94a7cd5ba93b9c7dacefd #x11609e06ad6c8fe2f287f3036037e8851318e8b08a0359a03b304ffca62e8284)
    (#x1166d9e554616dba9e753eea427c17b7fecd58c076dfe42708b08f5b783aa9af #x2de52989431a859593413026354413db177fbf4cd2ac0b56f855a888357ee466 #x3006eb4ffc7a85819a6da492f3a8ac1df51aee5b17b8e89d74bf01cf5f71e9ad)
    (#x2af41fbb61ba8a80fdcf6fff9e3f6f422993fe8f0a4639f962344c8225145086 #x119e684de476155fe5a6b41a8ebc85db8718ab27889e85e781b214bace4827c3 #x1835b786e2e8925e188bea59ae363537b51248c23828f047cff784b97b3fd800)
    (#x28201a34c594dfa34d794996c6433a20d152bac2a7905c926c40e285ab32eeb6 #x083efd7a27d1751094e80fefaf78b000864c82eb571187724a761f88c22cc4e7 #x0b6f88a3577199526158e61ceea27be811c16df7774dd8519e079564f61fd13b)
    (#x0ec868e6d15e51d9644f66e1d6471a94589511ca00d29e1014390e6ee4254f5b #x2af33e3f866771271ac0c9b3ed2e1142ecd3e74b939cd40d00d937ab84c98591 #x0b520211f904b5e7d09b5d961c6ace7734568c547dd6858b364ce5e47951f178)
    (#x0b2d722d0919a1aad8db58f10062a92ea0c56ac4270e822cca228620188a1d40 #x1f790d4d7f8cf094d980ceb37c2453e957b54a9991ca38bbe0061d1ed6e562d4 #x0171eb95dfbf7d1eaea97cd385f780150885c16235a2a6a8da92ceb01e504233)
    (#x0c2d0e3b5fd57549329bf6885da66b9b790b40defd2c8650762305381b168873 #x1162fb28689c27154e5a8228b4e72b377cbcafa589e283c35d3803054407a18d #x2f1459b65dee441b64ad386a91e8310f282c5a92a89e19921623ef8249711bc0)
    (#x1e6ff3216b688c3d996d74367d5cd4c1bc489d46754eb712c243f70d1b53cfbb #x01ca8be73832b8d0681487d27d157802d741a6f36cdc2a0576881f9326478875 #x1f7735706ffe9fc586f976d5bdf223dc680286080b10cea00b9b5de315f9650e)
    (#x2522b60f4ea3307640a0c2dce041fba921ac10a3d5f096ef4745ca838285f019 #x23f0bee001b1029d5255075ddc957f833418cad4f52b6c3f8ce16c235572575b #x2bc1ae8b8ddbb81fcaac2d44555ed5685d142633e9df905f66d9401093082d59)
    (#x0f9406b8296564a37304507b8dba3ed162371273a07b1fc98011fcd6ad72205f #x2360a8eb0cc7defa67b72998de90714e17e75b174a52ee4acb126c8cd995f0a8 #x15871a5cddead976804c803cbaef255eb4815a5e96df8b006dcbbc2767f88948)
    (#x193a56766998ee9e0a8652dd2f3b1da0362f4f54f72379544f957ccdeefb420f #x2a394a43934f86982f9be56ff4fab1703b2e63c8ad334834e4309805e777ae0f #x1859954cfeb8695f3e8b635dcb345192892cd11223443ba7b4166e8876c0d142)
    (#x04e1181763050e58013444dbcb99f1902b11bc25d90bbdca408d3819f4fed32b #x0fdb253dee83869d40c335ea64de8c5bb10eb82db08b5e8b1f5e5552bfd05f23 #x058cbe8a9a5027bdaa4efb623adead6275f08686f1c08984a9d7c5bae9b4f1c0)
    (#x1382edce9971e186497eadb1aeb1f52b23b4b83bef023ab0d15228b4cceca59a #x03464990f045c6ee0819ca51fd11b0be7f61b8eb99f14b77e1e6634601d9e8b5 #x23f7bfc8720dc296fff33b41f98ff83c6fcab4605db2eb5aaa5bc137aeb70a58)
    (#x0a59a158e3eec2117e6e94e7f0e9decf18c3ffd5e1531a9219636158bbaf62f2 #x06ec54c80381c052b58bf23b312ffd3ce2c4eba065420af8f4c23ed0075fd07b #x118872dc832e0eb5476b56648e867ec8b09340f7a7bcb1b4962f0ff9ed1f9d01)
    (#x13d69fa127d834165ad5c7cba7ad59ed52e0b0f0e42d7fea95e1906b520921b1 #x169a177f63ea681270b1c6877a73d21bde143942fb71dc55fd8a49f19f10c77b #x04ef51591c6ead97ef42f287adce40d93abeb032b922f66ffb7e9a5a7450544d)
    (#x256e175a1dc079390ecd7ca703fb2e3b19ec61805d4f03ced5f45ee6dd0f69ec #x30102d28636abd5fe5f2af412ff6004f75cc360d3205dd2da002813d3e2ceeb2 #x10998e42dfcd3bbf1c0714bc73eb1bf40443a3fa99bef4a31fd31be182fcc792)
    (#x193edd8e9fcf3d7625fa7d24b598a1d89f3362eaf4d582efecad76f879e36860 #x18168afd34f2d915d0368ce80b7b3347d1c7a561ce611425f2664d7aa51f0b5d #x29383c01ebd3b6ab0c017656ebe658b6a328ec77bc33626e29e2e95b33ea6111)
    (#x10646d2f2603de39a1f4ae5e7771a64a702db6e86fb76ab600bf573f9010c711 #x0beb5e07d1b27145f575f1395a55bf132f90c25b40da7b3864d0242dcb1117fb #x16d685252078c133dc0d3ecad62b5c8830f95bb2e54b59abdffbf018d96fa336)
    (#x0a6abd1d833938f33c74154e0404b4b40a555bbbec21ddfafd672dd62047f01a #x1a679f5d36eb7b5c8ea12a4c2dedc8feb12dffeec450317270a6f19b34cf1860 #x0980fb233bd456c23974d50e0ebfde4726a423eada4e8f6ffbc7592e3f1b93d6)
    (#x161b42232e61b84cbf1810af93a38fc0cece3d5628c9282003ebacb5c312c72b #x0ada10a90c7f0520950f7d47a60d5e6a493f09787f1564e5d09203db47de1a0b #x1a730d372310ba82320345a29ac4238ed3f07a8a2b4e121bb50ddb9af407f451)
    (#x2c8120f268ef054f817064c369dda7ea908377feaba5c4dffbda10ef58e8c556 #x1c7c8824f758753fa57c00789c684217b930e95313bcb73e6e7b8649a4968f70 #x2cd9ed31f5f8691c8e39e4077a74faa0f400ad8b491eb3f7b47b27fa3fd1cf77)
    (#x23ff4f9d46813457cf60d92f57618399a5e022ac321ca550854ae23918a22eea #x09945a5d147a4f66ceece6405dddd9d0af5a2c5103529407dff1ea58f180426d #x188d9c528025d4c2b67660c6b771b90f7c7da6eaa29d3f268a6dd223ec6fc630)
    (#x3050e37996596b7f81f68311431d8734dba7d926d3633595e0c0d8ddf4f0f47f #x15af1169396830a91600ca8102c35c426ceae5461e3f95d89d829518d30afd78 #x1da6d09885432ea9a06d9f37f873d985dae933e351466b2904284da3320d8acc)
    (#x2796ea90d269af29f5f8acf33921124e4e4fad3dbe658945e546ee411ddaa9cb #x202d7dd1da0f6b4b0325c8b3307742f01e15612ec8e9304a7cb0319e01d32d60 #x096d6790d05bb759156a952ba263d672a2d7f9c788f4c831a29dace4c0f8be5f)
    (#x054efa1f65b0fce283808965275d877b438da23ce5b13e1963798cb1447d25a4 #x1b162f83d917e93edb3308c29802deb9d8aa690113b2e14864ccf6e18e4165f1 #x21e5241e12564dd6fd9f1cdd2a0de39eedfefc1466cc568ec5ceb745a0506edc)
    (#x1cfb5662e8cf5ac9226a80ee17b36abecb73ab5f87e161927b4349e10e4bdf08 #x0f21177e302a771bbae6d8d1ecb373b62c99af346220ac0129c53f666eb24100 #x1671522374606992affb0dd7f71b12bec4236aede6290546bcef7e1f515c2320)
    (#x0fa3ec5b9488259c2eb4cf24501bfad9be2ec9e42c5cc8ccd419d2a692cad870 #x193c0e04e0bd298357cb266c1506080ed36edce85c648cc085e8c57b1ab54bba #x102adf8ef74735a27e9128306dcbc3c99f6f7291cd406578ce14ea2adaba68f8)
    (#x0fe0af7858e49859e2a54d6f1ad945b1316aa24bfbdd23ae40a6d0cb70c3eab1 #x216f6717bbc7dedb08536a2220843f4e2da5f1daa9ebdefde8a5ea7344798d22 #x1da55cc900f0d21f4a3e694391918a1b3c23b2ac773c6b3ef88e2e4228325161)))

(defconst *ingonyama-bn-254-mds*
  '((#x109b7f411ba0e4c9b2b70caf5c36a7b194be7c11ad24378bfedb68592ba8118b #x16ed41e13bb9c0c66ae119424fddbcbc9314dc9fdbdeea55d6c64543dc4903e0 #x2b90bba00fca0589f617e7dcbfe82e0df706ab640ceb247b791a93b74e36736d)
    (#x2969f27eed31a480b9c36c764379dbca2cc8fdd1415c3dded62940bcde0bd771 #x2e2419f9ec02ec394c9871c832963dc1b89d743c8c7b964029b2311687b1fe23 #x101071f0032379b697315876690f053d148d4e109f5fb065c8aacc55a0f89bfa)
    (#x143021ec686a3f330d5f9e654638065ce6cd79e28c5b3753326244ee65a1b1a7 #x176cc029695ad02582a70eff08a6fd99d057e12e58e7d7b6b16cdfabc8ee2911 #x19a3fc0a56702bf417ba7fee3802593fa644470307043f7773279cd71d25d5e0)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Made this a nullary function to make it easier to prevent expansion
; and easier to view in the debugger.
(define ingonyama-bn-254--parameters ()
  :short "Poseidon parameters for Ingonyama BN-254 rate 2 implementation."
  (make-param
   :prime primes::*bn-254-group-prime*
   :rate 2
   :capacity 1
   :alpha 5
   :full-rounds-half 4
   :partial-rounds 57
   :constants *ingonyama-bn-254-ark*
   :mds *ingonyama-bn-254-mds*
   :rate-then-capacity-p nil
   :ascending-p t
   :partial-first-p t)
  :returns (param paramp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define ingonyama-bn-254-hash ((inputs (fe-listp inputs (param->prime (ingonyama-bn-254--parameters)))))
  :guard (<= (len inputs) (param->size (ingonyama-bn-254--parameters)))
  :returns (output (fep output (param->prime (ingonyama-bn-254--parameters)))
                   :name fep-of-ingonyama-bn-254-hash
                    :hyp (and (fe-listp inputs (param->prime (ingonyama-bn-254--parameters)))
                              (<= (len inputs) (param->size (ingonyama-bn-254--parameters))))
                   :hints (("Goal" :use (:instance len-of-hashp
                                                   (inputs preimage)
                                                   (param (ingonyama-bn-254--parameters))
                                                   (count 2))
                                   :in-theory (e/d (len-of-hashp) ((:e INGONYAMA-BN-254--PARAMETERS)))
                                   :do-not-induct t)))
  :short "Hash no more than @($c + r$) inputs to a single field element output."
  (let ((preimage (append (repeat (- (param->size (ingonyama-bn-254--parameters)) (len inputs)) 0)
                          inputs)))
    
    ;; The state vector t is [c.., r..], and in this case (len c)=1 and (len r)=2.
    ;; The Ingonyama output is t[1].
    ;; Since hashp with a count argument of 1 returns t[0], that is insufficient.
    ;; We need to tell hashp to return 2 outputs and to take the second.
    (nth 1 (hashp preimage
                  (ingonyama-bn-254--parameters)
                  2)))
  :guard-hints  (("Goal" :in-theory (enable hashp)))
  )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ poseidon-ingonyama-bls-255
  :parents (poseidon-instantiations)
  :short "Instantiation of Poseidon used by Ingonyama for the BLS12-381 scalar field."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is an instantiation of the Poseidon permuation used by Ingonyama,
     as described in the URL below.
     It uses the Poseidon permutation but not any sponge absorb or squeeze steps.
     The number of input field elements is constrained to be no more than
     @($t = c + r$), and they are loaded directly into both the capacity and
     rate field elements, left padded with zeros if needed to bring the total
     to @($t$) field elements.
     The output is a single field element taken from @($t[1]$), which is
     equivalent to the first element of the rate @($r[0]$).
     See "
    (xdoc::ahref "https://github.com/ingonyama-zk/poseidon-hash" "the Github repository")
    ".")
   (xdoc::p
    "This particular instantiation has 1 capacity and 4 rate field elements,
     8 full rounds, 60 partial rounds, and an alpha of 5.  The field is
     the prime order of the group defined by BLS12-381, which is also the base
     field in which the Jubjub curve is defined.  Field elements require 255 bits.")
   (xdoc::p
    "There is no domain separation defined for this instantiation,
     so the caller must make sure to do their own padding and not
     to expect different results for inputs of [1] and [0,1], for example")
   (xdoc::p
    "Note, there is also a test case for this instantiation
     in the source file @('ingonyama-hash-tests.lisp')."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ark and mds matrices for ingonyama-bls-255-hash

; The ark matrix is #rounds rows, and t columns, where t = capacity + rate
; t = 5 and there are 8 + 60 = 68 rounds, so there are 5 * 68 = 340 elements.

(defconst *ingonyama-bls-255-ark*
  '((#x5ee52b2f39e240a4006e97a15a7609dce42fa9aa510d11586a56db98fa925158 #x3e92829ce321755f769c6fd0d51e98262d7747ad553b028dbbe98b5274b9c8e1 #x7067b2b9b65af0519cef530217d4563543852399c2af1557fcd9eb325b5365e4 #x725e66aa00e406f247f00002487d092328c526f2f5a3c456004a71cea83845d5 #x72bf92303a9d433709d29979a296d98f147e8e7b8ed0cb452bd9f9508f6e4711)
    (#x3d7e5deccc6eb706c315ff02070232127dbe99bc6a4d1b23e967d35205b87694 #x13558f81fbc15c2793cc349a059d752c712783727e1443c74098cd66fa12b78b #x686f2c6d24dfb9cddbbf717708ca6e04a70f0e077766a39d5bc5de5155e6fcb2 #x582bc59317a001ed75ffe1c225901d67d8d3764a70eb254f810afc895cbf231b #x076df166a42eae40f6df9e5908a54f69a77f4c507ea6dd07d671682cbc1a9534)
    (#x531f360b9640e565d580688ee5d09e2635997037e87129303bf8297459ab2492 #x30be41b5a9d8af19a5f922794008a263a121837bcbe113d59621ea30beefd075 #x39f57e4c8a1178d875210f820977f7fcd33812d444f88e471040676e3e591306 #x3514084b13bc0be636482204d9cddb072ee674c5cb1238890ee6206a3e7bf035 #x6372b6bc660daf6b04361caff785b46bbe59eb6a34ab93e23d6364e655dc3a36)
    (#x422af985e648814bec5af62c142828e002d4b014b702760106b0b90c50d11de5 #x3296e51f12e0f5c49747c1beb050ff320e2eb7422807eb0c157a372dba2ea013 #x3b76246abaf33b03dd5b589b80a7fac0ae7f1ad8a9623bb7cf7432c90e27358d #x0b40e7e02f5cb836c883c7cef72ec48e87c1808f7d829e2ee0bec0ee709f7409 #x2ee81b5c29c93b8a6e8871c01d0380a698e547475359b4a4befc22ed2232690f)
    (#x341ff90fc4a8afee9b74c464955ba9b357252e915b8d39ea7c1318eda718f54d #x55eddabde058f3b5e9dae90873ec9bd7b05927da36925e7dfb7bc290c1da125e #x6b34ad8cec56aae4595c403377cd2aa990a2f09b931f832781221965bb081b1c #x707de76df294fb845309d2160e1bdffebefd57a80c8658899e2c95e77254c752 #x05e9b152bfd4946b9c109f930eb01892f314597507d28c735a266f4277bb2a32)
    (#x1589a5cbcee13b696b6f0a1dbbabc08394ab00ed5a6ae6435020e9e3e2fc909a #x7116a5d027fe73fbc45bfc60fd875c3116fe3a567e830d1d2d38655223dbd7ec #x05382ee6ad97381eb3137f5a90ea13298dac6bc7c2204906044fafc01bfe6ae4 #x0900bcfe5e7c1b7d0aa80c714b7b2a0c1df7473362138a9dc5c552d11c1d0015 #x0513deb89d2e48fc729440dc08d0256a79cda84d511a04e0d92cce3c7e55a7c2)
    (#x6bbb5f1736d499fe3fda42ad40a2b124952ac35fe970ebde38c65cc20ad2afc8 #x5782ac68a8da0ba09f4d17e7e4b46caa4411a27e60be92168ce75bed95453e05 #x2d83f3324639c5d83a1ffcf6ac693eef98d8ea4877d547c62b304b0a9f4a0c28 #x16d3a13700ec503e29ca4d0c6342864595134408b6668bbf1766bb48d7f96cba #x318050e971e075931253b00430d35f89f40a88fc73d62150882a8e87149d7244)
    (#x7180760dd839d8bffbf9b1e26826cb4f6de65fa868a8143e1dc8c2b6ac6d1ac2 #x5cf2aa95907e59c4725cc17c8cf492f9a7eeef2de337ac227a983c444ae0e80e #x2b8345763484d7ec02d6ee267b7c737ca9de41e2186416bf91c65eb0cd11c0a4 #x055aa90aa60ef9b7f3c29c7500c64e6b85929220a6418dfad37ead3928059117 #x541d5e4be0967bf49a595c1d8290b750305a334f3347c01b57f8ba313170e1ca)
    (#x05c0a1f16f97f582caaf4338f018f869e8dd0fa32f007bad1a1a4780053d5817 #x01519e13858591aa93b9c1d7f849276ac1d2011b7fd19a475371c7968d9f52cd #x69c30d5a27f4dffa19c956c348287a704676d999f23044036b9e687a45a1a113 #x58c93b899aa53e06e82b6346e36338841ba7279d2b7a0ecd3aa20f292852936f #x06b8a12870a15479d41018fed6f1a29102ae23e13d0fbccec93ace48bdb9dc93)
    (#x33eda3c347379e61c2297aa1026682d22f95dc3c7e46e68ab3adb4b0939d76e2 #x187728045111275b93a1218a148ada85a1f6e2059c443ac7d61fe81e3130b89b #x397ec485c5a8b0c8a03ff543e9a9e5a4dc0dd4849fe955bb77b452e2e22c4f17 #x2f33f8de90f81248455d5a6592667092992be0468372addbaff664caa84cd2d5 #x061a1a458994ddf9f38c5edfbd737d3ceb05deaee685058b14943e7e9246ebca)
    (#x4b73ab5b9d35f47307b731e3cf1a1a22e7068e2744f2af0ef6bd78bf8aae4845 #x5578b7ad5f8d4f3b8e618af7d8d5ec8bf837d2d9486527fe2f9bf7464f8516ad #x50b4f055d860f89e12883209f847a4b1a2395fb419eb53c182dbb555c962255c #x0b2da770936d6c778be289557ddd2ca024b93fa38c5d4541344e883a69611813 #x47d8441e1ae7cb8ffc52a18c67afff3cf7543cad51605b2d4e2513f1e1868b68)
    (#x619da3bf44b42acd949ed572c9f3c195ed20b0b91bcd9e95ee3750d26f3b0ebd #x6c9e249e89b2b4cf9cd7772950e0cc9d06688d4f051095eafd116371ede49ab7 #x210bd3217a141c55877d4528a4e80d5d81d78de7addce85994082281a6250d4b #x4e1d8e4079c14c83847af6394d7dc23f33ebf71593379583ec574bf5c86ea9a6 #x699187330fc1d606e8b31b677651a2c7d1c87d4d001018031792cad0ad3f2826)
    (#x2946bfc0f45c1f1a0dc4c343a85259f6a6237f064481fe66eda76f01998a01ea #x5543e07588375c6d800e5e42d1bfd8b7a92a2a35d65b234ded85f879f82a3d66 #x660e9d0f2f866e8d12b40dd9d9c03cc8b9ca78600bd649f0fffb2c388dcc8b43 #x38f06c48d4dc53cb1b69619244cc2a610fdc4229ea316980dffe9131a72b4209 #x5c9a73a16521ddf463f9de314dd5f7255bc66add48297615b761f34e4636762d)
    (#x310931f0204c9936fe659e9ebbda832c930172130b3f5476c6c6ee5e7fef3e45 #x72eb1d833664d8989998af11441ac49654c12210b3465e5ac67a99679634a3af #x6981346585a2a466a9255841f710e1d083bdcc21c0aa6721745e158218767a94 #x0370a259836b3766d563ed3cdcf55ace52655111a1017d8c76eaf8f97e81d858 #x4f63c45a324b8b974c22a20a6c670eb62d47ef900541b63f1d362b8bbe4ec418)
    (#x6a4c7347121c2d4745ecffaad22281cc4d58ea74453b7d2b625b890190fdc7ad #x36d8869bb69a51ee99622af09d6878c5b715084b25f6e4560a7498557fe87fb5 #x18faa7f51e1b7a442f9123806872094c0de8a46a6d8402f31f0cde3fcb878394 #x3610d022aacbe58593e0d6aa7eefdca767f5ddfe7fa1fb9fb4f80225d82b617b #x3b5f13d6a8bbff31569bc6860087b2a4b361146a04ad5fc7396a3d0c59f68c1c)
    (#x40e919335051c6aaaee033745c41b6fa36739a097d94ce6eb075ec03da2a978b #x2f54586ab9b7886340f8ed5254f29128a85e2fb1e3725bf3c9cd8bddadc947f1 #x00606231b689a040363e5afc050f9fc9296d6c620a885eeaffe91be387cbe96c #x4b55696db6b0fa327527a76e6ab6b688561c879e53d858e4c90a1122210130e1 #x569c39bd78356991953aef4b1a01fdf71710bb05eea1f447c3e5efe13bd62894)
    (#x537f73fcaa256497a2582e45105f1dc10f39c7fce9b88cab5523af3f5f82dcd9 #x2d58d32120c25995cd0754ab9fdf9ad67d67623cfd1fcbf489f51fa6e6eee4a2 #x37cb0f655951fca18a4ccdddd4d8466f8839ba8e320a104cb47a59cd387d322f #x4e29d154430c9bced788d2eed8f3e01b5da24c1d3710e490bc40ee6d5903213c #x47597b7a9018192ef22d6dd24555af1c0c51d8a90b54d8a0bdc2df7967d7a28b)
    (#x4e01b43205fca0b4a32582abe600f3a326035fe7e028cb0569bac43c997b98ce #x0172ffdfba7e43ca807d5b5de7727b4e41706c1f2858c1e8a46c27ed3eae5ff2 #x2216dd907ab98c0d1e720a46ef83334a236d2c134ccf35ef8e889421e70ebe03 #x168709f668b635f03607a39390a0de71306d6430ce2babf7292d789d25c0f8d5 #x0ff6a3823440877dfd355dea80595e21115d0dfe3472cec4ad1437572cc6151d)
    (#x44e37699b3c72f50ec1a754c72e6fa3f5a074181dd63d189ba36447d34e536ff #x267298d2e46227f7f7f422e3059f18d83a8795731b13f6568ce54730cd3fe9ae #x1ecbe7a60848077203373441a5b09b44693a155fe226442259e37ac47209235a #x31cb23e6b5d7393577d5f5c3368c5bdd5b434ee6319f07e502031cc393d4eccb #x5d4c550c4a6eccd74b74d6279b3d9bc755084588156a1bef673657dc2116ecfc)
    (#x226056b5dec9afd19190ac48740c3b5ab1bb429b19f56894a3dec3f104d238c0 #x09077c021183dd37ad10451ded70d7ae6ec4819ae76ce23fb2a0be63e69907d9 #x53545c868ba0fbf0ed1ed7a24ec11b2ecfba5b37fd5cee80774e1ecdea991ed4 #x69521c33d148e678ca10b33103812cd27597c4a6cddbe83f4970d4b96e03304d #x01d5779be7477b96aac6532ef919e61c624072be54587e0698999dd5f460e446)
    (#x57875a44441d2f191ac7d8de42691ab55fd3401bbaf04b786ef0603b3edf2927 #x1d5c957da0832d5b94e76f7abdb190972774b594ed232810bfcafe5441839d37 #x1b678335a80fd045fc7ce1897aa129f67bd55ca9ca801bd88eb7cc868538bd7a #x31e69d706a5c1e011c1cb1809e5bf1857c90f9f50b9e1ae5ad36e4d3dcdbb7ed #x485df8462ed7a18de34aa6e99ecc9bbf2db075a096b56bc2943b76a99c4bb1a0)
    (#x1e46fdcbb3705f663a350e78f99024912d80c95779195807aae82cbb494ce9e4 #x441d0fa0e9cb86c3a2a1f87151681c603c3e028f1a0670be2149eed4f0a24f08 #x02a3caff274f40942062340ec1fae17c1b1e97c2f0fc7e847c90e9317fea2c0c #x4caf281080c0b2f2f638bf0f4859442f4c9da94e9994dada34c5c914130c1a9e #x444470c6c49b5b9a38181c3af20bcfea572450946135baea85cfd6b692fa6464)
    (#x6d5e07a13376fc883bea2dcdbad7f80b7780f231cdd33f5b98618f42cc49ec2f #x1b9470418a07d8c88c767d1e63e8d5cc7f810cc530db1340181ecbbb212e0f70 #x4134c8666c685b712f4aec72077c540ef4a041dcaa123caabd57b83fc6266f14 #x3d5d0489e27362db9bf0cc7217477d81d2a73e1a44edc43e32d43bb544287c9d #x71d7d4a91945e796f538f03b9324497489009ec1a0a403de062ed5bb4d7c2400)
    (#x646c3d732a94f722384ac266b41e06cf21bf24fb9426c9556d8ac9514f0875f7 #x4f860c9e5d9bb73057d93c207902d9e60fd6a7c779fde1ebf16b853dba1ea9ad #x05801566eb9e119e2f9ace565c9488cd999d66a5753eb4b9887363137baa09ab #x0263bdb8654cf1245ae4589370dfd5eeb109a50944eef54308566055b887ee01 #x4cc39561e65eb05cb8c83f9854750a9114a996eb23e6a0bb07d2d61f0baf0a62)
    (#x36b544778b2fdb94f808ad8d077b7f0b44f3bba515ecdf026919e2fed09a106d #x3fb1f7aec47cbe990151d4bf703c38349b95f409abdf0504e67c1a55ef82294c #x637e7eb19cf539aada7e48bc6b72e5ccb0e3f6913f18a0d55696dddfcb1b587a #x73bc630fcece6947fb81ac8e0f1f1671ed6042c3ef3bbb12ed554f28b48b46ec #x304b46f52d597b964fbec3fc0dceee442febe6131359e156c194ab7be2a11e6d)
    (#x067d85956dcfff7fd9f6a0fec505b7f4998e3d85672623677a6d974d6b111de6 #x65830d8053bf8afc0ba5274f1a4c4cce617fa624b480f13ed3eb369fbba78e67 #x6c32c101e08a962bd996d759a6c012a4d97aedaab9fc99c1fa735a16cd24dd44 #x11fb2d160e41a1845fd14578c617285081fb1a16a21b36cfd5065b30fac574e3 #x50aada39348c4736f6c59f7f053c488ed999a33ad23501d9c635aa03baf90db5)
    (#x5a5f0e3a32b260fbdfdc8c0eaf3a99396992b50b6dbb63a9d1e1ddf9c91d78d4 #x62c9f6d9aea355d358f2986ad487c2ae443122e1edfb076930865608d05c3b39 #x520cea06cee20150703a1c8000d4a5f22b3efeb9e34eb90bad0b4ff091b33683 #x6da4e4682545c1f4c0076f5845fbbcf48632a9c193a92593d12d248031f2c893 #x1ba5502cee2ea2d07a64f68f0a7492d2426382a5b9662d0410e086107399989b)
    (#x6ab843ca92240f8a82862da071d53f048272d55425907fc8d0e60dcccd5a1ea4 #x3f65c2dfa6bb39c1b291c40f810cc912015384a2a24fd322b6375e27bd069322 #x6a2df71a64cb0d9a548e3b65ba4e646ff5e519cab564b5f77b3fe08e038b9c3a #x64776bf2b66bcd09c8661ee6ca6b8251bb4aba5a7ba181464d905db561ca45e1 #x6d7bed0d258b518eda13368f00be2cc0a94d71cc203d5905c35b10a3ee53eea8)
    (#x371b958b5c79c889d1786edfe404119773f728822637fb4890b8847a93f97af1 #x56923182c33cb4dbf0988ba2314378dfd7491b3467b6134e6283c87a1478cbb8 #x3c4304994ef664d6aa19e3db492c306534281b5b6f857fa6ffae67bdba99c09e #x0d003bd3068fa94c4f7bbe6ba02993acd341a27ed2fd7ecaa4e6b0b9d0abd85a #x1073cb8c08510e7d88ed4cdf78e96b297cabe9d6677db47289b056c2a640da01)
    (#x5c57522580fbc75883658d4b7b8ea07e1a4fc75f453c09edd9d249ff1bd31ae0 #x2a5bec9b422b4dc64958f4752d0c091ffa7904e0ce4809728d16235bb41d707f #x379c4a9b4174c5878f72b60fa985f7aa86c1fd868683bdbe8fae194cda2e56c7 #x3634e042e79d046adb911d57b338e78f51ac7d212c5a5c6dc4fa1a05ddb58c82 #x3ace976310c5040e1484d1a6d42993ac5923d474ce5497a3fac468af25843a01)
    (#x3f5a856ab863b7584bc2e6e4c610b9df55a9306eb68894d630ff7d04f243e6f5 #x0d52822f5581fe9c5dab0b1f8d04eae183deb87c89504544a3d5558594b3149b #x3c119e173586c22059bb09d2af4fc1044c8fc44f709233f7625e5fffa6696596 #x3e154fd5a026d7c6584faf8c089d82fd560f138392a8d4a5fe287859994c96b5 #x47251339c44d737b21df0ed1e204a28b68c9abb58f1cf2232f8a2da433e24b0b)
    (#x73d84625f38db2f3842d7724d8e79d6d0349a93b8d6142603eea382ba6ed8692 #x42929bffc19bf9cd1c53d10440b0760a3be6442db20458b692b4ba3901e6003f #x39b16b0fc3700aa93e0cac53fcaf7e84495ac3b49553b2e1a5ff9f73fe74de50 #x2b715e21640cfb6f77b91a4f6d3dcaef9b5faa7c0bfe94c8d80b0824292603bc #x306bef0c637b5d7c8d6486915f6623f4e1ed81971f40772ec60feb5e243d32a0)
    (#x5287d6ece65ef5df6e1c65dddf1d97cfa019157a5c90c004527c9d7c7496d814 #x0d760a2132c9092b0c8c89cbdf4fb1bd282791ef6284b73a44b313e8118e7d0c #x5e830f4484268a349e4d9f6178ef745460f1f8456b04d0dc7814844052d51eb5 #x2468669481610965d8439f60a66aa61fbc7b18e82b35aa4755873ec4db82174e #x23b6ea9e4d1fde701c719c2afab1272ea22b172bf7afe0837364ad9a2f698bd4)
    (#x412024b2e86e9d5e903a5fbda26200be47003e3b0dcc322480d3079850606cc0 #x1f64c17825c1ce9333d211d45a555b5ceaa4608a354ed3237db56225b3a9459b #x0b66fa87587ab95d5d29dde50cd606a1bc2c45fd223c03d0693c88b13ae23039 #x3086c386026698e733e54e5e17f65cb26c17fe64e76f85902cc184d5dd8ef0cf #x72036acd9ef575414d5437327d902da6396cc70c0bcffcef2a82b4c296b5ea93)
    (#x53d89e4470b3ea1eb861717e47c08fda42f6e61fc08118b16645ae5e8fdd664f #x4ebea65d1fc5c5167b1412ffcbf8900a8df2096c25d7011e6c74f500662465f8 #x5ee6e1e0312e78e2e67b246a95afdd79e2e7a5b9b0ef6ee36c3d1639f9736e65 #x1d770c0cc2c2231213624d58b7875b1715554f6100784bb2b545e405c7fcb94e #x2ea5c9837af445988c480fc6a55b1e5640dbe38d5e8cf1ddd85bc42c3658d9ca)
    (#x6fb78d12c35235f738b1667749064d0066fa7cfe3a9624cb0944f16d37bc485e #x35b75e89e794282cee1e66991ccfb2499dce4366b88d7be5f7b5775c12103a22 #x50e83b08162e7ccfe2d0f19aea4753ba83ef5c40572d6e904cfe2419ee9d901d #x3fc5c93031cbcecf12d5831aaa6b2b3071657cd669f7b377b2fef4a7bfc9adf2 #x37895bdfe29a174b98cd4b49104e56ea09e41c7b50f9aa95b400b529c545f5b4)
    (#x695e405509a0981035ba77e27cdcf53f3bc15d20fe4e43a335aeb6406ae1837d #x104985a48aa7e0a668d8cc7140c255ed1b8482ac5febbd3d7a1cca0e96cf0682 #x118220b30330f1954e7d94d40fb1043a1a79ca83e68e9ef590601a86a4a917a4 #x098b3be7845a63543c13d211efac076b94a9528d34cb355faf0ff7a0d5ee9991 #x69ca1313dcddd8c2f5c5c7ee93a1d2a94726c0c0bc4a303fcf83109b23bf3621)
    (#x570c1bd286b258b8bf11e8b85a2eb0c6dbfc2e4cdf01a0cde5464aa009b5bd43 #x4f2921de3696018e0d1ca7cdd5a4064ebf51845ab25b2d395b71c341ea8527da #x19035c69cbaf0e0e7e02c5c524a8cc56de0e52d1936a9a10b7580f0c0555878f #x2b8fdad2064a6f58d01e8c48d49bb25730780055829c1faead0430afcfbc5669 #x60ef9a74bbf8b98cb8248856492257f30c7520b3353a6fec9d90d48be46070ba)
    (#x4c9a6bc8284e783afd6c425f8cbdab82c0db3eac060a2dc00eca48ed6d1d052b #x68e6d3a83ac8e60c92d2860ff7557e1fbe3b91c38fabbde8b28371dccce2a10b #x56e0e39848046f0305d268b28aa753a41d48586e8579d5f95f12dba60e181d4c #x5176824fd8c92fed23df24c382a9fdf86aeeecab0b6716bef53da57bd3f551eb #x3aaf796b71041e8b2b494bca3b030f56a0c5663149093c8a179c0f3e24d0f718)
    (#x101cd65865abc573f5382df3636f4d60bc669aaa70f09ba040d61ef8d09c5296 #x2581f83d616d932b438bfe0062082d4e1ed7d34b9a1cf63580199731d44a4b25 #x65d74f6d1320dd1dc9412547b130bc7ad03c4e80cd8a44c108f24ec7aa35489a #x0d5cb6e19c9aac7d9f51f176ed42d008317a189dc4f6fc5c36fc6d451a035916 #x0e367d17423501e62db9fd487f72076f2d1de6dabd3c175341ce35f925c9941e)
    (#x3f3f101f7c8abd6bebe6b81dadf0ff5fa31ec7140e317909a8d2f94ce4adc890 #x6d5f212b5f4775095ab1d20fffd41dd73ab69b4ac60e9de11693f8e6bab88e67 #x6b11154212e86e185a4cb17dc2b9dc061f72bf9cc3df5f95f7b87f1101d09f1c #x43f4cf980ff1a9101ca3c4601814f8de4124d108be2584ee9ffb9505188d35fd #x5d9be9303e3a25e8fa1abb6f2a7e3250231091100f9d7311b050b52666ec8f02)
    (#x1eb3b147885e1261d9034ca89a658817caef5ae629e1265cd32c6ef89ce704e9 #x1595d95dac2c4653d32b01c3fbc294b2922140e41b93c5e7f5702212226d7140 #x578b22f1f6d6eeb61507f0de1c817bb876b9cd079a18be9e99e2faa8e02618e2 #x4de38f88c5e8ba1890b3695c912ccacd63721298c9ba3d3668b44f2a13b40abd #x0b9df0b81af072be21be9f08df336d3babe6ed5bfc199c73f2e97ccc73de80ae)
    (#x2a1a8c6d54abda22954e90386d40cc7d5c4f54c592ec2c69a9574601e88b6559 #x5c5d96136cd1c4ae8fa1db9273083567345b407eb66f73a313ab8ad1a76cb157 #x1ade9e2b734e937fc2fa04ca445236faf24e6d47ad1a4baa3408102c0d1e6363 #x49354c394824998704e44eeb2ba6cb6fb431c334b648e6c87565e5fe133e8079 #x4ea258f019a8055902a696b85547652519b8d8d92de4bf18e2dbfa41264a9a6e)
    (#x008a5162adf5ebd8711fd8139418509e472abc02908084f2e494086232336808 #x6badee92872dcc00812a1cbc8081dd65ece0c7d3512af5a9df5fed7428557c81 #x324c64ef2693e966965246bb7bb8c04b57a21fbee9be8c4a10096222bc83cc51 #x3f14138eee87c93b0fbfe7efdcfa906525b0ce0f3b9a7431a492f8cb71514219 #x0db99fa5ce25d50f557415ad181f1399840574f678b2534cae8f774bc8703009)
    (#x23d984702589f3275211041a4bde9d79329967723ec029da095bdbe733e97381 #x6c5144ace155e976e287f1b95951194895bac2e5d54b07b62c3afe0eeafcbe39 #x57a3e420fe7e0638bfb4d0b2c6286c2946166a6eb17836571909da153c3204de #x156621c4691a9240863577f10e29dc66a37d1b94e756869984c22d9f9d284726 #x1b1e774a7ec903650adffe34f6aa8201d356e41e0951d38fb83a89413d078e4b)
    (#x514b940e5717c1ae53ea29b9a5a15998e294f69c1f553fe56124f66a16a78d53 #x16350c6898d04d355d966c1d7827eee076a1ebd90781639e120feab665391ea9 #x5b8b30d8c5ae46c4171d40478886c71c28fc86a3ae4a52ad1c05d8bcb9991b52 #x5226cdc8a40c229ea4fb08f2c10e0e24cd41f24ca5fa5b5ab73e7340f632e727 #x64383db664537c84a0a4030c3318f2f19cbeda46c70460035ad9d9240011639d)
    (#x61068a086ab73c87701b2642af25f6a430240936ba473a9a258cbf90db275277 #x5bf320a3e8a48c6a85e2dffc4740d1b381ec4aa0771d885dc16adee569403ad3 #x2603e0fd03264a856c1a7b8f1c5a22c3b98f4858c345e8e0a68e3f6424dd2dfb #x100d221342e64ed7e4f1520be70f5b0134031f8a31b4790ebb8e0a89e50b42e2 #x0e61bad85ce909438ecc028b55085ec2cee0dd3ac5a7bcaa79d96186747a4606)
    (#x570a2045ca0fa7288d7f372f36bd075c2517a9743c9baa46503c4396e1f316f4 #x1a64e108621e134020ea761d8f2c5bb42f24fab7641b095f1d164d1fc7b8be90 #x097f0f28fd299e3597ffd761e9ae8b0fa46526c9d78503dc9dd5f61df3a085d7 #x1d1063cb1be0f9f96aca5e5e39be9df69c96ff717c7d0c7cfe179cd6dce27505 #x3e30f5d48b3c2475b8f3ba08cba27caed32b1cf67f76ba9223803733e13ad863)
    (#x2b30db4198cd832506017fa26430d204476113cc791ee110cf5586af5ce3824c #x2b520e374519be203c022ec51dcf8d972dd01abfaea371de9b1532647fca7bfd #x183b9a8e45fd480e822f8a97a8d2f127d0ef561914903229fbc5602bea46cb37 #x4e01e6edf11ef4c94fe8589f9622a70709330a12e68591f6ea7dda994117bdc8 #x52ee256fb3031d20fc299de7fabd0d4ef2e7f12539760dafb0fbc8560a40ee16)
    (#x327f5e141e4758d3d9a94c1628a57c817cf84fc0082b8dc098adbe84c1430979 #x3d0e12036899e5be167de13913901831a714ea5617b94de6de070ddc117bac71 #x1d9466d50efd1be3080d0aec4b81dd5cdf1ad4681e3ac04d08057f8fe49cdf0b #x2360abd7728da2dcda3f495a9a4f0f2aaff1d2420b8f6a7fed6592e1463f3d00 #x23c1df4ddd6da863a1a2837e5222150278adfd4faf2fae7beaf64ed67a30736c)
    (#x1e98ec3b325a2a11738273f94516a9d56107f33062661e571342bc043764cf77 #x431de5d108f8f7109df3059abcc16ccbd17e18676ef64f8998498e4a3f331fde #x550937f2bf0f1adb53f412d49ffd2886158703c375f87d059461f740d655e3d0 #x1341fa99aca4bfc0f511dc9a9bc57c1e7aeb41ebb3a9140f5f93af1b3aeeb582 #x706889448219016f970b32463a87e355b55ce0a34401dbfe4dd19fb3f93dec2e)
    (#x28d6207e409ab1c6e8e196d9e363040070b6c6fc4685a5482f80ba38cb792dc5 #x6827087ecdf4e6bc7c396c59de859cbf08f92c361b5174e7f681ba0e72f83aaa #x553e112dab620286f6cf2d31325b971a6516dc7776a6e5ef37bcb11d1785299d #x40b44f7413d152f0d46460c54e9572fd91174b4b94a3595d709119e49925354c #x4d324dd7dfdf2380ef9f6d3c4f4bc4c5f90dbbbf2f1fd923256913f33a45cc09)
    (#x609b3ae79dcdc8a8379a690394c95805d862bc31068b572ac126bbc082ebf8b7 #x33973520a1d9fb67048d64a22ad1b75b081d88c135a556dbc1b6a8479f75eaa7 #x3bcb7630fc45d34b78fd253d0b5275ecfa85ce48125ef7275c3a9205d01b85d8 #x1287f419048e81322d73bb9333e9b854e4ceac4b993b5342547263a486b42e34 #x2a2f5a5a689471d5ef46d669e449ccdc1d37256618722f08cc2c7e75d06fc277)
    (#x38c913fdc729a28b7e354947f2b6449029976d442e349bc1c2acf3b0fa28bc92 #x421826bc690adac2b1f3637bc5e2333cb5e4bce3f9e8eac1a0a76df32a7ebff7 #x30ac2452c3a07bb924b6f7ed47cd6581499d532c5f90bf7fbc69556ff3bf6b09 #x40ce93f92b281e538efbe7cec9a22a9c005eef428dde3cdd46191630f563ba04 #x4fc3dd6720c87f672f7b6ff129e9b2a3236ec760a71f78aee84925d8e7616e97)
    (#x3f3ba6f9f12ca6f934f92b17f4f3bd8ec261e5870610557f687bc734eadaa2d4 #x11d9eedda8d94fcbed859f5787fe20b7d4483cd319d8215530e2e316c89ee635 #x29981cff92be6c882c89feb59849d014fcd163699b5b4fdafca335552c4581d1 #x4c4fe2838d175c666c0d3f20d8dfefdcbcdacebca86e013d8ad29b6a0cf6bb79 #x630428a99469c03f9027d3c601864185d360d920771ea950732cf000b869a09a)
    (#x46a776fbf1f36d7fdfa7a210cbb2ffa533540068c169e12f127cb14d9b587056 #x41a775960677e6c5fdf73c2a409b6e5c08e271cbb8c825f598a1801c84fde5ae #x3086af931c41d791deb57f7f82dc511e4d349f42b52c3e0080097c4e44373dc8 #x155516da7a229b61392a39cc10a67112f512203cab706428f5fbbb3a9fd89fbd #x41bdb1e32081ac55f42969658f78e308bdf50175b619c3ca8e3bfdf1ca984684)
    (#x01344d21e02b9c20d0d886a02167cf8502c3614ab909ae2fa7929b12d3e88519 #x733a3e92f74b793915beab78e87bd88a2227aa5406df54dc9a2c5e80a11f71e5 #x6a6cc17a31ba2fe1411cdebeb0809bf4ff0069b0d6ac681edf816ef4c59b6f64 #x0a77e0a85b06c1b152098066bd36933264641627192e3acdbf611bd002918820 #x3efb107ebed9b44672f679bffec0121fb509d19e97ae1bac3a86384e274c8c94)
    (#x3c0c4b441b0ea7ffe03c011db9aab4f86ec4849a0c783a3b7af21b05f5654482 #x28072c7bfa64f6cb97e4341cd18809ef5cd083374fbec26370c2b0ac02dcdafe #x1962306e92b3c7295b2f7435ed8f67dda3a15ec6d8b0786d7727d071663ab22b #x594dc533611f7f588838f894a26b1cd27432c63f9fbe03ef2d95d9a2d191ae3f #x3e287fec491c686222949bc16c2308ade64e3a0a1dccdb25d64f9d5b94ead6e7)
    (#x2a95d47fb725b3978a7f90e601f2a9ab39074b35594e0bd133f9c5f34d765d42 #x29c603ecc031a9750a4d826e4abf3874bc76c76cc7ea306b3b9636f9653ff58c #x0bbff6ba283aa42f01172bb82a2838e50941227abc3a2a5b1215b9a6d68de07c #x73c7ee55aaa453d36ed857353bc227375244a7e554ceeea2018eb9cb39a51e74 #x3ff41b13d4cb3140ac8426322e88ff6f16895d88e6de3336cc88c693e0d38175)
    (#x03043688d4c991763362912a460be95b668fe9b1823fe90febfb3ffc7652ab24 #x33a29a0d56a7a64d36a67da2c691ff3eaf8ec7f0d78b357e7d2254c5b0e28f73 #x185db562fc75b43ba2710ad5e9114486b3e9712fe4c88f98b333c0c6211ac882 #x147b89a0cff9083b8952b3ef292c683f75d523f932711c6e1db3f28f5163b1fb #x58ebc5d6b50bb1e4fdb4dcdfae1b69027978826f757ee4dc10d34f963f98fb59)
    (#x1318791367815809badf1f3ed677e50cef92021c65549b2dabaa52c7b424f5a9 #x5bce78553694ba32f793c8d7f8d09ac63d0d7ada32b888d61b87849f3eda9557 #x026bebcc38f0b2804ed21f2e2b16af2194375ff2559fbc588a8962caf0b684c0 #x494bceff689f9885a3998de0eaaa7ac71a04522700f2e067efdbb037c6e53c66 #x03ebaf5f0602347c4ed2bdb9a86eb955cb5cd5378f7a6f369dccb69792de8bd2)
    (#x3626d91f9f05334cb32d3a42eed03f7a553a0ed4cada2db08b45b548bd3b3655 #x63ee9e5c5cd3c83e93757ed93358ff0583d761e595b62f11df27bd4292ffb6e5 #x705dd80b2db4492c8b9984439b823681c4d9c8dcddcc04b9786a90051513a0e1 #x2636ac2ac559be8fe509641dbc67e55db47bb051e05ef06301020c9501f110f1 #x4781b8da302c7764951730e7ac0892de64537d94db2e19b84eec5a2d9539288e)
    (#x197852b9a62e16779725f35cd8daf52ffbc8cc9c902c16923f2ff8873795ca86 #x1c3e49f33fd73480b280dba7744cf67e244449048f8fc84f7b6e452b4ede9a35 #x41d20cdc6a15c07fd9735c89b155412fcbb7bd3cdfc27abaad2a3a8a90e99743 #x0c3a7aaeb5f65d907944d7aa48c27648be3d0371bd97a9c060e8ef4f573521b8 #x52ea7c3f75cba07991674295c4e1462108401b9a103736623943d42e4fbe334e)
    (#x1106537bf3150b442b0992ee517b69707c3042015e938f97a63d5c924e67f677 #x71de967042516a5b990ef18ae9956fab89f361b950e0639963017c237ee2a0cf #x664a4487e02f7bfa07a1db6ab94a0d1ed0f9e74002bde9cfcbb65f6f74dbfca0 #x1023721fd7285260935b5a347f167ce721dd6ae5004c4debc68066bac8f2c467 #x2d52fbc95404515f5456c74b65186c860a89dcda8c84bf68fbf715f3d58fe3f2)
    (#x6d987c9de419fb6e075441fd99606303e765d8696bcfe01a0d11aa0bd47c8601 #x422016ce4d744029b1440a288d7988e43d0f29d616c47f70322ff87cfbc69301 #x1f82afe8eb16611abc6600f7dc2a72c8e1d39643c189f3caa1ead08241a896c4 #x3bb8684cf815ae6d8a789e0e488c6fb2ac46883fe1cfeb8cfa6f3dbca0f954bd #x3d5a1a6e571306fac431b098cdb3c4518f5a8fc436535766fe9e1bb8bda95d1d)
    (#x5e36e175c5d7df42b86285f43b1e4c6bfbaca19f1019073d38d04de0d0647669 #x2c3b1b86ce90cb3fe74c5c99b20c3314e28e2f07ce8d932030caee4dfe5055f1 #x0bfba44d41c49044bce730d8af86fe0397fff85ec10288b847868d0e9834f754 #x0b79924b9e44662369c615cc8d7f36fe4a4b2a79045cee61c413eaf91d82e0c2 #x048a11ec75eb154b70223a40cc0db9104b13f6a4ca24e7b9707963ee6f9f74ef)
    (#x6dd58a400d366014e46b0b9785ce9d78516813ed2eb329dc4531bfbd8e80eec0 #x112844b7c50e7e676b616e72539d5751dec5a063456921b6b16f9e930cc35ebc #x217b616b50e729547af8ceef5008d1edf8d90bc9a7f3ce7c9bc71867e1c06471 #x3f9a0b8402ffa291bccbb46dcd2522dea790b35a8503da46717c63917dcb7b79 #x42a44fc114c0cad9badf62b911610bdc4b1a0ba9f656f66173a5476e63dfce86)
    (#x294223972f4c7e9c9ebefebf059eb90f44479956f5337b12a2eb803e313e96cc #x448101837874eb1bda92bc8a632cbf8f70a0664bbcf3a196609b14c53ee4dbcb #x53a26c6e2b3df0b17faf6a259bc5531d3ae79da59eb8fc5f594e0b886d8d97be #x207c7c32631a75fe8e0da895367176d24e32c5573ec91acf235f3c6c307807cd #x20f955773b13b160d3575eb2380b466f7d38cb4a0e12a15d43d147645c3944ca)))

(defconst *ingonyama-bls-255-mds*
  '((#x354423b163d1078b0dd645be56316e34a9b98e52dcf9f469be44b108be46c107 #x44778737e8bc1154aca1cd92054a1e5b83808403705f7d54da88bbd1920e1053 #x5872eefb5ab6b2946556524168a2aebb69afd513a2fff91e50167b1f6e4055e0 #x43dff85b25129835819bc8c95819f1a34136f6114e900cd3656e1b9e0e13f86a #x07803d2ffe72940596803f244ac090a9cf2d3616546520bc360c7eed0b81cbf8)
    (#x45d6bc4b818e2b9a53e0e2c0a08f70c34167fd8128e05ac800651ddfee0932d1 #x08317abbb9e5046b22dfb79e64c8184855107c1d95dddd2b63ca10dddea9ff1a #x1bb80eba77c5dcffafb55ccba4ae39ac8f94a054f2a0ee3006b362f709d5e470 #x038e75bdcf8be7fd3a1e844c4de7333531bbd5a8d2c3779627df88e7480e7c5c #x2dd797a699e620ea6b31b91ba3fad4a82f40cffb3e8a30c0b7a546ff69a9002b)
    (#x4b906f9ee339b196e958e3541b555b4b53e540a113b2f1cabba627be16eb5608 #x605f0c707b82ef287f46431f9241fe4acf0b7ddb151803cbcf1e7bbd27c3e974 #x100c514bf38f6ff10df1c83bb428397789cfff7bb0b1280f52343861e8c8737e #x2d40ce8af8a252f5611701c3d6b1e517161d0549ef27f443570c81fcdfe3706b #x3e6418bdf0313f59afc5f40b4450e56881110ea9a0532e8092efb06a12a8b0f1)
    (#x71788bf7f6c0cebae5627c5629d012d5fba52428d1f25cdaa0a7434e70e014d0 #x55cc73296f7e7d26d10b9339721d7983ca06145675255025ab00b34342557db7 #x0f043b29be2def73a6c6ec92168ea4b47bc9f434a5e6b5d48677670a7ca4d285 #x62ccc9cdfed859a610f103d74ea04dec0f6874a9b36f3b4e9b47fd73368d45b4 #x55fb349dd6200b34eaba53a67e74f47d08e473da139dc47e44df50a26423d2d1)
    (#x45bfbe5ed2f4a01c13b15f20bba00ff577b1154a81b3f318a6aff86369a66735 #x6a008906685587af05dce9ad2c65ea1d42b1ec32609597bd00c01f58443329ef #x004feebd0dbdb9b71176a1d43c9eb495e16419382cdf7864e4bce7b37440cd58 #x09f080180ce23a5aef3a07e60b28ffeb2cf1771aefbc565c2a3059b39ed82f43 #x2f7126ddc54648ab6d02493dbe9907f29f4ef3967ad8cd609f0d9467e1694607)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Made this a nullary function to make it easier to prevent expansion
; and easier to view in the debugger.
(define ingonyama-bls-255--parameters ()
  :short "Poseidon parameters for Ingonyama BLS-255 rate 4 implementation."
  (make-param
   :prime primes::*bls12-381-scalar-field-prime*
   :rate 4
   :capacity 1
   :alpha 5
   :full-rounds-half 4
   :partial-rounds 60
   :constants *ingonyama-bls-255-ark*
   :mds *ingonyama-bls-255-mds*
   :rate-then-capacity-p nil
   :ascending-p t
   :partial-first-p t)
  :returns (param paramp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define ingonyama-bls-255-hash ((inputs (fe-listp inputs (param->prime (ingonyama-bls-255--parameters)))))
  :guard (<= (len inputs) (param->size (ingonyama-bls-255--parameters)))
  :returns (output (fep output (param->prime (ingonyama-bls-255--parameters)))
                   :name fep-of-ingonyama-bls-255-hash
                    :hyp (and (fe-listp inputs (param->prime (ingonyama-bls-255--parameters)))
                              (<= (len inputs) (param->size (ingonyama-bls-255--parameters))))
                   :hints (("Goal" :use (:instance len-of-hashp
                                                   (inputs preimage)
                                                   (param (ingonyama-bls-255--parameters))
                                                   (count 2))
                                   :in-theory (e/d (len-of-hashp) ((:e INGONYAMA-BLS-255--PARAMETERS)))
                                   :do-not-induct t)))
  :short "Hash no more than @($c + r$) inputs to a single field element output."
  (let ((preimage (append (repeat (- (param->size (ingonyama-bls-255--parameters)) (len inputs)) 0)
                          inputs)))
    
    ;; The state vector t is [c.., r..], and in this case (len c)=1 and (len r)=2.
    ;; The Ingonyama output is t[1].
    ;; Since hashp with a count argument of 1 returns t[0], that is insufficient.
    ;; We need to tell hashp to return 2 outputs and to take the second.
    (nth 1 (hashp preimage
                  (ingonyama-bls-255--parameters)
                  2)))
  :guard-hints  (("Goal" :in-theory (enable hashp)))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ poseidon-ingonyama-bls-255-neptune
  :parents (poseidon-instantiations)
  :short "Instantiation of Poseidon used by Ingonyama for Neptune."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is an instantiation of the Poseidon permuation used by Ingonyama,
     as described in the URL below.
     It uses the Poseidon permutation but not any sponge absorb or squeeze steps.
     The number of input field elements is constrained to be no more than
     @($t = c + r$), and they are loaded directly into both the capacity and
     rate field elements, left padded with zeros if needed to bring the total
     to @($t$) field elements.
     The output is a single field element taken from @($t[1]$), which is
     equivalent to the first element of the rate @($r[0]$).
     See "
    (xdoc::ahref "https://github.com/ingonyama-zk/poseidon-hash" "the Github repository")
    ".")
   (xdoc::p
    "This particular instantiation has 1 capacity and 3 rate field elements,
     8 full rounds, 56 partial rounds, and an alpha of 5.  The field is
     the prime order of the group defined by BLS12-381, which is also the base
     field in which the Jubjub curve is defined.  Field elements require 255 bits.")
   (xdoc::p
    "There can be a domain separation field used with this instantiation.
     However, it is applied to the input before calling the permutation,
     so we do not model it here.  A caller must model it separately.")
   (xdoc::p
    "Note, there is also a test case for this instantiation
     in the source file @('ingonyama-hash-tests.lisp')."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ark and mds matrices for ingonyama-bls-255-neptune

; The ark matrix is #rounds rows, and t columns, where t = capacity + rate
; t = 4 and there are 8 + 56 = 64 rounds, so there are 4 * 64 = 256 elements.

(defconst *ingonyama-bls-255-neptune-ark*
  '((#x435dbb70fe9639bb3d2e7e1948b167bbcc7c29bed7d24e2ae783b7258c3b9b79 #x307afe4a167ba0d1d93f60f15346bda015fa08615bc785bd204aee1741264d5 #x5c0e30ebca2f181197c0f06e98379ea11ca0b657bf1dde1060041f9e959945d2 #x181b9f96bd7efa33178ba5316e4441a392c2bb1e0d5437a8ff1613f5997cc4cf)
    (#x428c5fb2bb090a1139904e862c27c1de0beb6790719c72495e4b83053dcdc51e #x6966e075558905b9ce6122ae7c5bd2cdd5dc93f394ce03b64e2a6a6267b81206 #x521a2ac09efda0859ac79dc6b7948ed9d184b70a93397757c7e1e78cb95e2ce6 #xe09bd2528ba0b94d6ed86121aa48abf769d47c65138c5993a573528d2f09837)
    (#x1360abae0b804b8f70007cc8953ee7697a650fba556e3f0bdacb8376f4ad5519 #x616b4bfdf7c135daf2dda0fc23649e67ecadd04666c0c20b25abcaf4325dea5 #x30d52aeb1848a121c102f3cd4c26c3a8b33a012d9a5d5d370ec2469d6bb621be #x29179ed83ab1110b93767b25101a0ed78388651d7926f80d0bedaee66e4dfc37)
    (#x71e1b7a0d4251e03b897b26d64e19230a94d5579de0114e99172feacaa50f1c #x8c69e6eddea48915949a0c7aa08d1d01b5128f7eb998d97b870d9f51f1aecdc #xec55a6b06550b25d3df32a9c2e24ae00a89b1a19d65226986033c2bbad66e02 #x38891749192cdac572a4eb14bdef515ebcfa720fd7170230bacfbb62862401cc)
    (#x210e13978dc5983b3efc3c3643b0df191cb26f2eb0085079578b1498f6c6f19a #xe7df63365732c5484104c7db6676a77ce32e0a4a06f78a54d70e5fd655adbc0 #x2378638e5e2c069daf8135babf89ce5f62d5fd5f374bc6a85d5d8e835df9a8c3 #x429b476c33c844ffc774fdb4573a05fcf60f6c62f881b21dc2bedd7ce3d699d9)
    (#xc1776202f964b1050a843ec1c09d0b2737a3562b0ac9a35d4d341a36fee7b5e #x214769f3449c3ec3a1d274c932b9b20b7d58606f1b39ec4d7c550f90f4980d7e #x2440ca310753171d1ecb3276fcb1abe78329665f1ee54031f62f02238467fe6e #x41d7fc31fb80088dd2e5050e9ca265bf0d3367272b97abe417e3889ec3edc8f6)
    (#x6acceda1e5b88a6d5e62c93a248225b3d019f22564f7b5dc2aacf619ae2f8647 #x3bd0c24df838361cf463d5346262b4a83fbdeda25ead4bdf3a0ec683247546a2 #x1c46861bb307c05ef9744ab2f3f1121944e4c45034ee6ac8f0272e755d4b6f6d #xaff9e92f3b43678fe551bbd8f25c2613b03c0b8246079e180a6cd2cbb4c9c24)
    (#x65c86e5cf041918dd50f1459eae4988cdd99a4bf7d1babfd946ffa235de2e73c #x34456093491393fc5ae59ab7c7b6169c0b0939a52e085f495559c23a7c7e6c14 #x39135f352442c5fa32c3400534c994e279913fabfcd9373bca18052fcef15803 #x6354a55ec8726101d11a749214be3c6418b1816e577e08f1c1a2f435614896cb)
    (#xd3b0448c50b68f7baceaccd94552a122b89b786ca620b863c02a7a5bfe774cd #x37de17de5d77d1aa293a04e1d2a9ccf8f5f475f1f6d00b2a9e279e20b6df70c3 #x3f06cd79850df6108be46661a92292eee0981c98d05ec2f553b8e7a6cdd15a13 #x1190dda4cbcc050860d69d9b0841f371e6ee0f1f3e318a193a3d254c8ea19930)
    (#x4d486da7382a041ff02f0d19ac57a8d10a46363a3ff70fe6166ecc52d073902d #x2b6542eaeb87fc42fcd92147064a91a0214004b6d838afd7346a2716b3ef92cf #x57511ad2e251bb9c3510331ef75a449de6ab76044785522e2c58118073734374 #x3d8b02b647073d6bb96f2e9b27b0fb80748c9fda99003c8529460644a3999544)
    (#x4d01c4c679e545449955e318ee0d32c66969b9ef7351cbefcff2be6db93d6340 #xd148334ee5f03afe680a0bfbc479ebba90c149c251c38dc7214d192adc23d59 #x51cb09abe4ca48849b8ed609a35530063d1e108bef461b57cb4ab256d23c8715 #x6cb836cc464e1def59ee2eb012ece332dbe6e8d40fe5be20d467f1aac6463a64)
    (#x4aa865b463081e81767d3a0bb6d996b80b46e60d252d4acdc6a58b686981a4e1 #x6d74e3ccfee94f471b8e2906f26117fa6ae0a5ecbeb8d5d6be7670950c7ad08e #x31de4d8889e45640c1ab6cb3c1e448a59251868082c8661e23ca2494fe20db41 #xcb45744245e16f06ceaa2fb2a7a867250aeb0b72c1362dbd59e8c42fc272403)
    (#x2022672836eb3a87b07d2b8e8d3d54e84ef074c3f3711f812ff4b917f724c9bd #x14a3a3c5e0772147a6099b58ddea285408e653738c665389206678f49cde7ed0 #x64aaf5f888bb77306894f4197dd84f69ee9def0215594abdce5ea7981fd7b04a #x5966360244e5804391c0a967aa24682ed0d253169803f1de4d7a74610ad696fd)
    (#x72ab5d880b24076b9ab2bde88b7f6c33b0f7dbebacc7fbcb34b7a8257a7d96d7 #x6710e136713dc87ff3f9cb210aa1db11394f00ce3b255a9c7c717edb53a62107 #xb4a708d71ca90ea133e0532a3328c3957ce53f2b9e3fc6d17271d47d9d35b67 #x4c2375cffbedf2ecde0634f867383938a70cec4f409f728859244d9df7d15154)
    (#x604027adcb15669476ccba80d2d5c279f7284cadcfca2e5fea7a3dbf25a07ff9 #x4b3f8fdba7c3c15a60e060fb40c89ac3774a5d05563fbabd283049c681dc3e88 #x3e4ef282d1a982e9d194441da324ad190ebc272c35f87da4b493731d145ed9e3 #x3ca36c11fab63fbde4d2b6acd740791e802ebe6888ff85be9be64a34afdb96e0)
    (#x59b6ac07a1c1e77c48969c3ff79699ce5c341a4dd7716bab3711caefd1e08983 #x585c41ab8870f47026fcb4920d5a1dfc46afd9193f71f2d6abe9e13f4a37c00 #x3306f364dce1e440ad2351d33fbe5f15fbdf4d9ff17d27e8b21d14f00e95320a #x30a82fe3a10afe039f3e8c2e7546ff76a4a3d4541b58125dbdb0d2516fe8a6ec)
    (#x688ffb1af4d60ba856a3b19eb147ce65cd68ace786afda1bfc574c9b93250f6e #x1ce1b8d7866ead6b29cdce992ab8960686cf3a6bc7feb49c3d460abbcb069c0e #x2cbff76a00d623515166d5b581f0adc4f7f713eabfc757e853f667893c90a827 #x63a260c6e46d75512b405b32c6ec7f8acdd64dd6d53eae4e2e395cdd9f5c520d)
    (#x42223d517e729c29cd0043929a48b7e3ad9251375c3a869ff7ac95aa98445a0b #x4db292dea742799cf5ec29f166d16703d01f66e48c0d2566dff402af4a680a32 #x61740f9a82c6a4fa9abc516991f0e6b87f0d5a319bb8256026f184513919dffe #x66c2eaa8ca804300d86a3972bddb6a53817d1c36af548c7362f79da78aa6816b)
    (#x1519e66cc576cc14ce2956470e8cdb70420740ecabfa6181c7176a70e79af414 #x6ebf120fa7c372ce9c7c17a07bdda63e9ea3b2eeb5f4a7bac8d908e7f1209103 #x5395c9ae6af6ee4858f9344f430ee8208843fde1cf235f97b60bdc5a3e848995 #x1fa295097c1d41e6fbd9f72fb0af9b5aa3d1a8196821b0305527bf62979b97b0)
    (#x42ddc7f384cf8f928cde8f88f4ee7d7b62b9163ac70aca740f2bf7e52e38228c #x54811c99d6c88a6277fa41dafe17b7a4f2b8ac8582fe266c751fdf7d2a88fd40 #x3ccbabd0849789cc1c434dd0c7356a0ec71d03db3ab227aec51f2a5abbd8aacb #x40d6918f0d6d75cae13b6cc63e771b27ee189301d97a2a207019f1646dfbf810)
    (#x25a0fac67dce64b86695c1f9986dfd547f51b603f6d73b5c849c9548547e2b32 #x2307d40b97903f0f59f5df01de9ac9a823edc818619d555ec3d9882bf5ad2fca #x398549f8aad34cb56e6f66bc76f30b7e0a0379292e0c9db49ef928719e83c4ae #x562b6d5e4cae80684baf14b45f5216ac45e8dd10ff8d99531a94332966a7d8d6)
    (#xde2f7224c7db546a7aa0b7097c6d48846cbe5ab35adbdbc14a134581a399168 #x6645f7389026a44284598c2ad679a73a487db8ce965577e57f51361acb684c1b #x47987c690cb45bd23189fe76ae6fe562ca7757ebbd5f861e97c6493f20e3058c #x6c158b77a9457918cf699d0759b2588e8c3e2083645e4acbc679ffe60b4ed081)
    (#x3a8f448b6dfdbab048b5f193acd1d89b3c0d8777449a7c9e82836d71fce09cb #x6f82e6cb42fa93680f1058179be9ff2463a16bc2af5cfe85bba7b6f78b057b6f #x6fdc6c0c05fba5fc05e81673a730002393e23f5d6ac931dcae63255550b941a2 #x18fd1168c293c5075de568971d285277010e8cc19dc1bf17b2dae0a3a38a0bda)
    (#x61f681ade3a5eb4308a786c967a76ad142ea814f5fcd788cd326a18d861855ef #x46351a8e970814672c6c2a5b8a2c4bf93291d4a7086b321e5d774a214f6cee42 #x2b48e2821ea22551541577ada2edc1066077c766c1c0a84abeed33b2862624c6 #x3af5569027cd396d262908fbcaedc7aaebd9ce44995833b312607ad33cfc958)
    (#x6771e0cbfcd5854b869555fa79c9b46f0704f1ceddb2003538864ac645887231 #xc60fc4ac4bab581a11fea9b1b5e1e1e4692e78231faedc6719ada9a395e5c84 #x553b99c71556f4d8ed95945a96d6cc1bf70a0d2307f77d5e6ddcffe852278d5 #x14b6aea2c0ba14365b3a0c56e937338bb8555c415400e287f8b7bbe67b59b971)
    (#x5b82bbc5b5c301000e0ce4a34777f6df59ab8540338a32d1eb50426c9bb95309 #x35f07a3ebc8947fcc9a72c3838bdf80000c0caa346aff7d1fa45ddf3b5892df9 #x158c739d41772ef413fb5b7b21fcf0e8bfb9fbf9a4318f99d5467d05ece81587 #x2f7ebdafde0bbe158a265dee8b18a1a4a0151952d06d15f389b2e517292b96eb)
    (#x41da27c7cb76adb43dc3e4d1afc732aace59618f0e2eb0923372f66504ca3076 #xa80d961878f34b45b371a87b2ebb1f9b58f3fd2719e5376d802c238796b11e9 #x6d3885078660fac763b5982bd1f401043b5ca158341d24aff76b84b1df818290 #x12cec3917acb30cc1e15281aa7bf53d90419d216986623613fb10aca9be742f7)
    (#x58289c672e594998792c12773554138fe3292caace0bfbadf01cee48ab72240a #x31c1bcad4eef1a0d7a73bfc7291951c337860dd5730ffb7400bc9ec72a76de9e #x1a28349b10c85633174b26436ec2cbc372ed1afd4b29af83cc89cb94ef457891 #x5d6e8ad13be5518ec892ed1520c066a64ae425bd392c77b8843b8e842512fbe0)
    (#xa2a6963b53495afbd3d312c684c5babc0037651d8f180d35d66aef892b9d2b #x5572d3d64b5bc3a507568493f71cf86208611fc997e323d8e92bf909b07afe89 #x2f4d74cea3c17f17d4778ce57e59bc217f8a5856d189079f0b60a681c0e325f3 #x6689968d8d529724311b6fe789bde980cb92392a91182edc36c57b09f4405304)
    (#x4d51b3403c6ddcec7ac976e62754c65bc338d5ea651555a876285dfc24702141 #x6349b82e8184a0f1da93e07a6901fc23cf25bed7d6bedf709249947ee7ed5a35 #x18a791400b6eefa2b19c6f83e019f170aa6bd445fcf4915f915b621c1573ba44 #x333d6751bc4ab5d168a2ae3c6ceb237e26ae948bfb205e8b060d383cdd674fe3)
    (#x374d51e4dee080abb1a2cc25a988ee15320ce11fad84e8eecfe3cf5bae13863f #x2b42981ce6690c373ff43ed24e5c784ff2e192d0911ccffdc8262dc532de4867 #x4a8002ec5feb1b6bafb30fb78d65023254751977478d7af2c7fa0e87b83d926 #x2f82302976ba29681c7cbcf49d0ad1e1609c1e727e2a40cf30260d3ce57dd6c3)
    (#x2bf08176879b9da45776d2fd57988de0f12f33ce757411f27dbaa181f9cd9a6b #x39851bff4e488eecb8e1f59168682fa47c2d057aaefec12315e2dfcf7df5ee0 #x605a4bdeadac0722061fbb098c05f47ccdc26ebdb45bda575f0c9495d27f6de4 #x609c5a5b62e3eb56edbe6dcf30f937502434ff935f5564c0a5d0f686b8c4583a)
    (#x4d041c087422144212ec8978bcccbf1b8f471799bb8fb5c527f7bf4902fdee49 #x17e6b7fa91ce16fa1818c61b4871acbbed0e3dd902aae03afaedb7a14c65132f #x528644c51eee10eb8373d43a3980b12df839fa178de5791f4cc53fc7cfae9614 #x60616ade433fdfe048b713947f8b8475954331d393be923e772ef97ba70ebe31)
    (#x668f41b34ea071e41aad40bc6b04e5102eb2a15500d515c0d6e7013677005d0b #x143b1f278815a0285a453ac96e232fd54dc965b7af415a474cbf204a6f4c2654 #x493726bef63266b200039437e47ee655704f641c785c72de2a7f852653b2aa9f #x13dc1ab1544f728cccb110b5d8a3b47b3331e57cfa9882786adb86cc70e55e73)
    (#x6fdb4a1b633c3e8b52910d143bc78493505fc721a469ebae764aa9d5f2fbd2e3 #x10c11599636ea832ab3aca48b1a46c1b9a97dd0d5df68cfe2e6ebb2a0d5c1b2d #x3d333054b81e7f1a488baaf2972d6b41cd228e7e2fc44bd0dcadde4c50520882 #x416549b06324046bdc8ea26e337a0a6ddeb83bb76e85ed609f4ffcfee9556fdf)
    (#x4fc2c645eb8ef43ac68aaef6992cbb7ef2215af392f7cf0bb326efe04b029c3b #x3615cdc9e839c98533ce09ec9139c9fb703d65bee903667cdfbeaa374032becf #x3723d731de7ebc74575fbe5701cd94ba51dbdd04a37afdbfb917e8f75d52b9ef #x4a061269abad785b7d1e3910c2ce47baa807df2be6b8f5154a823c225ba575c8)
    (#x83a8a784a984f22564334b099cc5db6297ea2831f6c98b92222b48b221fa872 #x618235c3469a8443bf0c02a5113d258ffa6397c348b5e96369a4ad47781e4f7f #x20a22918f833d921243c3d40791e5532bb96632e4c1e6fd65b4a6edb9d12d1a9 #x15f31ec89c9f83b4681857307fc8b97a4ece66c5049e73000dcb3117c570648e)
    (#x1458e387ac0fea846f84590101b081bf90185387542d5cc0c097f440d51ec533 #x3bd2626d79f9b1d5689ceadd8e91ff146972b91bf85006efefd2b01e497ac3bc #x1d6a3cb53bc3be5050418257b01d9d5d0b1022538d8d5426cf2ad130a84bda92 #x408f60ba780bf6331f410c132919f21a91dffb689c2dfd1c8a1e74a202f41f49)
    (#x6ba975d7b853cc0321a7a236d8cf0dfdb056588ddeaf82f9b82778f6d0a78ede #x18a3b273d1e8715a336b3f412a928b81bff10be01094711965c6375a3fbc0730 #x36fc46cd7f322b010550bc1036f1c7abec5aec8e1a5cf6ef77d7913b662a9704 #x3418d0dfcab4f80176020091b5a8a8f98e8803122929e5cc9ff632fdb40d8b1a)
    (#xd910826913820c68af0552f51222e023bef305453ffc848d52796855b43b5f1 #x636f59eeb8428abed226c38296e39524fff76afb1525959108c0efaf049a47b8 #x6d5f15e9c5bddcfd6f6f86b9a078f227bcf841a7d908786f5abcee128827233 #x6e21dbe0b1ca7c453fc8629f3f0ad4e9d01aed47f6cc9d5ed8166c6b00367d22)
    (#x9d9434f8dc085dfc74a3d15f73d89a1bde46f2c79659015e87a86d4f61e451c #x4ffbf816fd76132cc64d02bf952c538b5fb3adc51a31421e718e8f8aaebb8b2f #x7061bca7b176fc8893957c8ad8bda36d884c4104895c815cd8e78f49f90e4630 #x21ddfccc092128048874a0c53156c08f9504a21d94dfc3ebda4d13073363c89a)
    (#x25e8ae5cc161160ea672beebb4ab2746950cd878072e1264df907ae9eb8fbace #x121c45420abf8e5cdab3ec9632c1becb3c42fe32aea83fa5265855d4ce4b51ad #x14a00fbc9c1a0319a19fe7dab909e800433556a4d394d50810dd9e42f79cd019 #x7261bcd2cab92a568d0f0f6d063161d3988579c3d37618775243eff31aa2e748)
    (#x108a5bb864bef87c1ebcc6dfbe909eb7a792e180fcaa4fb60ce02a48e3f0b03e #x34d973e68a2b1179db3fe9937c5454b3adf89e495219ea8df5faaeaa67d30415 #x1cc5702ef8cb0d489d447d08f7b66d22deed85b2e9e891317717fb1a8e8ac826 #x1276800cddffa60515a5e57cc2cddcc3419cf1eed8791474c2188ee27eef0a28)
    (#x26a5bb35746471b63ef933bca24a49d366a58ca54150f0e524592795e1f4f695 #x25c0a52e174ddcc11b5a3713702824a0608baabe32b55d293b201a0a27c2964d #x2b8cda9580ce03aabcd2508cce9bdbdf5b48fa7563c89dbd2db16cba4575e3b0 #x48f8b225f65fa4f33d31e10698af06c59e387647d421d2005437d3a2a250ca2d)
    (#x112f07705f98748cc06f8ffd03e31f467d969fd05884db149a21e48537b6b0ef #x1a4923cd0a20ff2e2c0c3b89fc6069408e6b594b73061b26d781035a1c3d0872 #x83130d75669dc95c0ca482e1a10caaf51863cf55af33ac10520e63cc8fc27ad #x492227265ac1e9f4b7679ee9975ce8d0b1adb3f9bd0b009c0a740b1de8de057d)
    (#x56c011941817140f9d524e7bab7c6ba0141bb007ec3699f8b998ce206816fc78 #x34033f30606670727f5aedfff6be8d2013f3694b614965ed91cac1d8081187ad #x1ee164f8a35c64a2c356cc7db807cd83778914344b3549941090bda2c4ba39b #x42bb3cf5995d0f35e0214d8c0de0f569ab41e9ef75e71d0674012892d68ac68c)
    (#x283963400a7600040fe3e0fef299baad9b753a3a95132383e941453fe608cf69 #x5645c81a6c057a50c7f415d774883607953b3865578cd9ba28101724b1751d05 #x380a6b4359b921620795b3da872d508b935b811cb6d02c805667ffcb28af4d24 #x2bad04565d6cc61669143b6ece9a1a77093ddd963c55e3a3cc11c53844041ea6)
    (#xcf4a15499ae28bad732654915dc1d5b67a511845737909b1cccf50680f5ba31 #x17c34ab6a23a149f249d490c9e88afc957ede5efbc18eb787890f63327e628e1 #x38383a5e3b0fbf5350d048c4f16997705e8b9aa0e64da87252a3b21979e11ac5 #x44f69eb8e9f7e53ea84bfe91293ab3a65f1fe6ab8bf7494e026afac6ac2bf484)
    (#x69a6655b628cb6e9c6e3b3b94f1ea8a2641c17a561e4a37eccbde34972624e3c #xe072d0b971d68d536b76352ed16c6dab0440eca585cab64a4e0ba1e08615410 #x4733e876b3b3f74ba58cf56b64faf89a95f1e7052599606c84b0b50e476ba880 #x6d802356b1b924cee10d0bf2f0b480b54306b994218dcb9225e5400eb81b65ef)
    (#x60865619d73e9c6425faae1db8f9f3a491401b32b578fed491191d4dfaf2f3ac #xb04d062c0757ef7ba7f8e54af428969c013d257e1f3d41d0dbbd0bad7711723 #x22ac7eecc33347bc1c2abd26c1a9b35853bce937c75c542932aa1e87141f3ce0 #x3ae3a7684111702cbeb86ceb67667cb3abc3afa2c93f0d8d8f2805fdf42c7594)
    (#x3e6b658310803f709adf8954c9b3b02864b37a490a84597bf74d380c06d0287e #x273dc3b92d063f1d67c3e4d61e7bf0dc505f84aafa1243e9ebd12120317abf1 #x1475088d8c356e610e5f4763d203c62e5bb941bf73a1716c43af9f1d50509f96 #x1386d879763a44d259f99582124ef6d7302d3da8fd135fc4d8304117bc067fe3)
    (#x1daa41aa1644ebb8216172bcd30fb28e4e13ae754bd5bb08245fd3e39434e687 #x124218390fad40862326265e6de2a7651ec5c5a6d422e568952b8fbee667caff #x1504f873a252e20b48d74663014532b2eff87628ab2955c97d0c48892c90b9cf #x1110bb10c565de009979db75bbe47c466d4cf37b822113f2ce77453e0a87f621)
    (#x316437b96f9cc8c18cd320e919e0d024fa4a4c487f7d78a2ad03e35df9b24d49 #x70b15b8c41cb5fe087542b0abb05767d41fe184e67b489f4f0483265c45ae8d6 #x24699333c2f08060595bea2f67d5c69e016d4dcd6399e9d1dbb6e69de7bc82d8 #x5f6868c658998b59347d630ff37f438b8ac1cee1156c87a34972c6632d770f22)
    (#x328922bcbaa152406015cae1a2a1ca9b15519c6cd36588d5f51856182fd5a232 #x5f1e4cff7c62b3a5bc89ba6fc70e7d20e61502f0b57ee51442b4d40da806b1e0 #x59b1e88b79a00ed114839aebfd02b95dc11c61e02d871ff73c0ea986b95b409b #x4544ccfe8fb21d6a49db579aaa2d23939b6afd73bc8bfaccb9f8e13052638f9a)
    (#x47a0138447359ad695a898bb9f8f3d7770e74d3d1a382ae786af997d281c20d1 #x3e385dcf88eca0c3c48bb0efb87d79b7f8da55e4c511f178af6b2d1fac2e12a3 #x183b5d3b2d59067eab3a4422c7377c8e3f2e752eb9b22c97a8b7389e4fd01113 #x37562ce51991499f8ad4cfe18ccd472052bda02c9ee4dbdfe3463dae34445d75)
    (#x421b21857527aa12f79a0d1e1712b31ee3c12f4f82c2e87954cee5bad9899d00 #x34446ddd54ad408830dac6f68080e4b7d3999eb1cf32484e47cb58dcb4efca49 #x52f2e208640f3d96cce87b89d5a1f1e100340ee6b59740b613c736e1dbbb7f5 #x3d4fe52bc2ea3d112fbdabd937c13094cfb2d2914fbb21f6b417c5e08db84af7)
    (#x23957ffc11794cafe82485702c3e550f9c4254ea1c170acb5f297ada1d6160dd #x74918240e1ea0c22f1cb2ab87f019c60c93f97b3bac35766ad9675a9b698880 #x72a6cb8d1ab3e1c262e67b7383af5989833135965ef0939d3ef5b9a772cb2075 #x6ed13064351b11a6edb5fad36c09dd2c8ca35a81bed8429699813f0f52c676)
    (#x14b9a2ca056d4af215d7c4e767a773934bb878263244cdf2fe3d91a923037824 #x17b7b4546ab94959ae33fef218a61010cee1bef29fefc4bec925b29004c931e8 #x127647ad48ff4c00766303f7f11ab9e50d3e454a12fdf6e1f0a65d3ffcfed25d #x156e8df1f40957f9cddb9109a36990e6435f8f3192475a0421a24b50831169ee)
    (#x14a93d5ec7b96474fcbb753a9705225989e7222185d3ba3ea98425288f61f947 #x342428fa567003be2117f74f6e2b64f92d1852b3a19785c85a9dcdc6414739a3 #x69728785a386d98f1415897eef703c1fd12ea0ebbb08d1a4b86ca4a876970e05 #x4205dbfd44798d218413a9ad69d1bba8c2171fe686f8d5e5161ef70e470a2dc8)
    (#x110e7c6c09c770034a4a3159af1552b93b6a5dcca36d4fe17c4d1ad25ef98a60 #x2995c20c83bb958c655d9d7331ce04fdefbe3795bb41af31346836db93857aa5 #x2f638c37f8c9b2fd506acc1a6716cbc2e035e6e3c6b7a18bd656bfab5f44bb7a #x9163acefb6a19112feb57a0d5dfa40ec9be98bc7393fa651447c2a08b2e92b9)
    (#x0a14e2951283469d27d64ff53de85bfec39491c9cc17d7867d34f4ac2c7ed1 #x17cfd1f18d6a19c63f3eba4742556382118ab17655e3dc2a2db314448afb563d #x4329d7e65590b914a826cd44e44e9bf40bcb7795dbc9967cdf22f7c235e54944 #x27ae8b032c813b4068cf9ed7a33e4412c2c2a542c0b852c223488e1835e090f2)
    (#x71d2902bb26a43c8a22c8e1b0e2c90c0b9e6855ebac5d2759842fea1576d85a #x5f85259bbd72609259df9d1eab84634decdd42bca99a7915b1ce0d2d13bc72fe #x2694ef88035366770006ac71a69237683c9338cc3170d8ba53af6abcffc0e029 #x5b9711d919c871e2856571ed2de34f76bcebf1f6338b3149465f1b266fd36c9d)
    (#x16880111cfebe378ce8f8a9f7ed97a8579533bac735afc9e40c06f5495409075 #x4d836438f0db187658eed75bae31ce7d978a1932aa894665235fee6e49b77abe #x2c03d6f3484ca56c256ba91a1d71a4210d840f07749196ff752c491c79e427f8 #x350594d4c210d6e66ef4fcbda72c80cd55b3dee5e91250a63d4cad3b14b77478)
    (#x5d50f9c8ff9b251d03b712c6f8d60f6137d192e69be0a92380a64d310a084e02 #x1375498a707afe69452fc093e90d23a54563582fa18813f02bfe996d35569f02 #x23e23d46ea3bc5cfb9aff1642e998710880b414a2048193b5dc642001a9fe1aa #x9dbae187a8543426c478b15560ca1838400236b1bf598dbdb1878b35eefe09e)))

(defconst *ingonyama-bls-255-neptune-mds*
  '((#x56f23d7e5f361df6266b620607396203fece3b023ffec4ff3fffffff40000001 #x458e97984c2b4b2b51ef819e6c2de803323e959b66656a65cccccccc33333334 #x609b60c54d5893118005895c0806deaf1b1e08ad2aa94ca9d555555480000001 #x211f5460e751918257c7624b7077624aaa362edc49241a48db6db6db24924925)
    (#x458e97984c2b4b2b51ef819e6c2de803323e959b66656a65cccccccc33333334 #x609b60c54d5893118005895c0806deaf1b1e08ad2aa94ca9d555555480000001 #x211f5460e751918257c7624b7077624aaa362edc49241a48db6db6db24924925 #x656ff268c469cd9f2cd29d07086d9d04a945ef829ffe907f1fffffff20000001)
    (#x609b60c54d5893118005895c0806deaf1b1e08ad2aa94ca9d555555480000001 #x211f5460e751918257c7624b7077624aaa362edc49241a48db6db6db24924925 #x656ff268c469cd9f2cd29d07086d9d04a945ef829ffe907f1fffffff20000001 #x19c308bd25b13848eef068e557794c72f62a247271c6bf1c38e38e38aaaaaaab)
    (#x211f5460e751918257c7624b7077624aaa362edc49241a48db6db6db24924925 #x656ff268c469cd9f2cd29d07086d9d04a945ef829ffe907f1fffffff20000001 #x19c308bd25b13848eef068e557794c72f62a247271c6bf1c38e38e38aaaaaaab #x22c74bcc2615a595a8f7c0cf3616f401991f4acdb332b532e66666661999999a)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Made this a nullary function to make it easier to prevent expansion
; and easier to view in the debugger.
(define ingonyama-bls-255-neptune--parameters ()
  :short "Poseidon parameters for Ingonyama BLS-255-neptune rate 3 implementation."
  (make-param
   :prime primes::*bls12-381-scalar-field-prime*
   :rate 3
   :capacity 1
   :alpha 5
   :full-rounds-half 4
   :partial-rounds 56
   :constants *ingonyama-bls-255-neptune-ark*
   :mds *ingonyama-bls-255-neptune-mds*
   :rate-then-capacity-p nil
   :ascending-p t
   :partial-first-p t)
  :returns (param paramp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define ingonyama-bls-255-neptune-hash ((inputs (fe-listp inputs (param->prime (ingonyama-bls-255-neptune--parameters)))))
  :guard (<= (len inputs) (param->size (ingonyama-bls-255-neptune--parameters)))
  :returns (output (fep output (param->prime (ingonyama-bls-255-neptune--parameters)))
                   :name fep-of-ingonyama-bls-255-neptune-hash
                   :hyp (and (fe-listp inputs (param->prime (ingonyama-bls-255-neptune--parameters)))
                             (<= (len inputs) (param->size (ingonyama-bls-255-neptune--parameters))))
                   :hints (("Goal" :use (:instance len-of-hashp
                                                   (inputs preimage)
                                                   (param (ingonyama-bls-255-neptune--parameters))
                                                   (count 2))
                                   :in-theory (e/d (len-of-hashp) ((:e INGONYAMA-BLS-255-neptune--PARAMETERS)))
                                   :do-not-induct t)))
  :short "Hash no more than @($c + r$) inputs to a single field element output."
  (let ((preimage (append (repeat (- (param->size (ingonyama-bls-255-neptune--parameters)) (len inputs)) 0)
                          inputs)))
    
    ;; The state vector t is [c.., r..], and in this case (len c)=1 and (len r)=2.
    ;; The Ingonyama output is t[1].
    ;; Since hashp with a count argument of 1 returns t[0], that is insufficient.
    ;; We need to tell hashp to return 2 outputs and to take the second.
    (nth 1 (hashp preimage
                  (ingonyama-bls-255-neptune--parameters)
                  2)))
  :guard-hints  (("Goal" :in-theory (enable hashp)))
  )
