/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <ctype.h>
#include <stdlib.h>
#include <gnome.h>
#include <limits.h>

#include "kiss.h"

#define FK_ERRORS &fkiss_errors

static int parse(char *args, char *argv[], char types[], int max);
static KissActionList *find_event(gchar *event, gchar *params);
static KissAction *new_action(KissActionType type);
static void *int_helper(gchar *integer, long min, long max);
static void parse_error(char *function, char *args);
static void check_and_add(char *method, char *params,
                          KissAction *action, int required);

static KissActionList *script;

static GMemChunk *action_chunk= NULL;

void reset_french() {
  if (action_chunk != NULL) {
    g_mem_chunk_destroy(action_chunk);
  }
  action_chunk= g_mem_chunk_create(KissAction, 100, G_ALLOC_ONLY);

  script= NULL;
}

static int parse(char *args, char *argv[], char types[], int max) {

  int start, end, offset, count= 0, quote= 0;
  char type= '?';

  for (start= end= offset= 0; args[offset] != '\0'; ++offset) {

    if (args[offset] == ',' && !quote) {
      /* safety */
      if (count == max) return count + 1;

      strncpy(argv[count], args + start, end - start);
      argv[count][end - start]= '\0';
      types[count++]= type;

      start= end= offset + 1;
      type= '?';

    } else if (type == '?' && start == offset && args[offset] == '#') {
      type= 'O';
      start= end= offset + 1;
    } else if (type == '?' && start == offset && args[offset] == '\"') {
      type= 'S';
      quote= !quote;
      start= end= offset + 1;
    } else if (type == 'S' && args[offset] == '\"' && quote) {
      quote= !quote;
      end= offset;
    } else if (type == '?' && start == offset && isspace(args[offset])) {
      start= end= offset + 1;
    } else if (!isspace(args[offset]) && !quote) {
      end= offset + 1;
    }

  }

  if (count > 0 || start < offset) {
    if (count == max) return count + 1;

    strncpy(argv[count], args + start, end - start);
    argv[count][end - start]= '\0';
    types[count++]= type;
  }
  
  return count;
}

/* There aren't any unimplemented features in GnomeKiSS :) */

#if 0
static void unimplemented(char *function, char *args) {
  log_error(_("%s(%s) is not yet implemented"), function, args);
}
#endif

static void parse_error(char *function, char *args) {
  log_error(_("%s(%s) has an incorrect number or type of parameters"),
            function, args);
}

static KissActionList *find_event(gchar *event, gchar *params) {
  KissCell *cell;
  KissObject *object;
  KissCollisionEvents *cell_hit;
  KissObjectCollision *object_hit;
  int count, val;
  gchar first[256], second[256], third[256];
  gchar *argv[3] = { first, second, third };
  gchar types[3];

  /* @EventHandler is dealt with elsewhere */

  count= parse(params, argv, types, 3);

  if (!strcmp("alarm", event)) {
    if (count == 1 && types[0] == '?') {
      val= numeric(first);
      if (val >= 0 && val <= INT_MAX) {
        return event_find_or_insert(config.alarms, val);
      } else {
        parse_error(event, params);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("apart", event)) { /* FKiSS 2.1 */
    if (count == 2 && types[0] == 'S' && types[1] == 'S') {
      cell_hit = collision_find(first, second);
      if (cell_hit == NULL)
        return FK_ERRORS; /* Reason logged in collision_find()  */
      else
        return &(cell_hit->apart);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("begin", event)) {
    if (count == 0) {
      return &(config.begin);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("catch", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->catch;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->catch);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("col", event)) {
    if (count == 1 && types[0] == '?') {
      val= numeric(first);
      if (val >= 0 && val < COLS) {
        return &(config.cols[val]);
      } else {
        parse_error(event, params);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("collide", event)) { /* FKiSS 2.1 */
    if (count == 2 && types[0] == 'S' && types[1] == 'S') {
      cell_hit = collision_find(first, second);
      if (cell_hit == NULL)
        return FK_ERRORS; /* Reason logged in collision_find()  */
      else
        return &(cell_hit->collide);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("drop", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->drop;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->drop);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("end", event)) {
    if (count == 0) {
      return &(config.end);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("fixcatch", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->fixcatch;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->fixcatch);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("fixdrop", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->fixdrop;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->fixdrop);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("in", event)) { /* FKiSS 2 */
    if (count == 2 && types[0] == 'O' && types[1] == 'O') {
      object_hit = impact_find(first, second);
      if (object_hit == NULL)
        return FK_ERRORS; /* Reason logged in impact_find()  */
      else
        return &(object_hit->in);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("initialize", event)) {
    if (count == 0) {
      return &(config.init);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("label", event)) { /* FKiSS 3 */
    if (count == 1 && types[0] == '?') {
      val= numeric(first);
      if (val >= 0 && val <= INT_MAX) {
        return event_find_or_insert(config.labels, val);
      } else {
        parse_error(event, params);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("never", event)) {
    if (count == 0) {
      return &(config.never);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("out", event)) { /* FKiSS 2 */
    if (count == 2 && types[0] == 'O' && types[1] == 'O') {
      object_hit = impact_find(first, second);
      if (object_hit == NULL)
        return FK_ERRORS; /* Reason logged in impact_find()  */
      else
        return &(object_hit->out);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("overflow", event)) /* FKiSS 3 */ {
    if (count == 0) {
      return &(config.overflow);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("press", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->press;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->press);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("release", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->release;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->release);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }


  } else if (!strcmp("set", event)) {
    if (count == 1 && types[0] == '?') {
      val= numeric(first);
      if (val >= 0 && val < SETS) {
        return &(config.sets[val]);
      } else {
        parse_error(event, params);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("stillin", event)) { /* FKiSS 2 */
    if (count == 2 && types[0] == 'O' && types[1] == 'O') {
      object_hit = impact_find(first, second);
      if (object_hit == NULL)
        return FK_ERRORS; /* Reason logged in impact_find()  */
      else
        return &(object_hit->stillin);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("stillout", event)) { /* FKiSS 2 */
    if (count == 2 && types[0] == 'O' && types[1] == 'O') {
      object_hit = impact_find(first, second);
      if (object_hit == NULL)
        return FK_ERRORS; /* Reason logged in impact_find()  */
      else
        return &(object_hit->stillout);
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("unfix", event)) {
    if (count == 1 && types[0] == 'S') {
      if ((cell= cell_find(first)) != NULL) {
        return &cell->events->unfix;
      } else {
        log_error(_("cell \"%s\" not loaded"), first);
        return FK_ERRORS;
      }
    } else if (count == 1 && types[0] == 'O') {
      if ((object= object_find(first)) != NULL) {
        return &(object->unfix);
      } else {
        log_error(_("invalid object #%s"), first);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  } else if (!strcmp("version", event)) { /* FKiSS 2 */
    if (count == 1 && types[0] == '?') {
      if (numeric(first) <= 4) {
        return &(config.version);       /* We claim FKiSS 3 compliance */
      } else {
        log_warning(_("FKiSS version %s not available"), params);
        return FK_ERRORS;
      }
    } else {
      parse_error(event, params);
      return FK_ERRORS;
    }

  }

  log_error(_("%s(%s) is not a recognised action or event"), event, params);
  return NULL;
}

static KissAction *new_action(KissActionType type) {
  KissAction *action;

  action= g_chunk_new0(KissAction, action_chunk);
  action->type= type;
  action->line= get_line();

  return action;
}

static void *int_helper(gchar *integer, long min, long max) {
  KissVariable *variable;
  long value = 0; 

  if (isdigit(integer[0]) || integer[0]=='-') {
    if (max == 0L && min == 0L)
      return NULL; /* constants are not permitted on the LHS */

    value= numeric(integer);
    if (value < min || value > max) {
      log_warning(_("value \"%s\" out of range"), integer);
    }
  }

  variable= var_find(integer);
  if (variable->value == 0) {
    variable->value= value;
  }
  return (void *) variable;
}

static void check_and_add(char *method, char *params, KissAction *action,
                                                             int required) {

  if (script == NULL) {
    log_error(_("%s(%s) actions must be inside an event"), method, params);
  } else if (required > 0 && action->args[0] == NULL) {
    log_error(_("%s(%s) first parameter incorrect"), method, params);
  } else if (required > 1 && action->args[1] == NULL) {
    log_error(_("%s(%s) second parameter incorrect"), method, params);
  } else if (required > 2 && action->args[2] == NULL) {
    log_error(_("%s(%s) third parameter incorrect"), method, params);
  } else if (required > 3 && action->args[3] == NULL) {
    log_error(_("%s(%s) fourth parameter incorrect"), method, params);
  } else {
    *script= g_slist_append(*script, action);
  }
}

void parse_action(gchar *text) {
  gchar method[32], params[256], rest[512];
  gchar first[256], second[256], third[256];
  gchar *argv[3] = { first, second, third };
  gchar types[3];
  KissAction *action;
  KissActionList *event;
  int count;

  rest[0]= '\0'; /* slack */

  count = sscanf(text, " %31[A-Za-z] (%255[^)]) %512[^;]",
                         method,        params,  rest);

  if (count == 1) {
    count = sscanf(text, " %31[A-Za-z] () %512[^;]", method, rest);
    params[0]= '\0';
  }

  if (count < 1) {
    log_warning(_("trailing garbage or syntax error"));
    return;
  }

  if (prefs.fkiss_case) {
    for (count= 0; method[count] != '\0'; ++count) {
      method[count] = tolower(method[count]);
    }
  }

  count = parse(params, argv, types, 3);

  if (!strcmp("add", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_ADD);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("altmap", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_ALTMAP_CELL);
      action->args[0]= (void *) cell_find(first);
      check_and_add(method, params, action, 1);
    } else if (count == 1 && types[0] == 'O') {
      action= new_action(ACTION_ALTMAP_OBJECT);
      action->args[0]= (void *) object_find(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("changeset", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_CHANGESET);
      action->args[0]= int_helper(first, 0, SETS - 1);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("changecol", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_CHANGECOL);
      action->args[0]= int_helper(first, 0, COLS - 1);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("debug", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_DEBUG);
      action->args[0]= (void *) g_strdup(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }


  } else if (!strcmp("div", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_DIV);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("else", method)) {
    if (count == 0) {
      action= new_action(CONTROL_ELSE);
      check_and_add(method, params, action, 0);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("endif", method)) {
    if (count == 0) {
      action= new_action(CONTROL_ENDIF);
      check_and_add(method, params, action, 0);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("exitevent", method)) {
    if (count == 0) {
      action= new_action(CONTROL_EXIT_EVENT);
      check_and_add(method, params, action, 0);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ghost", method)) {
    if (count == 2 && types[0] == 'S' && types[1] == '?') {
      action= new_action(ACTION_GHOST_CELL);
      action->args[0]= (void *) cell_find(first);
      action->args[1]= int_helper(second, 0, 1);
      check_and_add(method, params, action, 2);
    } else if (count == 2 && types[0] == 'O' && types[1] == '?') {
      action= new_action(ACTION_GHOST_OBJECT);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, 1);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("gosub", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(CONTROL_GOSUB);
      action->args[0]= int_helper(first, 0, INT_MAX);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("gosubrandom", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(CONTROL_GOSUB_RANDOM);
      action->args[0]= int_helper(first, 0, 100);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("goto", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(CONTROL_GOTO);
      action->args[0]= int_helper(first, 0, INT_MAX);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("gotorandom", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(CONTROL_GOTO_RANDOM);
      action->args[0]= int_helper(first, 0, 100);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }


  } else if (!strcmp("ifequal", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(CONTROL_EQUAL);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("iffixed", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {

      action= new_action(ACTION_IF_FIXED);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("iflessthan", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(CONTROL_LESS_THAN);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifgreaterthan", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(CONTROL_GREATER_THAN);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifmapped", method)) {
    if (count == 3 && types[0] == 'S' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_IF_MAPPED);
      action->args[0]= (void *) cell_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifmoved", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_IF_MOVED);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifnotequal", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(CONTROL_NOT_EQUAL);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifnotfixed", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_IF_NOT_FIXED);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifnotmapped", method)) {
    if (count == 3 && types[0] == 'S' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_IF_NOT_MAPPED);
      action->args[0]= (void *) cell_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("ifnotmoved", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_IF_NOT_MOVED);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("let", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(ACTION_VAR_LET);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letcatch", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_VAR_CATCH);
      action->args[0]= int_helper(first, 0, 0);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letcollide", method)) {
    if (count == 3 && types[0] == '?' && types[1] == 'S' && types[2] == 'S') {
      action= new_action(ACTION_VAR_COLLIDE);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) cell_find(second);
      action->args[2]= (void *) cell_find(third);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letfix", method)) {
    if (count == 2 && types[0] == '?' && types[1] == 'O') {
      action= new_action(ACTION_VAR_OBJECT_FIX);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) object_find(second);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letinside", method)) {
    if (count == 3 && types[0] == '?' && types[1] == 'O' && types[2] == 'O') {
      action= new_action(ACTION_VAR_INSIDE);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) object_find(second);
      action->args[2]= (void *) object_find(third);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letmapped", method)) {
    if (count == 2 && types[0] == '?' && types[1] == 'S') {
      action= new_action(ACTION_VAR_CELL_MAPPED);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) cell_find(second);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letmousex", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_VAR_MOUSE_X);
      action->args[0]= int_helper(first, 0, 0);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letmousey", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_VAR_MOUSE_Y);
      action->args[0]= int_helper(first, 0, 0);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letobjectx", method)) {
    if (count == 2 && types[0] == '?' && types[1] == 'O') {
      action= new_action(ACTION_VAR_OBJECT_X);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) object_find(second);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letobjecty", method)) {
    if (count == 2 && types[0] == '?' && types[1] == 'O') {
      action= new_action(ACTION_VAR_OBJECT_Y);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) object_find(second);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letpal", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_VAR_PAL);
      action->args[0]= int_helper(first, 0, 0);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("letset", method)) {
    if (count == 1 && types[0] == '?') {
      action= new_action(ACTION_VAR_SET);
      action->args[0]= int_helper(first, 0, 0);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("lettransparent", method)) {
    if (count == 2 && types[0] == '?' && types[1] == 'S') {
      action= new_action(ACTION_VAR_CELL_TRANSPARENT);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= (void *) cell_find(second);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("map", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_MAP_CELL);
      action->args[0]= (void *) cell_find(first);
      check_and_add(method, params, action, 1);
    } else if (count == 1 && types[0] == 'O') {
      action= new_action(ACTION_MAP_OBJECT);
      action->args[0]= (void *) object_find(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("mod", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_MOD);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("move", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_MOVE_OBJECT);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("movebyx", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == 'O' && types[2] == '?') {
      action= new_action(ACTION_MOVE_BYX);
      action->args[0]= (void *) object_find(first);
      action->args[1]= (void *) object_find(second);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("movebyy", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == 'O' && types[2] == '?') {
      action= new_action(ACTION_MOVE_BYY);
      action->args[0]= (void *) object_find(first);
      action->args[1]= (void *) object_find(second);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("moverandx", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_MOVE_RANDX);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("moverandy", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_MOVE_RANDY);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("moveto", method)) {
    if (count == 3 && types[0] == 'O' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_MOVE_TO);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("movetorand", method)) {
    if (count == 1 && types[0] == 'O') {
      action= new_action(ACTION_MOVE_TORAND);
      action->args[0]= (void *) object_find(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("mul", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_MUL);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("music", method)) {
    if (count == 1 && types[0] == 'S') {
      if (first[0] == '\0') {
        action= new_action(ACTION_MUSIC_CANCEL);
        check_and_add(method, params, action, 0);
      } else {
        action= new_action(ACTION_MUSIC);
        if (case_fixup(first))
          action->args[0]= (void *) g_strdup(first);
        check_and_add(method, params, action, 1);
      }
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("nop", method)) {
    if (count == 0) {
      action= new_action(ACTION_NOP);
      check_and_add(method, params, action, 0);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("notify", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_NOTIFY);
      action->args[0]= (void *) g_strdup(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("quit", method)) {
    if (count == 0) {
      action= new_action(ACTION_QUIT);
      check_and_add(method, params, action, 0);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("random", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_RANDOM);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("randomtimer", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_RANDOMTIMER);
      action->args[0]= int_helper(first, 0, INT_MAX);
      action->args[1]= int_helper(second, 0, INT_MAX);
      action->args[2]= int_helper(third, 0, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("setfix", method)) {
    if (count == 2 && types[0] == 'O' && types[1] == '?') {
      action= new_action(ACTION_SETFIX);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, 0, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("shell", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_SHELL);
      action->args[0]= (void *) g_strdup(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("sound", method)) {
    if (count == 1 && types[0] == 'S') {
      if (first[0] == '\0') {
        action= new_action(ACTION_SOUND_CANCEL);
        check_and_add(method, params, action, 0);
      } else {
        action= new_action(ACTION_SOUND);
        if (case_fixup(first))
          action->args[0]= (void *) g_strdup(first);
        check_and_add(method, params, action, 1);
      }
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("sub", method)) {
    if (count == 3 && types[0] == '?' && types[1] == '?' && types[2] == '?') {
      action= new_action(ACTION_VAR_SUB);
      action->args[0]= int_helper(first, 0, 0);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      action->args[2]= int_helper(third, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 3);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("timer", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(ACTION_TIMER);
      action->args[0]= int_helper(first, 0, INT_MAX);
      action->args[1]= int_helper(second, 0, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("transparent", method)) {
    if (count == 2 && types[0] == 'S' && types[1] == '?') {
      action= new_action(ACTION_TRANSPARENT_CELL);
      action->args[0]= (void *) cell_find(first);
      action->args[1]= int_helper(second, -255, 255);
      check_and_add(method, params, action, 2);
    } else if (count == 2 && types[0] == 'O' && types[1] == '?') {
      action= new_action(ACTION_TRANSPARENT_OBJECT);
      action->args[0]= (void *) object_find(first);
      action->args[1]= int_helper(second, -255, 255);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("unmap", method)) {
    if (count == 1 && types[0] == 'S') {
      action= new_action(ACTION_UNMAP_CELL);
      action->args[0]= (void *) cell_find(first);
      check_and_add(method, params, action, 1);
    } else if (count == 1 && types[0] == 'O') {
      action= new_action(ACTION_UNMAP_OBJECT);
      action->args[0]= (void *) object_find(first);
      check_and_add(method, params, action, 1);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("viewport", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(ACTION_VIEWPORT);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else if (!strcmp("windowsize", method)) {
    if (count == 2 && types[0] == '?' && types[1] == '?') {
      action= new_action(ACTION_WINDOW_SIZE);
      action->args[0]= int_helper(first, INT_MIN, INT_MAX);
      action->args[1]= int_helper(second, INT_MIN, INT_MAX);
      check_and_add(method, params, action, 2);
    } else {
      parse_error(method, params);
    }

  } else {
    event= find_event(method, params);

    if (event != NULL) {
      script= event;
      if (event != FK_ERRORS && *event != NULL) {
        log_warning(_("%s(%s) event used more than once"), method, params);
      }
    }
  }

  if (*rest) {
    parse_action(rest);
  }

}
