/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2024 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "OffComponent.h"

#include <sstream>

#include <QFileInfo>
#include <QVector3D>

// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkOBJReader.h>
#include <vtkProperty.h>
#include <CamiTKReEnableWarnings>

#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>

using namespace camitk;

// -------------------- default constructor  --------------------
OffComponent::OffComponent(const QString& file) : MeshComponent(file) {

    // use the file name without extension as component name
    setName(QFileInfo(file).baseName());

    vtkPolyData* offMesh    = vtkPolyData::New();
    vtkPoints* points       = vtkPoints::New();
    vtkCellArray* polys     = vtkCellArray::New();

    std::ifstream inputFile(file.toStdString().c_str(), std::ios::in);
    std::string line;

    // pass the comments and the first line
    while (!std::getline(inputFile, line).eof() && line.at(0) == '#' && line.compare("OFF") != 0) {
    }

    //read the numbers of vertices and of cells
    long        nbVertex = 0;
    long        nbPolygon = 0;
    vtkIdType countVertex = 0;
    double    pt[3];

    std::getline(inputFile, line);
    std::istringstream stream(line, std::istringstream::in);
    stream >> nbVertex;
    stream >> nbPolygon;

    long nbCells     = nbPolygon;
    long nbPoints    = nbVertex;

    while (std::getline(inputFile, line) && countVertex < nbPoints) {
        std::istringstream stream(line, std::istringstream::in);

        if (line.compare("#") != 0) {
            for (int i = 0; i < 3; i++) {
                stream >> pt[i];
            }

            points->InsertPoint(countVertex, pt[0], pt[1], pt[2]); // insert point
            ++countVertex;
        }
    }

    offMesh->SetPoints(points);

    //read connectivity
    vtkIdType eltType = 0;
    vtkIdType pts[3];
    int elemCount = 0;

    do {
        std::istringstream stream(line, std::istringstream::in);

        if (line.compare("#") != 0) {
            stream >> eltType;          // the first character is the number of points of the element

            for (int i = 0; i < eltType; i++) {
                stream >> pts[i];    // points coordinates
            }

            polys->InsertNextCell(eltType, pts); // add the element to the vtkCellArray
            ++elemCount;
        }
    }
    while (std::getline(inputFile, line) && elemCount < nbCells);

    offMesh->SetPolys(polys);

    // instantiate the Geometry
    initRepresentation(offMesh);

    // add the represention in the 3D viewer
    setVisibility("3D Viewer", true);
}
