(* :Title: Package Utilities *)
(* :Context: ExtendGraphics`DataIO` *)

(* :Author: Tom Wickham-Jones *)

(* :Summary: 
	This package provides functions to read and write
	tab-delimited files of numbers.
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(* :History:
	V. 1.0 April 93, by Tom Wickham-Jones
*)

(* :Keywords:
	tab-delimited, input, output, import, export
*)

(* :Limitations:  *)

(* :Discussion:
	
*)


BeginPackage["ExtendGraphics`DataIO`"]

ImportNumberData::usage = 
"ImportNumberData[ \"file\"] will read the contents of file and attempt
to return them in a matrix.  Each line of the file should consist of 
tab-delimited numbers and will form one row of the resulting matrix."

ExportNumberData::usage = 
"ExportNumberData[ \"file\", data] will write the matrix of numbers
into file placing each row of data into a seperate line of the file
with a tab character between each number."

Begin["`Private`"]


ImportNumberData[ file_String] :=
    	ReadList[ file, Number, RecordLists -> True]

ExportNumberData[ file_String, 
                  data_List /; VectorQ[ data, NumberQ]] :=
	ExportNumberData[ file, Transpose[ {data}]]

ExportNumberData[ file_String, 
                  data_List /; MatrixQ[ data, NumberQ]] :=
    Module[{stm, i, j},
    	stm = OpenWrite[ file] ;
	If[ Head[ stm] =!= OutputStream, Return[ $Failed]] ;
	Do[
	  Do[ 
	    WriteString[ 
	    	stm, 
	    	ToString[ Part[ data, i, j], FormatType -> FortranForm
		], "\t"], 
	    {j, Length[ Part[ data, i]]-1}];
	  WriteString[ 
	  	stm, 
		ToString[ Part[ data, i, -1], FormatType -> FortranForm
		], "\n"],
	  {i, Length[ data]}] ;
	Close[ stm]
	]
	
End[]

EndPackage[]


(*
	Examples

data = Table[ i j, {i, 5}, {j, 1, 2,.5}]

ExportNumberData[ "file.dat", data]

ImportNumberData[ "file.dat"]

data = {1,2,3,4}

data = Table[ 10.^-(i+j), {i,5}, {j,5}]


*)
