/*

 Author: Tom Wickham-Jones WRI
	
 A MathLink application to construct contour plots
 of irregular data. 

 The application is described in the book
 Mathematica Graphics: Techniques and Applications.
 Tom Wickham-Jones, TELOS/Springer-Verlag 1994.


 To build on Unix systems enter
 
 	mcc -o contour contour.tm

 to the shell.
 
 This presupposes an ANSI compiler called cc.

 If you have an ANSI compiler that is not called cc you
 must edit the mcc script and make the necessary change.
 (Future versions of mcc will do this with a variable).
 
 If you do not have an ANSI compiler then you must compile
 it without prototypes:
 
	mcc -o contour -DNOPROTO contour.tm


  To build this on the Macintosh or Windows consult your MathLink
  documentation.   There are Macintosh and Windows binaries
  available.
  
*/

:Begin:
:Function:       contour
:Pattern:        ExtendGraphics`Contour`Private`ContourL[pts_List, tri_List, hull_List, cont_List, smooth_]
:Arguments:      { pts, tri, hull, cont, smooth}
:ArgumentTypes:  {Manual}
:ReturnType:     Manual 
:End:
#include <math.h>
#include "mathlink.h"
#include <assert.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>



#ifdef NOPROTO
#define PROTO 0
#endif

#ifndef PROTO
#define PROTO 1
#endif

#define TRUE	1
#define FALSE	0

struct pointstruct
  	{
  	double x, y, z;
  	};

typedef struct pointstruct spoint;
typedef struct pointstruct * point;

#define Coord( i)	pnts[i]
#define XCoord( i)	pnts[i].x
#define YCoord( i)	pnts[i].y
#define ZCoord( i)	pnts[i].z
#define XCoordPt( pt)	(pt).x
#define YCoordPt( pt)	(pt).y
#define ZCoordPt( pt)	(pt).z

struct csegstruct
	{
	double	x, y;
	struct csegstruct * prev;
	int	p0;
	int	p1;
	};
	
typedef struct csegstruct * cseg;

#define CSegPrev( l) 	l->prev
#define CSegX( l) 	l->x
#define CSegY( l) 	l->y
#define CSegP0( l) 	l->p0
#define CSegP1( l) 	l->p1

struct clevelstruct
	{
	int	cont;
	cseg	line;
	struct clevelstruct * prev;
	};

typedef struct clevelstruct * clevel;

#define CLevelCont( c)	c->cont
#define	CLevelLine( c)	c->line
#define	CLevelPrev( c)	c->prev

struct tristruct
	{
	int 	pts[3];
	int	neigh[3];
	int	look;
	};

typedef struct tristruct stri;

#define TriPt( ti, i) 		tris[ti].pts[i]
#define TriPtsX( ti, i) 	XCoord( TriPt( ti, i))
#define TriPtsY( ti, i) 	YCoord( TriPt( ti, i))
#define TriPtsZ( ti, i) 	ZCoord( TriPt( ti, i))
#define TriNeigh( ti, i)	tris[ti].neigh[i]
#define TriLook( ti) 		tris[ti].look
#define NonTri			-1

point 	pnts 		= NULL;
int	npts		= 0;
double *contours	= NULL;
int	ncontours	= 0;
int    *hull		= NULL;
int	nhull		= 0;
int	smooth		= 0;
stri *	tris		= NULL;
int	ntri		= 0;

static void 
#if PROTO
  cons_points( int len)
#else
  cons_points( len)
  int len;
#endif
{
  pnts = (point) malloc( sizeof( struct pointstruct)*len);
}

static void
#if PROTO
  des_points( void)
#else
  des_points( )
#endif
{
  if ( pnts != NULL)
  	free( pnts);
  pnts = NULL;
  npts = 0;
}

static void 
#if PROTO
  cons_contours( int len)
#else
  cons_contours( len)
  int len;
#endif
{
  contours = (double *) malloc( sizeof( double)*len);
}

static void 
#if PROTO
  des_contours( void)
#else
  des_contours()
#endif
{
  if ( contours != NULL)
	free( contours);
  contours = NULL;
  ncontours = 0;
}

static void 
#if PROTO
  cons_hull( int len)
#else
  cons_hull( len)
  int len;
#endif
{
  hull = (int *) malloc( sizeof( int)*len);
}

static void 
#if PROTO
  des_hull( void)
#else
  des_hull()
#endif
{
  if ( hull != NULL)
	free( hull);
  hull = NULL;
  nhull = 0;
}

static void 
#if PROTO
  cons_tris( int len)
#else
  cons_tris( len)
  int len;
#endif
{
  tris = (stri *) malloc( sizeof( stri)*len);
}

static void 
#if PROTO
  des_tris( void)
#else
  des_tris()
#endif
{
  if ( tris != NULL)
  	free( tris);
  tris = NULL;
  ntri = 0;
}

static void 
#if PROTO
  des_data( void)
#else
  des_data()
#endif
{
  des_tris();
  des_hull();
  des_contours();
  des_points();
}



static cseg 
#if PROTO
  cons_cseg(void)
#else
  cons_cseg()
#endif
{
  cseg s;
  
  s = (cseg) malloc( sizeof( struct csegstruct));
  CSegPrev( s) = NULL;
  return s;
}

static void 
#if PROTO
  des_cseg(cseg s)
#else
  des_cseg(s)
  cseg s;
#endif
{
  free( s);
}

static int 
#if PROTO
  len_cseg(cseg s)
#else
  len_cseg(s)
  cseg s;
#endif
{
  int i;
  cseg sn;
  
  sn = s;
  i = 0;
  while ( sn != NULL)
  	{
	i++;
	sn = CSegPrev( sn);
	}
  return i;
}

static int 
#if PROTO
  len_clevel(clevel c)
#else
  len_clevel(c)
  clevel c;
#endif
{
  int i;
  clevel cn;
  
  cn = c;
  i = 0;
  while ( cn != NULL)
  	{
	i++;
	cn = CLevelPrev( cn);
	}
  return i;
}



static clevel 
#if PROTO
  cons_clevel( void)
#else
  cons_clevel()
#endif
{
  clevel c;
  
  c = (clevel) malloc( sizeof( struct clevelstruct));
  CLevelLine( c) = NULL;
  CLevelPrev( c) = NULL;
  return c;
}

static void 
#if PROTO
  des_clevel( clevel c)
#else
  des_clevel( c)
  clevel c;
#endif
{
  free( c);
}

static clevel 
#if PROTO
  append_clevel( clevel c)
#else
  append_clevel( c)
  clevel c;
#endif
{
  clevel cn;
  
  cn = cons_clevel( );
  
  CLevelPrev( cn) = c;
  CLevelCont( cn) = CLevelCont( c);
  return cn;
}

static cseg 
#if PROTO
  append_cseg(cseg s, double x, double y, int a, int b)
#else
  append_cseg(s, x, y, a, b)
  cseg s;
  double x, y;
  int a, b;
#endif
{
  cseg sn;
  
  sn = cons_cseg();
  CSegPrev( sn) = s;
  CSegX( sn) = x;
  CSegY( sn) = y;
  CSegP0(sn) = a;
  CSegP1(sn) = b;
  return sn;
}

static void
#if PROTO
  fill_tris(void)
#else
  fill_tris()
#endif
{
  int ti, to, i, j, p0, p1, tk;
  
  for ( ti = 0; ti < ntri; ti++)
  	{
 	for ( j = 0; j < 3; j++)
		{
		p0 = TriPt( ti, j);
		p1 = TriPt( ti, (j+1)%3);
 		for ( tk = 0; tk < ntri; tk++)
			{
			if ( tk == ti)
				continue;
			if ( (p0 == TriPt( tk, 0) ||
			      p0 == TriPt( tk, 1) ||
			      p0 == TriPt( tk, 2)) &&
			     (p1 == TriPt( tk, 0) ||
			      p1 == TriPt( tk, 1) ||
			      p1 == TriPt( tk, 2)))
			      	{
			     	TriNeigh( ti, j) = tk;
 				break;
				}
			}
		if ( tk == ntri)
			TriNeigh( ti, j) = NonTri;
 
 		}
	assert( TriNeigh( ti, 0) != TriNeigh( ti, 1) ||
		TriNeigh( ti, 0) == NonTri);
	assert( TriNeigh( ti, 1) != TriNeigh( ti, 2) ||
		TriNeigh( ti, 1) == NonTri);
	assert( TriNeigh( ti, 0) != TriNeigh( ti, 2) ||
		TriNeigh( ti, 2) == NonTri);
	}
	
	for (ti = 0; ti < ntri; ti++)
		{
		for ( j = 0; j < 3; j++)
			{
			to = TriNeigh( ti, j);
			if ( to != NonTri)
				{
				for ( i = 0; i < 3; i++)
					{
					if ( TriNeigh( to, i) == ti)
						break;
					}
				assert( i != 3);
				}
			}
		}
}

static int
#if PROTO
  cross_tri( int t, int i, double val)
#else
  cross_tri( t, i, val)
  int t;
  int i;
  double val;
#endif
{

  if ( (val <= TriPtsZ( t, i) && val <= TriPtsZ( t, (i+1)%3)) ||
       (val > TriPtsZ( t, i) && val > TriPtsZ( t, (i+1)%3)))
  	return FALSE;
  return TRUE;
}

static cseg
#if PROTO
  seg_build( cseg s, int t, int coord, double val)
#else
  seg_build( s, t, coord, val)
  cseg s;
  int t;
  int coord;
  double val;
#endif
{
  double x0, x1, y0, y1, z0, z1, fact;
  int coordn;
  
  coordn = (coord+1)%3;
  z0 = TriPtsZ( t, coord);
  z1 = TriPtsZ( t, coordn);
  
  assert( (z0 < val && val <= z1) ||
          (z1 < val && val <= z0));
  
  fact = (z1 - z0);
  assert( fact != 0.0);
  fact = (val - z0)/fact;
  
  x0 = TriPtsX( t, coord);
  x1 = TriPtsX( t, coordn);
  y0 = TriPtsY( t, coord);
  y1 = TriPtsY( t, coordn);

  return append_cseg( s, 
  		x0 + fact*(x1-x0), 
  		y0 + fact*(y1-y0), 
		TriPt( t, coord),
		TriPt( t, coordn));
}


/*
 c is contour to join to
 t_ind is triangle to test
 there is a crossing between seg_pt and seg_pt+1
 we must find the other crossing.
*/
 
static clevel 
#if PROTO
  contour_follow( clevel c, int t_ind, int seg_pt)
#else
  contour_follow( c, t_ind, seg_pt)
  clevel c;
  int t_ind;
  int seg_pt;
#endif
{
  double val;
  int i, coord, n_ind;
  
  
  val = contours[ CLevelCont( c)] ;
  
  assert( cross_tri( t_ind, seg_pt, val));

  if ( TriLook( t_ind))
  	return append_clevel( c);	/* Closed loop */

  TriLook( t_ind) = TRUE;
    
  for ( i = 1; i < 3; i++)
  	{
	coord = (seg_pt+i)%3;
  	if ( cross_tri( t_ind, coord, val))
		{
  		n_ind = TriNeigh( t_ind, coord);
		CLevelLine( c) = seg_build( CLevelLine(c), t_ind, coord, val);
		break;
		}
	}
  assert( i != 3);

  if ( n_ind == NonTri)		/* hit the outside */
 	return append_clevel( c);
  
  for ( i = 0; i < 3; i++)
  	if ( TriNeigh( n_ind, i) == t_ind)
		break;
  assert( i != 3);
  return contour_follow( c, n_ind, i);
}


static clevel 
#if PROTO
  contour_start( clevel c, int t_ind)
#else
  contour_start( c, t_ind)
  clevel c;
  int t_ind;
#endif
{

  double val;
  int i;
  
  if ( TriLook( t_ind))
  	return c;	/* been here before */
  
  val = contours[ CLevelCont( c)] ;
  
  if ( ! cross_tri( t_ind, 0, val) && !cross_tri( t_ind, 1, val) &&
  				      !cross_tri( t_ind, 2, val))
  	return c;
/*
 crosses the triangle
*/


  for ( i = 0; i < 3; i++)
	if ( cross_tri( t_ind, i, val))
		{
		CLevelLine( c) = seg_build( CLevelLine(c), t_ind, i, val);
		break;
		}
  
  return contour_follow( c, t_ind, i);
}

static int 
#if PROTO
  on_hull( int pt)
#else
  on_hull( pt)
  int pt;
#endif
{
  int i;
  
  for ( i = 0; i < nhull; i++)
	if ( pt == hull[ i])
		return TRUE;
  return FALSE;	
}

static clevel 
#if PROTO
  contour_start_hull( clevel c, int t_ind)
#else
  contour_start_hull( c, t_ind)
  clevel c;
  int t_ind;
#endif
{

  double val;
  int i;

  if ( TriLook( t_ind))
  	return c;

  val = contours[ CLevelCont( c)] ;

  for ( i = 0; i < 3; i++)
	if ( cross_tri( t_ind, i, val) &&
	     on_hull( TriPt( t_ind, i)) &&
	     on_hull( TriPt( t_ind, (i+1)%3)))
		{
		CLevelLine( c) = seg_build( CLevelLine(c), t_ind, i, val);
  		return contour_follow( c, t_ind, i);
		}
  return c;
}

static clevel 
#if PROTO
  contour_build( clevel c)
#else
  contour_build( c)
  clevel c;
#endif
{
  int i;
    
  for ( i = 0; i < ntri; i++)
  	TriLook( i) = FALSE;

  for ( i = 0; i < ntri; i++)
	c = contour_start_hull( c, i); 

  for ( i = 0; i < ntri; i++)
	c = contour_start( c, i);

  return c;
}

static int
#if PROTO
  find_inter(
  	double x0, double y0, double x1, double y1,
  	double x2, double y2, double x3, double y3,
  	double * xn, double * yn)
#else
  find_inter(
  	x0, y0, x1, y1,
  	x2, y2, x3, y3,
  	xn, yn)
  double x0, y0, x1, y1, x2, y2, x3, y3, * xn, * yn;
#endif
{
  double nx, ny, dx, dy;
  double t, den;
  
  nx = y3-y2;
  ny = x2-x3;
  
  dx = x1 - x0;
  dy = y1 - y0;

  den = dx*nx + dy*ny;
  if ( den == 0.0)
  	{
	return FALSE;
	}
  
  t = ((x2 - x0)*nx + (y2 - y0)*ny)/den;
  
  if ( t < 0.0)
  	t = 0.0;
  if ( t > 1.0)
  	t = 1.0;
  
  *xn = dx*t + x0;
  *yn = dy*t + y0;
  
  return TRUE;
}

static cseg
#if PROTO
  smooth_seg( cseg s)
#else
  smooth_seg( s)
  cseg s;
#endif
{
  cseg st, sn, sp;
  double 
  	x0, y0, x1, y1, 
	x2, y2, x3, y3,
	xn, yn;
  
  st = s;
  
  sn = s;
  s = CSegPrev( s);
  sp = CSegPrev( s); 
  while( sp != NULL)
  	{
	x0 = XCoord( CSegP0( s));
	y0 = YCoord( CSegP0( s));
	x1 = XCoord( CSegP1( s));
	y1 = YCoord( CSegP1( s));
	x2 = CSegX( sp);
	y2 = CSegY( sp);
	x3 = CSegX( sn);
	y3 = CSegY( sn);
	
	if ( find_inter( x0, y0, x1, y1, x2, y2, x3, y3, &xn, &yn))
		{
		CSegX(s) = xn;
		CSegY(s) = yn;
		}
		
	sn = s;
	s = CSegPrev( s);
  	sp = CSegPrev( s); 
	}
  return st;
}

static clevel
#if PROTO
  smooth_levels( clevel c)
#else
  smooth_levels( c)
  clevel c;
#endif
{
  clevel c1;
  
  c1 = c;
  
  while ( c1 != NULL)
  	{
	CLevelLine( c1) = smooth_seg( CLevelLine( c1));
	c1 = CLevelPrev( c1);
	}
  return c;
}

void
#if PROTO
  contour( void)
#else
  contour()
#endif
{
  long lenl, tmp;
  int i, j, num;
  clevel c, cd;
  cseg s, sd;

  if (!MLCheckFunction( stdlink, "List", &lenl))
  	goto error;
  npts = lenl;
  cons_points( npts);
  for ( i = 0; i < npts; i++) 
  	{
  	if ( ! MLCheckFunction( stdlink, "List", &tmp))
		goto error;
	if ( tmp != 3)
		goto error;
	if ( ! MLGetDouble( stdlink, &XCoord( i)) ||
	     ! MLGetDouble( stdlink, &YCoord( i)) ||
	     ! MLGetDouble( stdlink, &ZCoord( i)))
	    	goto error;
  	}

  if (! MLCheckFunction( stdlink, "List", &lenl))
  	goto error;
  ntri = lenl;
  cons_tris( ntri);
  for ( i = 0; i < ntri; i++)
  	{
  	if ( ! MLCheckFunction( stdlink, "List", &tmp))
		goto error;
	if ( tmp != 3)
		goto error;
	if ( ! MLGetInteger( stdlink, &TriPt( i, 0)) ||
	     ! MLGetInteger( stdlink, &TriPt( i, 1)) ||
	     ! MLGetInteger( stdlink, &TriPt( i, 2)))
	    	goto error;
	for ( j = 0; j < 3; j++)
		TriPt( i, j) -= 1;
  	}
  fill_tris();
  	
  if (! MLCheckFunction( stdlink, "List", &lenl))
  	goto error;
  nhull = lenl;
  cons_hull( nhull);
  for ( i = 0; i < nhull; i++)
  	{
  	if ( ! MLGetInteger( stdlink, &hull[i]))
		goto error;
	hull[i]--;
  	}

  if (! MLCheckFunction( stdlink, "List", &lenl))
  	goto error;
  ncontours = lenl;
  cons_contours( ncontours);
  for ( i = 0; i < ncontours; i++)
  	if ( ! MLGetDouble( stdlink,&contours[i]))
		goto error;
  if ( ! MLGetInteger( stdlink, &smooth))
  	goto error;

  c = cons_clevel();
  for ( i = 0; i < ncontours; i++)
    	{
      	CLevelCont(c) = i;
  	c = contour_build( c);
	}
  cd = c;
  c = CLevelPrev( c);
  des_clevel( cd);
  if (smooth)
  	c = smooth_levels( c);
  num = len_clevel( c);
  MLPutFunction( stdlink, "List", num);
  while ( c != NULL)
  	{
	MLPutFunction( stdlink, "List", 2);
	MLPutInteger( stdlink, CLevelCont( c)+1);
	s = CLevelLine( c);
	num = len_cseg( s);
	MLPutFunction( stdlink, "List", num);
	while ( s != NULL)
		{
		MLPutFunction( stdlink, "List", 2);
		MLPutDouble( stdlink, CSegX( s));
		MLPutDouble( stdlink, CSegY( s));
		sd = s;
		s = CSegPrev( s);
		des_cseg( sd);
		}
	cd = c;
	c = CLevelPrev( c);
	des_clevel( cd);
	}

  des_data();
  return;

error:
  des_data();
  MLPutSymbol( stdlink, "$Failed");
  return;
}


#if !WINDOWS_MATHLINK

int main(argc, argv)
        int argc; char* argv[];
{
        return MLMain(argc, argv);
}

#else

int PASCAL WinMain( HANDLE hinstCurrent, HANDLE hinstPrevious, LPSTR lpszCmdLine, int nCmdShow)
{
	char  buff[512];
	char FAR * argv[32];
	int argc;

	if( !MLInitializeIcon( hinstCurrent, nCmdShow)) return 1;
	argc = MLStringToArgv( lpszCmdLine, buff, argv, 32);
	return MLMain( argc, argv);
}
#endif
