/* $Id: fors_dark_impl.c,v 1.18 2013-09-10 19:12:03 cgarcia Exp $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-10 19:12:03 $
 * $Revision: 1.18 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_dark_impl.h>

#include <fors_stack.h>
#include <fors_dfs.h>
#include <fors_utils.h>

#include <cpl.h>

/**
 * @addtogroup fors_dark
 */

/**@{*/

const char *const fors_dark_name = "fors_dark";
const char *const fors_dark_description_short = "Compute master dark frame";
const char *const fors_dark_author = "Jonas M. Larsen";
const char *const fors_dark_email = PACKAGE_BUGREPORT;
const char *const fors_dark_description =
"This recipe is used to combine input raw DARK frames into a master dark\n"
"frame by subtracing the master bias and using the given stacking method.\n"
"The overscan regions, if present, are removed from the result.\n"
"\n"
"Input files:\n"
"\n"
"  DO category:               Type:       Explanation:         Required:\n"
"  DARK                       Raw         Dark frame              Y\n"
"  MASTER_BIAS                FITS image  Master bias             Y\n"
"\n"
"Output files:\n"
"\n"
"  DO category:               Data type:  Explanation:\n"
"  MASTER_DARK                FITS image  Master dark frame\n"
"\n";

/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */
void fors_dark_define_parameters(cpl_parameterlist *parameters)
{
    const char *context = cpl_sprintf("fors.%s", fors_dark_name);
    
    fors_stack_define_parameters(parameters, context, "median");

    cpl_free((void *)context);

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(dark_frames); \
    cpl_frameset_delete(master_bias_frame); \
    fors_image_delete_const(&master_bias); \
    fors_stack_method_delete(&sm); \
    fors_image_delete(&master_dark); \
    fors_image_list_delete_const(&darks, fors_image_delete); \
    fors_setting_delete(&setting); \
    cpl_free((void *)context); \
} while (0)
/**
 * @brief    Do the processing
 *
 * @param    frames         input frames
 * @param    parameters     parameters
 *
 * @return   0 if everything is ok
 */

void fors_dark(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    /* Raw */
    cpl_frameset *dark_frames      = NULL;
    const fors_image_list *darks   = NULL;

    /* Calibration */
    cpl_frameset *master_bias_frame = NULL;
    const fors_image *master_bias   = NULL; 

    /* Product */
    fors_image *master_dark = NULL;

    /* Parameters */
    stack_method *sm    = NULL;

    /* Other */
    fors_setting *setting = NULL;
    const char *context = cpl_sprintf("fors.%s", fors_dark_name);

    /* Get parameters */
    sm = fors_stack_method_new(parameters, context);
    assure( !cpl_error_get_code(), return, "Could not get stacking method");
    
    /* Find raw */
    dark_frames = fors_frameset_extract(frames, DARK);
    assure( cpl_frameset_get_size(dark_frames) > 0, return, 
            "No %s provided", DARK);

    /* Find calibration */
    master_bias_frame = fors_frameset_extract(frames, MASTER_BIAS);
    assure( cpl_frameset_get_size(master_bias_frame) == 1, return, 
            "One %s required. %"CPL_SIZE_FORMAT" found", 
            MASTER_BIAS, cpl_frameset_get_size(master_bias_frame));

    /* Get instrument setting */
    setting = fors_setting_new(cpl_frameset_get_position(dark_frames, 0));
    assure( !cpl_error_get_code(), return, "Could not get instrument setting" );

    master_bias = fors_image_load(cpl_frameset_get_position(master_bias_frame, 0), 
                                  NULL, setting,
                                  NULL);
    assure( !cpl_error_get_code(), return, 
            "Could not load master bias");
    
    /* Load dark, subtract bias */
    darks = fors_image_load_list_const(dark_frames, master_bias, setting, NULL);
    assure( !cpl_error_get_code(), return, "Could not load dark images");

    /* Stack */
    master_dark = fors_stack_const(darks, sm);
    assure( !cpl_error_get_code(), return, "Dark stacking failed");
    
    /* Save product */
    fors_dfs_save_image_err(frames, master_dark, MASTER_DARK,
                        NULL, parameters, fors_dark_name, 
                        cpl_frameset_get_position(dark_frames, 0));
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            MASTER_DARK);
    
    cleanup;
    return;
}

/**@}*/
